#line 1 "./src/sync/lock.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : Locking Synchronization & Transaktion                            */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* PROBLEMS: A transaction might be awoken and woke up                       */
/*           another transacation and then failed.                           */
/*           The other transaction will read and maybe                       */
/*           misinterpret the first ones error code.                         */
/*                                                                           */
/* VERSION: 1.1                                                              */
/* ========================================================================= */

#include "obst_progstd.h"
#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "trc_tsy.h"
#include "sync_trc.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: LockingSync                                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: LockingSync::grant()                                     */
/*                                                                  */
/* Purpose: This method is called whenever a transaction gains      */
/*          access to data not accessed in that mode before.        */
/*          It marks the data accessed with the accessing           */
/*          transaction and notes the accessed data in the          */
/*          appropriate access sets of the transaction.             */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingSync::grant (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta, Lock_Mode lm)
const{
   T_PROC("LockingSync::grant"); 
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Lock_Mode"); TI(lm);
             TXT("Container"); TI(get_soscontainer(_OBSThis)));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   BlockedTA entry;
   BlockedTA bta;


   set_lock(_OBSThis,lm);

   entry = get_accesses(_OBSThis)[ta.get_ts()];
   if VALID(entry) 
   {
      // Upgrade _TA's lock.
      // It's assumed that grant is called only if _TA doesn't
      // hold a lock on this data item or holds a weaker lock.
      // Presently this can only be a shared lock.
      entry.set_am(WRITING);

      // Update transaction's data.
      ta.get_writeset().insert(SyncObj::make(LockingSync::make(_OBSThis,this)));
      ta.get_readset().remove(SyncObj::make(LockingSync::make(_OBSThis,this))); 
      ta.get_openset().insert(SyncObj::make(LockingSync::make(_OBSThis,this)));
   }
   else
   {
      // Create new entry using BlockedTA type.
      
      if (lm == READLOCK)
      {
	 bta = BlockedTA::create(SYNC_CONTAINER, ta, READING);
	 ta.get_readset().insert(SyncObj::make(LockingSync::make(_OBSThis,this)));
      }
      else
      {
	 bta = BlockedTA::create(SYNC_CONTAINER, ta, WRITING);
	 ta.get_writeset().insert(SyncObj::make(LockingSync::make(_OBSThis,this)));
      }

      get_accesses(_OBSThis).insert(ta.get_ts(), bta);
      ta.get_openset().insert(SyncObj::make(LockingSync::make(_OBSThis,this)));
   }

   set_error(_OBSThis,SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingSync::tpl_readlock()                              */
/*                                                                  */
/* Purpose: Check for lock conflict and                             */
/*          (1) either grant lock or                                */
/*          (2) block transaction or                                */
/*          (3) return error to non-waiting transactions.           */
/*                                                                  */
/*              The check is done within a loop to recheck after    */
/*              the transaction has been blocked. This is necessary */
/*              because of the wake-up mechanism of timestamp _TA.  */
/*                                                                  */
/*          At present, only two lock modes (shared, exclusive)     */
/*          are supported. Thus, if data is already locked in a     */
/*          lockmode different to READLOCK, a lock conflict may     */
/*          have occured.                                           */
/*                                                                  */
/*          (4)                                                     */
/*          After waiting for a lock, a transaction has to recheck  */
/*          that the lock may be granted. This is because a         */
/*          successfully woke up transaction tries to wake up the   */
/*          next in queue. Wakeup may be done before lock           */
/*          aquisition is finished because the woke up transaction  */
/*          has to wait for the SyncContainer.                      */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK, SYNC_DEADLOCK                                    */
/* ================================================================ */

sos_Bool _LockingSync::tpl_readlock (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta, sos_Sync_mode sm)
const{

   T_PROC("LockingSync::tpl_readlock");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Sync_mode"); TI(sm));
   TT(tsy_L, TXT("Container"); TI(get_soscontainer(_OBSThis)));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   sos_Bool waited = FALSE;
   Index    pos    = 0;

   while (TRUE)
   {
      if (get_lock(_OBSThis) == NOLOCK)
      {
	 // (4)
	 if ( (waited) && (!get_queue(_OBSThis).is_empty()) )
	    dequeue(_OBSThis);

	 // (1)
	 grant(_OBSThis,ta, READLOCK);
         
         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      }

      if (get_lock(_OBSThis) != READLOCK)
      {
	 // does transaction hold write lock itself?
	 if  (get_accesses(_OBSThis).is_key(ta.get_ts()) )
	 {
	    // Yes, accesses holds only one writing transaction at
	    // a time and this is the accessing _TA.
	    // (1)
	    set_error(_OBSThis,SYNC_OK);
            TT(tsy_H, TXT("TRUE"); T_LEAVE);
	    return(TRUE);
	 }
	 else if (sm == TESTING)
	 {
	    // Inform transaction of failed access.
	    // (3)
	    set_error(_OBSThis,SYNC_OK);
            TT(tsy_H, TXT("TESTING->FALSE"); T_LEAVE);
	    return(FALSE);
	 }
	 else
	 {
	    // Wait until lock is available.
            // When the enqueue call returns, the old position in the 
            // queue is stored in the variable pos, so that it is possible to
            // insert the ta again, if the access is still not possible. 
	    // (2)
	    waited = TRUE;
            ta.set_waits_for(SyncObj::make(LockingSync::make(_OBSThis,this)));
	    pos = enqueue(_OBSThis,ta, READING, pos);
            ta.set_waits_for(SyncObj::make(NO_OBJECT));
	    if (get_error(_OBSThis) == SYNC_DEADLOCK)
	      { TT(tsy_H, TXT("FALSE"); T_LEAVE);
                return(FALSE); // for example deadlock
              }
	 }
      }
      else
      {
	 // The data is locked in shared mode. 
	 // Maybe _TA holds read lock already.
	 if ( ! ( get_accesses(_OBSThis).is_key(ta.get_ts()) ) )
	    // No. (1)
	    grant(_OBSThis,ta, READLOCK);

	 // (4)
	 if ( (waited) && (!get_queue(_OBSThis).is_empty()) )
	    dequeue(_OBSThis);

	 // (1)
	 set_error(_OBSThis,SYNC_OK);
         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      } 
   } 
} 

/* ================================================================ */
/* Method: LockingSync::tpl_writelock()                             */
/*                                                                  */
/* Purpose: see LockingSync::tpl_readlock()                         */
/*                                                                  */
/* Error: SYNC_OK, SYNC_DEADLOCK                                    */
/* ================================================================ */

sos_Bool _LockingSync::tpl_writelock (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta, sos_Sync_mode sm)
const{

   T_PROC("LockingSync::tpl_writelock");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Sync_mode"); TI(sm));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   sos_Bool waited = FALSE;
   Index    pos    = 0;

   while (TRUE)
   {
      if ((get_lock(_OBSThis) == NOLOCK) && (get_accesses(_OBSThis).card() ==0))
      {
	 // (1)
	 grant(_OBSThis,ta, WRITELOCK);

         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      }

      // Someone holds a lock on data item; _TA itself?
      if  (get_accesses(_OBSThis).is_key(ta.get_ts()) )
      {
	 // Does _TA hold sole lock on data?
	 if (get_accesses(_OBSThis).card() == 1)
	 {
	    // (1)
	    // holds already write lock?
	    if (get_lock(_OBSThis) == WRITELOCK)
	    {
	       set_error(_OBSThis,SYNC_OK);
	       TT(tsy_H, TXT("TRUE"); T_LEAVE);
               return(TRUE);
	    }
	    else
	    {
	       grant(_OBSThis,ta, WRITELOCK);
 	       TT(tsy_H, TXT("TRUE"); T_LEAVE);
               return(TRUE);
            }
	 }
      }

      // Somebody else holds a lock (, too).
      if (sm == TESTING)
      {
	 // Inform transaction of failed access.
	 // (3)
	 set_error(_OBSThis,SYNC_OK);
         TT(tsy_H, TXT("TESTING->FALSE"); T_LEAVE);
	 return(FALSE);
      }
      else
      {
	 // Wait until lock is available.
         // When the enqueue call returns, the old position in the
         // queue is stored in the variable pos, so that it is possible to
         // insert the ta again, if the access is still not possible.
	 // (2)
	 waited = TRUE;
         ta.set_waits_for(SyncObj::make(LockingSync::make(_OBSThis,this)));
	 pos = enqueue(_OBSThis,ta, WRITING, pos);
         ta.set_waits_for(SyncObj::make(NO_OBJECT));
	 if (get_error(_OBSThis) == SYNC_DEADLOCK)
           { TT(tsy_H, TXT("FALSE"); T_LEAVE);
	     return(FALSE); // for example deadlock
           }
      }
   } 
}

/* ================================================================ */
/* Method: LockingSync::tpl_release_n_wakeup()                      */
/*                                                                  */
/* Purpose: The transaction is removed from the list of accessing   */
/*          transactions for this data item. If some other trans-   */
/*          actions are waiting for the data item, the first of     */
/*          them is activated.                                      */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingSync::tpl_release_n_wakeup (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta)
const{

   T_PROC("LockingSync::tpl_release_n_wakeup");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Container"); TI(get_soscontainer(_OBSThis)));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   sos_AccessMap access = get_accesses(_OBSThis);
   BlockedTA     bta;
   sos_Timestamp ts;
   sos_Bool      found;

   // remove ta from accesses
   access.remove(ta.get_ts());

   // test for existing locks
   if (get_lock(_OBSThis) == READLOCK)
     {
       found = FALSE;
       agg_iterate_association (access, ts, bta) 
         { if ((bta.get_am() == READING) &&
               (bta.get_ta().get_type() == LOCKING))
 // <af> break      
             found = TRUE; }
       agg_iterate_association_end (access, ts , bta);
       
       if (!found)
         set_lock(_OBSThis,NOLOCK);   // last Readlock released
        else
         if ((access.card() == 1) && (!get_queue(_OBSThis).is_empty())) 
           { // If a transaction tries to upgrade its lock and it waits
             // in the head of the queue and its lock is the only lock
             // on the syncobj, the transaction can be dequeued 
             ts = get_queue(_OBSThis).get_nth(1).get_ta().get_ts();
             if (access.is_key(ts))
               dequeue(_OBSThis); 
           }
     }
    else
         set_lock(_OBSThis,NOLOCK);   // Writelock released

       
   // Release blocked transactions.
   if (get_lock(_OBSThis) == NOLOCK)
      dequeue(_OBSThis);
   
   set_error(_OBSThis,SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
} 


/* ================================================================ */
/* Method: LockingSync::tpl_commit()                                */
/*                                                                  */
/* Purpose: Tells the data item that a locking transaction has      */
/*          committed, which means, its lock can be released.       */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingSync::tpl_commit (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta)
const{
   T_PROC("LockingSync::tpl_commit");
   TT(tsy_H, T_ENTER);
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   sos_Container ct;


   // *all* transactions are listed by their timestamps
   if ( get_accesses(_OBSThis).is_key(ta.get_ts()) )
   {
      ct = get_soscontainer(_OBSThis);

      if (get_lock(_OBSThis) == WRITELOCK)
	ct.write_close(); 
      else
	ct.read_close(); 

      tpl_release_n_wakeup(_OBSThis,ta);
      set_error(_OBSThis,SYNC_OK);
   }
   else
      // _TA doesn't exist.
      err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE);

   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingSync::tpl_abort()                                 */
/*                                                                  */
/* Purpose: Tells the data item that a locking transaction has      */
/*          aborted, which means, its lock can be released.         */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingSync::tpl_abort (const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta)
const{

   T_PROC("LockingSync::tpl_abort");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Container"); TI(get_soscontainer(_OBSThis)));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(LockingSync::make(_OBSThis,this)); TTA(ta);

   sos_Container ct;


   // *all* transactions are listed by their timestamps
   if ( get_accesses(_OBSThis).is_key(ta.get_ts()) )
   {
      ct = get_soscontainer(_OBSThis);

      ct.read_close();  

      tpl_release_n_wakeup(_OBSThis,ta);
      set_error(_OBSThis,SYNC_OK);
   }
   else
      // _TA doesn't exist.
      err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE); 

   TT(tsy_H, T_LEAVE);
   return;
}

/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: LockingTA                                                     */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

void _LockingTA::local_initialize (OBST_PARDECL(LockingTA) ta)

{
  T_PROC("LockingTA::local_initialize");
  TT(tsy_H, T_ENTER);

  // set type of ta
  ta.set_type(LOCKING);

  ta.set_waits_for(SyncObj::make(NO_OBJECT));

  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: LockingTA::commit                                        */
/*                                                                  */
/* Purpose: Closes all opended containers, if ta not implicit.      */
/*          Calls commit method on every accessed data item,        */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingTA::commit(const sos_Typed_id&_OBSThis)
const{
   T_PROC("LockingTA::commit");
   TT(tsy_H, T_ENTER);
   TTA(LockingTA::make(_OBSThis,this));

   sos_SOSet       opset;
   sos_SOSet       clset;
   SyncObj         so;
   sos_Bool        implta = get_implicit(_OBSThis);
   sos_Bool        pcom;        // partial commit ? 


   // is ta aborted internally ?
   if (get_status(_OBSThis) == ABORTED)
     { set_error(_OBSThis,SYNC_OK);
       TT(tsy_H, TXT("Intern aborted"); T_LEAVE);
       return; }

   set_error(_OBSThis,SYNC_OK);

   if (!implta)
     { // close used containers of explicit started transactions
       opset  = sos_SOSet::clone(get_openset(_OBSThis), TEMP_CONTAINER); 
       agg_iterate(opset, SyncObj so)
       {
         so.release();
         if (so.get_error() != SYNC_OK)
           { set_error(_OBSThis,so.get_error());
             TT(tsy_H, TXT("SyncObj-Error: "); TI(so.get_error()));
             break; }
       }
       agg_iterate_end(opset, so)
       if (get_error(_OBSThis) != SYNC_OK)
         { TT(tsy_H, T_LEAVE);
           return; }

       opset.clear();
     }

   if ((!implta) ||
      (( implta) && (get_openset(_OBSThis).is_empty())))
     { // commit explicit started transactions or
       // commit implicit started transactions with all containers closed
 
       agg_iterate(get_readset(_OBSThis), so)
       {
          LockingSync::make(so).tpl_commit(LockingTA::make(_OBSThis,this));
       }
       agg_iterate_end(get_readset(_OBSThis), so)
    
       get_readset(_OBSThis).clear();



       agg_iterate(get_writeset(_OBSThis), so)
       {
          LockingSync::make(so).tpl_commit(LockingTA::make(_OBSThis,this));
       }
       agg_iterate_end(get_writeset(_OBSThis), so)

       pcom = FALSE;
     }

    else
     { // partial commit implicit started transactions with all
       // containers opened writing closed
       // all closed containers are committed

       clset = sos_SOSet::create(TEMP_CONTAINER);
       clset += get_writeset(_OBSThis);
       clset += get_readset(_OBSThis);
       clset -= get_openset(_OBSThis);

       agg_iterate(clset, so)
       {
          LockingSync::make(so).tpl_commit(LockingTA::make(_OBSThis,this));
       }
       agg_iterate_end(clset, so)

       // only open container remain in readset
       get_readset(_OBSThis) *= get_openset(_OBSThis);
    
       pcom = TRUE;
     }

   if (!get_writeset(_OBSThis).is_empty())
      get_writeset(_OBSThis).clear();

   // set status
   if (!pcom)
     { set_status(_OBSThis,COMMITTED);
       TT(tsy_M, TXT("Status: COMMITTED"));
     }
    else
     { set_status(_OBSThis,ACTIVE);
       TT(tsy_M, TXT("Status: ACTIVE"));
     }

   // end commit
   set_error(_OBSThis,SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingTA::abort()                                       */
/*                                                                  */
/* Purpose: Resets all opended containers.                          */
/*          Calls abort method on every accessed data item,         */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingTA::abort(const sos_Typed_id&_OBSThis)
const{
   T_PROC("LockingTA::abort");
   TT(tsy_H, T_ENTER);
   TTA(LockingTA::make(_OBSThis,this));

   set_error(_OBSThis,SYNC_OK);

   if (get_status(_OBSThis) != ABORTED)
      intern_abort(_OBSThis);

   // end abort
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingTA::intern_abort()                                */
/*                                                                  */
/* Purpose: Calls abort method on every accessed data item,         */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _LockingTA::intern_abort(const sos_Typed_id&_OBSThis)
const{
   T_PROC("LockingTA::intern_abort");
   TT(tsy_H, T_ENTER);
   TTA(LockingTA::make(_OBSThis,this));

   sos_SOSet    helpset;                       // read/writeset of _TA
   sos_Cursor   cursor;                        // to iterate the sets
   SyncObj      sobj;                          // entry in set
   int          i;                             // counter
   sos_Bool     okay;


   // abort containers and clear sets
   for (i=0; i<=1; i++)
    {
      if (i==0)
        helpset = get_readset(_OBSThis);
       else
        helpset = get_writeset(_OBSThis);
 
      TSOSET(helpset);
 
      if (!helpset.is_empty())
       {
         // iterate set
         TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset":"writeset")));
         okay   = TRUE;
         cursor = helpset.open_cursor();
         while ((helpset.is_valid(cursor)) && (okay))
           { sobj = helpset.get(cursor);
             TSOBJ(sobj);
 
             // abort container
             LockingSync::make(sobj).tpl_abort(LockingTA::make(_OBSThis,this));
 
             okay = helpset.to_succ(cursor);
           }
         helpset.close_cursor(cursor);
         helpset.clear();
      }
    }

   // clear openset
   get_openset(_OBSThis).clear();

   // set status
   set_status(_OBSThis,ABORTED);
   TT(tsy_M, TXT("Status: ABORTED"));
 
   // end intern_abort
   set_error(_OBSThis,SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}


/* ================================================================ */
/* Method: LockingTA::extern_abort()                                */
/*                                                                  */
/* Purpose: Calls tpl_release_n_wakeup on every accessed data item, */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */
 
void _LockingTA::extern_abort(const sos_Typed_id&_OBSThis)
const{
   T_PROC("LockingTA::extern_abort");
   TT(tsy_H, T_ENTER);
   TTA(LockingTA::make(_OBSThis,this));

   sos_SOSet    helpset;                       // read/writeset of _TA
   sos_Cursor   cursor;                        // to iterate the sets
   SyncObj      sobj;                          // entry in set
   int          i;                             // counter
   int          pos;                           // takeout position
   sos_Bool     okay;
 
 
   // abort containers and clear sets
   for (i=0; i<=1; i++)
    {
      if (i==0)
        helpset = get_readset(_OBSThis);
       else
        helpset = get_writeset(_OBSThis);
 
      TSOSET(helpset);
 
      if (!helpset.is_empty())
       {
         // iterate set
         TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset":"writeset")));
         okay   = TRUE;
         cursor = helpset.open_cursor();
         while ((helpset.is_valid(cursor)) && (okay))
          { sobj = helpset.get(cursor);
            TSOBJ(sobj);
 
            // release_n_wakeup container
            if (sobj.get_accesses().is_key(get_ts(_OBSThis)))
              LockingSync::make(sobj).tpl_release_n_wakeup(LockingTA::make(_OBSThis,this));
 
            okay = helpset.to_succ(cursor);
          }
         helpset.close_cursor(cursor);
         helpset.clear();
       }
    }
 
   // clear sets
   get_openset(_OBSThis).clear();

   // remove from queue of SyncObj 
   if VALID(get_waits_for(_OBSThis))
     { // remove from queue
       pos = Queued::make(get_waits_for(_OBSThis)).q_takeout(LockingTA::make(_OBSThis,this), FALSE);
       if (pos == 1)
          // awake next, if ta was at head of the queue, because 
          // it might be possible for it to access the SyncObj
          Queued::make(get_waits_for(_OBSThis)).dequeue();  
     }
   
   // end extern_abort
   set_error(_OBSThis,SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
} 



/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: SyncObj (Locking Part)                                        */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: SyncObj::tpl_readlock()                                  */
/*                                                                  */
/* Purpose: This method mainly calls LockingSync::tpl_readlock and  */
/*          additionally asures deadlocks abort the transaction.    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED, SYNC_IMPLTA_ABORTED             */
/* ================================================================ */

sos_Bool _SyncObj::tpl_readlock(const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta, sos_Sync_mode sm) 
const{
   T_PROC("SyncObj::tpl_readlock");
   TT(tsy_H, T_ENTER);
   TSOBJ(SyncObj::make(_OBSThis,this)); TTA(ta); TT(tsy_L, TSM(sm));

   sos_Bool granted; //states whether access is granted or not

   granted = _LockingSync::tpl_readlock(_OBSThis,ta,sm);
   if (! granted)
     { if ((get_error(_OBSThis) == SYNC_DEADLOCK) || (sm == WAITING))
       { // transaction has to be aborted
         ta.intern_abort();
         if (ta.get_implicit())
           { set_error(_OBSThis,SYNC_IMPLTA_ABORTED);
             TT(tsy_H, TXT("SYNC_IMPLTA_ABORTED -> FALSE")); }
          else
           { set_error(_OBSThis,SYNC_TA_ABORTED);
             TT(tsy_H, TXT("SYNC_TA_ABORTED -> FALSE")); }
       }
      else
	 set_error(_OBSThis,SYNC_OK); // just tell user that access not possible
     }
    else
       set_error(_OBSThis,SYNC_OK); 

   TT(tsy_H, TB(granted); T_LEAVE);
   return(granted);
} 

/* ================================================================ */
/* Method: SyncObj::tpl_writelock()                                 */
/*                                                                  */
/* Purpose: This method mainly calls LockingSync::tpl_writelock and */
/*          additionally asures deadlocks abort the transaction.    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED, SYNC_IMPLTA_ABORTED             */
/* ================================================================ */

sos_Bool _SyncObj::tpl_writelock(const sos_Typed_id&_OBSThis,OBST_PARDECL(LockingTA) ta, sos_Sync_mode sm) 
const{
   T_PROC("SyncObj::tpl_writelock");
   TT(tsy_H, T_ENTER);
   TSOBJ(SyncObj::make(_OBSThis,this)); TTA(ta); TT(tsy_L, TSM(sm));

   sos_Bool granted; //states whether access is granted or not


   granted = _LockingSync::tpl_writelock(_OBSThis,ta,sm);
   if (! granted)
     { if ((get_error(_OBSThis) == SYNC_DEADLOCK) || (sm == WAITING))
       { // transaktion has to be aborted
         ta.intern_abort();
         if (ta.get_implicit())
           { set_error(_OBSThis,SYNC_IMPLTA_ABORTED);
             TT(tsy_H, TXT("SYNC_IMPLTA_ABORTED -> FALSE")); }
          else
           { set_error(_OBSThis,SYNC_TA_ABORTED);
             TT(tsy_H, TXT("SYNC_TA_ABORTED -> FALSE")); }
       }
      else
         set_error(_OBSThis,SYNC_OK); // just tell user that access not possible
     }
    else
       set_error(_OBSThis,SYNC_OK);

   TT(tsy_H, TB(granted); T_LEAVE);
   return(granted);

} 

