/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module smg                                                Juergen Uhl (ju)
//
// **************************************************************************
// implements methods of classes: smg_String
// **************************************************************************
//
// tracing conventions: see trc_smg.h

#ifdef __GNUG__
#pragma implementation
#endif

#define OBST_IMP_STREAM
#define OBST_IMP_FORMATTED_IO
#include "obst_stdinc.h"
#include "obst_progstd.h"


#include "knl_use.h"
#include "trc_smg.h"

#include "smg.h"

/* ------------  Auxiliary Definitions  ----------------------------------- */

#define TT_smg_String_enter(name, enter_cmds) \
		T_PROC(name); TT(smg_H, T_ENTER); TT(smg_M, enter_cmds);

#define TT_smg_String_leave(leave_cmds) \
		TT (smg_L, TXT ("smg_String"); TP ((char*)this); 	\
	                   TXT ("smg_String_ptr"); TP ((char*)ptr);	\
			   TXT ("RefCtr");         TI (ptr->rc);	\
			   TXT ("managed");        TB (ptr->managed); 	\
			   TXT ("string");         TP (ptr->c);		\
			   TXT ("size");	   TI (ptr->size);	\
			   TXT ("length");	   TI (ptr->length));	\
		TT (smg_H, T_LEAVE; leave_cmds);

#define TT_nil TXT("")


smg_String_ptr smg_empty_string_ptr;

// This function must only be invoked once - in _obst_basic_init!
EXPORT void _smg_init()
{  smg_empty_string_ptr.managed = 0;
   smg_empty_string_ptr.c	= "";
   smg_empty_string_ptr.length  = 0;
   smg_empty_string_ptr.size    = 1;

   ++ smg_empty_string_ptr.rc;
   // 'rc' must not be set to a fixed initial value since any invocation
   // of the smg_String default constructor increases 'rc' starting from
   // the initial value 0.
   // 'rc' is increased here once more to prevent the deletion of
   // smg_empty_string_ptr.
}

/* ------------  Constructors / Assignment  ------------------------------- */
				    
smg_String::smg_String (register char*	 c,
			const smg_access access /*=SMG_BORROW */)
{  TT_smg_String_enter ("smg_String::smg_String (char*, smg_access)",
		        TXT ("params"); TS (c); TI (access));

   register smg_String_ptr* p = ptr
			      = new smg_String_ptr;
   p->rc      = 1;
   p->managed = (access != SMG_BORROW);
   p->size    = (p->length = strlen (c)) + 1;

   if (access == SMG_COPY)
   {  register char* pc = p->c
			= new char [p->size];
      while (*(pc ++) = *(c ++))
	 ;
   }
   else
      p->c = c;

   TT_smg_String_leave (TT_nil);
}

smg_String::smg_String (sos_String s)
{  TT_smg_String_enter ("smg_String::smg_String (sos_String)",
			TXT ("param (ct,of)");
			TI ((int)s.container()); TU ((unsigned)s.offset()));
   if (s == NO_OBJECT)
   {  ptr = &smg_empty_string_ptr;
      ++ smg_empty_string_ptr.rc;
   }
   else
   {  register smg_String_ptr* p = ptr
				 = new smg_String_ptr;
      p->rc	 = 1;
      p->managed = 1;
      p->c       = s.make_Cstring();
      p->size    = (p->length = strlen (p->c)) + 1;
   }
   TT_smg_String_leave (TT_nil);
}

smg_String::smg_String (const sos_Int i, const int decimal_mode /* = TRUE */)
{  TT_smg_String_enter ("smg_String::smg_String (sos_Int)",
			TXT ("param"); TI((int)i));
   
   register smg_String_ptr* p = ptr
			      = new smg_String_ptr;

				// 2^32 takes 10 digits + sign + \0 (base 10)
   sprintf (p->c = new char[p->size = 12],
	    (decimal_mode ? "%ld" : "%lx"), (long)i);

   p->rc      = 1;
   p->managed = 1;
   p->length  = strlen (p->c);

   TT_smg_String_leave (TT_nil);
}

smg_String& smg_String::operator= (const smg_String &s)
{  TT_smg_String_enter ("smg_String::op= (smg_String&)",
		        TXT ("param"); TP ((char*)&s));

   register smg_String_ptr *p;

   if (--(p = ptr)->rc == 0)
   {  if (p->managed)
	 delete p->c;
      delete p;
   }
   ptr = p = s.ptr;
   ++ p->rc;

   TT_smg_String_leave (TT_nil);
   return *this;
}

smg_String& smg_String::operator= (char* c)
{  TT_smg_String_enter ("smg_String::operator= (char*)", TXT("param"); TS(c));

   register smg_String_ptr *p;

   if ((p = ptr)->rc == 1 && p->managed)
   {  register int  sz = 1;
      register char *cptr, *pcptr;
      for (cptr = c;  *cptr;  ++ cptr)
	 ++ sz;
      if (sz > p->size)
      {  delete p->c;
	 p->c = new char[p->size = sz];
      }
      p->length = sz - 1;
      for (pcptr = p->c, cptr = c;  *(pcptr ++) = *(cptr ++); )
	 ;
   }
   else
      *this = smg_String(c, SMG_BORROW);

   TT_smg_String_leave (TT_nil);
   return *this;
}

smg_String smg_String::clone () const
{  TT_smg_String_enter ("smg_String::clone", TT_nil);

   smg_String result (ptr->c, SMG_COPY);

   TT_smg_String_leave (TP ((char*)&result));
   return result;
}

void smg_String::empty ()
{  TT_smg_String_enter ("smg_String::clear", TT_nil);

   register smg_String_ptr* p;

   if ((p = ptr)->managed)
   {  if (p->length)
      {  p->length = 0;
	 *(p->c)   = '\0';
      }
   }
   else
      *this = smg_String();

   TT_smg_String_leave (TT_nil);
}

/* ------------  Conversions  --------------------------------------------- */

char* smg_String::make_Cstring (const smg_access access /* = SMG_COPY */) const
{  TT_smg_String_enter ("smg_String::make_Cstring",
			TXT ("param"); TI (access));
   register char *result;

   if (access == SMG_COPY)
   {  register char *rptr,
		    *pcptr = ptr->c;
      for (result = rptr = new char [length() + 1];  *(rptr ++) = *(pcptr ++); )
	 ;
   }
   else
   {  if (access == SMG_TRANSFER)
	 ptr->managed = 0;
      result = ptr->c;
   }
   TT_smg_String_leave (TP (result));
   return result;
}

sos_String smg_String::make_String (const sos_Container cnt) const
{  TT_smg_String_enter ("smg_String::make_String",
			TXT ("param"); TI ((int)cnt));

   sos_String s = sos_String::create (cnt, make_Cstring (SMG_BORROW));

   TT_smg_String_leave (TXT ("(ct,of)");
			TI ((int)s.container()); TU ((unsigned)s.offset()));
   return s;
}

/* ------------  String Manipulation  ------------------------------------- */

smg_String& smg_String::operator += (const char *c)
{  TT_smg_String_enter ("smg_String::operator+=", TXT ("param"); TS (c));

   if (*c)
   {  register smg_String_ptr* p      = ptr;
      register int             newlen = p->length;
      register char*	       cptr   = (char*)c;

      while (*(cptr ++))
	 ++ newlen;			       // newlen = length() + strlen(c)
	 
      if (p == &smg_empty_string_ptr)
      {  register char *pcptr;

	 ptr = p    = new smg_String_ptr;
	 p->rc      = 1;
	 p->managed = 1;
	 p->length  = newlen;
	 p->c       =
	 pcptr	    = new char[p->size = newlen+1];

	 for (cptr = (char*)c;  *(pcptr ++) = *(cptr ++); ) // strcpy (p->c, c)
	    ;
      }
      else
      {  int len = p->length;

	 if ((p->length = newlen) >= p->size)
	 {  if (newlen >= ( p->size += (p->size < 512) ? p->size : 512 ))
	       p->size = newlen + 1;

	    register char* pcptr = p->c;
	    register char* cat   = cptr
			         = new char [p->size];
	    while (*(cptr ++) = *(pcptr ++))	// strcpy (cat, p->c)
	       ;
	    if (p->managed)
	    {  TT (smg_H, TXT ("deleting"); TP (p->c));
	       delete p->c;
	    }
	    else
	       p->managed = 1;
	    
	    p->c = cat;
	 }
	 register char* pcptr = p->c + len;	      // strcpy (p->c + len, c)
	 for (cptr = (char*)c;  *(pcptr ++) = *(cptr ++); )
	    ;
      }
   }
   TT_smg_String_leave (TT_nil);
   return (*(smg_String*)this);
}

smg_String smg_String::operator + (const char *c) const
{  TT_smg_String_enter ("smg_String::operator+", TXT ("param"); TS (c));

   smg_String result(ptr->c, (*c ? SMG_BORROW : SMG_COPY));
   result += c;

   TT_smg_String_leave (TP ((char*)&result));
   return result;
}

/* ------------  String I/O  ---------------------------------------------- */

ostream& operator<< (ostream& o, const smg_String &s)
{  return o << s.ptr->c;
}

ostream& operator<< (ostream& o, const sos_String& s)
{  if (s != NO_OBJECT)
   {  char *str = s.make_Cstring();
      o << str;
      delete str;
   }
   return o;
}
