/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// implementation of incremental loading for BSD systems
// **************************************************************************

// --------------------------------------------------------------------------
// Tracing conventions: see trc_cci.h
// --------------------------------------------------------------------------

#include "_obst_config.h"

#define OBST_IMP_STDCONST
#define OBST_IMP_STRINGOP
#define OBST_IMP_FILE
#define OBST_IMP_MALLOC
#define OBST_IMP_PROCEXEC
#define OBST_IMP_PROCRESOURCE
#define OBST_IMP_A_OUT
#include "obst_stdinc.h"

#ifdef MUST_MPROTECT
#  define OBST_IMP_MEMMGR
#  include "obst_stdinc.h"
#endif

#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "cci_err.h"
#include "trc_cci.h"
#include "cci_obst.h"
#include "cci_incrload.h"

#if OBST_INCRLD == LD_COLLECT  
#  define LDPARAMXX		"GNUcollect"
#  define SYMBOL1	 	"___CTOR_LIST__"
#  define SYMBOL2	 	"___DTOR_LIST__"
#  define SYMBOL3	 	"___main_reference"
#  define SYMBOL_COUNT		3
#  define INCRLD_CLEAR_EXT    	1
#  define INCRLD_CLEAR_MAIN	1
#endif

#if OBST_INCRLD == LD_GNU  
#  define LDPARAMXX		"GNUld"
#  define SYMBOL1	 	"___CTOR_LIST__"
#  define SYMBOL2	 	"___DTOR_LIST__"
#  define SYMBOL_COUNT		2
#  define INCRLD_CLEAR_EXT    	0
#  define INCRLD_CLEAR_MAIN	1
#endif

#if OBST_INCRLD == LD_PATCH  
#  define LDPARAMXX		"ATTpatch"
#  define SYMBOL1		"___link"
#  define SYMBOL_COUNT		1
#  define INCRLD_CLEAR_EXT	0
#  define INCRLD_CLEAR_MAIN	0
#  define FIND_ALL_SYMS
#endif

#if OBST_INCRLD == LD_MUNCH  
#  define LDPARAMXX		"ATTmunch"
#  define SYMBOL1		"__ctors"
#  define SYMBOL2		"__dtors"
#  define SYMBOL_COUNT		2
#  define INCRLD_CLEAR_EXT	1
#  define INCRLD_CLEAR_MAIN	1
#endif
 
// --------------------------------------------------------------------------

	// SYMTAB_FILE: name of temporary file holding symbol table of the
	//		executable as extended by previous loads
	// TMPFILE:	name of temporary file used to build the next
        //		version of SYMTAB_FILE
#define SYMTAB_FILE cci_incr_ld.tmpf1
#define TMPFILE	    cci_incr_ld.tmpf2


// --------------------------------------------------------------------------

// --------------------------------------------------------------------------
// module initialization
// --------------------------------------------------------------------------

extern "C" cci_LoadFct incrld_BSD;    // must be referenced to load this module

LOCAL void init_BSD()
{  T_PROC ("BSD_incrld : init_BSD")
   TT (cci_M, T_ENTER);

   cci_load_fct = incrld_BSD;

   TT (cci_M, T_LEAVE);
}
LOCAL int _dummy = (init_BSD(),0);


// --------------------------------------------------------------------------
// auxiliary functions: ?tor handling
// --------------------------------------------------------------------------

// **************************************************************************
LOCAL void make_xtorlist (cci_XtorList* xtl,
			  void* addr1, void* addr2, void* start, void* end,
			  sos_Bool get_ctors)
// **************************************************************************
{  T_PROC ("BSD_incrld : make_xtorlist")
   TT (cci_VL, T_ENTER);
#ifndef NO_TT
   int xtor_count = 0;
#endif

#if (OBST_INCRLD == LD_COLLECT) || (OBST_INCRLD == LD_GNU) 
   // GCC 2.3.3 Manual (sec. 14.16.5) - xtor lists have the following structure
   // - function pointer (cast from int) optionally holding the list length,
   // - function pointers to the xtor functions,
   // - NULL pointer.
   // Hence, we skip the first field and scan until the NULL pointer is found.

   // However, the NULL check is necessary since this is the encoding for
   // "no such entry in symbol table" (see cci_IncrLd::read_xtors).

   cci_Xtor_p* xp_list = (cci_Xtor_p *)((get_ctors) ? addr1 : addr2);

   if (xp_list)
      while (*(++ xp_list))
	 if (start <= (void*)*xp_list && (void*)*xp_list <= end)
	 {  xtl->add (*xp_list);
	    TTN (cci_VL, ++ xtor_count);
	 }
#endif
#if OBST_INCRLD == LD_PATCH  
   // patch seems to generate an NULL terminated list holding ctor/dtor pairs.

   struct Patch_l
   {  Patch_l*   next;
      cci_Xtor_p ctor, dtor;
   } *xtor_list;

   for (xtor_list = (Patch_l*)addr1;  xtor_list;  xtor_list = xtor_list->next)
   {  cci_Xtor_p xtor = (get_ctors) ? xtor_list->ctor
      				    : xtor_list->dtor;
      if (xtor  &&  start <= (void*)xtor  &&  (void*)xtor <= end)
      {  xtl->add (xtor);
	 TTN (cci_VL, ++ xtor_count);
      }
   }
#endif
#if OBST_INCRLD == LD_MUNCH  
   // munch seems to generate NULL terminated xtor vectors.

   cci_Xtor_p* xp_list = (cci_Xtor_p *)((get_ctors) ? addr1 : addr2);

   if (xp_list)
      for ( ;  *xp_list;  xp_list ++)
	 if (start <= (void*)*xp_list && (void*)*xp_list <= end)
	 {  xtl->add (*xp_list);
	    TTN (cci_VL, ++ xtor_count);
	 }
#endif
   TT (cci_VL, T_LEAVE; TXT("no. of xtors"); TI(xtor_count));
}


// **************************************************************************
LOCAL void read_xtors (const char*   symfile,
		       cci_XtorList& ctors,   cci_XtorList& dtors,
		       void* 	     start,   void* 	    end)
// **************************************************************************
{  T_PROC ("BSD_incrld - read_xtors")
   TT (cci_VL, T_ENTER);

   struct exec header;
   int fd;
   
   if ((fd = open (symfile, O_RDWR, 0)) < 0)
      err_raise (err_SYS, err_CCI_OPEN_FAILED, symfile);
   if (read (fd, (char*)&header, sizeof(header)) != sizeof(header))
      err_raise (err_SYS, err_CCI_READ_FAILED, symfile);
   
   off_t sym_offset = N_SYMOFF(header),
         stringbase = N_STROFF(header);

   // Current search times seem to be well below 5% of the overall load time.
   // nlist(3V) is not used since we might want to modify the original
   // symtable entries and hence need their (file) position.

#define SEARCHMASK(i)     (0x1 << (i))

   struct nlist sym[SYMBOL_COUNT+1];
   int		searchfor = 0;

   sym[0].n_un.n_name = SYMBOL1;	sym[0].n_value = 0;
#if SYMBOL_COUNT > 1
   sym[1].n_un.n_name = SYMBOL2;	sym[1].n_value = 0;
#endif
#if SYMBOL_COUNT > 2
   sym[2].n_un.n_name = SYMBOL3;	sym[2].n_value = 0;
#endif

   const int READBUF_SIZE = 20;
   char      readbuf[READBUF_SIZE];

   readbuf[READBUF_SIZE - 1] = '\0';
   for (int i = 0; i < SYMBOL_COUNT;  ++ i)
   {  searchfor |= SEARCHMASK(i);
      err_assert (strlen (sym[i].n_un.n_name) < READBUF_SIZE,
		  "BSD_incrld - read_xtors");
   }
   const int    SYMBUF_SIZE = 1024 / sizeof(struct nlist);
   struct nlist symbuf[SYMBUF_SIZE];

#ifndef NO_TT
   int syms_read = 0;
#endif

   do
   {  if (lseek (fd, sym_offset, SEEK_SET) < 0)
	 err_raise (err_SYS, err_CCI_SEEK_FAILED, symfile);

      if (read (fd, (char*)&symbuf[0], sizeof(symbuf)) != sizeof(symbuf))
	 err_raise (err_SYS, err_CCI_READ_FAILED, symfile);
      	 // Assume the string table is at least sizeof(symbuf) bytes in size!

      struct nlist *sym_ptr = symbuf + SYMBUF_SIZE;

      if ((sym_offset += sizeof(symbuf)) > stringbase)
	 sym_ptr -= (sym_offset - stringbase) / sizeof(struct nlist);

      while (--sym_ptr >= symbuf)
      {
#ifndef NO_TT
	 ++ syms_read;
#endif
	 if (   !(sym_ptr->n_type & N_STAB)
	     && sym_ptr->n_type & (N_TEXT | N_EXT)
	     && sym_ptr->n_un.n_strx > 0)
	 {  
	    if (lseek(fd,(off_t)(stringbase+sym_ptr->n_un.n_strx),SEEK_SET) <0)
	       err_raise (err_SYS, err_CCI_SEEK_FAILED, symfile);
	    if (read (fd, (char*)&readbuf[0], READBUF_SIZE - 1) <= 0)
	       err_raise (err_SYS, err_CCI_READ_FAILED, symfile);

	    for (int idx = 0;  idx < SYMBOL_COUNT;  ++ idx)
	    {  if (   end > start
		   && (   start > (void*)sym_ptr->n_value
		       || end   < (void*)sym_ptr->n_value))
		  continue;

	       if (
#ifndef FIND_ALL_SYMS
		   SEARCHMASK(idx) & searchfor &&
#endif
		   streql (readbuf, sym[idx].n_un.n_name))
	       {  TT (cci_VL, TXT(readbuf); TXT("symbol found"));

   		  sym[idx].n_value = sym_ptr->n_value;
#if INCRLD_CLEAR_EXT
		  if (sym_ptr->n_type & N_EXT)
		  {  sym_ptr->n_type &= ~N_EXT;   // clear external bit

		     off_t sympos = sym_offset
				    - (sizeof(symbuf)
				       - ((char*)sym_ptr - (char*)symbuf));
		     
		     if (lseek (fd, sympos, SEEK_SET) < 0)
			err_raise (err_SYS, err_CCI_SEEK_FAILED, symfile);
		     if (write (fd, (char*)sym_ptr, sizeof(struct nlist))
			 < sizeof(struct nlist))
			err_raise (err_SYS, err_CCI_WRITE_FAILED, symfile);
		  }
#endif
#ifdef FIND_ALL_SYMS
		  make_xtorlist (&ctors,
				 (void*)sym[0].n_value,
				 (void*)sym[1].n_value, start, end, TRUE);
		  make_xtorlist (&dtors,
				 (void*)sym[0].n_value,
				 (void*)sym[1].n_value, start, end, FALSE);
#else
		  if (!(searchfor &= ~SEARCHMASK(idx)))
		     break;
#endif
	       }
	    }
#ifndef FIND_ALL_SYMS
	    if (!searchfor)
	       break;
#endif
	 }
      }
   }
   while (sym_offset < stringbase && searchfor);

   TT (cci_VL, TXT("symbols scanned"); TI(syms_read));

   close (fd);

#ifndef FIND_ALL_SYMS
   if (end > start)
   {  make_xtorlist (&ctors, (void*)sym[0].n_value, (void*)sym[1].n_value,
		     start, end, TRUE);
      make_xtorlist (&dtors, (void*)sym[0].n_value, (void*)sym[1].n_value,
		     start, end, FALSE);
   }
#endif
   TT (cci_VL, T_LEAVE);
}


// --------------------------------------------------------------------------
// main functions for incremental loading
// --------------------------------------------------------------------------

// **************************************************************************
EXPORT void incrld_BSD (const sos_Object_List &schemas,
		        const sos_String_List &objects,
		        const sos_String_List &libs)
// **************************************************************************
{  T_PROC ("incrld_BSD")
   TT (cci_M, T_ENTER);

   static const char*  here = "incrld_BSD";
   static cci_XtorList ctors;
   static int	       pagesize_min1;
   static const char*  ld_prog;
   static const char*  ld_kind;

   static sos_Bool first_load = TRUE;
   if (first_load)
   {  first_load = FALSE;

      ctors.init();

      ld_prog = (cci_incr_ld.ld_param1) ? cci_incr_ld.ld_param1 : LDXX;
      ld_kind = (cci_incr_ld.ld_params) ? cci_incr_ld.ld_params : LDPARAMXX;

      pagesize_min1 =
#ifdef EXEC_PAGESIZE
                      EXEC_PAGESIZE - 1;
#else
#if defined(PAGSIZ)
                      PAGSIZ - 1;
#else
                      getpagesize() - 1;
#endif
#endif

#if INCRLD_CLEAR_MAIN
      // Extract the symbol table from the running executable (by writing
      // an executable with zero sized text,data segment) and clear
      // the relevant external definitions contained therein.

      // (May also make sense in order to reduce the size of the symtab
      //  file -- if the employed ld program can not properly handle -x).

      if (cci_incr_ld.echo)
	 cerr << "...preparing " << cci_incr_ld.executable << "\n";

      smg_String cmd = smg_String("/bin/ld -A ") + cci_incr_ld.executable
	 	       		    + " -x -o "  + SYMTAB_FILE;
      if (system (cmd.make_Cstring (SMG_BORROW)))
	 err_raise (err_SYS, err_CCI_LOAD_FAILED, here, FALSE);

      read_xtors (SYMTAB_FILE, ctors, cci_incr_ld.dtors, (void*)1, (void*)0);
      ctors.clear();
      cci_incr_ld.dtors.clear();
#else
      // No external definitions to clear --> symbol table does not have
      // to be modified initially.

      smg_String cmd = smg_String("ln -s ") + cci_incr_ld.executable + " "
	 				    + SYMTAB_FILE;
      if (system (cmd.make_Cstring (SMG_BORROW)))
	 err_raise (err_SYS, err_CCI_LINK_FAILED, here, FALSE);
#endif
   }
   unsigned long size     = 0;
   smg_String    cmd_head = smg_String((char*)ld_prog)
      			    + " " + SYMTAB_FILE
			    + " " + TMPFILE
			    + " " + (char*)ld_kind;
   int	         fd;
   struct exec   header;
   smg_String    cmd_tail = smg_String(" '") + cci_incr_ld.searchpath + "' '";

   if (objects != NO_OBJECT)
      agg_iterate (objects, sos_String on)
      {  smg_String obj_path = on;

	 fd = cci_incr_ld.find_file (obj_path);

	 if (read (fd, (char*)&header, sizeof (header)) != sizeof (header))
	    err_raise (err_SYS, err_CCI_READ_HEADER_FAILED,
				obj_path.make_Cstring(SMG_BORROW));
	 
	 size     += header.a_text + header.a_data + header.a_bss;
	 cmd_tail += smg_String(" ") + obj_path;
	 
	 close (fd);
	 
	 if (cci_incr_ld.echo)
	    cerr << "...loading " << obj_path << "\n";
      }
      agg_iterate_end (objects, on);

   cmd_tail += "' '";

   if (libs != NO_OBJECT)
      agg_iterate (libs, sos_String ln)
	 cmd_tail += smg_String(" ") + ln;
      agg_iterate_end (libs, ln);

   cmd_tail += "' '";

   if (schemas != NO_OBJECT)
      agg_iterate (schemas, sos_Object _sm)
	 cmd_tail += smg_String(" ") + sos_Named::make(_sm).get_name();
      agg_iterate_end (schemas, _sm)

   cmd_tail += "'";
   
   		// Increase the chance that the first size guess will suffice.
   size += (size > 81920) ? 8192 : 512;

   unsigned long actual_size;// Repeat until size >= actual size.
   char          *obj_addr;  // (The first guess might be to small because the
   for (;;) 		     //  size of the BSS-Segment and the size of code
			     //  loaded from libraries is not yet known.)
   {  size += pagesize_min1;
      char *allocated = new char[size];

      obj_addr = (char*)( ((unsigned)allocated + pagesize_min1)
			  & ~pagesize_min1);
      size    -= obj_addr - allocated;
      // place additional object modules on page boundary

      smg_String ldcmd = cmd_head + " " + smg_String((sos_Int)obj_addr, FALSE)
				  + cmd_tail;

      TT (cci_M, TXT("load command"); TS(ldcmd.make_Cstring (SMG_BORROW)));

#ifdef HAVE_GETRLIMIT
      struct rlimit core_limit;
      getrlimit (RLIMIT_CORE, &core_limit);	// suppress core file
      int cur_limit = core_limit.rlim_cur;	// creation temporarily
      core_limit.rlim_cur = 0;
      setrlimit (RLIMIT_CORE, &core_limit); 
#endif
      int error_status = system (ldcmd.make_Cstring (SMG_BORROW));

#ifdef HAVE_GETRLIMIT
      core_limit.rlim_cur = cur_limit;
      setrlimit (RLIMIT_CORE, &core_limit);
#endif
      if (error_status) 
      {  unlink (TMPFILE);
	 err_raise (err_SYS, err_CCI_LOAD_FAILED, here, FALSE);
      }

      fd = open (TMPFILE, O_RDONLY, 0);
      read (fd, (char*)&header, sizeof (header));
      close (fd);

      actual_size = header.a_text + header.a_data + header.a_bss;

      TT (cci_M, TXT("size guess");   TI((int)size);
	  	 TXT("actual size");  TI((int)actual_size);
	  	 TXT("text segment"); TI((int)header.a_text);
	  	 TXT("data segment"); TI((int)header.a_data);
	  	 TXT("bss segment");  TI((int)header.a_bss));

      if (actual_size <= size)
      {  TT (cci_M, TXT("start address"); TP((void*)obj_addr);
	     	    TXT("end address");   TP((void*)(obj_addr + size - 1)));
	 break;
      }
      TT (cci_M, TXT("...repeat loading"));

      delete allocated;
      size = actual_size;
   }
   smg_String cmd = smg_String("rm ") + SYMTAB_FILE
			     + ";mv " + TMPFILE
			     + " "    + SYMTAB_FILE;
   if (system (cmd.make_Cstring (SMG_BORROW)))
      err_raise (err_SYS, err_CCI_MOVE_FAILED, NULL, FALSE);

   fd = open (SYMTAB_FILE, O_RDONLY, 0);
   if (lseek (fd, (off_t)N_TXTOFF(header), SEEK_SET) < 0)
      err_raise (err_SYS, err_CCI_SEEK_FAILED, here, FALSE);

   if (read (fd, obj_addr, (int)actual_size) != actual_size)
      err_raise (err_SYS, err_CCI_READ_FAILED, here, FALSE);
   close (fd);

#ifdef MUST_MPROTECT
   if (mprotect (obj_addr, actual_size, PROT_READ|PROT_EXEC|PROT_WRITE))
      err_raise (err_SYS, err_CCI_MKEXEC_FAILED, here, FALSE);
#endif

   read_xtors (SYMTAB_FILE, ctors, cci_incr_ld.dtors,
			    obj_addr, obj_addr + actual_size - 1);
   ctors.process (FALSE);
   ctors.clear();

   TT (cci_M, T_LEAVE);
}
