/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : Timestamp Synchronisation                                        */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_FILE
#define OBST_IMP_PROCESS
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "knl_use.h" 	/* includes psm.h */
#include "ext_sync_obst.h"
#include "sync_decl.h"
#include "trc_tsy.h"
#include "sync_trc.h"
#include "sync_err.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: Timemark                                                      */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

void Timemark::local_initialize(Timemark tm)

{
  T_PROC("Timemark::local_initialize");
  TT(tsy_H, T_ENTER);
  tm.set_active(TRUE);
  TT(tsy_H, T_LEAVE);
}


/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: TimestampTA                                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

void TimestampTA::local_initialize (TimestampTA ta)

{
  T_PROC("TimestampTA::local_initialize");
  TT(tsy_H, T_ENTER);

  // set type of ta
  ta.set_type(TIMESTAMP);

  ta.set_waits_for(SyncObj::make(NO_OBJECT));

  TT(tsy_H, T_LEAVE);  
  return;
}

/* ================================================================ */
/* Method: TimestampTA::commit()                                    */
/*                                                                  */
/* Error: SYNC_OK, (SyncObj::release)                               */
/* ================================================================ */


void TimestampTA::commit() 

{
  T_PROC("TimestampTA::commit");
  TT(tsy_H, T_ENTER);
  TTA(self);

  SyncCont        syct;                           // syncronization data
  sos_SOSet       helpset;                        // set to commit container
  sos_SOSet       openset;                        // openset of _TA
  sos_PidSet      pid_set;
  SyncObj         sobj;                           // entry in read/writeset
  sos_Cursor      cursor;                         // to iterate sets
  sos_HostProc_Id pid;
  sos_Timestamp   acts;
  Transaction     ta;
  OptimisticTA    opta;
  sos_Bool        okay;
  int             i;
 

  // is ta intern aborted ?
  if (self.get_status() == ABORTED)
    { // everything is done
      self.set_error(SYNC_OK);
      TT(tsy_H, TXT("Intern aborted"); T_LEAVE);
      return; }

  // close all container that are still open
  // if there is an open container, which has been opened by this transaction
  // for writing, then the commit may not be successfull and the transaction
  // may be aborted. So release all open containers.
  if (!self.get_openset().is_empty())
    {
      // iterate openset
      openset  = sos_SOSet::clone(self.get_openset(), TEMP_CONTAINER);

      TT(tsy_L, TXT("iterate openset"));
      TSOSET(openset);

      okay   = TRUE;
      cursor = openset.open_cursor();
      while ((openset.is_valid(cursor)) && (okay))
        { sobj = openset.get(cursor);
          TSOBJ(sobj);

          // release container
          sobj.release();
          if (sobj.get_error() != SYNC_OK)
            { self.set_error(sobj.get_error());
              openset.close_cursor(cursor);
              TT(tsy_H, TXT("SyncObj-Error: "); TI(self.get_error()); T_LEAVE);
              return;
            }

          okay = openset.to_succ(cursor);
        }
      openset.close_cursor(cursor);

      // remove container from readset
      openset.clear();
    }


  // commit containers and clear sets
  for (i=0; i<=1; i++)
   {
     if (i==0)
        helpset = self.get_readset();
       else
        helpset = self.get_writeset();
     TSOSET(helpset);
 
     if (!helpset.is_empty())
      {
        // iterate set
        TT(tsy_L, TXT("iterate set:");TXT((i==0 ? "readset":"writeset")));
        okay   = TRUE;
        cursor = helpset.open_cursor();
        while ((helpset.is_valid(cursor)) && (okay))
          { sobj = helpset.get(cursor);
            TSOBJ(sobj);
 
            // commit container
            TimestampSync::make(sobj).tso_commit(self);
 
            okay = helpset.to_succ(cursor);
          }
        helpset.close_cursor(cursor);
      }
   }

  
  if (!self.get_writeset().is_empty())
    {
      // distribute sets among optimistic transactions
      TT(tsy_L, TXT("distribute writeset"));
      syct    = SyncCont::get_root();
 
      if (!syct.get_opta_tab().is_empty())
        { // there are optimistic ta
          acts    = syct.get_ts() - SOS_OLD_TA_TS;
          pid_set = sos_PidSet::create(TEMP_CONTAINER);
 
          agg_iterate_association (syct.get_opta_tab(), pid, ta)
           { TT(tsy_VL, TI(pid); TI(ta.get_type()));
 
            opta = OptimisticTA::make(ta);
            if ((opta.get_ts() < acts) &&
                (!syct.process_check(pid)))
              { // process of old optimistic ta is aborted
                pid_set.insert(pid);
              }
             else
              { // distribute
                opta.get_commitset() += self.get_writeset();
                TSOSET(opta.get_commitset());
              }
           } agg_iterate_association_end (syct.get_opta_tab(), pid, ta);
 
          // remove aborted process
          if (!pid_set.is_empty())
            { agg_iterate ( pid_set, pid )
               {
                syct.process_remove(pid);
               }
              agg_iterate_end (pid_set, pid);
              pid_set.clear();
            }
        }
    }
 
  
  // clear sets
  self.get_writeset().clear();
  self.get_readset().clear();


  // set status
  self.set_status(COMMITTED);
  TT(tsy_M, TXT("Status: COMMITTED"));

  // end commit
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}  

/* ================================================================ */
/* Method: TimestampTA::abort                                       */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

 
void TimestampTA::abort()

{ 
  T_PROC("TimestampTA::abort");
  TT(tsy_H, T_ENTER);
  TTA(self);
 
  self.set_error(SYNC_OK);

  // if _TA wasn't aborted by the system
  if (self.get_status() != ABORTED)
     self.intern_abort ();

  // end abort      
  TT(tsy_H, T_LEAVE);
  return;
}  

/* ================================================================ */
/* Method: TimestampTA::intern_abort                                */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */


void TimestampTA::intern_abort ()

{ 
  T_PROC("TimestampTA::intern_abort");
  TT(tsy_H, T_ENTER);
  TTA(self);

  sos_SOSet    helpset;                       // read/writeset of _TA
  sos_Cursor   cursor;                        // to iterate the sets
  SyncObj      sobj;                          // entry in set
  int          i;                             // counter 
  sos_Bool     okay;
 

  // abort containers and clear sets
  for (i=0; i<=1; i++)
   {
     if (i==0)
        helpset = self.get_readset();
       else
        helpset = self.get_writeset();
     TSOSET(helpset);
 
     if (!helpset.is_empty())
      {
        // iterate set
        TT(tsy_L, TXT("iterate set:");TXT((i==0 ? "readset":"writeset")));
        okay   = TRUE;
        cursor = helpset.open_cursor();
        while ((helpset.is_valid(cursor)) && (okay))
          { sobj = helpset.get(cursor);
            TSOBJ(sobj);
 
            // abort container
            TimestampSync::make(sobj).tso_abort(self);
 
            okay = helpset.to_succ(cursor);
          }
        helpset.close_cursor(cursor);
        helpset.clear();
      }
   }

  // clear sets
  self.get_openset().clear();


  // set status
  self.set_status(ABORTED);
  TT(tsy_M, TXT("Status: ABORTED"));
  
  // end intern_abort
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: TimestampTA::extern_abort                                */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void TimestampTA::extern_abort()
{
  T_PROC("TimestampTA::extern_abort");
  TT(tsy_H, T_ENTER);
  TTA(self);

  sos_SOSet    helpset;                       // read/writeset of TA
  sos_Cursor   cursor;                        // to iterate the sets
  SyncObj      sobj;                          // entry in set
  int          i;                             // counter
  int          pos;                           // takeout position
  sos_Bool     okay;
 
 
  // abort containers and clear sets
  for (i=0; i<=1; i++)
   {
     if (i==0)
        helpset = self.get_readset();
       else
        helpset = self.get_writeset();
     TSOSET(helpset);
 
     if (!helpset.is_empty())
      {
        // iterate set
        TT(tsy_L, TXT("iterate set:");TXT((i==0 ? "readset":"writeset")));
        okay   = TRUE;
        cursor = helpset.open_cursor();
        while ((helpset.is_valid(cursor)) && (okay))
          { sobj = helpset.get(cursor);
            TSOBJ(sobj);
 
            // release_n_wakeup container
            if (sobj.get_accesses().is_key(self.get_ts()))
              TimestampSync::make(sobj).tso_release_n_wakeup(self, FALSE);
 
            okay = helpset.to_succ(cursor);
          }
        helpset.close_cursor(cursor);
        helpset.clear();
      }
   }
 
  // clear sets
  self.get_openset().clear();
 
  // remove from queue of SyncObj
  if VALID(self.get_waits_for())
    { // remove from queue
      pos = Queued::make(self.get_waits_for()).q_takeout(self, FALSE);
      if (pos == 1)
          // awake next, if ta was at head of the queue, because 
          // it might be possible for it to access the SyncObj
          Queued::make(self.get_waits_for()).dequeue();  
    }
 
  // end extern_abort
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}


/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: TimestampSync                                                 */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

void TimestampSync::local_initialize(TimestampSync tsync)

{ 
  T_PROC("TimestampSync::local_initialize");
  TT(tsy_H, T_ENTER);

  // Create RTM-List
  tsync.set_rtm_list (sos_TimeList::create(SYNC_CONTAINER));
  tsync.get_rtm_list().insert(1, Timemark::create(SYNC_CONTAINER, NO_TS));
  tsync.get_rtm_list().get_nth(1).set_active(FALSE);

  // Set WTM
  tsync.set_wtm (NO_TS);

  TT(tsy_H, T_LEAVE); 
}

//-----------------------------------------------------------------------------


void TimestampSync::local_finalize (TimestampSync tsync)

{
  T_PROC("TimestampSync::local_finalize");
  TT(tsy_H, T_ENTER);
  TSOBJ(tsync);

  sos_TimeList   list  = tsync.get_rtm_list();     // timemark list
  sos_Cursor     cursor;
  sos_Bool       okay;


  // remove timemarks from list
  cursor = list.open_cursor();
  okay   = TRUE;
  while ((list.is_valid(cursor)) && (okay))
    { list.get(cursor).destroy();
      okay = list.to_succ(cursor);
    }
  
  list.close_cursor(cursor);

  // destroy RTM_List
  list.clear();
  list.destroy();


  //end local_finalize
  TT(tsy_H, T_LEAVE);
  return;
}
  
/* ================================================================ */
/* Method: TimestampSync::tso_read                                  */
/*                                                                  */
/* Error: SYNC_OK, SYNC_NOVALID_TS, SYNC_DEADLOCK                   */
/* ================================================================ */

sos_Bool TimestampSync::tso_read (TimestampTA ta, sos_Sync_mode sm)

{ 
  T_PROC("TimestampSync::tso_read");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TSM(sm); TTA(ta) );

  sos_BlockList   queue   = self.get_queue();      // waiting queue
  sos_AccessMap   access  = self.get_accesses();   // access mapping
  Lock_Mode       lock    = self.get_lock();       // lock on container
  sos_Timestamp   write_ts;                        // _TS of WRITE in accesses
  sos_Cursor      cursor;                          // cursor for mapping
  BlockedTA       blockta;                        
  sos_Bool        found;                     
  int             pos;                             // insert position in queue
  int             qpos;                            // position counter

 
  // Timestamp okay ?
  TT(tsy_VL, TI(self.get_wtm()); TI(ta.get_ts())); 
  if (self.get_wtm() > ta.get_ts()) 
    { self.set_error(SYNC_NOVALID_TS);
      TT(tsy_H, TXT("SYNC_NOVALID_TS"); TXT("FALSE"); T_LEAVE); 
      return FALSE; 
    }


  // Access immediately okay ?
   // look for WRITE in accesses
   TT(tsy_VL, TSL(lock)); 
   if (lock == WRITE)
     { 
       // search in mapping for timestamp of WRITE
       cursor = access.open_cursor();
       while ((access.get_info(cursor).get_am() != WRITING) ||
              (access.get_info(cursor).get_ta().get_type() != TIMESTAMP))
            access.to_succ(cursor);
       write_ts = access.get_info(cursor).get_ta().get_ts();
       access.close_cursor(cursor);
     }



  // search insert position in waiting queue
  pos     = 1;

  if (!queue.is_empty())
    { 
      // The queue is iterated to find the right position to insert the read
      // into the queue. The read is inserted either in front of a timestamp
      // request with a larger/younger timestamp or in front of a writelock
      // if the youngest timestamp transaction which has read the container
      // is younger then the requesting one. The reason for the second insertion
      // point is that there will possibly a conflict between the writelock and
      // a write corresponding to the youngest read and then there would be a
      // problem with the older timestamp requests in the queue.

      TT(tsy_VL, TI(queue.card()));
      found = FALSE;
      for (qpos = 1; (!found) && (qpos<=queue.card()); qpos++)
      { blockta = queue.get_nth(qpos);
        TBTA(blockta);
        if (((blockta.get_ta().get_type() == TIMESTAMP) &&
             (blockta.get_ta().get_ts()               >  ta.get_ts()))
           ||
            ((blockta.get_ta().get_type() == LOCKING) &&
             (blockta.get_am()            == WRITING) &&
             (self.get_rtm_list().get_nth(1).get_tm() > ta.get_ts())))
           { found = TRUE;
             pos  = qpos; }
      }
      if (!found)
         pos = queue.card()+1;
      TT(tsy_VL, TXT("pos/qpos"); TI(pos); TI(qpos); TB(found));
   
      // trace
      TT(tsy_VL, if (pos>1) 
                   { TXT("after:"); TBTA(queue.get_nth(pos-1));}
                 if (pos<queue.card())
                   { TXT("before:"); TBTA(queue.get_nth(pos)); } )
    }
  
   
  // access permitted ?
  if (((lock == WRITE) && (write_ts < ta.get_ts())) ||
       (lock == WRITELOCK) || (pos > 1))
    { // in the moment no access possible
      TT(tsy_L, TXT("NO ACCESS POSSIBLE NOW"));
      if (sm == TESTING)
        { self.set_error(SYNC_OK);
          TT(tsy_H, TXT("TESTING->FALSE"); T_LEAVE);     
          return FALSE;
        }

      // go to sleep and wait till it's on you
      do {
           TT(tsy_L, TXT("GO SLEEP"); TI(pos));
           ta.set_waits_for(SyncObj::make(self));
           self.enqueue (ta, READING, pos);
           ta.set_waits_for(SyncObj::make(NO_OBJECT));
           if (self.get_error() != SYNC_OK)
             { // deadlock occurred
               TT(tsy_H, TXT("SYNC_DEADLOCK"); TXT("FALSE"); T_LEAVE);     
               return FALSE; }
           pos  = 1; 
           lock = self.get_lock();
           TT(tsy_VL, TSL(lock));
         }
      while ((lock == WRITE) || (lock == WRITELOCK));


      // awake the next in the queue, when someone's there
      self.dequeue(); 
    }


  // insert into accesses
  TT(tsy_L, TXT("ACCESS"));
  blockta = BlockedTA::create(SYNC_CONTAINER, ta, READING);
  access.insert(ta.get_ts(), blockta);  
  TBTA(blockta);


  self.rtm_active(ta.get_ts());
  self.set_error(SYNC_OK);
  TT(tsy_H, TXT("TRUE"); T_LEAVE);
  return TRUE;
}

/* ================================================================ */
/* Method: TimestampSync::tso_write                                 */
/*                                                                  */
/* Error: SYNC_OK, SYNC_NOVALID_TS, SYNC_DEADLOCK                   */
/* ================================================================ */

void TimestampSync::tso_write (TimestampTA ta)

{ 
  T_PROC("TimestampSync::tso_write");
  TT(tsy_H, T_ENTER);
  TSOBJ(self);
  TTA(ta);

  sos_BlockList   queue   = self.get_queue();      // waiting queue
  BlockedTA       blockta;                        
  sos_Bool        found;                     
  int             pos;                             // insert position in queue
  int             qpos;                            // position counter


  // has this container been reopened ?
  if ((self.get_accesses().is_key(ta.get_ts())) && 
      (self.get_accesses()[ta.get_ts()].get_am() == WRITING))
    { // container has been reopened and there is alreaady an entry
      // in accesses, so there is nothing to do
      self.set_error(SYNC_OK);
      return; }

 
  // Timestamp okay ?
  TT(tsy_VL, TI(self.get_wtm()); TI(ta.get_ts()); 
            TI(self.get_rtm_list().get_nth(1).get_tm()));
  if ((self.get_wtm() > ta.get_ts()) ||
      (self.get_rtm_list().get_nth(1).get_tm() > ta.get_ts()))
    { self.set_error(SYNC_NOVALID_TS);
      TT(tsy_H, TXT("SYNC_NOVALID_TS"); T_LEAVE);     
      return; 
    }

  
  // insert in waiting queue
  pos     = 1;

  if (!queue.is_empty())
    { 
      // A position is searched to insert the write into the queue. The write
      // is either inserted in front of a younger timestamp request or in front
      // of the first writelock from the front of the queue. The reason for the
      // second insertion point is that the read which corresponds to the write
      // will never allow the writelock to be admitted before the write is 
      // executed. But the write is never executed when it is inserted behind
      // the writelock. Problems with other tso-ta are avoided because of the
      // way of insertion of the reads of tso-ta.

      TT(tsy_VL, TI(queue.card()));
      found = FALSE;
      for (qpos = 1; (!found) && (qpos<=queue.card()); qpos++)
         { blockta = queue.get_nth(qpos);
           TBTA(blockta);
           if (((blockta.get_ta().get_type() == TIMESTAMP) &&
                (blockta.get_ta().get_ts()   >  ta.get_ts()))
             ||
               ((blockta.get_ta().get_type() == LOCKING) &&
                (blockta.get_am()            == WRITING)))
              { found = TRUE;
                pos   = qpos; }
         }
      if (!found)
         pos = queue.card()+1;
      TT(tsy_VL, TXT("pos/qpos"); TI(pos); TI(qpos); TB(found));

      // Trace
      TT(tsy_VL, if (pos>1)
                   { TXT("after:"); TBTA(queue.get_nth(pos-1));}
                 if (pos<queue.card())
                   { TXT("before:"); TBTA(queue.get_nth(pos)); } )
    }


  // need to block the ta ?
  if ((self.get_lock() != NOLOCK) || (pos > 1))
    {  
      // go to sleep and wait till it's on you
      TT(tsy_L, TXT("NO ACCESS POSSIBLE NOW"));
      do {
           TT(tsy_L, TXT("GO SLEEP"); TI(pos));
           ta.set_waits_for(SyncObj::make(self));
           self.enqueue (ta, WRITING, pos);
           ta.set_waits_for(SyncObj::make(NO_OBJECT));
           if (self.get_error() != SYNC_OK)
             { // deadlock occurred
               TT(tsy_H, TXT("SYNC_DEADLOCK"); T_LEAVE);
               return;  }
           pos = 1; 
           TT(tsy_VL, TSL(self.get_lock()));
         }
      while (self.get_lock() != NOLOCK);     
    }


  // insert into accesses
  blockta  = self.get_accesses()[ta.get_ts()];
  blockta.set_am(WRITING);
  self.get_accesses().remove(ta.get_ts());
  self.get_accesses().insert(ta.get_ts(), blockta);  
  TBTA(blockta);


  // allow write access
  self.set_lock(WRITE);
  
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: TimestampSync::tso_commit                                */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void TimestampSync::tso_commit (TimestampTA ta)

{
  T_PROC("TimestampSync::tso_commit");
  TT(tsy_H, T_ENTER);
  TSOBJ(self);
  TTA(ta);

  sos_Container  ct;

  
  // commit container
  ct = self.get_soscontainer();
  TT(tsy_VL, TCONTAINER(ct));
  if (self.get_accesses()[ta.get_ts()].get_am() == WRITING)
    { 
      ct.write_close();
      self.tso_release_n_wakeup(ta, TRUE);
      self.set_wtm(ta.get_ts());
      TT(tsy_VL, TI(self.get_wtm()));
    }
   else
    {
      ct.read_close();
      self.tso_release_n_wakeup(ta, TRUE);
    }

  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: TimestampSync:tso_abort                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void TimestampSync::tso_abort (TimestampTA ta)

{
  T_PROC("TimestampSync::tso_abort");
  TT(tsy_H, T_ENTER);
  TSOBJ(self);
  TTA(ta);

  sos_Container   ct;


  // commit container
  ct = self.get_soscontainer(); 
  TT(tsy_VL, TCONTAINER(ct));
  ct.read_close();

  // repair accesses
  self.tso_release_n_wakeup(ta, FALSE);

  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: TimestampSync:tso_release_n_wakeup                       */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */
 
void TimestampSync::tso_release_n_wakeup (TimestampTA ta, sos_Bool rtmcom)
 
{
  T_PROC("TimestampSync::tso_release_n_wakeup");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TB(rtmcom));
  TSOBJ(self);
  TTA(ta);

  sos_AccessMap access = self.get_accesses();
  sos_Bool      change;
 
 
  // correct rtm_list
  // if rtmcom = TRUE, the rtm is deactivated, otherwise it is removed
  self.rtm_inactive(ta.get_ts(), rtmcom);
 
  // correct accesses
  if (access.is_key(ta.get_ts()))
    { if (access[ta.get_ts()].get_am() == WRITING)
        { self.set_lock(NOLOCK);
          change = TRUE;
        }
       else
        { if ((!self.get_queue().is_empty()) &&
              (self.get_queue().get_nth(1).get_ta().get_type() == LOCKING))
            change = TRUE;
           else
            change = FALSE;
        }
      TT(tsy_VL, TXT("access has to be changed ?"); TB(change);
                 TSL(self.get_lock()));
      access.remove(ta.get_ts());
      if (change)
         self.dequeue();
    }
 
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}


/* ================================================================ */
/* Method: TimestampSync::rtm_active                                */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void TimestampSync::rtm_active (sos_Timestamp ts)

{
  T_PROC("TimestampSync::rtm_active");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(self.get_soscontainer()); TI(ts));

  sos_TimeList    list = self.get_rtm_list();   // read timemarks
  sos_Bool        found;
  int             pos;


  // insert _TS of _TA in RTM-list
  found = FALSE;
  for (pos = 1; (!found) && (pos <= list.card()); pos++)
     { if (list.get_nth(pos).get_tm() < ts)
           found = TRUE;
     }

  if (found) pos--;
  TT(tsy_VL, TB(found); TI(pos));
  list.insert(pos, Timemark::create(SYNC_CONTAINER, ts));
 
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;       
}

/* ================================================================ */
/* Method: TimestampSync::rtm_inactive                              */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

 void TimestampSync::rtm_inactive (sos_Timestamp ts, sos_Bool com)

{
  T_PROC("TimestampSync::rtm_inactive");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(self.get_soscontainer()); TI(ts); TB(com));

  sos_TimeList   list =self.get_rtm_list();   // read timemarks
  sos_Bool       found;
  int            pos;


  // search list for rtm
  found = FALSE;
  for (pos = list.card()-1; (!found) && (pos>=1); pos--)
     { if (list.get_nth(pos).get_tm() == ts)
          found = TRUE; }
  pos++;
  TT(tsy_VL, TB(found); TI(pos));


  if (com)
    { // commit transaction
      list.get_nth(pos).set_active(FALSE); }
   else
    { // abort transaction
      list.remove(pos);
       pos--;
    }

 
  // remove inactive transactions from end of list
  while ((pos>=1) && (list.get_nth(pos  ).get_active() == FALSE)
                  && (list.get_nth(pos+1).get_active() == FALSE))
       { if ((pos+1) != list.card()) list.remove(pos+1);
         pos--; }

  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: ExtSyncObj (Timestamp part)                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: ExtSyncObj::tso_read                                     */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED                                  */
/* ================================================================ */

sos_Bool ExtSyncObj::tso_read (TimestampTA ta, sos_Sync_mode sm)

{
  T_PROC("ExtSyncObj::tso_read");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(self.get_soscontainer());TSM(sm));
  TT(tsy_VL, TI(ta.get_hostproc_id()));

  sos_Bool  okay;
  sos_Error err;
  

  // execute read
  okay = self.TimestampSync::tso_read (ta, sm);
  err  = self.get_error();
 
  if ((sm == WAITING) && ((err == SYNC_NOVALID_TS) || (err == SYNC_DEADLOCK)))
    {
      // transaction has to be aborted because of TSO or DEADLOCK
      ta.intern_abort();
      self.set_error(SYNC_TA_ABORTED); 
      TT(tsy_H, TXT("SYNC_TA_ABORTED -> FALSE"); T_LEAVE);
      return FALSE;
    }
  
  TT(tsy_H, T_LEAVE);
  return okay;
}  

/* ================================================================ */
/* Method: ExtSyncObj::tso_write                                    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED                                  */
/* ================================================================ */

void ExtSyncObj::tso_write (TimestampTA ta)

{
  T_PROC("ExtSyncObj::tso_write");
  TT(tsy_H, T_ENTER);

  TT(tsy_L, TI(self.get_soscontainer()));
  TT(tsy_VL,  TI(ta.get_hostproc_id()));

  BlockedTA  blockta;
  sos_Error  err;


  // test whether container was reopened
  blockta = self.get_accesses()[ta.get_ts()];
  TBTA(blockta);
  if (VALID(blockta) && (blockta.get_am() == WRITING))
    { // everything is done
      self.set_error(SYNC_OK);
      TT(tsy_H, T_LEAVE);
      return; }
  

  // execute write
  self.TimestampSync::tso_write(ta);
  err = self.get_error();
  if ((err == SYNC_NOVALID_TS) || (err == SYNC_DEADLOCK))
    {
      // transaction has to be aborted because of TSO
      ta.intern_abort();
      self.set_error(SYNC_TA_ABORTED);
      TT(tsy_H, TXT("SYNC_TA_ABORTED"));
    }
  TT(tsy_H, T_LEAVE);
}

