/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : Optimistic Synchronisation & Transaction                         */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 08.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */
#include "obst_progstd.h"
#include "knl_use.h"
#include "ext_sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "trc_tsy.h"
#include "sync_trc.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: OptimisticSync                                                */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 08.04.1992              */
/*                                                                           */
/* ========================================================================= */

void OptimisticSync::opt_read (OptimisticTA)
{
}

void OptimisticSync::opt_write (OptimisticTA)
{
}


/* ================================================================ */
/* Method: OptimisticSync::opt_commit                               */
/*                                                                  */
/* Errors:  SYNC_OK                                                 */
/* ================================================================ */

void OptimisticSync::opt_commit (OptimisticTA ta, ta_Access_mode am)
{
  T_PROC("OptimisticSync::opt_commit");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TTM(am));
  TSOBJ(self);
  TTA(ta);

  sos_Container ct;


  ct = self.get_soscontainer();

  // commit container  
  if (am == WRITE_AC)
      ct.write_close();
   else
      ct.read_close();

  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}
  
/* ================================================================ */
/* Method: OptimisticSync::opt_abort                                */
/*                                                                  */
/* Errors:  SYNC_OK                                                 */
/* ================================================================ */

void OptimisticSync::opt_abort (OptimisticTA ta)
{
  T_PROC("OptimisticSync::opt_abort");
  TT(tsy_H, T_ENTER);
  TSOBJ(self);
  TTA(ta);

  sos_Container ct;


  ct = self.get_soscontainer();

  // abort container  
  ct.read_close();

  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}


/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: ExtSyncObj (Optimistic part)                                  */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 08.04.1992              */
/*                                                                           */
/* ========================================================================= */

void ExtSyncObj::opt_read (OptimisticTA)
{
  self.set_error(SYNC_OK);
}


void ExtSyncObj::opt_write (OptimisticTA)
{
  self.set_error(SYNC_OK);
}


/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: OptimisticTA                                                  */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 08.04.1992              */
/*                                                                           */
/* ========================================================================= */


void OptimisticTA::local_initialize (OptimisticTA ta)
{
    T_PROC("OptimisticTA::local_initialize");
    TT(tsy_H, T_ENTER);

    // set type of ta
    ta.set_type (OPTIMISTIC);

    ta.set_commitset (sos_SOSet::create (SYNC_CONTAINER));

    // get root_obj of SYNC_CONTAINER;
    SyncCont syct = SyncCont::get_root();
 
    // put ta in OptTa mapping
    syct.insert_TA(ta);
    if (syct.get_error() != SYNC_OK)
       ta.set_error(syct.get_error());

    TT(tsy_H, T_LEAVE);
    return;
}

//--------------------------------------------------------------------

void OptimisticTA::local_finalize (OptimisticTA ta)
{
    T_PROC("OptimisticTA::local_finalize");
    TT(tsy_H, T_ENTER);

    ta.get_commitset().destroy ();
 
    // get root_obj of SYNC_CONTAINER;
    SyncCont syct = SyncCont::get_root();
 
    // remove ta from OptTa mapping
    syct.remove_TA(ta);
 
    TT(tsy_H, T_LEAVE);
    return;
}

/* ================================================================ */
/* Method: OptimisticTA::validateCommit                             */
/*                                                                  */
/* Errors: SYNC_OK                                                  */
/* ================================================================ */

sos_Bool OptimisticTA::validateCommit ()
{
    T_PROC("OptimisticTA::validateCommit()");
    TT(tsy_H, T_ENTER);

    SyncCont        sc;
    Transaction     ta;
    Transaction     opta;
    sos_HostProc_Id pid;
    sos_Timestamp   acts;
    sos_SOSet       readset;
    sos_SOSet       intersection;
    sos_PidSet      pid_set;
 
 
    // Note:
    // - readset and writeset should remain unchanged
 
   // Test for an intersection between the readset and the commitset
 
    // implementation detail:
    // - readset contains only SyncObj, which were opened for reading only
    // - all Syncobj, which are part of the writeset, are not part of the
    //   readset, but in spite of that, they have been read
 
    TT(tsy_VL, TXT("Commitset:"));
    TSOSET(self.get_commitset());
 
    if (!self.get_commitset().is_empty())
     { TT(tsy_L, TXT("--> BOCC"));
 
       // the complete readset contains the read- and the writeset
       readset = sos_SOSet::clone (self.get_readset(), TEMP_CONTAINER);
       readset += self.get_writeset();
       TT(tsy_VL, TXT("complete Readset:"));
       TSOSET(readset);
 
       // intersect the complete readset with the commitset
       // "readset" contains intersection set
       readset *= self.get_commitset();
       if (!readset.is_empty ())
        { // intersection is empty => invalid
 
          readset.clear ();
 
          self.set_error(SYNC_OK);
          TT(tsy_H, TB(FALSE); T_LEAVE);
          return FALSE;
        }
     }
    else TT(tsy_L, TXT("--> no BOCC"));
 
   // Tests if something shall be modified, what other transaction have
   // already read
 
    TT(tsy_VL, TXT("Writeset:"));
    TSOSET(self.get_writeset());
 
    if (!self.get_writeset().is_empty())
     { // intersect the writeset with the read- and writesets of all other
       // (active) transactions, which don't use the synchronization method
       // OPTIMISTIC. The intersection with the optimistic ta's is done by
       // distributing the writeset to their commitsets.
       TT(tsy_L, TXT("--> FOCC"));
 
       sc      = SyncCont::get_root ();
       pid_set = sos_PidSet::create(TEMP_CONTAINER);
 
       TT(tsy_L, TXT("Cut with Readsets of other transactions :"));
       agg_iterate_association (sc.get_ta_tab(), pid, ta)
       {
         TTA(ta);
 
         if (ta.get_type() != OPTIMISTIC)
         {
           // the complete readset contains the readset and the writeset
           intersection = sos_SOSet::clone (ta.get_readset(), TEMP_CONTAINER);
           intersection += ta.get_writeset();
           TT(tsy_VL, TXT("vollstaendiger Readset:"));
           TSOSET(intersection);
 
           intersection *= self.get_writeset();
           TT(tsy_VL, TXT("intersection:"));
           TSOSET(intersection);
 
           if (!intersection.is_empty())
           {  // intersection not empty => invalid
 
              intersection.clear();
 
              if (sc.process_check(ta.get_hostproc_id()))
               { // process still alive
                 self.set_error(SYNC_OK);
                 TT (tsy_H, TXT("intersection not empty => "); TB(FALSE); T_LEAVE);
                 return FALSE;
               }
              else
               { // process is aborted
                 pid_set.insert(ta.get_hostproc_id());
               }
           }
         }
       } agg_iterate_association_end (sc.get_ta_tab(), pid, ta);
 
       // remove aborted process with locking or timestamp ta's
       if (!pid_set.is_empty())
         {  agg_iterate ( pid_set, pid )
             {
                sc.process_remove(pid);
             }
            agg_iterate_end (pid_set, pid);
            pid_set.clear();
         }
 
 
       // all intersections empty => valid
       // distribute writeset on other optimistic transactions
       TT(tsy_L, TXT("Distribute Writeset on other optimistic transactions :"));
       acts = sc.get_ts() - SOS_OLD_TA_TS;
       agg_iterate_association (sc.get_opta_tab(), pid, opta)
       {
         TT(tsy_VL, TI(pid); TTT(opta.get_type()));
 
         if (opta != self)
           { // different optimistic ta => distribute writeset
             if ((opta.get_ts() < acts) &&
                (!sc.process_check(pid)))
              { // process of old optimistic ta is aborted
                pid_set.insert(pid);
              }
             else
              { // distribute
                OptimisticTA::make(opta).get_commitset() += self.get_writeset();
                TSOSET(OptimisticTA::make(opta).get_commitset());
              }
           }
       } agg_iterate_association_end (sc.get_opta_tab(), pid, opta);
 
       // remove aborted process with optimistic ta's
       if (!pid_set.is_empty())
        { agg_iterate ( pid_set, pid )
           {
             sc.process_remove(pid);
           }
          agg_iterate_end (pid_set, pid);
          pid_set.clear();
        }
     }
    else TT(tsy_L, TXT("--> no FOCC"));
 
    self.set_error(SYNC_OK);
    TT (tsy_H, TB(TRUE); T_LEAVE);
    return TRUE;
}

/* ================================================================ */
/* Method: OptimisticTA::commit                                     */
/*                                                                  */
/* Errors: SYNC_OK, SYNC_TA_ABORTED                                 */
/* ================================================================ */

void OptimisticTA::commit ()
{
    T_PROC("OptimisticTA::commit()");
    TT(tsy_H, T_ENTER);
    TTA(self);

    SyncObj        sobj;
    sos_SOSet      opset;
    sos_Bool       okay;
    sos_Cursor     cursor;
    ta_Access_mode opac;
    int            i;

    // close all open container
    opset = sos_SOSet::clone(self.get_openset(),TEMP_CONTAINER);
    agg_iterate(opset, SyncObj so)
      {
        so.release ();       // free SyncObj and delete it from openset 
      }
    agg_iterate_end (opset, so);
    opset.clear();


    // execute commit validation 
    if (self.validateCommit ())
      {   
        // commit containers and clear sets
        for (i=0; i<=1; i++)
        { if (i==0)
            { opset = self.get_readset();
              opac  = READ_AC;
            }
          else
            { opset = self.get_writeset();
              opac  = WRITE_AC;
            }
 
          if (!opset.is_empty())
           {
             // iterate set
             TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset":"writeset")));
             okay   = TRUE;
             cursor = opset.open_cursor();
             while ((opset.is_valid(cursor)) && (okay))
               { sobj = opset.get(cursor);
                 TSOBJ(sobj);
 
                 // commit container
                 OptimisticSync::make(sobj).opt_commit(self, opac);
 
                 okay = opset.to_succ(cursor);
               }
             opset.close_cursor(cursor);
             opset.clear();
           }
        }
 
        // clear commitset
        self.get_commitset().clear();

        // set status
        self.set_status(COMMITTED); 
        TT(tsy_M, TTS(COMMITTED));

        // end commit
        self.set_error (SYNC_OK);
      }
     else
      {
        // no success in validating
        self.abort ();
        self.set_error(SYNC_TA_ABORTED);
        TT(tsy_H, TXT("SYNC_TA_ABORTED")); 
      } 

    TT (tsy_H, T_LEAVE);
    return;
}


/* ================================================================ */
/* Method: OptimisticTA::abort                                      */
/*                                                                  */
/* Errors: SYNC_OK                                                  */
/* ================================================================ */

void OptimisticTA::abort ()
{
    T_PROC("OptimisticTA::abort()");
    TT(tsy_H, T_ENTER);
    TTA(self);

    SyncObj    sobj;
    sos_SOSet  opset;
    sos_Bool   okay;
    sos_Cursor cursor;
    int        i;

    // abort containers and clear sets
    for (i=0; i<=1; i++)
     {
       if (i==0)
          opset = self.get_readset();
         else
          opset = self.get_writeset();
 
       if (!opset.is_empty())
        {
          TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset" : "writeset")));
         okay   = TRUE;
          cursor = opset.open_cursor();
          while ((opset.is_valid(cursor)) && (okay))
           { sobj = opset.get(cursor);
             TSOBJ(sobj);
 
             // abort container
             OptimisticSync::make(sobj).opt_abort(self);
 
             okay = opset.to_succ(cursor);
           }
          opset.close_cursor(cursor);
          opset.clear();
        }
     }
 
    // remove container from openset and commitset
    self.get_openset().clear();
    self.get_commitset().clear();

    // set status
    self.set_status(ABORTED);
    TT(tsy_M, TTS(ABORTED));

    self.set_error (SYNC_OK);
    TT (tsy_H, T_LEAVE);
    return;
}


/* ================================================================ */
/* Method: OptimisticTA::extern_abort                               */
/*                                                                  */
/* Errors: SYNC_OK                                                  */
/* ================================================================ */

void OptimisticTA::extern_abort()
{
  T_PROC("OptimisticTA::extern_abort");
  TT(tsy_H, T_ENTER)
  TTA(self);
 
  self.get_readset().clear();
  self.get_writeset().clear();
  self.get_openset().clear();
  self.get_commitset().clear();
 
  self.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: OptimisticTA::intern_abort                               */
/*                                                                  */
/* Errors: SYNC_OK                                                  */
/* ================================================================ */

void OptimisticTA::intern_abort()
{
 // Dummy
}
