/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module cfe_generic                                        Juergen Uhl (ju)
//
// **************************************************************************
// OBST schema compiler front end (cfe)
// **************************************************************************
//
// tracing conventions: see trc_cfe.h

#define OBST_IMP_STDCONST
#include "obst_stdinc.h"

#include "cfe.h"

// LOCAL

LOCAL void cfe_fill_inst_class_type
	  (sos_Class_type, sos_Class_type, sos_Type_descr_List);
LOCAL void copy_offsets_and_size
	  (sos_Class_type ct1, sos_Class_type ct2);
LOCAL sos_Type_descr subst_actual
          (sos_Gen_param, sos_Gen_param_List, sos_Type_descr_List);
LOCAL void subst_methods
          (sos_Method_List, sos_Class_type,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Method subst_method 
	  (sos_Method, sos_Class_type, sos_Method_table, sos_Gen_param_List,
           sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Bool must_subst_components
          (sos_Comp_descr_List,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Bool must_subst_params
          (sos_Param_List,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Param_List subst_params
          (sos_Param_List, sos_Gen_param_List,
	   sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Bool must_subst_type_descr
          (sos_Type_descr,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Type_descr subst_type_descr
          (sos_Type_descr,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Bool must_subst_type_descrs
          (sos_Aggregate,
           sos_Gen_param_List, sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_Aggregate subst_type_descrs
          (sos_Aggregate, sos_Gen_param_List,
	   sos_Type_descr_List, sos_Type_descr_List);
LOCAL sos_String instantiation_name
	  (sos_String, sos_Type_descr_List, sos_Bool = TRUE);


EXPORT sos_Bool cfe_contains_formal_gen_params (sos_Type_descr_List agpl)
// Checks, whether the list contains an element of type sos_Gen_param.
{
   T_PROC ("cfe_contains_formal_gen_params");
   TT (cfe_H, T_ENTER);

   sos_Bool result = FALSE;

   agg_iterate (agpl, sos_Type_descr agp)
      if (agp.has_type(sos_Gen_param_type))
      {
	 result = TRUE;
	 break;
      }
   agg_iterate_end (agpl, agp);

   TT (cfe_H, T_LEAVE);
   return result;
}

EXPORT sos_Type_descr cfe_lookup_generic_instantiation
		 			(sos_String          gen_class_name,
		  		 	 sos_Type_descr_List agpl,
		  		 	 sos_Import_mode     imp_mode,
		  		 	 sos_Type_descr_List type_decls)
/*
 * agpl will be destroyed, if not stored somewhere.
 */
{  T_PROC ("cfe_lookup_generic_instantiation")
   TT (cfe_H, T_ENTER);

   sos_Type_descr result;
   sos_Type_descr t = cfe_schema.lookup_type (gen_class_name, imp_mode);

   if (t == NO_OBJECT)
      result = cfe_create_and_enter_forward (gen_class_name, agpl);
   else if (NOT t.has_type(sos_Class_type_type))
   {
      cfe_error (err_USE, err_CFE_CLASS_EXPECTED, gen_class_name);
      result = ERROR_TYPE;
   }
   else
   {  sos_Class_type instct,
      		     genct = sos_Class_type::make(t);

      if (genct.get_methods() == NO_OBJECT) // forward ?
	 result = cfe_create_and_enter_forward (gen_class_name, agpl);
      else
      {  sos_Bool destroy_params = TRUE;
	 sos_Gen_param_List fgpl = genct.get_formal_gen_params();
	 if (fgpl == NO_OBJECT)
	 {
	    cfe_error (err_USE,err_CFE_GENERIC_TYPE_EXPECTED, gen_class_name);
	    result = ERROR_TYPE;
	 }
	 else if (fgpl.card() != agpl.card())
	 {  cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH, gen_class_name);
	    result = ERROR_TYPE;
	 }
	 else
	 {  sos_Bool has_formals = cfe_contains_formal_gen_params(agpl);

	    instct = cfe_create_inst_class_type (gen_class_name, agpl,
						 (sos_Bool)(NOT has_formals),
						 type_decls);
	    cfe_fill_inst_class_type (genct, instct, type_decls);

   	    if (agpl == instct.get_actual_gen_params())
	       destroy_params = FALSE;

	    if (has_formals)
	    {  sos_Generic_instantiation gi
		      = cfe_get_instantiation (genct, instct, agpl);
	       result = gi;
	       if (destroy_params AND agpl == gi.get_act_gen_params())
                  destroy_params = FALSE;
	    }
	    else
	       result = instct;
	 }
	 if (destroy_params AND agpl != NO_OBJECT)
	    agpl.destroy();
      }
   }
   TT (cfe_H, T_LEAVE);
   return result;
}


EXPORT void cfe_complete_forward_insts ()
{
   T_PROC ("cfe_complete_forward_insts")
   TT (cfe_H, T_ENTER);

   agg_iterate_association (cfe_forward_tab, sos_String name,
			    sos_Class_type_List ctl)
   {  // Get first element of the class type list, which is the generic
      // class in case of a forward generic instantiation.
      sos_Class_type gen_class = sos_Class_type::make(NO_OBJECT);
      agg_iterate (ctl, sos_Class_type ct)
      {
	 if (gen_class == NO_OBJECT) // first element can be generic class
	 {
	    gen_class = ct;
	    if (!gen_class.is_generic_class ())
	       break;                // non-generic classes are ignored
	 }
         else
	    if (gen_class.get_methods() == NO_OBJECT) //class not yet defined?
	    {  // ERROR: cannot complete the forward instantiations!
	       smg_String s = gen_class.get_name();
	       cfe_error (err_USE, err_CFE_UNKNOWN_GENERIC_CLASS,
			     	   s.make_Cstring (SMG_BORROW));
	    }
	    else
	       cfe_fill_inst_class_type (gen_class, ct, cfe_types);
      }
      agg_iterate_end (ctl, ct);
   }
   agg_iterate_association_end (cfe_forward_tab, name, ctl);

   TT (cfe_H, T_LEAVE);
}


EXPORT void cfe_check_generic_instantiations (sos_Type_descr_List tdl)
{  
   // Elements of list are expected to be sos_Class_type's or
   // sos_Generic_instantiation's.
   T_PROC ("cfe_check_generic_instantiations");
   TT (cfe_H, T_ENTER);

   agg_iterate (tdl, sos_Type_descr td)
   {  sos_Class_type gen_class;
      if (td.has_type (sos_Class_type_type))
	 gen_class = sos_Class_type::make(td).get_generic_class();
      else 
	 gen_class = sos_Generic_instantiation::make(td).get_gen_class();

      if (gen_class != NO_OBJECT) // don't check for instantiations, of which
				  // the generic class was undefined!
      {  sos_Type_descr_List agpl;
	 sos_Gen_param_List  fgpl = gen_class.get_formal_gen_params();

	 if(td.has_type (sos_Class_type_type))
	    agpl = sos_Class_type::make(td).get_actual_gen_params();
	 else
	    agpl = sos_Generic_instantiation::make(td).get_act_gen_params();

	 int comp;
	 agg_iterate_double (agpl, sos_Type_descr agp,
			     fgpl, sos_Gen_param  fgp, comp)
	 {  if (fgp.get_super_type() != NO_OBJECT AND
		NOT agp.make_type().is_derived_from (fgp.make_type()))
	    {  smg_String s = smg_String (gen_class.get_name()) + "<...," +
			      agp.make_type().get_name() + ",...>";
	       cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH,
			     	   s.make_Cstring (SMG_BORROW));
	    }
	 }
	 agg_iterate_double_end (agpl, agp, fgpl, fgp, comp);
      }
   }
   agg_iterate_end (tdl, tn);

   TT (cfe_H, T_LEAVE);
}


EXPORT sos_Gen_param cfe_lookup_gen_params (sos_String         name, 
					    sos_Gen_param_List gpl)

// Search for a generic parameter 'name' in the generic parameter list 'gpl'.
// Return the generic parameter found if any or NO_OBJECT otherwise.

{  T_PROC ("cfe_lookup_gen_params")
   TT (cfe_H, T_ENTER);

   sos_Gen_param result = sos_Gen_param::make (NO_OBJECT);

   agg_iterate (gpl, sos_Gen_param gp)
   {  if (name.equal (gp.get_name()))
      {  result = gp;
	 break;
      }
   }
   agg_iterate_end (gpl, gp);

   TT (cfe_H, T_LEAVE);
   return result;
}


LOCAL sos_String instantiation_name (sos_String          gen_class_name,
				     sos_Type_descr_List act_params,
				     sos_Bool		 make_tp /* = TRUE */)
{
   /* if make_tp is set (the default), then make_type() is applied to
      the type description, even if it is a typedef type. Otherwise
      for typedef types the defined name is taken into the instantiation
      name. This is useful for generating the second name for a generic
      instantiation, in which typedef names are kept. For an example see
      comments in cfe_create_inst_class_type.				 */

   T_PROC ("instantiation_name")
   TT (cfe_M, T_ENTER);

   smg_String s;

   agg_iterate (act_params, sos_Type_descr td)
   {  if (make_tp || !td.has_type (sos_Typedef_type_type))
	 s += td.make_type().get_name();
      else
	 s += sos_Typedef_type::make(td).get_name();
      s += "_";
   }
   agg_iterate_end (act_params, td);
   s += gen_class_name;

   TT (cfe_M, T_LEAVE);
   return s.make_String (TEMP_CONTAINER);
}

sos_Bool cfe_is_universal (sos_Class_type gen_t, sos_Type_descr_List tdl)
{
   T_PROC ("cfe_is_universal")
   TT (cfe_H, T_ENTER);

   sos_Gen_param_List gpl = gen_t.get_formal_gen_params();

   sos_Bool is_universal = TRUE;
   		   // each `normal` class is its own most general instantiation

   if(gen_t.get_formal_gen_params() != NO_OBJECT) // generic class ?
   {
      // length of parameter lists need not be compared, since the compiler
      // did it when encountering the generic instantiation.
      int c;
      agg_iterate_double (gpl, sos_Gen_param gp, tdl, sos_Type_descr td, c)
	 if (NOT gp.make_type().equal (td.make_type()))
	 {  is_universal = FALSE;
	    break;
	 }
      agg_iterate_double_end (gpl, gp, tdl, td, c);
   }
   TT (cfe_H, T_LEAVE);
   return is_universal;
}

EXPORT void cfe_set_generic_stuff(sos_Class_type     ct,
				  sos_Gen_param_List fgpl)
// Enters the formal generic parameter list into the classtype ct
// and sets the actual generic parameter list to NO_OBJECT.
// If the generic class was instantiated forward, the cardinality of
// the generic parameter list is compared.
{
   T_PROC ("cfe_set_generic_stuff");
   TT (cfe_M, T_ENTER);

   sos_Gen_param_List old_fgpl = ct.get_formal_gen_params();

   if (old_fgpl != NO_OBJECT)  // Was a generic class used forward ?
	// Then the temporary fgpl must be destoyed, since it's in cfe_cnt.
      old_fgpl.destroy();
   ct.set_formal_gen_params (fgpl);
   ct.set_actual_gen_params (sos_Type_descr_List::make(NO_OBJECT));

   if (fgpl == NO_OBJECT)
   {  ct.set_generic_class (sos_Class_type::make(NO_OBJECT));
      ct.set_root_class (ct);
   }
   else
   {  sos_Class_type_List ctl = cfe_forward_tab [ct.get_name()];
      if (ctl != NO_OBJECT) // class was instantiated forward
      {
	 sos_Gen_param_List forward_fgpl = ctl.get_nth (1)
					      .get_formal_gen_params ();
	 if (forward_fgpl == NO_OBJECT)
	    // error, forward use was no instantiation
	    cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH,
			  	ct.get_name().make_Cstring());
	 else 	// was instantiation, then check cardinalities
	    if (fgpl.card() != forward_fgpl.card())
	       cfe_error (err_USE, err_CFE_GEN_PARAM_MISMATCH,
			     	   ct.get_name().make_Cstring());
      }

      sos_Type_descr_List act_params = sos_Type_descr_List::create(cfe_cnt);
      agg_iterate(fgpl, sos_Gen_param gp)
         act_params.append (sos_Type_descr::make (gp.make_type()));
      agg_iterate_end (fgpl, gp);

      ct.set_generic_class (ct);
      ct.set_root_class (
	  cfe_create_inst_class_type(ct.get_name(),act_params,TRUE,cfe_types));
   }
   TT (cfe_M, T_LEAVE);
}


EXPORT void cfe_create_universal_inst (sos_Class_type ct)
{
   T_PROC ("cfe_create_universal_inst")
   TT (cfe_H, T_ENTER);

   sos_Gen_param_List fgpl = ct.get_formal_gen_params();

   if (fgpl != NO_OBJECT) // generic class, look for universal instantiation!
   {
      sos_Class_type inst_class = ct.get_root_class();

      // Universal instantiation used forward? Then fill it in now.
      if (inst_class.get_methods() == NO_OBJECT)
	 cfe_fill_inst_class_type (ct, inst_class, cfe_types);
   }

   TT (cfe_H, T_LEAVE);
}


EXPORT sos_Class_type cfe_create_inst_class_type
			 (sos_String          gen_class_name,
			  sos_Type_descr_List act_params,
			  sos_Bool	      may_delete_params,
			  sos_Type_descr_List type_decls)

/* If the instantiation was C<T1,...,Tn> then a new class type with
   name T1'_..._Tn'_C is searched (created if not found), where Ti' is the 
   base name of type Ti (i.e typedef's are evaluated (see function
   'instantiation_name')). The created class type is declared to be forward
   by setting methods to NO_OBJECT. This is important for the function
   cfe_fill_inst_class_type to work properly on this class. 
   The class type is entered into the type table and the type list, if it had
   to be created.
   Note that neither the superclasses nor the methods etc. are initialized.
   This has to be done with cfe_fill_inst_class_type, if the generic class
   is fully declared.
*/ 

{  T_PROC ("cfe_create_inst_class_type")
   TT (cfe_M, T_ENTER);

   sos_Class_type aux_c;
   sos_String	  visible_name;
   sos_String     base_name = instantiation_name (gen_class_name, act_params);
   sos_Type_descr tp        = cfe_schema.lookup_type (base_name, sos_IMP_TYPE);

   if (tp == NO_OBJECT)
   {  sos_Bool must_replace = FALSE;

      sos_Type_descr_List tdl = sos_Type_descr_List::create (TEMP_CONTAINER);
      agg_iterate (act_params, sos_Type_descr td)
	 sos_Type td_tp = td.make_type();
         tdl.append (sos_Type_descr::make (td_tp)); // use only `real` types
         if (td_tp != td)
	    must_replace = TRUE;
      agg_iterate_end (act_params, td);

      if (must_replace)
      {  visible_name = instantiation_name (gen_class_name, act_params, FALSE);

	 if (may_delete_params)
	    act_params.destroy();
	 
	 act_params  = sos_Type_descr_List::create (cfe_cnt);
	 act_params += tdl;
      }
      else
	 visible_name = base_name;

      tdl.destroy();

      sos_String bnm = sos_String::copy (base_name, cfe_cnt);

      aux_c = sos_Class_type::create (cfe_cnt);
      // Mark aux_c to be forward by setting methods to NO_OBJECT.
      aux_c.set_methods(sos_Method_table::make(NO_OBJECT));
      aux_c.set_name(bnm);
      aux_c.set_actual_gen_params(act_params);
      aux_c.set_formal_gen_params(sos_Gen_param_List::make(NO_OBJECT));
      cfe_type_tab.insert (bnm, aux_c);
      type_decls.append (aux_c);
   }
   else if (tp.has_type (sos_Class_type_type))
   {  aux_c 	   = sos_Class_type::make (tp);
      visible_name = instantiation_name (gen_class_name, act_params, FALSE);
   }
   else // error
      aux_c = sos_Class_type::make (NO_OBJECT);

   if (aux_c != NO_OBJECT)
   {  if (    NOT base_name.equal (visible_name)
	  AND cfe_schema.lookup_type (visible_name) == NO_OBJECT)
      {
	 // In order to allow the use of 'visible_name' in the application
	 // generate a typedef of the form "typedef base_name visible_name;".
	 // Example:  OBST declarations
	 //
	 //            typedef sos_Int T;
	 //            class C1<X> { ... X ... };
	 //            class C2    { ... C1<T> ... }     generate
	 //
	 //            typedef sos_Int T;
	 //            class C1 { ... sos_Object ... };
	 //            class sos_Int_C { ... sos_Int ... };
	 //            typedef sos_Int_C T_C; !!
	 //            class C2 { ... T_C ... };

	 sos_Typedef_type aux_td = sos_Typedef_type::create (cfe_cnt);
	 sos_String       _nm    = sos_String::copy (visible_name, cfe_cnt);

	 aux_td.set_name (_nm);
	 aux_td.set_named_type (aux_c);

	 cfe_type_tab.insert (_nm, aux_td);
	 type_decls.append (aux_td);
      }
   }
   TT (cfe_M, T_LEAVE);
   return aux_c;
}


LOCAL sos_Super_class_List cfe_del_double_sc (sos_Super_class_List scl)
{
   /* The superclass list may contain some entries which are equal.
      The reason for this is an instantiation like the following:
	 class A <E> { ... }; class B <F, G> : A <F>, A <G> { ... };
      In this case an instantiation of B <X, X> leads to two equal
      superclasses, namely A <X>. This method deletes double superclasses
      from the provided superclass list. 
      The method is intended to be used in cfe_fill_inst_class_type to
      correct the super_classes component of the filled-in class.
      Precondition: The sos_Super_class_List has list_cursor=TRUE. */

   T_PROC ("cfe_del_double_sc");
   TT (cfe_H, T_ENTER);

   if (scl != NO_OBJECT)
   {
      err_assert (scl.get_list_cursor (), err_CFE_INTERNAL);
      sos_Cursor c = scl.open_cursor();
      if (scl.is_valid (c))
	 do
	 {
	    sos_Super_class sc1 = scl.get (c);
	    sos_Cursor d = scl.duplicate (c);
	    scl.to_succ (d);
	    if (scl.is_valid (d))
	       do
	       {
		  sos_Super_class sc2 = scl.get (d);
		  if (sc1.equal (sc2, EQ_STRONG))
		     scl.remove_at (d);
		  else
		     scl.to_succ (d);
	       } while (scl.is_valid (d));
	    scl.to_succ (c);
	 } while (scl.is_valid (c));
   }

   TT (cfe_H, T_LEAVE);
   return scl;
}


LOCAL void cfe_fill_inst_class_type (sos_Class_type      gen_class,
				     sos_Class_type      aux_c,
				     sos_Type_descr_List type_decls)

/* Fills in the class type of a generic instantiation by substituting the
   actual types of aux_c for the formal parameters within the class.
   The class type is expected to be already inserted into the type table
   and appended to the type list. Note that the substitution of types
   in the declaration may lead to a recursive call of cfe_fill_inst_class_type
   due to the evaluation of class instances within the class declaration.
   Classes not being forward (with component <methods> holding a valid object) 
   are not filled in. */

{  T_PROC ("cfe_fill_inst_class_type")
   TT (cfe_M, T_ENTER);

   // Check, if aux_c is really not filled in
   if (aux_c.get_methods() == NO_OBJECT)
   {
      sos_Type_descr_List act_params  = aux_c.get_actual_gen_params ();
      sos_Method_table    tmp_methods = sos_Method_table::create(cfe_cnt);

      /* The methods component is initialized with a temporary method table.
	 This must be done, since otherwise cfe_fill_inst_class_type
	 might run into an endless recursion. The method table is
	 replaced afterwards by the function cfe_build_methodtable,
	 hence the tmp_methods can be destroyed at the function's end. */
      aux_c.set_methods(tmp_methods);

      sos_Gen_param_List gen_params = gen_class.get_formal_gen_params();
      sos_Comp_descr_List cdl = gen_class.get_components();
      if (must_subst_components(cdl, gen_params, act_params, type_decls))
      {  // new component list needed, copy the old elementwise.
	 sos_Comp_descr_List new_cdl = sos_Comp_descr_List::create (cfe_cnt);
	 agg_iterate(cdl, sos_Comp_descr cd)
	    new_cdl.append(cd);
	 agg_iterate_end(cdl, cd);
	 aux_c.set_components(new_cdl);
      }
      else
	 aux_c.set_components(cdl);

      aux_c.set_root_class(gen_class.get_root_class());
      aux_c.set_generic_class (gen_class);
      aux_c.set_super_classes (
	 cfe_del_double_sc (sos_Super_class_List::make(
	    subst_type_descrs(gen_class.get_super_classes(),
			      gen_params, act_params, type_decls))));

      if (gen_class.get_create_params() == NO_OBJECT)
	 aux_c.set_create_params (sos_Param_List::make (NO_OBJECT));
      else
      {  sos_Param_List new_pl
	    = subst_params (gen_class.get_create_params(),
			    gen_params, act_params, type_decls);
	 aux_c.set_create_params (new_pl);
      }
      cfe_set_super_closure (aux_c);

      /* offsets and size cannot be copied, since the superclosure may get
	 smaller. This happens, if two different instantiations of the same
	 generic class are provided as superclasses, which are merged into
	 the same instantiation in case of a fitting instantiation of the
	 current class. */
      cfe_set_offsets_and_size (aux_c, gen_class.get_local_size());
      aux_c.set_has_init_comps (gen_class.get_has_init_comps());
      aux_c.set_is_abstract (gen_class.get_is_abstract());

      sos_Type_List fl;
      if ((fl = gen_class.get_friends()) != NO_OBJECT)
         aux_c.set_friends (sos_Type_List::make(
			       subst_type_descrs (fl, gen_params, act_params,
	     					  type_decls)));

      cfe_init_methods (aux_c);

      subst_methods (gen_class.get_local_methods(), aux_c,
		     gen_params, act_params, type_decls);

      cfe_build_methodtable (aux_c);

      tmp_methods.destroy(); // was replaced by cfe_build_methodtable
   }

   TT (cfe_M, T_LEAVE);
}


LOCAL void copy_offsets_and_size (sos_Class_type ct1, sos_Class_type ct2)
{
   T_PROC ("copy_offsets_and_size")
   TT (cfe_M, T_ENTER);

   ct1.set_local_size (ct2.get_local_size());

   sos_Super_class_List scl1 = ct1.get_super_closure();
   sos_Super_class_List scl2 = ct2.get_super_closure();

   err_assert (scl1.card() == scl2.card(), err_CFE_SCRAMBLED_INSTANTIATION);

   int comp;
   agg_iterate_double (scl1, sos_Super_class sc1,
		       scl2, sos_Super_class sc2, comp)
   {
      err_assert (sc1.get_super_class().make_type().root()
		  == sc1.get_super_class().make_type().root(),
		  err_CFE_SCRAMBLED_INSTANTIATION);

      sc1.set_offset (sc2.get_offset());
   }
   agg_iterate_double_end (scl1, sc1, scl2, sc2, comp);

   ct1.set_object_size (ct2.get_object_size());

   TT (cfe_M, T_LEAVE);
}


LOCAL sos_Type_descr subst_actual (sos_Gen_param       gp,
			           sos_Gen_param_List  gpl,
			           sos_Type_descr_List tpl)
{  T_PROC ("subst_actual")
   TT (cfe_M, T_ENTER);

   sos_Type_descr td = sos_Type_descr::make(NO_OBJECT);

   int comp;
   agg_iterate_double (gpl, sos_Gen_param  gp1,
		       tpl, sos_Type_descr td1, comp)
   {  if (gp == gp1)
      {  td = td1; break; }
   }
   agg_iterate_double_end (gpl, gp1, tpl, td1, comp);

   err_assert (td != NO_OBJECT, "subst_actual");

   TT (cfe_M, T_LEAVE);
   return td;
}


LOCAL sos_Method subst_method (sos_Method          old,
			       sos_Class_type      aux_c,
			       sos_Method_table	   ref_tbl,
			       sos_Gen_param_List  gpl,
			       sos_Type_descr_List tdl,
			       sos_Type_descr_List type_decls)
{
   // This function does not handle component descriptions. These are
   // expected to be treated on an above level.
   T_PROC ("subst_method")
   TT (cfe_M, T_ENTER);

   sos_Method     new_m;
   sos_Type_descr old_td = old.get_result_type();
   sos_Type_descr new_td = subst_type_descr (old_td, gpl, tdl, type_decls);
   sos_Param_List old_pl = old.get_params();
   sos_Param_List new_pl = subst_params (old_pl, gpl, tdl, type_decls);

					  //ATT2.0 QUEERNESS: explicit operator
   if (new_pl.operator==(old_pl) AND new_td.operator==(old_td))
      new_m = old;
   else
   {  new_m = sos_Method::clone (old, cfe_cnt);
      new_m.set_params (new_pl);
      new_m.set_result_type (new_td);
      new_m.set_defined_in (aux_c);
      new_m.set_impls (sos_Method_impl_List::make (NO_OBJECT));

      // The generated create, copy methods are specific for each instantiation
      // and not implemented by calling a root class method.
      sos_Bool ignore_reftbl;
      if (    NOT (ignore_reftbl = (ref_tbl == NO_OBJECT))
	  AND old.get_is_generated()
	  AND old.get_is_static())
      {  smg_String mn = old.get_name();
	 if (mn.equal ("create")  OR  mn.equal ("copy"))
	    ignore_reftbl = TRUE;
      }
      if (ignore_reftbl)
         new_m.set_generated_from (old);
      else
      {  sos_Method ref_m = ref_tbl.lookup (old);
	 new_m.set_generated_from (ref_m);
	 err_assert (ref_m != NO_OBJECT, "subst_method");
      }
   }
   TT (cfe_M, T_LEAVE);
   return new_m;
}


LOCAL void subst_methods (sos_Method_List     old,
			  sos_Class_type      aux_c,
			  sos_Gen_param_List  gpl,
			  sos_Type_descr_List tdl,
			  sos_Type_descr_List type_decls)
{  T_PROC ("subst_methods")
   TT (cfe_M, T_ENTER);
   				// defined for non-universal instantiations
   sos_Method_table root_tbl
	= (aux_c.is_generic_class() || aux_c.get_root_class() == aux_c)
	       ? sos_Method_table::make (NO_OBJECT)
	       : aux_c.get_root_class().get_methods();

   sos_Method_List  new_methods = aux_c.get_local_methods();
   sos_Bool	    is_root     = aux_c == aux_c.get_root_class();

   // First make a duplicate of the old method list to enable removing
   // elements from it. Note, that we need cursor stability.
   sos_Method_List tmp_old = sos_Method_List::create (TEMP_CONTAINER, TRUE);
   tmp_old += old;

   agg_iterate (tmp_old, sos_Method old_m)
   {
      // copy the method only if it is not declared private. Otherwise
      // copying not needed, since private methods only are used for implemen-
      // ting the root class' methods.

      if (is_root || old_m.get_kind() != sos_PRIVATE)
      {
	 sos_Comp_descr cd = old_m.get_comp_descr();

	 if (cd == NO_OBJECT)
	    new_methods.append(
	       subst_method (old_m, aux_c, root_tbl, gpl, tdl, type_decls));
	 else
	 {			// Note, that this must be a local component.

	    sos_Method ogm = cd.get_get_method(),
		       osm = cd.get_set_method(),
		       ngm,
		       nsm = subst_method (osm, aux_c, root_tbl, gpl, tdl,
					   type_decls);

	    if (osm.identical (nsm))	//ATT2.0 QUEERNESS: explicit operator
	       ngm = ogm;
	    else
	    {  sos_Comp_descr ncd = sos_Comp_descr::clone (cd, cfe_cnt);

	       sos_Comp_descr_List cdl = aux_c.get_components();
	       cdl.remove (cdl.find (cd));
	       cdl.append (ncd);

	       ngm = subst_method (ogm, aux_c, root_tbl, gpl, tdl, type_decls);

	       ngm.set_comp_descr (ncd);
	       nsm.set_comp_descr (ncd);
	       ncd.set_get_method (ngm);
	       ncd.set_set_method (nsm);
	    }
	    new_methods.append (ngm);
	    new_methods.append (nsm);

	    // Now search the corresponding second method in the old list
	    // and remove it, since it is already substituted.
	    tmp_old.remove (tmp_old.find (old_m == ogm ? osm : ogm));
	 }
      }
   }
   agg_iterate_end (tmp_old, old_m);
   tmp_old.destroy();

   TT (cfe_M, T_LEAVE);
}

LOCAL sos_Bool must_subst_components (sos_Comp_descr_List cdl,
				      sos_Gen_param_List  gpl,
				      sos_Type_descr_List tdl,
				      sos_Type_descr_List type_decls)
{
   // This function checks, whether one of the get_- or set_methods contained
   // in the component descriptions, will be replaced by generic substitution.
   // In this case a new sos_Comp_descr_List will be necessary and TRUE is 
   // returned.
   T_PROC ("must_subst_components")
   TT (cfe_H, T_ENTER);

   sos_Bool result = FALSE;

   agg_iterate (cdl, sos_Comp_descr cd)
   {
      sos_Method     m   = cd.get_get_method();
      sos_Param_List plm = m.get_params();

      result
	= (sos_Bool)(
	    result
	    OR must_subst_params(m.get_params(),gpl,tdl,type_decls)
	    OR must_subst_type_descr(m.get_result_type(),gpl,tdl,type_decls));

      if ((m = cd.get_set_method()) != NO_OBJECT)
      {  plm = m.get_params();
	 result = (sos_Bool)(
		     result
		     OR must_subst_params(m.get_params(),gpl,tdl,type_decls)
		     OR must_subst_type_descr(m.get_result_type(),gpl,tdl,
					      type_decls));
      }
      if (result) break;
   } agg_iterate_end(cdl, cd);

   TT (cfe_H, T_LEAVE);
   return result;
}


LOCAL sos_Bool must_subst_params (sos_Param_List      old,
			          sos_Gen_param_List  gpl,
			          sos_Type_descr_List tdl,
			          sos_Type_descr_List type_decls)
{  T_PROC ("must_subst_params")
   TT (cfe_M, T_ENTER);

   sos_Bool result = FALSE;

   if (old != NO_OBJECT)
      agg_iterate (old, sos_Param old_p)
	 if (must_subst_type_descr (old_p.get_type(), gpl, tdl, type_decls))
	 {  result = TRUE;  break;
	 }
      agg_iterate_end (old, old_p);

   TT (cfe_M, T_LEAVE);
   return result;
}


LOCAL sos_Param_List subst_params (sos_Param_List      old,
                                   sos_Gen_param_List  gpl,
                                   sos_Type_descr_List tdl,
                                   sos_Type_descr_List type_decls)
{  T_PROC ("subst_params")
   TT (cfe_M, T_ENTER);

   sos_Param_List new_pl;

   if (NOT must_subst_params (old, gpl, tdl, type_decls))
      new_pl = old;
   else
   {  new_pl = sos_Param_List::create (cfe_cnt);
      agg_iterate (old, sos_Param old_p)
      {  sos_Type_descr old_td = old_p.get_type();
	 sos_Type_descr new_td = subst_type_descr (old_td,
						   gpl, tdl, type_decls);

	 if (new_td.operator==(old_td))	//ATT2.0 QUEERNESS: explicit operator
	    new_pl.append (old_p);
	 else
	 {  sos_Param new_p = sos_Param::clone (old_p, cfe_cnt);
	    new_p.set_type (new_td);
	    new_pl.append (new_p);
	 }
      }
      agg_iterate_end (old, old_p);
   }

   TT (cfe_M, T_LEAVE);
   return new_pl;
}


LOCAL sos_Bool must_subst_type_descr (sos_Type_descr      old,
				      sos_Gen_param_List  gpl,
				      sos_Type_descr_List tdl,
				      sos_Type_descr_List type_decls)
{  T_PROC ("must_subst_type_descr")
   TT (cfe_M, T_ENTER);

   sos_Bool result;

   if (old.has_type (sos_Generic_instantiation_type))
   {  sos_Type_descr_List old_tdl
	 = sos_Generic_instantiation::make(old).get_act_gen_params();
      result = must_subst_type_descrs (old_tdl, gpl, tdl, type_decls);
   }
   else if (old.has_type (sos_Super_class_type))
   {  sos_Type_descr old_twpn = sos_Super_class::make(old).get_super_class();
      result = must_subst_type_descr (old_twpn, gpl, tdl, type_decls);
   }
   else
      result = old.has_type (sos_Gen_param_type);

   TT (cfe_M, T_LEAVE);
   return result;
}


LOCAL sos_Type_descr subst_type_descr (sos_Type_descr      old,
                                       sos_Gen_param_List  gpl,
                                       sos_Type_descr_List tdl,
                                       sos_Type_descr_List type_decls)
{  T_PROC ("subst_type_descr")
   TT (cfe_M, T_ENTER);

   sos_Type_descr new_td;

   if (old.has_type (sos_Generic_instantiation_type))
   {  sos_Generic_instantiation gi = sos_Generic_instantiation::make(old); 
      sos_Type_descr_List old_tdl = gi.get_act_gen_params();
      sos_Type_descr_List new_tdl
	 = sos_Type_descr_List::make(
	      subst_type_descrs (old_tdl, gpl, tdl, type_decls));

      if (new_tdl.operator==(old_tdl))	//ATT2.0 QUEERNESS: explicit operator
	 new_td = old;
      else
      {  sos_Class_type gen_class  = gi.get_gen_class();
	 new_td = cfe_lookup_generic_instantiation (
		gen_class.get_name(), new_tdl, sos_IMP_NONE, type_decls);
      }
   }
   else if (old.has_type (sos_Super_class_type))
   {  sos_Super_class twp = sos_Super_class::make(old);
      sos_Type_descr old_twpd = twp.get_super_class();
      sos_Type_descr new_twpd = subst_type_descr (old_twpd,
						  gpl, tdl, type_decls);

      if (new_twpd.operator==(old_twpd))  //ATT2.0 QUEERNESS: explicit operator
	 new_td = old;
      else
      {  sos_Super_class new_twp = sos_Super_class::clone (twp, cfe_cnt);
	 new_twp.set_super_class (new_twpd);
	 new_td = new_twp;
      }
   }
   else if (old.has_type (sos_Gen_param_type))
      new_td = subst_actual (sos_Gen_param::make(old), gpl, tdl);
   else
      new_td = old;

   TT (cfe_M, T_LEAVE);
   return new_td;
}


LOCAL sos_Bool must_subst_type_descrs (sos_Aggregate       old_list,
				       sos_Gen_param_List  gpl,
				       sos_Type_descr_List tdl,
				       sos_Type_descr_List type_decls)
// old must be a list instantiated with a type inheriting from sos_Type_descr
// or from sos_Type.
{
   T_PROC ("must_subst_type_descrs")
   TT (cfe_M, T_ENTER);

   sos_Bool result = FALSE;

   if (old_list.has_type (sos_Type_descr_List_type))
   {
      sos_Type_descr_List old = sos_Type_descr_List::make(old_list);
      agg_iterate (old, sos_Type_descr old_td)
      {  if (result = must_subst_type_descr (old_td, gpl, tdl, type_decls))
	    break;
      }
      agg_iterate_end (old, old_td);
   }
   else if (old_list.has_type (sos_Super_class_List_type))
   {  sos_Super_class_List old = sos_Super_class_List::make(old_list);
      agg_iterate (old, sos_Type_descr old_td)
      {  if (result = must_subst_type_descr (old_td, gpl, tdl, type_decls))
	    break;
      }
      agg_iterate_end (old, old_td);
   }
   else if (old_list.has_type (sos_Class_type_List_type))
   {  sos_Class_type_List old = sos_Class_type_List::make(old_list);
      agg_iterate (old, sos_Type_descr old_td)
      {  if (result = must_subst_type_descr (old_td, gpl, tdl, type_decls))
	    break;
      }
      agg_iterate_end (old, old_td);
   }
   else if (old_list.has_type (sos_Type_List_type)) 
   {  sos_Type_List old = sos_Type_List::make(old_list);
      agg_iterate (old, sos_Type old_td)
      {  if (result = must_subst_type_descr (sos_Type_descr::make(old_td),
					     gpl, tdl, type_decls))
	    break;
      }
      agg_iterate_end (old, old_td);
   }

   TT (cfe_M, T_LEAVE);
   return result;
}


LOCAL sos_Aggregate subst_type_descrs (sos_Aggregate       old_list,
                                       sos_Gen_param_List  gpl,
                                       sos_Type_descr_List tdl,
                                       sos_Type_descr_List type_decls)
// old must be a list instantiated with a type inheriting from sos_Type_descr
// or from sos_Type.
{
   T_PROC ("subst_type_descrs");
   TT (cfe_M, T_ENTER);

   sos_Aggregate new_tdl;

   if (NOT must_subst_type_descrs (old_list, gpl, tdl, type_decls))
      new_tdl = old_list;
   else
   {
      if (old_list.has_type (sos_Type_descr_List_type))
      {  sos_Type_descr_List old = sos_Type_descr_List::make(old_list);
	 sos_Type_descr_List new_temp_list = 
			     sos_Type_descr_List::create (cfe_cnt);
	 agg_iterate (old, sos_Type_descr old_td)
	 {  new_temp_list.append (subst_type_descr (old_td, 
						    gpl, tdl, type_decls));
	 }
	 agg_iterate_end (old, old_td);
	 new_tdl = new_temp_list;
      }
      else if (old_list.has_type (sos_Super_class_List_type))
      {  sos_Super_class_List old = sos_Super_class_List::make(old_list);
	 // Take over the list_cursor of the old list, since for
	 // sos_Super_class_Lists TRUE may be necessary as well as FALSE.
	 // (see function cfe_del_double_sc).
	 sos_Super_class_List new_temp_list = 
	       sos_Super_class_List::create (cfe_cnt,
	       old_list.get_list_cursor());
	 agg_iterate (old, sos_Type_descr old_td)
	 {  new_temp_list.append (
		sos_Super_class::make(
		   subst_type_descr (old_td, gpl, tdl, type_decls)));
	 }
	 agg_iterate_end (old, old_td);
	 new_tdl = new_temp_list;
      }
      else if (old_list.has_type (sos_Class_type_List_type))
      {  sos_Class_type_List old = sos_Class_type_List::make(old_list);
	 sos_Class_type_List new_temp_list = 
			     sos_Class_type_List::create (cfe_cnt);
	 agg_iterate (old, sos_Type_descr old_td)
	 {  new_temp_list.append (
		sos_Class_type::make(
		   subst_type_descr (old_td, gpl, tdl, type_decls)));
	 }
	 agg_iterate_end (old, old_td);
	 new_tdl = new_temp_list;
      }
      else if (old_list.has_type (sos_Type_List_type))
      {  sos_Type_List old = sos_Type_List::make(old_list);
	 sos_Type_List new_temp_list = sos_Type_List::create (cfe_cnt);
	 agg_iterate (old, sos_Type old_td)
	 {  new_temp_list.append (
		sos_Type::make(subst_type_descr (sos_Type_descr::make(old_td),
			    	  		 gpl, tdl, type_decls)));
	 }
	 agg_iterate_end (old, old_td);
	 new_tdl = new_temp_list;
      }
      else // error !!!
	 err_raise (err_SYS, err_CFE_INVALID_TYPE_LIST, NULL, FALSE);
   }

   TT (cfe_M, T_LEAVE);
   return new_tdl;
}
