/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module cci                       03/07/90                 Juergen Uhl (ju)
// **************************************************************************
// implements methods of classes: cci_Schema_impl, cci_Method_impl
// **************************************************************************

// --------------------------------------------------------------------------
// Tracing conventions: see trc_cci.h
// --------------------------------------------------------------------------

#define OBST_IMP_STDCONST
#define OBST_IMP_CHARTYPE
#define OBST_IMP_STRINGOP
#define OBST_IMP_FILE
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_MALLOC
#include "obst_stdinc.h"

#include "_obst_config.h"
#include "obst.h"
#include "obst_progstd.h"
#include "obst_stats.h"
#include "smg.h"
#include "trc_cci.h"
#include "cci_err.h"
#include "cci_obst.h"

#ifdef OBST_HAVE_INCRLD
#  include "cci_incrload.h"

cci_IncrLd   cci_incr_ld;
cci_LoadFct* cci_load_fct;

// --------------------------------------------------------------------------
// module initialization / finalization
// --------------------------------------------------------------------------
// **************************************************************************
EXPORT void cci_IncrLd::fini ()
// **************************************************************************
{  T_PROC ("cci_IncrLd::fini")
   TT (cci_M, T_ENTER);

   if (tmpf1) unlink (tmpf1);	// ignore error(s) if files do not exist
   if (tmpf2) unlink (tmpf2);

   dtors.process (TRUE);

   TT (cci_M, T_LEAVE);
}

// **************************************************************************
EXPORT void cci_IncrLd::init (char* argv[])
// **************************************************************************
{  T_PROC ("cci_IncrLd::init")
   TT (cci_M, T_ENTER);

   char cwd[MAXPATHLEN];

   cci_incr_ld.initial_wd = obst_strdup (obst_getwd(cwd));

        // argv[0] is expected to contain either the absolute pathname of
        // the executable, or a relative pathname anchored at the current
        // working directory.
   if (argv[0][0] EQ '/')
      cci_incr_ld.executable = obst_strdup (argv[0]);
   else
   {  smg_String tmp = smg_String(cwd) + "/" + argv[0];

      cci_incr_ld.executable = tmp.make_Cstring (SMG_TRANSFER);
   }
   TT (cci_M, T_LEAVE);
}

// **************************************************************************
EXPORT void cci_IncrLd::final_init ()
// **************************************************************************
{  T_PROC ("cci_IncrLd::final_init")
   TT (cci_M, T_ENTER);

   static sos_Bool initialized = FALSE;
   if (initialized)
   {  TT (cci_M, T_LEAVE);
      return;
   }
   else
      initialized = TRUE;

   if (!cci_incr_ld.executable)
      err_raise (err_SYS, err_CCI_NOT_INITIALIZED, NULL, FALSE);

   	// initialize names for temporary files
   smg_String tmpf_nm;
   tmpf_nm = smg_String((char*)obst_tempdir) + "/obst1XXXXXX";
   tmpf1   = tmpf_nm.make_Cstring (SMG_TRANSFER);
   mktemp (tmpf1);

   tmpf_nm = smg_String((char*)obst_tempdir) + "/obst2XXXXXX";
   tmpf2   = tmpf_nm.make_Cstring (SMG_TRANSFER);
   mktemp (tmpf2);

   ld_param1  = NULL;
   ld_params  = NULL;
   echo	      = ECHOXX;
   ld_context = sos_String::create (TEMP_CONTAINER, LOADCONTEXTXX);

   // assumptions about strtok:
   //  - always yields NULL after NULL was returned once
   //  - tokens contain at least 1 character

   char* settings = obst_getenv (ENVVAR_INCRLOAD);
   if (settings)
   {  settings = obst_strdup (settings);

      sos_Bool	  is_ok = TRUE;
      const char* sep   = " \t";
      char* 	  field = strtok (settings, sep);

      if (field)
      {  if (streql (field, "+") OR streql (field, "-"))
	    echo = (sos_Bool)(*field EQ '+');
	 else
	 {  ld_context.assign_Cstring (field);

	    if (field = strtok (NULL, sep))
	       if (streql (field, "+") OR streql (field, "-"))
		  echo  = (sos_Bool)(*field EQ '+');
	       else
		  is_ok = FALSE;
	 }
      }
      if (is_ok && (field = strtok (NULL, sep)))
      {  ld_param1 = obst_strdup (field);
	 
	 while (*(++ field))
	    ;
	 int off = strspn (++ field, sep);
	 field  += off;

	 if (*field)
	    ld_params = obst_strdup (field);
      }
      if (NOT is_ok)
	 err_raise (err_USE, err_CCI_WRONG_INIT, obst_getenv (ENVVAR_INCRLOAD));
      delete settings;
   }

   if (!(searchpath = obst_getenv (ENVVAR_LOADPATHS)))
       searchpath = LOADPATHSXX;
   if (!searchpath OR !*searchpath)
   {  searchpaths = NULL;
      searchpath  = "";
   }
   else
   {  settings = obst_strdup (searchpath);

      smg_String path;

      int   count = 2;
      char* pptr  = settings;
      while (*pptr)
	 if (*(pptr ++) == ':')
	    ++ count;
      searchpaths = new char* [count];

      char* field;
      for (count = 0, pptr = settings;
	   field = strtok (pptr, ":");
	   pptr = NULL, ++ count)
      {  if (*field EQ '.')
	 {  smg_String s = smg_String((char*)initial_wd) + "/" + field;
	    field = s.make_Cstring (SMG_TRANSFER);
	 }
	 else
	    field = obst_strdup (field);

	 if (path.length())
	    path += ":";
	 path += field;

	 searchpaths[count] = field;
	 TT (cci_M, TXT("loadpath"); TS(field));
      }
      searchpath	 = path.make_Cstring (SMG_TRANSFER);
      searchpaths[count] = NULL;

      delete settings;
   }
   TT (cci_M, T_LEAVE; TXT("main");    TS(executable);
       	      	       TXT("context"); TsS(ld_context);
	      	       TXT("param1");  TS(ld_param1);
	      	       TXT("params");  TS(ld_params);
	      	       TXT("echo");    TB(echo));
}

// --------------------------------------------------------------------------
// utility functions
// --------------------------------------------------------------------------

// **************************************************************************
EXPORT int cci_IncrLd::find_file (smg_String& fpath,
				  sos_Bool raise_err /* = TRUE */)
// **************************************************************************
{  T_PROC ("cci_IncrLd::find_file")
   TT (cci_VL, T_ENTER; TS(fpath.make_Cstring(SMG_BORROW)));

   int   fd     = -1;
   char* _fpath = fpath.make_Cstring(SMG_BORROW);
   
   if (*_fpath == '/')
      fd = open (_fpath, O_RDONLY, 0);

   else if (searchpaths)
      for (char** prefix = (char**)searchpaths;  *prefix;  ++ prefix)
      {  smg_String path = smg_String(*prefix) + "/" + fpath;

	 if ((fd = open (path.make_Cstring(SMG_BORROW), O_RDONLY, 0)) >= 0)
	 {  fpath = path;
	    break;
	 }
      }

   if (fd < 0 && raise_err)
      err_raise (err_SYS, err_CCI_OPEN_FAILED, fpath.make_Cstring(SMG_BORROW));

   TT (cci_VL, T_LEAVE; TS(fpath.make_Cstring(SMG_BORROW)));
   return fd;
}

// **************************************************************************
EXPORT void cci_XtorList::add (cci_Xtor_p xtor)
// **************************************************************************
{  T_PROC ("cci_XtorList::add")
   TT (cci_VL, T_ENTER);

   if (card == size)
      if (card)
	 list = (cci_Xtor_p*)REALLOC (list, sizeof(cci_Xtor_p)*(size += 10));
      else
	 list = new cci_Xtor_p[size = 10];

   list[card ++] = xtor;

   TT (cci_VL, T_LEAVE);
}

// **************************************************************************
EXPORT void cci_XtorList::process (sos_Bool reverse)
// **************************************************************************
{  T_PROC ("cci_XtorList::process")
   TT (cci_VL, T_ENTER);

   if (reverse)
   {  for (int i = card;  --i >= 0; )
         (* list[i])();
   }
   else
   {  for (int i = 0;  i < card;  ++ i)
         (* list[i])();
   }
   TT (cci_VL, T_LEAVE);
}
#endif /* OBST_HAVE_INCRLD */


// --------------------------------------------------------------------------
// Universal access functions for those methods which are generated for each
// class and may be executed via the interface of sos_Object.
// --------------------------------------------------------------------------

sos_Object _cci_univ_assign (const sos_Object& _x, const sos_Object_Array& _p)
{  _x.assign (_p[0]);
   return NO_OBJECT;
}

sos_Object _cci_univ_clone (const sos_Object&, const sos_Object_Array& _p)
{  return sos_Object::clone (_p[0], make_sos_Container(_p[1]));
}

sos_Object _cci_univ_destroy (const sos_Object& _x, const sos_Object_Array&)
{  _x.destroy();
   return NO_OBJECT;
}

sos_Object _cci_univ_equal (const sos_Object& _x, const sos_Object_Array& _p)
{  return make_sos_Bool_object(_x.equal (_p[0],
					 make_sos_Eq_kind(_p[1])));
}

sos_Object _cci_univ_hash_value (const sos_Object& _x, const sos_Object_Array&)
{  return make_sos_Int_object(_x.hash_value());
}


// --------------------------------------------------------------------------
// conversions for external type cci_Fun
// --------------------------------------------------------------------------

sos_Cstring make_string_from_cci_Fun_object (const sos_Object& f)
{  cci_Fun p1	  = make_cci_Fun (f);
   char*   result = smg_String ((sos_Int)p1, FALSE).make_Cstring(SMG_TRANSFER);
   return result;
}

sos_Object make_cci_Fun_object_from_string (const sos_Cstring s)
{  sos_Cstring ptr;
   long        l      = strtol (s, &ptr, 16);
   sos_Object  result = (ptr != s AND ptr[0] == 0)
			   ? make_cci_Fun_object (cci_Fun (l))
			   : NO_OBJECT;
   return result;
}


// --------------------------------------------------------------------------
// LOCAL: Management of loaded schema implementations
//	  (A schema is identified by its associated schema container.)
// --------------------------------------------------------------------------
#define CNT_HTAB_SIZE	128
#define HASH_CT(ct)	sos_Int(ct) % CNT_HTAB_SIZE

struct cci_CntEntry {
   cci_CntEntry *next;
   sos_Int      ct;
};
LOCAL cci_CntEntry *cnt_htab[CNT_HTAB_SIZE];

#ifdef OBST_HAVE_INCRLD
// The following array is used in 'lookup_files' to prevent from processing
// a schema (container) twice, and afterwards in 'cci_Schema_impl::load' to
// enter the newly loaded schemas (resp. their containers) into the hash table.
LOCAL struct
{  sos_Container *array;
   int	   	 size,
	   	 last_entry; } cnt_array = { NULL, 0, -1 };

// *************************************************************************
LOCAL sos_Bool enter_ct (sos_Container ct)
// *************************************************************************
{  T_PROC ("enter_ct")
   TT (cci_L, T_ENTER);

   for (int i = 0;  i <= cnt_array.last_entry;  ++ i)
      if (ct EQ cnt_array.array[i])
      {  TT (cci_L, T_LEAVE; TB(FALSE));
	 return FALSE;
      }

   if (++cnt_array.last_entry EQ cnt_array.size)
   {  if (cnt_array.size)
         cnt_array.array
	    = (sos_Container*)
		  REALLOC (cnt_array.array,
			   (cnt_array.size *= 2) 
			    * sizeof(sos_Container));
      else
	 cnt_array.array = new sos_Container[cnt_array.size = 4];
   }
   cnt_array.array[cnt_array.last_entry] = ct;

   TT (cci_L, T_LEAVE; TB(TRUE));
   return TRUE;
}
#endif /* OBST_HAVE_INCRLD */

// *************************************************************************
LOCAL
sos_Bool enter_schema (register sos_Int ct_id, sos_Bool enterIfNew = TRUE)
// *************************************************************************
{  T_PROC ("enter_schema")
   TT (cci_L, T_ENTER);

   register sos_Bool new_schema;

   for (register cci_CntEntry **e = &cnt_htab[ HASH_CT(ct_id) ];
        *e AND (*e)->ct != ct_id;
        e = &(*e)->next)
      ;

   if (*e)
      new_schema = FALSE;
   else
   {  new_schema = TRUE;

      if (enterIfNew)
      {  *e = new cci_CntEntry;
	 (*e)->next = NULL;
	 (*e)->ct   = ct_id;
      }
   }
   TT (cci_L, T_LEAVE; TI((int)ct_id); TB(new_schema));
   return new_schema;
}

// *************************************************************************
LOCAL void load_schema (register sos_Int ct_id)
// *************************************************************************
{  T_PROC ("cci - load_schema")
   TT (cci_L, T_ENTER);

   // This routine is the entry point to the incremental loading facility
   // for the generated code, in particular via the C::make() calls.

   for (register cci_CntEntry **e = &cnt_htab[ HASH_CT(ct_id) ];
	*e AND (*e)->ct != ct_id;
	e = &(*e)->next)
      ;

   if (! *e)					// schema not yet loaded
   {  sos_Container ct = sos_Container::make(ct_id);

      if (NOT ct.exists())
	 err_raise (err_SYS, err_CCI_INVALID_OBJ, "cci - load_schema", FALSE);

      sos_Bool found = FALSE;
      sos_Schema_module    sm  = sos_Schema_module::retrieve (ct);
      sos_Schema_impl_List sil = sm.get_impls();
      if (sil != NO_OBJECT)
	 agg_iterate (sil, sos_Schema_impl si)
	    if (si.isa (cci_Schema_impl_type))
	    {  cci_Schema_impl::make (si).load();
	       found = TRUE;
	       break;
	    }
	 agg_iterate_end (sil, si);
      
      if (NOT found)
	 err_raise (err_SYS, err_CCI_INC_LOAD_FAILED,
			     sm.get_name().make_Cstring());
   }
   TT (cci_L, T_LEAVE);
}

// *************************************************************************
#define schema_is_loaded(ct)  (void)enter_schema ((sos_Int)ct, TRUE)
// *************************************************************************
// (g++ 2.3.3 generates wrong code when optimization (`-O`) is turned on,
//  if _schema_is_loaded is implemented as an inline function.)


#ifdef OBST_HAVE_INCRLD
// *************************************************************************
LOCAL void lookup_files (sos_Container		ct,
			 const sos_Object_List& schemas,
			 const sos_String_List& objects,
			 const sos_String_List& libraries)
// *************************************************************************
{  T_PROC ("cci - lookup_files")
   TT (cci_L, T_ENTER; TXT("container"); TI((int)ct));

   sos_Schema_module sm = sos_Schema_module::retrieve (ct);

	// first step: (recursively) add file names for imported schemas
	//	       not yet loaded
   sos_Import_List impl = sm.get_imports();
   if (impl != NO_OBJECT)
      agg_iterate (impl, sos_Import imp)
	 sos_Schema_module imp_mdl = imp.get_module();
	 sos_Container     imp_ct  = imp_mdl.container();

	 if (enter_schema (imp_ct, FALSE)  AND  enter_ct (imp_ct))
	    lookup_files (imp_ct, schemas, objects, libraries);
      agg_iterate_end (impl, imp);

   cci_Schema_impl      si  = cci_Schema_impl::make (NO_OBJECT);
   sos_Schema_impl_List sil = sm.get_impls();
   if (sil != NO_OBJECT)
      agg_iterate (sil, sos_Schema_impl i)
	 if (i.isa (cci_Schema_impl_type))
	 {  si = cci_Schema_impl::make (i);
	    break;
	 }
      agg_iterate_end (sil, i);
   
   if (si EQ NO_OBJECT)
      err_raise (err_SYS, err_CCI_INC_LOAD_FAILED,
			  sm.get_name().make_Cstring());

	// second step: (recursively) add file names for schemas this schema
	//		depends on and which are not yet loaded
   sos_String_List sl = si.get_schemas();
   if (sl != NO_OBJECT)
      agg_iterate (sl, sos_String s)
	 sos_Schema_module sm = sos_Schema_module::lookup (s);
	 if (sm EQ NO_OBJECT)
	    err_raise (err_SYS, err_CCI_UNKNOWN_SCHEMA, s.make_Cstring());

	 sos_Container imp_ct = sm.container();
	 if (enter_schema (imp_ct, FALSE)  AND  enter_ct (imp_ct))
	    lookup_files (imp_ct, schemas, objects, libraries);
      agg_iterate_end (sl, s);

	// third step: process the direct dependencies of this schema on
	//	       object modules, libraries
   schemas.append (sm);

   sos_Directory dir;
   if ((dir = si.get_object_files()) != NO_OBJECT)
   {  sl = sos_String_List::make (dir[cci_incr_ld.ld_context]);
      if (sl != NO_OBJECT)
         objects += sl;
   }
   if ((dir = si.get_libraries()) != NO_OBJECT)
   {  sl = sos_String_List::make (dir[cci_incr_ld.ld_context]);
      if (sl != NO_OBJECT)
         libraries += sl;
   }
   TT (cci_L, T_LEAVE);
}
#endif /* OBST_HAVE_INCRLD */


// --------------------------------------------------------------------------
// class cci_Schema_impl
// --------------------------------------------------------------------------
// *************************************************************************
EXPORT cci_Schema_impl cci_Schema_impl::make_impl (sos_Schema_module s)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::make_impl")
   TT (cci_H, T_ENTER);

   cci_Schema_impl result;
   sos_Bool found = FALSE;

   sos_Schema_impl_List impls = s.get_impls();
   if (impls == NO_OBJECT)
   {  impls = sos_Schema_impl_List::create (s.container());
      s.set_impls (impls);
   }
   else
   {  agg_iterate (impls, sos_Schema_impl impl)
         if (impl.isa (cci_Schema_impl_type))
	 {  found = TRUE;
	    result = cci_Schema_impl::make (impl);
	    break;
	 }
      agg_iterate_end (impls, impl);
   }
   if (NOT found)
   {  result = cci_Schema_impl::create (s.container());
      impls.append (result);
   }
   TT (cci_H, T_LEAVE);
   return result;
}


// **************************************************************************
EXPORT void cci_Schema_impl::load ()
// **************************************************************************
{  T_PROC ("cci_Schema_impl::load")
   TT (cci_H, T_ENTER);

   if (NOT enter_schema (self.container(), FALSE))
   {  TT (cci_H, T_LEAVE);
      return;
   }
#ifndef OBST_HAVE_INCRLD
   err_raise (err_SYS,
	      err_CCI_INC_LOAD_NOT_IMPL, "cci_Schema_impl::load", FALSE);
#else
   if (NOT cci_load_fct)
      err_raise (err_SYS,
		 err_CCI_INC_LOAD_NO_LOADFCT, "cci_Schema_impl::load", FALSE);

   cci_incr_ld.final_init();

   cnt_array.last_entry = -1;
   enter_ct (self.container());

   static sos_Object_List schemas = sos_Object_List::create (TEMP_CONTAINER);
   static sos_String_List objects = sos_String_List::create (TEMP_CONTAINER);
   static sos_String_List libs	  = sos_String_List::create (TEMP_CONTAINER);

   lookup_files (self.container(), schemas, objects, libs);

   int card = objects.card() + libs.card();
   if (card > 0)
      (*cci_load_fct) (schemas, objects, libs);

   schemas.clear();
   objects.clear();
   libs.clear();

   if (card EQ 0)
   {  sos_Schema_module sm = sos_Schema_module::retrieve (self.container());
      err_raise (err_SYS,
		 err_CCI_NO_DEPENDENCIES, sm.get_name().make_Cstring());
   }
   for (int i = 0;  i <= cnt_array.last_entry;  ++ i)
      schema_is_loaded (cnt_array.array[i]);

#endif /* OBST_HAVE_INCRLD */
   TT (cci_H, T_LEAVE);
}


// --------------------------------------------------------------------------
// class cci_Schema_impl : Management of dependency lists
// --------------------------------------------------------------------------

// *************************************************************************
LOCAL void remove_deps_from_dir (sos_Directory &dir, sos_String cntxt)
// *************************************************************************
{  T_PROC ("cci - remove_deps_from_dir")
   TT (cci_M, T_ENTER);

   if (dir != NO_OBJECT)
      if (cntxt == NO_OBJECT)
      {  agg_iterate_association (dir, sos_String s, sos_Object _sl)
	    _sl.destroy();
	 agg_iterate_association_end (dir, s, _sl)

	 dir.destroy();
	 dir = sos_Directory::make (NO_OBJECT);
      }
      else
      {  sos_String_List sl = sos_String_List::make (dir[cntxt]);
	 if (sl != NO_OBJECT)
	 {  sl.destroy();
	    dir.remove (cntxt);

	    if (dir.is_empty())
	    {  dir.destroy();
	       dir = sos_Directory::make (NO_OBJECT);
	    }
	 }
      }
   TT (cci_M, T_LEAVE);
}

// *************************************************************************
EXPORT void cci_Schema_impl::remove_deps (cci_Dependency kind,
					  sos_String	 cntxt /*=NO_OBJECT*/)
// *************************************************************************
{  T_PROC("cci_Schema_impl::remove_deps")
   TT (cci_M, T_ENTER);

   if (kind == cci_NONE || kind == cci_SCHEMA)
   {  sos_String_List sl = self.get_schemas();
      if (sl != NO_OBJECT)
      {  sl.destroy();
	 self.set_schemas (sos_String_List::make (NO_OBJECT));
      }
   }
   if (kind == cci_NONE || kind == cci_OBJECT)
   {  sos_Directory dir = self.get_object_files();
      remove_deps_from_dir (dir, cntxt);
      self.set_object_files (dir);
   }
   if (kind == cci_NONE || kind == cci_LIB)
   {  sos_Directory dir = self.get_libraries();
      remove_deps_from_dir (dir, cntxt);
      self.set_libraries (dir);
   }
   TT (cci_M, T_LEAVE);
}

// *************************************************************************
LOCAL void add_dep_to_dir (sos_Container cnt, sos_Directory &dir,
			   sos_String    dep, sos_String    cntxt, Index pos)
// *************************************************************************
{  T_PROC("cci - add_dep_to_dir")
   TT (cci_M, T_ENTER);

   if (dir == NO_OBJECT)
      dir = sos_Directory::create (cnt);

   sos_String_List sl = sos_String_List::make (dir[cntxt]);
   if (sl == NO_OBJECT)
   {  sl = sos_String_List::create (cnt, FALSE, TRUE);
      dir.insert (cntxt, sl);
   }
   if (NOT sl.is_element (dep))
   {  if (dep.container() == cnt)
	 sl.insert (pos, dep);
      else
	 sl.insert (pos, sos_String::clone (dep, cnt));
   }
   TT (cci_M, T_LEAVE);
}

// *************************************************************************
EXPORT void cci_Schema_impl::add_dep (cci_Dependency kind,
			   	      sos_String     dep,
				      sos_String     _cntxt /* = NO_OBJECT */,
				      Index	     pos /* = 0 */)
// *************************************************************************
{  T_PROC("cci_Schema_impl::add_dep")
   TT (cci_M, T_ENTER);

   err_assert (kind != cci_NONE, "cci_Schema_impl::add_dep");

   sos_String cntxt = _cntxt;
   if (cntxt == NO_OBJECT)
      cntxt = cci_incr_ld.ld_context;

   if (kind == cci_SCHEMA)
   {  sos_String_List sl = self.get_schemas();
      if (sl == NO_OBJECT)
      {  sl = sos_String_List::create (self.container(), FALSE, TRUE);
	 self.set_schemas (sl);
      }
      if (NOT sl.is_element (dep))
      {  if (dep.container() == self.container())
	    sl.insert (pos, dep);
	 else
	    sl.insert (pos, sos_String::clone (dep, self.container()));
      }
   }
   else if (kind == cci_OBJECT)
   {  sos_Directory dir = self.get_object_files();
      add_dep_to_dir (self.container(), dir, dep, cntxt, pos);
      self.set_object_files (dir);
   }
   else /* kind == cci_LIB */
   {  sos_Directory dir = self.get_libraries();
      add_dep_to_dir (self.container(), dir, dep, cntxt, pos);
      self.set_libraries (dir);
   }
   TT (cci_M, T_LEAVE);
}


// --------------------------------------------------------------------------
// class cci_Schema_impl : Management of dynamic type representation objects
// --------------------------------------------------------------------------
#define IMPL_OBJ_HTAB_SIZE	1993
#define PREP_HASH_IDs(id1,id2)	( (id1).offset() ^ (id2).offset()<<12 )
#define HASH_IMPL_IDs(id1,id2)	PREP_HASH_IDs(id1,id2) % IMPL_OBJ_HTAB_SIZE


struct cci_ImplObjEntry {
   cci_ImplObjEntry *next;
   sos_Id           f, t;
   cci_Impl_obj     obj;
};

LOCAL cci_ImplObjEntry *impl_obj_htab[IMPL_OBJ_HTAB_SIZE];

// *************************************************************************
EXPORT cci_Impl_obj _cci_lookup_impl_obj (const sos_Id& t, const sos_Id& f)
// *************************************************************************
{  T_PROC ("_cci_lookup_impl_obj")
   TT (cci_L, T_ENTER);
   
   for (register cci_ImplObjEntry **e = &impl_obj_htab[ HASH_IMPL_IDs(f,t) ];
	*e AND ((*e)->f != f OR (*e)->t != t);
	e = &(*e)->next)
      ;
   cci_Impl_obj result = (*e) ? (*e)->obj
			      : (cci_Impl_obj)NULL;

   TT (cci_L, T_LEAVE);
   return result;
}

// *************************************************************************
EXPORT cci_Impl_obj _cci_get_impl_obj (const sos_Id& t, const sos_Id& f)
// *************************************************************************
{  T_PROC ("_cci_get_impl_obj")
   TT (cci_L, T_ENTER);
   
   cci_Impl_obj      result;
   register sos_Bool first_lookup = TRUE;

   while (1)
   {  for (register cci_ImplObjEntry **e = &impl_obj_htab[ HASH_IMPL_IDs(f,t) ];
	   *e AND ((*e)->f != f OR (*e)->t != t);
	   e = &(*e)->next)
	 ;
      if (*e)
      {  result = (*e)->obj;
	 break;
      }
      else if (f.offset() == 0  AND  (sos_Int)f.container() == 0)
	 // f == ( 0, 0 ) --> type ID of NO_OBJECT
	 // Possibility 1): Return always NULL, i.e. NO_OBJECT has no
	 //		    implementation object.
	 // Possibility 2): Return the implementation object associated with
	 //		    type t.
	 //		    This requires a proper initialization of NO_OBJECT
	 //		    (see _obst_basic_init) and the existence of
	 //		    implementation objects for any class type
	 //		    (see genCC).

	 return NULL;	// Possibility #1
         //f = t;	// possibility #2

      else if (first_lookup)
      {  first_lookup = FALSE;
#if BOOT
	 err_raise(err_SYS, err_CCI_NO_LOAD_IN_BOOT,"_cci_get_impl_obj",FALSE);
#else
	 load_schema (f.container());
#endif
      }
      else
      {  sos_Type tp = sos_Type::make (sos_Typed_id::make (f));
	 sos_Type rt = tp.root();

	 if (tp.operator!=(rt))	
	 {   // for CC 2.1
	     sos_Id tmp=rt._self_id(); // temporary needed by some compilers
	     result = _cci_get_impl_obj (t, tmp);
	 }
	 else
	 {  sos_Type   ttp = sos_Type::make (sos_Typed_id::make (t));
	    smg_String why = smg_String (ttp.get_name())
				+ "::make() on instance of "
				+ tp.get_name();
	    err_raise (err_SYS,
		       err_CCI_TYPE_ERROR, why.make_Cstring (SMG_BORROW));
	 }
	 break;
      }
   }
   TT (cci_L, T_LEAVE; TP (result));
   return result;
}

// *************************************************************************
EXPORT
void _cci_enter_impl_obj (const sos_Id& t, const sos_Id& f, cci_Impl_obj obj)
// *************************************************************************
{  T_PROC ("_cci_Senter_impl_obj")
   TT (cci_L, T_ENTER; TI((int) t.container()); TI(t.offset()));

   schema_is_loaded (f.container());

   for (register cci_ImplObjEntry **e = &impl_obj_htab[ HASH_IMPL_IDs(f,t) ];
	*e AND ((*e)->f != f OR (*e)->t != t);
        e = &((*e)->next))
      ;

   if (! *e)
   {  *e = new cci_ImplObjEntry;
      (*e)->next = NULL;
      (*e)->f    = f;
      (*e)->t    = t;
      (*e)->obj  = obj;
   }
   TT (cci_L, T_LEAVE);
}

// *************************************************************************
EXPORT cci_Impl_obj cci_Schema_impl::get_impl_obj (sos_Id t, sos_Id f)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::get_impl_obj")
   TT (cci_L, T_ENTER);

   cci_Impl_obj result = _cci_get_impl_obj (t, f);

   TT (cci_L, T_LEAVE);
   return result;
}

// *************************************************************************
EXPORT
void cci_Schema_impl::enter_impl_obj (sos_Id t, sos_Id f, cci_Impl_obj obj)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::enter_impl_obj")
   TT (cci_L, T_ENTER);

   _cci_enter_impl_obj (t, f, obj);

   TT (cci_L, T_LEAVE);
}


// --------------------------------------------------------------------------
// class cci_Schema_impl : C++ method implementation management
// --------------------------------------------------------------------------
#define FUN_HTAB_SIZE		1993
#define PREP_HASH_ID(id)	( m.offset() ^ (sos_Int)m.container()<<16 )
#define HASH_FUN_ID(id)		PREP_HASH_ID(id) % FUN_HTAB_SIZE

struct cci_Fun_entry {
   cci_Fun_entry* next;
   sos_Id  m;
   cci_Fun fun;
};

      
LOCAL cci_Fun_entry *fun_htab[FUN_HTAB_SIZE];


// *************************************************************************
EXPORT void _cci_enter_fun (const sos_Id& m, cci_Fun fun)
// *************************************************************************
{  T_PROC ("_cci_enter_fun")
   TT (cci_L, T_ENTER; TI((int) m.container()); TI(m.offset()); TP(fun));

   for (register cci_Fun_entry **e = &fun_htab[ HASH_FUN_ID(m) ];
	*e AND NOT ((*e)->m == m);
        e = &((*e)->next))
      ;

   if (! *e)
   {  *e = new cci_Fun_entry;
      (*e)->next = NULL;
      (*e)->m = m;
      (*e)->fun = fun;
   }
   TT (cci_L, T_LEAVE);
}

// *************************************************************************
EXPORT cci_Fun _cci_get_fun (const sos_Id& m)
// *************************************************************************
{  T_PROC ("_cci_get_fun")
   TT (cci_VL, T_ENTER; TI((int) m.container()); TI(m.offset()));

   for (register cci_Fun_entry **e = &fun_htab[ HASH_FUN_ID(m) ];
	*e AND (*e)->m != m;
	e = &(*e)->next)
      ;

   cci_Fun fun;
   if (*e)
      fun = (*e)->fun;
   else
      err_raise (err_SYS, err_CCI_INVALID_METHOD,
		 "cci_Schema_impl::get_fun", FALSE);

   TT (cci_VL, T_LEAVE; TP(fun));
   return fun;
}

// *************************************************************************
EXPORT cci_Fun cci_Schema_impl::get_fun (sos_Id m)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::get_fun")
   TT (cci_L, T_ENTER);

   load_schema (m.container());

   cci_Fun fun = _cci_get_fun (m);

   TT (cci_L, T_LEAVE);

   return fun;
}

// *************************************************************************
EXPORT void cci_Schema_impl::enter_fun (sos_Id m, cci_Fun fun)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::enter_fun")
   TT (cci_L, T_ENTER);

   _cci_enter_fun (m, fun);

   TT (cci_L, T_LEAVE);
}

// *************************************************************************
EXPORT
void cci_Schema_impl::enter_extern_cvt (sos_Id      et,
				        cci_Cvt_fun from_extern,
				        cci_Cvt_fun to_extern)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::enter_extern_cvt")
   TT (cci_L, T_ENTER);

   _cci_enter_fun (_cci_ext2obj_key(et), (cci_Fun)from_extern);
   _cci_enter_fun (_cci_obj2ext_key(et), (cci_Fun)to_extern);

   TT (cci_L, T_LEAVE);
}

// *************************************************************************
EXPORT
void cci_Schema_impl::enter_string_io (sos_Id     et,
				       cci_IO_fun from_string,
				       cci_IO_fun to_string)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::enter_string_io")
   TT (cci_L, T_ENTER);

   _cci_enter_fun (_cci_str2obj_key(et), (cci_Fun)from_string);
   _cci_enter_fun (_cci_obj2str_key(et), (cci_Fun)to_string);

   TT (cci_L, T_LEAVE);
}


// --------------------------------------------------------------------------
// class cci_Schema_impl : generic conversions for external types
// --------------------------------------------------------------------------

// *************************************************************************
EXPORT
sos_Object cci_Schema_impl::extern_object_from_string (sos_Type   tp,
						       sos_String s)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::extern_object_from_string")
   TT (cci_H, T_ENTER);

   char*      str = s.make_Cstring();
   sos_Object result
      = (* (cci_str2obj_fun)
	   cci_Schema_impl::get_fun (_cci_str2obj_key (tp._self_id())) ) (str);

   delete str;

   TT (cci_H, T_LEAVE);
   return result;
}

// *************************************************************************
EXPORT sos_String cci_Schema_impl::string_from_extern_object (sos_Object o)
// *************************************************************************
{  T_PROC ("cci_Schema_impl::string_from_extern_object")
   TT (cci_H, T_ENTER);

   char* str
      = (* (cci_obj2str_fun)
           cci_Schema_impl::get_fun (_cci_obj2str_key (o._type_id())) ) (o);

   sos_String result = (str) ? sos_String::create (TEMP_CONTAINER, str)
      			     : sos_String::make (NO_OBJECT);

   TT (cci_H, T_LEAVE);
   return result;
}


// --------------------------------------------------------------------------
// class cci_Method_impl
// --------------------------------------------------------------------------
// *************************************************************************
EXPORT cci_Method_impl cci_Method_impl::make_impl (sos_Method m)
// *************************************************************************
{  T_PROC ("cci_Method_impl::make_impl")
   TT (cci_H, T_ENTER);

   cci_Method_impl result;
   sos_Bool found = FALSE;

   sos_Method_impl_List impls = m.get_impls();
   if (impls == NO_OBJECT)
   {  impls = sos_Method_impl_List::create (m.container());
      m.set_impls (impls);
   }
   else
   {  agg_iterate (impls, sos_Method_impl impl)
         if (impl.isa (cci_Method_impl_type))
	 {  found = TRUE;
	    result = cci_Method_impl::make (impl);
	    break;
	 }
      agg_iterate_end (impls, impl);
   }
   if (NOT found)
   {  result = cci_Method_impl::create (m.container());
      impls.append (result);
   }
   TT (cci_H, T_LEAVE);
   return result;
}

// *************************************************************************
EXPORT void cci_Method_impl::enter_fun (register cci_Fun fun)
// *************************************************************************
{  T_PROC ("cci_Method_impl::enter_fun")
   TT (cci_L, T_ENTER);

   cci_Schema_impl::enter_fun (self._self_id(), fun);

   TT (cci_L, T_LEAVE);
}

// *************************************************************************
EXPORT sos_Object cci_Method_impl::execute (sos_Object o, sos_Object_Array p)
// *************************************************************************
{  T_PROC ("cci_Method_impl::execute")
   TT (cci_H, T_ENTER);

   register cci_Fun f = cci_Schema_impl::get_fun (self._self_id());

   sos_Object result  = (*f) (o, p);

   TT (cci_H, T_LEAVE);
   return result;
}

// *************************************************************************
EXPORT sos_String cci_Method_impl::operator_string (sos_String op_name)
// *************************************************************************
{  T_PROC ("cci_Method_impl::operator_string")
   TT (cci_H, T_ENTER);

   smg_String op_string = op_name;
   char	      *result_string;
   sos_String result;

        if (op_string.equal ("+")  ) result_string = "__plus";
   else if (op_string.equal ("-")  ) result_string = "__minus";
   else if (op_string.equal ("*")  ) result_string = "__times";
   else if (op_string.equal ("/")  ) result_string = "__div";
   else if (op_string.equal ("%")  ) result_string = "__rem";
   else if (op_string.equal ("^")  ) result_string = "__xor";
   else if (op_string.equal ("&")  ) result_string = "__and";
   else if (op_string.equal ("|")  ) result_string = "__or";
   else if (op_string.equal ("~")  ) result_string = "__not";
   else if (op_string.equal ("!")  ) result_string = "__logical_not";
   else if (op_string.equal ("=")  ) result_string = "__assign";
   else if (op_string.equal ("<")  ) result_string = "__less";
   else if (op_string.equal (">")  ) result_string = "__greater";
   else if (op_string.equal ("+=") ) result_string = "__plus_assign";
   else if (op_string.equal ("-=") ) result_string = "__minus_assign";
   else if (op_string.equal ("*=") ) result_string = "__times_assign";
   else if (op_string.equal ("/=") ) result_string = "__div_assign";
   else if (op_string.equal ("%=") ) result_string = "__rem_assign";
   else if (op_string.equal ("^=") ) result_string = "__xor_assign";
   else if (op_string.equal ("&=") ) result_string = "__and_assign";
   else if (op_string.equal ("|=") ) result_string = "__or_assign";
   else if (op_string.equal ("<<") ) result_string = "__shift_left";
   else if (op_string.equal (">>") ) result_string = "__shift_right";
   else if (op_string.equal (">>=")) result_string = "__shift_right_assign";
   else if (op_string.equal ("<<=")) result_string = "__shift_left_assign";
   else if (op_string.equal ("==") ) result_string = "__equal";
   else if (op_string.equal ("!=") ) result_string = "__not_equal";
   else if (op_string.equal ("<=") ) result_string = "__less_equal";
   else if (op_string.equal (">=") ) result_string = "__greater_equal";
   else if (op_string.equal ("&&") ) result_string = "__logical_and";
   else if (op_string.equal ("||") ) result_string = "__logical_or";
   else if (op_string.equal ("++") ) result_string = "__increment";
   else if (op_string.equal ("--") ) result_string = "__decrement";
   else if (op_string.equal (",")  ) result_string = "__comma";
   else if (op_string.equal ("->*")) result_string = "__member_access";
   else if (op_string.equal ("->") ) result_string = "__access";
   else if (op_string.equal ("()") ) result_string = "__function";
   else if (op_string.equal ("[]") ) result_string = "__index";
   else result_string = op_string.make_Cstring (SMG_BORROW);

   result = sos_String::create (TEMP_CONTAINER, result_string);

   TT (cci_H, T_LEAVE);
   return result;
}

// ---------------------------------------------------------------------------
// schema cci: (hash table) statistics
// ---------------------------------------------------------------------------
// *************************************************************************
EXPORT void _cci_collect_stats (obst_htabstat* htstat)
// *************************************************************************
{  obst_htabinfo* htinfo;
   register int   size;

   htinfo	   = obst_add_htabinfo (htstat);
   htinfo->descr   = "loaded schemata (key: ID schema container)";
   htinfo->buckets = CNT_HTAB_SIZE;

   for (register int i = CNT_HTAB_SIZE;  i --; )
   {  register cci_CntEntry* queue;
      for (queue = cnt_htab[i], size = 0;  queue;  queue = queue->next)
	 ++ size;
      obst_add_histdata (&htinfo->histogram, &htinfo->histsize, size);
   }

   htinfo	   = obst_add_htabinfo (htstat);
   htinfo->descr   = "type lattice (key: sos_Ids of related types)";
   htinfo->buckets = IMPL_OBJ_HTAB_SIZE;

   for (i = IMPL_OBJ_HTAB_SIZE;  i --; )
   {  register cci_ImplObjEntry* queue;
      for (queue = impl_obj_htab[i], size = 0;  queue;  queue = queue->next)
	 ++ size;
      obst_add_histdata (&htinfo->histogram, &htinfo->histsize, size);
   }

   htinfo	   = obst_add_htabinfo (htstat);
   htinfo->descr   = "method table (key: sos_Id method object)";
   htinfo->buckets = FUN_HTAB_SIZE;

   for (i = FUN_HTAB_SIZE;  i --; )
   {  register cci_Fun_entry* queue;
      for (queue = fun_htab[i], size = 0;  queue;  queue = queue->next)
	 ++ size;
      obst_add_histdata (&htinfo->histogram, &htinfo->histsize, size);
   }
}
