/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// Module bundle_main
// **************************************************************************

#define OBST_IMP_FORMATTED_IO
#include "obst_stdinc.h"

#include "_obst_config.h"
#include "obst_progstd.h"
#include "obst.h"
#include "obst_trc.h"
#include "obst_err.h"
#include "smg.h"
#include "cci_use.h"

// --------------------------------------------------------------------------

/*
 * Checks the basic assumptions of the OBST core about the installation.
 *
 * The following tests should be transfered analogously to the standalone
 * test program ./src/obst/check_machdeps.C.
 */

LOCAL void wrong_IntProperty (char *prop, char* tpname, int wanted, int real)
{  smg_String msg = smg_String("property") + " " + prop
					   + "(" + tpname + ") == "
					   + real + ", required: " + wanted;
   err_assert (0, msg.make_Cstring (SMG_BORROW));
}

LOCAL void wrong_Property (char *prop, char* tpname)
{  smg_String msg = smg_String("wrong ") + prop + " for type " + tpname;
   err_assert (0, msg.make_Cstring (SMG_BORROW));
}

#define assert_typesize(tpname, wanted, real) \
   if ((wanted)!=(real)) wrong_IntProperty ("sizeof", tpname, wanted, real)


/*
 * sos_Short is not an external type. But the equivalent to the following
 * implementation is used in the kernel routines which convert from/to an
 * external type.
 * These are: _obst_Int_from_enum, _obst_enum_from_sos_Int plus the component
 *	      access methods generated in genCC.
 */
LOCAL void bcopy_from_sos_Short (void* s, void* c)
{  *(sos_Short *)c = *(sos_Short *)s;
}

LOCAL void assert_installation()
{
   /*
    * external types of schemas knl, cci
    */
   assert_typesize ("sos_Char",      SOS_CHAR_SIZE,	 sizeof(sos_Char));
   assert_typesize ("sos_Short",     SOS_SHORT_SIZE,	 sizeof(sos_Short));
   assert_typesize ("sos_Int",       SOS_INT_SIZE, 	 sizeof(sos_Int));
   assert_typesize ("sos_Float",     SOS_FLOAT_SIZE, 	 sizeof(sos_Float));
   assert_typesize ("sos_Pointer",   SOS_POINTER_SIZE,   sizeof(sos_Pointer));
   assert_typesize ("sos_Cstring",   SOS_CSTRING_SIZE,   sizeof(sos_Cstring));
   assert_typesize ("sos_Offset",    SOS_OFFSET_SIZE,    sizeof(sos_Offset));
   assert_typesize ("sos_Container", SOS_CONTAINER_SIZE, sizeof(sos_Container));
   assert_typesize ("sos_Id", 	     SOS_ID_SIZE,        sizeof(sos_Id));
   assert_typesize ("sos_Typed_id",  SOS_TYPED_ID_SIZE,  sizeof(sos_Typed_id));
   assert_typesize ("cci_Fun",	     CCI_FUN_SIZE,	 sizeof(cci_Fun));

   /*
    * check alignment: overly restrictive.
    * (Note: some requirements stem from the implementation of Mapping<X,Y>.)
    */
   typedef double maxalign_t;
   int dummyforal;

#define compute_alignment(tp, var) \
   {  struct { maxalign_t _x; char reffield; tp tpfield;} s;	\
      var = (int)&s.tpfield - (int)&s.reffield; }

#define op_identical(a,b)  ((a)==(b))
#define op_multiple(a,b)   ((b)%(a)==0)
#define assert_alignment(tp, tpname, wanted, cmp_op) \
   compute_alignment (tp, dummyforal);	\
      if (!cmp_op(dummyforal,wanted))	\
	 wrong_IntProperty ("alignment", tpname, wanted, dummyforal);

/* requirements (a <= b means: align(b) mod align(a) == 0,
 *			   or: any b-alignment fullfills any a-constraint)
 *
 * standard assumption:
 *	char <= <all types>
 * _obst_Int_from_enum, _obst_enum_from_sos_Int:
 *	sos_Char <= sos_Id, sos_Short <= sos_Id
 * _obst_object_from_extern, _obst_extern_from_object:
 *	<any extern> <= _obst_maxalign_t
 * Mapping_Hash.C {read_/write_ht_entry, read_/write_page_header}:
 *	sos_Char <= sos_Offset
 * Mapping_Hash.C {read_/write_page (5 args), read_/write_page_entry}:
 *	sos_Typed_id == sos_Int
 */
   int obst_maxal, int_al, offset_al, id_al;
   
   compute_alignment (_obst_maxalign_t, obst_maxal);
   compute_alignment (sos_Int,		int_al);
   compute_alignment (sos_Offset,	offset_al);
   compute_alignment (sos_Id,		id_al);

   assert_alignment (char,	    "char",	     1, 	 op_identical);
   assert_alignment (sos_Char,	    "sos_Char",      1, 	 op_identical);
   assert_alignment (sos_Short,	    "sos_Short",     id_al, 	 op_multiple);
   assert_alignment (sos_Int,	    "sos_Int",       obst_maxal, op_multiple);
   assert_alignment (sos_Float,	    "sos_Float",     obst_maxal, op_multiple);
   assert_alignment (sos_Pointer,   "sos_Pointer",   obst_maxal, op_multiple);
   assert_alignment (sos_Cstring,   "sos_Cstring",   obst_maxal, op_multiple);
   assert_alignment (sos_Offset,    "sos_Offset",    obst_maxal, op_multiple);
   assert_alignment (sos_Container, "sos_Container", obst_maxal, op_multiple);
   assert_alignment (sos_Id,	    "sos_Id",	     obst_maxal, op_multiple);
   assert_alignment (sos_Typed_id,  "sos_Typed_id",  int_al, 	 op_identical);
   assert_alignment (cci_Fun, 	    "cci_Fun",	     obst_maxal, op_multiple);

#ifdef OBST_STD_FORMAT
   /*
    * persistent standard representation for externals?
    */
   char tstbuf[4];
   int  i;

#define assert_order(tp, tpname, bcopy_from, tstval)\
   {  tp tstvar = (tp)tstval;			 \
      memset (tstbuf, 0, 4);			 \
      bcopy_from ((void*)&tstvar, (void*)tstbuf);\
      for (i = 0;  i < sizeof(tp);  ++ i)	 \
      {  if (tstbuf[i] != i+1)			 \
	    wrong_Property ("byteorder", tpname); }}
   
   // Note, that sos_Short is only needed for enumeration types with more than
   // 256 literals. That means, that a failure of the following assertion
   // can be ignored in practice, IFF the role of sos_Short has not been
   // extended in the meantime. I.e. sos_Short must only be used in the two
   // kernel functions mentioned above plus the genCC function which generates
   // component access methods.
   assert_order (sos_Short,  "sos_Short",  bcopy_from_sos_Short,  0x0102);

   assert_order (sos_Int,    "sos_Int",    bcopy_from_sos_Int,    0x01020304);
   assert_order (sos_Offset, "sos_Offset", bcopy_from_sos_Offset, 0x01020304);

   // Note that the following test might fail simply due to differences in
   // the string --> float conversion routines.
   // If the test fails, check if that is indeed the reason.
   assert_order (sos_Float, "sos_Float", bcopy_from_sos_Float, 2.38793926e-38);
#endif
}

// --------------------------------------------------------------------------

extern int cnt_main  (int argc, char* argv[]);
extern int cfe_main  (int argc, char* argv[]);
extern int dmp_main  (int argc, char* argv[]);
extern int gen_main  (int argc, char* argv[]);
extern int init_main (int argc, char* argv[]);
extern int scp_main  (int argc, char* argv[]);

#ifdef OBST_HAVE_INCRLD
extern int sil_main  (int argc, char* argv[]);

#  define INCRLDTOOL(main) main(argc,argv)
#else
#  define INCRLDTOOL(main) (err_raise(err_USE,err_CCI_INC_LOAD_NOT_IMPL,argv[0]),1)
#endif

#ifdef OBST_HAVE_JOYCE
extern int sync_switch_main (int argc, char* argv[]);

#  define JOYCETOOL(main) main(argc,argv)
#else
#  define JOYCETOOL(main) (err_raise(err_USE,"JOYCE not installed",argv[0]),1)
#endif


int main (int argc, char *argv[])
{  obst_init (argc, argv);

   int status = 0;

   char *basename, *ptr;
   basename = ptr = argv[0];
   while (*ptr)
      if (*(ptr ++) == '/')
	 basename = ptr;

#ifdef OBST_HAVE_JOYCE
   sos_Bool TA_started = FALSE;

   if (!streql (basename, "init")  &&  !streql (basename, "sync_switch"))
   {  TRANSACTION.start(LOCKING);
      TA_started = TRUE;
   }
#endif

   if (streql (basename, "cfe"))
   {  assert_installation();
      status = cfe_main (argc, argv);
   }
   else if (streql (basename, "cnt"))   status = cnt_main  (argc, argv);
   else if (streql (basename, "genCC")) status = gen_main  (argc, argv);
   else if (streql (basename, "scp"))   status = scp_main  (argc, argv);
   else if (streql (basename, "init"))  status = init_main (argc, argv);
   else if (streql (basename, "dmp"))   status = dmp_main  (argc, argv);
   else if (streql (basename, "sil"))   status = INCRLDTOOL(sil_main);
   else if (streql (basename, "sync_switch"))
      status = JOYCETOOL(sync_switch_main);
   else
   {  err_raise (err_USE, "unknown main routine", argv[0]);
      status = 1;
   }

#ifdef OBST_HAVE_JOYCE
   if (TA_started)
      TRANSACTION.commit();
#endif

   T_EXIT ();
   return status;
}
