#line 1 "./src/agg/MapList.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "agg.h"

// **************************************************************************
// Module Mapping_List              20/10/92                  Jochen Alt (ja)
// **************************************************************************
// implements methods of classes: Mapping_List
// **************************************************************************

// Vorsicht !!!
// wildester Hack !
// Die Knoten werden direkt ueber psm abgespeichert, um bei einheitlicher
// Typinformation bzw. einheitlichem Container viiiiel Platz zu sparen. 
// Auf die Methode kann guenstigstenfalls ein Mapping_List von 44 Bytes auf
// 16 Bytes pro Eintrag Schrumpfen !!!
// 
// Es wird fuer den Key und den Entity ein default Typ und ein default
// Container abgespeichert, beide werden beim Eintragen des ersten Elementes 
// gesetzt. Entspricht ein folgender Eintrag einem solchen Defaultwert,
// wird nur das unbedingt notwendige gespeichert. In der Standardanwendung,
// in der der Key immer denselben Typ hat und im selben Container liegt,
// braucht ein Knoten dann 28 Bytes (immer noch eine Platzersparnis von 40%)
// 
// 
// Ein Knoten wird in folgendem Format abgespeichert:
//   sos_Offset : Verweis auf Vorgaenger
//   sos_Offset : Verweis auf naechsten Knoten
//   sos_Int    : Typinformation des Knotens und Groesse des Knotens
// 
//   falls type != std-type: sos_Id Key-Typ Information
//   falls cnt != cnt-type : sos_Container Key-Container Information
//   sos_Offset Key-Offset
//
//   falls type != std-type: sos_Id Info-Typ Information
//   falls cnt != cnt-type : sos_Container Info-Container Information
//   sos_Offset Info-Offset
// 
// Der Typ eines Knotens wird durch die folgenden Konstanten identifiziert,
// die die Wertigkeit des Bits angeben, welches bestimmt, ob ein jeweiliges
// Datum abgespeichert wurde. Beispiel: Ist in einem Knoten
// der Key-Typ abgespeichert, weil er dem Standard-Keytyp nicht entspricht, 
// so ist das KEY_TYPE_DIFF.te Bit gesetzt.
// Die Groesse des Knotens ist im 2.Byte abgespeichert (Koennte man sich
// aus dem Type bequem berechnen, den Platz im sos_Int kriegt man jedoch 
// geschenkt).

LOCAL const sos_Offset NULL_OFFSET=0;
#define KEY_CNT_DIFF 1
#define KEY_TYPE_DIFF 2
#define INFO_CNT_DIFF 4
#define INFO_TYPE_DIRR 8
// Die Groesse befindet sich im 2.ten Byte der Typinformation eines Knotens
#define SIZE_POS 256  

// Erzeuge den Typ Node_type mit _AGG_NODE_TYPE_SIZE und den entsprechenden bcopies
#define Node_type sos_Int
#define _AGG_NODE_TYPE_SIZE SOS_INT_SIZE
#define bcopy_from_Node_type bcopy_from_sos_Int
#define bcopy_to_Node_type bcopy_to_sos_Int


LOCAL const int _AGG_MAX_NODE_SIZE = 2*SOS_OFFSET_SIZE + _AGG_NODE_TYPE_SIZE + 
			  2*(SOS_CONTAINER_SIZE + 
			     SOS_OFFSET_SIZE +
			     SOS_ID_SIZE);

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::set_next (const sos_Typed_id&_OBSThis,sos_Offset offset,
						   sos_Offset next)
// ************************************************************************** 
// Setze in dem Knoten am Offset offset den Vorwaertsverweis auf den Knoten
// am Offset next
const{  if (offset == NULL_OFFSET)
   { set_first (_OBSThis,next);
     return;
   }

   err_assert (offset != next,"Mapping:set_next:circle");
   union {sos_Offset dummy;
          char mem[_AGG_MAX_NODE_SIZE]; } u;

   char* ptr = &u.mem[0];
   ptr += SOS_OFFSET_SIZE;
   bcopy_from_sos_Offset (&next, ptr);

   _OBSThis.container().write (offset+SOS_OFFSET_SIZE, SOS_OFFSET_SIZE, ptr);
}  // *** Mapping_List::set_next ***

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::set_prev (const sos_Typed_id&_OBSThis,sos_Offset offset,
                                                   sos_Offset prev)
// ************************************************************************** 
// Setze in dem Knoten am Offset offset den Vorwaertsverweis auf den Knoten
// am Offset next
const{  if (offset == NULL_OFFSET)
   { set_last (_OBSThis,prev);
     return;
   }

   err_assert (offset != prev,"Mapping:set_prev:circle");
   union {sos_Offset dummy;
          char mem[_AGG_MAX_NODE_SIZE]; } u;

   char* ptr = u.mem;
   bcopy_from_sos_Offset (&prev, ptr);
   
   _OBSThis.container().write (offset, SOS_OFFSET_SIZE, &u);
}  // *** Mapping_List::set_prev ***


// ************************************************************************** 
sos_Offset _sos_Object_sos_Object_Mapping_List::create_node 
                               (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) key,OBST_PARDECL( sos_Object) info, 
                                sos_Offset prev, sos_Offset next)
// ************************************************************************** 
// create a node with contents and return the offset
const{  // garuantee correct Byte position
   union {sos_Offset dummy;
          char mem[_AGG_MAX_NODE_SIZE]; } u;

   sos_Id key_default_type = get_key_default_type(_OBSThis);
   sos_Id info_default_type = get_info_default_type(_OBSThis);
   sos_Container key_default_container = get_key_default_cnt(_OBSThis);
   sos_Container info_default_container = get_info_default_cnt(_OBSThis);

   sos_Container cnt = _OBSThis.container();
   if (card(_OBSThis) == 0)
   {  set_key_default_type (_OBSThis,key_default_type = key._type_id());
      set_key_default_cnt(_OBSThis,key_default_container=key.container());
      set_info_default_type (_OBSThis,info_default_type = info._type_id());
      set_info_default_cnt(_OBSThis,info_default_container=info.container());
   }
   
   char* ptr = u.mem;
   // ueberspringe die Vorgaenger, Nachfolger und den Knotentyp
   ptr += 2*SOS_OFFSET_SIZE; 
   ptr += _AGG_NODE_TYPE_SIZE;
   Node_type node_type = 0;

   sos_Id key_type = key._type_id();
   if (key_type != key_default_type)
   {  bcopy_from_sos_Id (&key_type, ptr);
      ptr += SOS_ID_SIZE;
      node_type += KEY_TYPE_DIFF;
   }
   sos_Container key_container = key.container();
   if (key_container != key_default_container)
   {  bcopy_from_sos_Container (&key_container, ptr);
      ptr += SOS_CONTAINER_SIZE;
      node_type += KEY_CNT_DIFF;
   }
   sos_Offset key_offset = key.offset();
   bcopy_from_sos_Offset (&key_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   sos_Id info_type = info._type_id();
   if (info_type != info_default_type)
   {  bcopy_from_sos_Id (&info_type, ptr);
      ptr += SOS_ID_SIZE;
      node_type += INFO_TYPE_DIRR;
   }
   sos_Container info_container = info.container();
   if (info_container != info_default_container)
   {  bcopy_from_sos_Container (&info_container, ptr);
      ptr += SOS_CONTAINER_SIZE;
      node_type += INFO_CNT_DIFF;
   }
   sos_Offset info_offset = info.offset();
   bcopy_from_sos_Offset (&info_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   sos_Int size = ptr - u.mem;
   node_type += size*SIZE_POS;
   ptr = u.mem;
   bcopy_from_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_from_sos_Offset (&next, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_from_Node_type (&node_type, ptr);

   sos_Offset offset = cnt.allocate (size);
   err_assert (size<=_AGG_MAX_NODE_SIZE,"create_node:node too big");
   cnt.write (offset, size, &u);

   set_size( _OBSThis,get_size(_OBSThis) + size);
   set_cardinality( _OBSThis,get_cardinality(_OBSThis) + 1);

   set_prev (_OBSThis,next, offset);
   set_next (_OBSThis,prev, offset);
   return offset;
}  // *** Mapping_List::create_node ***


// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::remove_node(const sos_Typed_id&_OBSThis,sos_Offset offset, sos_Int size)
// ************************************************************************** 
// remove a node at position with the size of size
const{  _OBSThis.container().deallocate (offset,size);
   set_size (_OBSThis,get_size(_OBSThis) - size);
   set_cardinality( _OBSThis,get_cardinality(_OBSThis) - 1);
} // ** Mapping_List::remove_node ***

// **************************************************************************
void _sos_Object_sos_Object_Mapping_List::read_node
                       (const sos_Typed_id&_OBSThis,sos_Offset offset,
                        sos_Object &key, sos_Object &info,
                        sos_Offset &prev, sos_Offset &next,
                        sos_Int &size)
// **************************************************************************
// read a node at position offset, and set the parameters
const{  sos_Id key_type = get_key_default_type(_OBSThis);
   sos_Id info_type = get_info_default_type(_OBSThis);
   sos_Container key_container = get_key_default_cnt(_OBSThis);
   sos_Container info_container = get_info_default_cnt(_OBSThis);

   union {sos_Offset dummy;
          char mem[_AGG_MAX_NODE_SIZE]; } u;

   sos_Container cnt = _OBSThis.container();
   cnt.read (offset, _AGG_NODE_TYPE_SIZE +2*SOS_OFFSET_SIZE, &u);

   char* ptr = u.mem;
   Node_type node_type;
   bcopy_to_sos_Offset (&prev, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_sos_Offset (&next, ptr);
   ptr += SOS_OFFSET_SIZE;
   bcopy_to_Node_type(&node_type, ptr);
   ptr += _AGG_NODE_TYPE_SIZE;
   size = node_type/SIZE_POS;

   err_assert (size<=_AGG_MAX_NODE_SIZE,"read_node:internal error");
   cnt.read (offset, size, &u);
   if (node_type BITAND KEY_TYPE_DIFF)
   {  bcopy_to_sos_Id (&key_type, ptr);  
      ptr += SOS_ID_SIZE;
   }
   if (node_type BITAND KEY_CNT_DIFF)
   {  bcopy_to_sos_Container (&key_container, ptr);  
      ptr += SOS_CONTAINER_SIZE;
   }
   sos_Offset key_offset;
   bcopy_to_sos_Offset (&key_offset, ptr);
   ptr += SOS_OFFSET_SIZE;

   if (node_type BITAND INFO_TYPE_DIRR)
   {  bcopy_to_sos_Id (&info_type, ptr);  
      ptr += SOS_ID_SIZE;
   }
   if (node_type BITAND INFO_CNT_DIFF)
   {  bcopy_to_sos_Container (&info_container, ptr);  
      ptr += SOS_CONTAINER_SIZE;
   }
   sos_Offset info_offset;
   bcopy_to_sos_Offset (&info_offset, ptr);
   ptr += SOS_OFFSET_SIZE;
   
 
   sos_Typed_id key_tpid = 
      sos_Typed_id::make (sos_Id::make (key_container, key_offset),key_type);
   sos_Typed_id info_tpid = 
      sos_Typed_id::make (sos_Id::make (info_container, info_offset),info_type);
   key = sos_Object::make  (key_tpid);
   info = sos_Object::make (info_tpid);
} // *** Mapping_List::read_node ***

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::local_initialize
                                    (OBST_PARDECL(sos_Object_sos_Object_Mapping_List) map)
// ************************************************************************** 
{  T_PROC("Mapping_List::local_initialize");
   TT(agg_H, T_ENTER);

   map.set_first(NULL_OFFSET);
   map.set_last(NULL_OFFSET);
   map.set_cardinality (0);
   map.set_size (0);

   TT(agg_H,T_LEAVE);
}; // ** Mapping_List::local_initialize **


// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::local_finalize
                                       (OBST_PARDECL(sos_Object_sos_Object_Mapping_List) map)
// ************************************************************************** 
{  T_PROC("Mapping::local_finalize");
   TT (agg_H, T_ENTER);

   map.clear();

   TT (agg_H, T_LEAVE);
} // ** Mapping_List::local_finalize **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::is_key (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) key)
// ************************************************************************** 
const{  T_PROC("Mapping_List::is_key");
   TT(agg_H, T_ENTER);

   sos_Bool found = FALSE;
   sos_Bool key_based_on_equal = get_role1_based_on_equal(_OBSThis);
   sos_Offset next;
   sos_Offset prev;
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Object tmp_key;
      sos_Object tmp_info;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (key, tmp_key,key_based_on_equal, EQ_STRONG))
      {  found = TRUE;
         break; 
      }
      list = next;
   }
   
   TT(agg_H, T_LEAVE);
   return found;
} // ** Mapping_List::is_key **


// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::is_info (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) info)
// ************************************************************************** 
// TRUE, falls info mit insert(*,info) in die Struktur
// aufgenommen wurde.
const{  T_PROC("Mapping_List::is_info");
   TT(agg_H,T_ENTER);

   sos_Bool found = FALSE;
   sos_Bool info_based_on_equal = get_role2_based_on_equal(_OBSThis);
   sos_Offset next;
   sos_Offset prev;
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Object tmp_key;
      sos_Object tmp_info;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (info, tmp_info, info_based_on_equal, EQ_STRONG))
      {  found = TRUE;
         break; 
      }
      list = next;
   }

   TT(agg_H, T_LEAVE);
   return found;
}  // Mapping_List::is_info

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::insert
                             (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) Key,OBST_PARDECL( sos_Object) Entity)
// ************************************************************************** 
const{  T_PROC("Mapping_List::insert");
   TT(agg_H, T_ENTER);

   err_assert (Key != NO_OBJECT, "Mapping_List::Don't insert NO_OBJECT");

   sos_Offset prev = NULL_OFFSET;
   sos_Bool found = FALSE;
   sos_Offset next;
   sos_Bool key_based_on_equal = get_role1_based_on_equal(_OBSThis);
   for (sos_Offset list = get_first(_OBSThis); (NOT found) AND (list != NULL_OFFSET); )
   {  sos_Object tmp_key;
      sos_Object tmp_info;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (Key, tmp_key, key_based_on_equal, EQ_STRONG))
      {  remove_node (_OBSThis,list,size);
         create_node (_OBSThis,Key, Entity, prev, next);
         found = TRUE;
         break; 
      }
      list = next;
   }

   if (NOT found)
   {  sos_Offset prev = get_last(_OBSThis);
      sos_Offset next = NULL_OFFSET;
      create_node (_OBSThis,Key,Entity, prev, next);
   }

   TT(agg_H,T_LEAVE);   
} // ** Mapping_List::insert **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::remove (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) key)
// ************************************************************************** 
const{  T_PROC("Mapping_List::remove");

   sos_Offset prev = NULL_OFFSET;
   sos_Offset next;
   sos_Bool key_based_on_equal = get_role1_based_on_equal(_OBSThis);
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Object tmp_key;
      sos_Object tmp_info;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (key, tmp_key, key_based_on_equal, EQ_STRONG))
      {  remove_node (_OBSThis,list,size);
         set_next (_OBSThis,prev, next);
         set_prev (_OBSThis,next, prev);
         break;
      }
      list = next;
   }

   TT(agg_H,T_LEAVE)
}// ** Mapping_List::remove **

// ************************************************************************** 
sos_Object _sos_Object_sos_Object_Mapping_List::__index (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) key)
// ************************************************************************** 
const{  T_PROC("Mapping_List::operator[]");
   TT(agg_H,T_ENTER);

   sos_Object info = NO_OBJECT;
   sos_Offset next, prev;
   sos_Bool key_based_on_equal = get_role1_based_on_equal(_OBSThis);
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Object tmp_key = NO_OBJECT;
      sos_Object tmp_info = NO_OBJECT;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (key, tmp_key, key_based_on_equal,EQ_STRONG))
      {  info = tmp_info;
	 break;
      }
      list = next;
   }

   TT(agg_H,T_LEAVE);
   return info;
} // ** Mapping_List::operator[] **

// ************************************************************************** 
sos_Object _sos_Object_sos_Object_Mapping_List::get_key(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  T_PROC("Mapping_List::get_key(sos_Cursor)");
   TT(agg_H, T_ENTER);

   err_assert (is_valid(_OBSThis,c), "Mapping_List:get_key:cursor invalid");

   sos_Object key = sos_Mapping_List_node::make (c.get_current()).get_key();

   TT(agg_H,T_LEAVE);
   return key;
} // ** Mapping_List::get_key **

// ************************************************************************** 
sos_Object _sos_Object_sos_Object_Mapping_List::get_info(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  T_PROC("Mapping_List::get_info(sos_Cursor)");
   TT(agg_H, T_ENTER);

   err_assert(is_valid(_OBSThis,c),"Mapping_List:get_info:cursor invalid");
   // err_assert (c.defined_for (self), "Mapping_List:get_info");

   sos_Object info = sos_Mapping_List_node::make (c.get_current()).get_info();

   TT(agg_H, T_LEAVE);
   return info;
} // ** Mapping::get_info **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::get_succ_pred
                                      (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Mapping_List_node) node, 
	                               sos_Offset &next, sos_Offset &prev)
// ************************************************************************** 
// if list_cursor == FALSE, use the forward and backward references
// in the sos_Mapping_node, otherwise read the node from psm
const{  sos_Bool lc = get_list_cursor(_OBSThis);
   if (lc)
   {  node.check_list_cursor (sos_Object_sos_Object_Mapping_List::make(_OBSThis,this));
      prev = node.get_prev();
      next  = node.get_next();
   }
   else
   {  sos_Int size;
      sos_Object key, info;
      read_node (_OBSThis,node.get_offset(), key, info, prev, next, size);
   }
} // *** get_succ_pred

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::set_info(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL( sos_Object) o)
// ************************************************************************** 
const{  T_PROC ("Mapping_List::set_info");
   TT (agg_H, T_ENTER);

   // err_assert ((c.defined_for (self)), "Mapping_List:set_info");
   err_assert(is_valid(_OBSThis,c),"Mapping_List:set_info:cursor invalid");

   sos_Mapping_List_node node = sos_Mapping_List_node::make(c.get_current());
   sos_Offset prev, next;
   get_succ_pred (_OBSThis,node, next, prev);
   create_node (_OBSThis,node.get_key(), o, prev, next);
   node.set_info (o);

   TT (agg_H, T_LEAVE);
} // ** Mapping::set_info **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::move_cursor
					(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL( sos_Object) key)
// ************************************************************************** 
// sets the cursor c to the key object in the mapping that corresponds
// to the given key
const{  T_PROC ("Mapping_List::move_cursor");
   TT( agg_H, T_ENTER);

   // err_assert ((c.defined_for (self)), "Mapping_List:move_cursor");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current()); 
   sos_Offset prev = NULL_OFFSET;
   sos_Offset next;
   sos_Bool found = FALSE;
   sos_Bool key_based_on_equal  = get_role1_based_on_equal(_OBSThis);
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Object tmp_key;
      sos_Object tmp_info;
      sos_Int size;
      read_node (_OBSThis,list, tmp_key,tmp_info, prev, next, size);
      if (agg_same_entity (key, tmp_key, key_based_on_equal, EQ_STRONG))
      {  found = TRUE;
         node.set_key (tmp_key);
         node.set_info (tmp_info);
	 node.set_next (next);
	 node.set_prev (prev);
	 node.set_offset (list);
         break;
      }
      list = next;
   }
   if (NOT found)
      node.set_key (NO_OBJECT);

   TT(agg_H, T_LEAVE);
   return found;
} // ** Mapping_List::move_cursor **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::insert_before (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL(
                                                        sos_Object) Key,OBST_PARDECL(
                                                        sos_Object) Entity)
// ************************************************************************** 
const{  T_PROC("Mapping_List::insert_before");
   TT(agg_H, T_ENTER);

   err_assert (is_valid(_OBSThis,c), "Mapping_List:insert_before:cursor invalid");
   // err_assert (c.defined_for (self), "Mapping_List:insert_before");
   err_assert (NOT is_key (_OBSThis,Key), 
	       "Mapping_List:insert_before:key already inserted");
   err_assert (Key != NO_OBJECT, "Mapping_List::Don't insert NO_OBJECT");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Offset next, prev;
   sos_Offset node_offset = node.get_offset();
   get_succ_pred (_OBSThis,node, next, prev);
   create_node (_OBSThis,Key,Entity,prev, node_offset);
 
   TT(agg_H, T_LEAVE);
} // ** Mapping_List::insert_before **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::insert_after (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c,OBST_PARDECL(
                                                       sos_Object) Key,OBST_PARDECL(
                                                       sos_Object) Entity)
// ************************************************************************** 
const{  T_PROC("Mapping_List::insert_after");
   TT(agg_H, T_ENTER);
 
   err_assert (is_valid(_OBSThis,c), "Mapping_List:insert_after");
   // err_assert (c.defined_for (self), "Mapping_List:insert_after");
   err_assert (NOT is_key (_OBSThis,Key), 
	       "Mapping_List:insert_after:key already inserted");
   err_assert (Key != NO_OBJECT, "Mapping_List::Don't insert NO_OBJECT");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Offset prev, next;
   get_succ_pred (_OBSThis,node, next, prev);
   sos_Offset node_offset = node.get_offset();
   create_node (_OBSThis,Key,Entity,node_offset, next);

   TT(agg_H, T_LEAVE);
} // ** Mapping::insert_after **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::local_assign
                                       (OBST_PARDECL(sos_Object_sos_Object_Mapping_List) x,OBST_PARDECL(
                                        sos_Object)                    o)
// ************************************************************************** 
{  T_PROC("Mapping_List::local_assign");

   sos_Object_sos_Object_Mapping_List y = 
      sos_Object_sos_Object_Mapping_List::make (o);
   x.clear();

   agg_iterate_association (y, sos_Object key, sos_Object info)
   {  x.insert (key,info); }
   agg_iterate_association_end (y, key, info)

   TT(agg_H,T_LEAVE); 
} // ** Mapping_List::local_assign **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::local_equal
                                  (OBST_PARDECL(sos_Object_sos_Object_Mapping_List) x,OBST_PARDECL(
                                   sos_Object)                         o,
                                   sos_Eq_kind                        eq_kind) 
// ************************************************************************** 
{  sos_Bool result;

   sos_Object_sos_Object_Mapping_List y = 
      sos_Object_sos_Object_Mapping_List::make (o);

   if (y.card() != x.card())
      result = FALSE; 
   else
   {  result = TRUE;
      sos_Bool info_based_on_equal = x.get_role2_based_on_equal();
      agg_iterate_association (x, sos_Object key, sos_Object info)
      {  if (NOT agg_same_entity (y[key],info,info_based_on_equal,eq_kind))
	 {  result = FALSE;
	    break;
	 }
      } 
      agg_iterate_association_end (x, key, info);
   }
   return result;
} // ** Mapping_List::local_equal **

// ************************************************************************** 
sos_Int _sos_Object_sos_Object_Mapping_List::local_hash_value
                                       (OBST_PARDECL(sos_Object_sos_Object_Mapping_List) m)
// ************************************************************************** 
{  
   return m.get_cardinality();
} // ** Mapping::local_hash_value **

// ************************************************************************** 
sos_Int _sos_Object_sos_Object_Mapping_List::size(const sos_Typed_id&_OBSThis)
// ************************************************************************** 
const{  return get_size(_OBSThis) + _sos_Object::size(_OBSThis);
} // ** Mapping_List::size **

// **************************************************************************
sos_Bool _sos_Object_sos_Object_Mapping_List::is_role1 (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) key)
// **************************************************************************
const{  return is_key (_OBSThis,key);
} // ** Mapping::is_role1 **

// **************************************************************************
sos_Bool _sos_Object_sos_Object_Mapping_List::is_role2 (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Object) info)
// **************************************************************************
const{  return is_info (_OBSThis,info);
} // ** Mapping::is_role2 **

// **************************************************************************
sos_Object _sos_Object_sos_Object_Mapping_List::get_role1 (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{  return get_key (_OBSThis,c);
} // ** Mapping::get_role1 **

// **************************************************************************
sos_Object _sos_Object_sos_Object_Mapping_List::get_role2 (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{  return get_info (_OBSThis,c);
} // ** Mapping::get_role2 **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::remove_at(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  T_PROC("Mapping_List::remove_at");
   TT(agg_H, T_ENTER);

   err_assert (is_valid(_OBSThis,c), "Mapping_List:remove_at:cursor invalid");
   // err_assert (c.defined_for (self), "Mapping_List:remove_at");
   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());

   sos_Offset offset = node.get_offset();
   sos_Offset prev, next;
   get_succ_pred (_OBSThis,node, next, prev);
   to_succ (_OBSThis,c);
   node = sos_Mapping_List_node::make (c.get_current());

   sos_Int size;
   sos_Object key;
   sos_Object info;

   read_node (_OBSThis,offset, key,info, prev, next,size);
   remove_node (_OBSThis,offset, size);
   set_next (_OBSThis,prev, next);
   set_prev (_OBSThis,next, prev);
   node.set_prev (prev);

   TT(agg_H, T_LEAVE);
} // ** Mapping_List::remove_at **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::clear(const sos_Typed_id&_OBSThis)
// ************************************************************************** 
const{  T_PROC("Mapping_List::clear");
   TT(agg_H, T_ENTER);

   sos_Offset next, prev;
   for (sos_Offset list = get_first(_OBSThis); list != NULL_OFFSET; )
   {  sos_Int size;
      sos_Object tmp_key;
      sos_Object tmp_info;

      read_node (_OBSThis,list, tmp_key, tmp_info, prev, next, size);
      remove_node (_OBSThis,list, size);
      list = next;
   }
   set_first(_OBSThis,NULL_OFFSET);
   set_last(_OBSThis,NULL_OFFSET);
   set_cardinality (_OBSThis,0);
   set_size (_OBSThis,_sos_Object::size(_OBSThis));

   TT(agg_H, T_LEAVE);
} // ** Mapping_List::clear **

// **************************************************************************
sos_Int _sos_Object_sos_Object_Mapping_List::card (const sos_Typed_id&_OBSThis)
// **************************************************************************
const{  T_PROC ("Mapping::card");
   TT (agg_H, T_ENTER);

   sos_Int crd = get_cardinality(_OBSThis);

   TT (agg_H, T_LEAVE);
   return crd;
} // ** Mapping_List::card **

// ************************************************************************** 
sos_Cursor _sos_Object_sos_Object_Mapping_List::open_cursor(const sos_Typed_id&_OBSThis,sos_Container cnt)
// ************************************************************************** 
const{  T_PROC ("Mapping_List::open_cursor");
   TT( agg_H, T_ENTER);

   sos_Cursor c = sos_Cursor::create (cnt, sos_Object_sos_Object_Mapping_List::make(_OBSThis,this));
   sos_Mapping_List_node node = 
      sos_Mapping_List_node::create (cnt,get_list_cursor(_OBSThis));
   c.set_current (node);
   to_first (_OBSThis,c);

   TT(agg_H, T_LEAVE);
   return c;
} // ** Mapping::open_cursor **

// ************************************************************************** 
void _sos_Object_sos_Object_Mapping_List::close_cursor(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{     
   err_assert (c != NO_OBJECT, "Mapping_List:close_cursor:cursor invalid");
   c.get_current().destroy();
   c.destroy();
} // ** Mapping_List::close_cursor **

// ************************************************************************** 
sos_Cursor _sos_Object_sos_Object_Mapping_List::duplicate
(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, sos_Container cnt /* = TEMP_CONTAINER */)
// ************************************************************************** 
const{  // err_assert ((c.defined_for (self)), "Mapping_List:duplicate");

   sos_Cursor new_c = open_cursor (_OBSThis,cnt);
   new_c.get_current().assign (c.get_current());
   return new_c;
} // ** Mapping::duplicate **

// **************************************************************************
sos_Bool _sos_Object_sos_Object_Mapping_List::is_valid (const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// **************************************************************************
const{  sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   return (node.get_key() != NO_OBJECT);
} // ** is_valid **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::to_first(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  err_assert (c != NO_OBJECT, "Mapping_List::to_first:cursor invalid");
   // err_assert ((c.defined_for (self)), "Mapping_List:to_first");
  
   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Object key = NO_OBJECT;
   sos_Object info = NO_OBJECT;
   sos_Int size = 0;
   sos_Offset next = NULL_OFFSET;
   sos_Offset prev = NULL_OFFSET;
   sos_Offset offset = get_first(_OBSThis);
   if (offset != NULL_OFFSET)
      read_node (_OBSThis,offset, key,info, prev, next,size);

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_key (key);
   node.set_info (info);

   return is_valid (_OBSThis,c);
} // ** Mapping::to_first **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::to_last(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c)
// ************************************************************************** 
const{  err_assert (c != NO_OBJECT, "Mapping_List::to_last:cursor invalid");
   // err_assert ((c.defined_for (self)), "Mapping_List:to_last");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Object key = NO_OBJECT;
   sos_Object info = NO_OBJECT;
   sos_Int size = 0;
   sos_Offset next = NULL_OFFSET;
   sos_Offset prev = NULL_OFFSET;
   sos_Offset offset = get_last(_OBSThis);
   if (offset != NULL_OFFSET)
      read_node (_OBSThis,offset, key,info, prev, next,size);

   node.set_offset (offset);
   node.set_prev (prev);
   node.set_next (next);
   node.set_key (key);
   node.set_info (info);


   return is_valid (_OBSThis,c);
} // ** Mapping_List::to_last **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::to_succ(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, Index i) 
// ************************************************************************** 
const{  err_assert (c != NO_OBJECT, "Mapping_List::to_succ:cursor invalid");
   err_assert (is_valid(_OBSThis,c), "Mapping_List:to_succ:cursor invalid");
   // err_assert (c.defined_for (self), "Mapping_List:to_succ");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Offset prev, next;
   get_succ_pred (_OBSThis,node, next, prev);
   sos_Offset list = node.get_offset();
   sos_Object key = node.get_key();
   sos_Object info = node.get_info();
   for (;((i>0) AND (next != NULL_OFFSET));)
   {  sos_Int size;
      list = next;
      read_node (_OBSThis,list, key,info,prev,next,size);
      i--;
   }
   if (i>0)
   {  list = NULL_OFFSET;   // End of List
      key = NO_OBJECT;
   }
   node.set_prev (prev);
   node.set_next (next);
   node.set_offset (list);
   node.set_key (key);
   node.set_info (info);

   return is_valid (_OBSThis,c);
} // ** Mapping_List::to_succ **

// ************************************************************************** 
sos_Bool _sos_Object_sos_Object_Mapping_List::to_pred(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Cursor) c, Index i)
// ************************************************************************** 
const{  err_assert (c != NO_OBJECT, "Mapping_List::to_pred:cursor invalid");
   err_assert (is_valid(_OBSThis,c), "Mapping_List:to_pred:cursor invalid");
   // err_assert (c.defined_for (self), "Mapping_List:to_pred");

   sos_Mapping_List_node node = sos_Mapping_List_node::make (c.get_current());
   sos_Offset prev, next;
   sos_Object key, info;
   get_succ_pred (_OBSThis,node, next, prev);
   sos_Offset list = node.get_offset();
   for (;i>0 AND prev != NULL_OFFSET;)
   {  list = prev;
      sos_Int size;
      read_node (_OBSThis,list, key,info,prev, next,size);
      i--;
   } 
   if (i>0)
   {  list = NULL_OFFSET;
      key = NO_OBJECT;
   }
   node.set_prev (prev);
   node.set_next (next);
   node.set_offset (list);
   node.set_key (key);
   node.set_info (info);

   return is_valid (_OBSThis,c);
} // ** Mapping_List::to_pred **


