#ifndef _PSM_H
#define _PSM_H 1
/* -----------------------------------------------------------------------------
* Copyright 1992-1993 by Forschungszentrum Informatik(FZI)
*
* You can use and distribute this software under the terms of the license
* you should have received along with this software.
* If not or if you want additional information, write to
* Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
* D-76131 Karlsruhe , Germany.
* ------------------------------------------------------------------------------
*/
// *****************************************************************************
// Module psm                  Emil Sekerinski, Oliver Spatscheck, Walter Zimmer
//
// *****************************************************************************
//  persistent storage manager: header 
// *****************************************************************************

// *****************************************************************************
// CLASS DECLARATIONS
// *****************************************************************************

class sos_Object;
class sos_Container_set;
class sos_Container_cursor;
class sos_Container;

// *****************************************************************************
// PUBLIC INCLUDEFILES
// *****************************************************************************

#include "knl_use.h"
 
// *****************************************************************************
// TYPEDEFS AND ENUMS
// *****************************************************************************


typedef short                   sos_Short;
typedef unsigned                sos_Offset;
typedef int                     sos_Int;
typedef void*                   sos_Pointer;
 
// The next tow lines are implemented for our STONE project partners ...
// ... from dresden. See psm.C for more details.

typedef void(*psm_write_err_fct_type)(sos_Int id);
extern psm_write_err_fct_type psm_write_err_fct_ptr;

#define SOS_SHORT_SIZE          2
#define SOS_OFFSET_SIZE         4
#define SOS_INT_SIZE            4
#define SOS_POINTER_SIZE        4
#define SOS_CONTAINER_SIZE      SOS_INT_SIZE          

#ifndef NO_TT
   extern sos_Container _psm_checked_cnt;
#endif NO_TT

extern const sos_Container &TEMP_CONTAINER;
extern const sos_Container &ROOT_CONTAINER;
extern const sos_Container &SYNC_CONTAINER;	
extern const sos_Container &UNUSED_CONTAINER;
extern const sos_Offset    ROOT_OFFSET;

extern const char*	   psm_cntdir;

// *****************************************************************************
// CLASS DEFINITIONS
// *****************************************************************************

// ================= class sos_Container =======================================

class sos_Container
{
   sos_Int id; // 0<=id<2^32-1

public:

   static sos_Container make(sos_Int i) {sos_Container ct; ct.id=i; return ct;}
   static sos_Container create();

   operator sos_Int() const {return id;}
   sos_Bool operator==(const sos_Container c) const 
   { 
      return(sos_Bool)(id == c.id); 
   }

   sos_Container_status status() const;
   sos_Bool modified() const;
   sos_Bool exists() const;
   sos_Int occupied() const;
   sos_Existing_status object_exists(sos_Offset, sos_Int) const;

   sos_Open_result open(sos_Access_mode, sos_Sync_mode) const;
   sos_Open_result access(sos_Access_mode, sos_Sync_mode) const;
   void close() const;
   void read_close() const;                //for JOYCE
   void write_close() const;               //for JOYCE
   void destroy() const;

   void commit() const;
   void reset() const;
   void squeeze() const;
   void clear() const;
   void update() const;                        
   void compress() const;
   void print_freelists() const;                  // only for tests
   sos_Bool equal_container(sos_Container) const; // only for tests

   sos_Offset allocate(sos_Int) const;         //allocates ... 
                                               // ... realsize(size) bytes
   void deallocate(sos_Offset, sos_Int) const; //deallocates ...
                                               // ... realsize(size) bytes
   sos_Int realsize(sos_Int) const; 

   void read(sos_Offset, sos_Int, sos_Pointer) const;
   void write(sos_Offset, sos_Int, sos_Pointer) const;
   void copy(sos_Offset, sos_Int, sos_Container, sos_Offset) const;
   sos_Bool equal(sos_Offset, sos_Int, sos_Container, sos_Offset) const;
   sos_Int  hash_value(sos_Offset, sos_Int) const;

   sos_Object root_object() const;
};

// ================= class sos_Container_list ==================================

class sos_Container_list 
{
public:
  sos_Container_list();
  ~sos_Container_list();

  void add_cnt(sos_Container c);
  void rm_cnt(sos_Container c);
  sos_Container get_cnt(sos_Int n)
  { 
     return sos_Container::make(set[n]);
  };
  sos_Int get_cnt_n(){ return number_of_cnt; };

  void add_ref(){ number_of_references++ ; };
  void rm_ref(){ number_of_references-- ; };
  sos_Bool is_referenced();

private:
  unsigned sz, number_of_cnt, number_of_references;
  sos_Int* set;
};


// ================= class sos_Container_set ===================================

class sos_Container_set 
{
public:
  
   sos_Container_set(const sos_Container_set&);
   sos_Container_set& operator=(const sos_Container_set&);
   sos_Container_set clone();

   sos_Container_set();
   ~sos_Container_set();
   sos_Container_set& operator+=(sos_Container);
   sos_Container_set& operator-=(sos_Container);
   sos_Container_set& operator+=(sos_Container_set);
   sos_Container_set& operator-=(sos_Container_set);
   sos_Int card() const { return set->get_cnt_n(); }

   static sos_Container_set& open_containers(sos_Container_status); 
   static sos_Container_set& open_containers(sos_Container_mod_status);
   static sos_Open_result open(const sos_Container_set&, 
                               const sos_Container_set&,
                               const sos_Container_set&,
                               sos_Sync_mode);
   sos_Open_result open(sos_Access_mode, sos_Sync_mode) const;
   void close() const;

   static sos_Open_result op(int i, const sos_Container_set&, 
                             const sos_Container_set&, 
                             const sos_Container_set&, 
                             sos_Sync_mode); 

   static void cl(int i, const sos_Container_set&, 
                  const sos_Container_set&, 
                  const sos_Container_set&); 
  
   void commit() const;
   void reset() const;

   sos_Container_cursor open_cursor() const;
   void close_cursor(sos_Container_cursor&) const;
   sos_Bool to_succ(sos_Container_cursor&) const;
   sos_Bool is_valid(sos_Container_cursor) const;
   sos_Container get(sos_Container_cursor) const;

private:
   sos_Container_list * set;
};

// ================= sos_Container_cursor ======================================

class sos_Container_cursor 
{
friend class sos_Container_set;

private:
   int idx;

public:
   sos_Bool is_valid() const { return(sos_Bool)(idx >= 0); };
};

// ================= cnt_iterate ===============================================

#define cnt_iterate(s, e) \
{sos_Container_cursor _cnt_cursor = s.open_cursor(); \
for (sos_Bool _cnt_valid = s.is_valid(_cnt_cursor); \
      _cnt_valid; \
      _cnt_valid = s.to_succ(_cnt_cursor)) \
 { e = s.get(_cnt_cursor);

#define cnt_iterate_end(s, e) } s.close_cursor(_cnt_cursor);}

 
#endif
