/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 *
 * --------------------------------------------------------------------------
 * Implementation of the Tcl --> OBST Interface: Utilities
 * --------------------------------------------------------------------------
 * ORIGINAL: D. Theobald				DATE: 2/3/93
 * --------------------------------------------------------------------------
 */
/* tclOBST LIBRARY MODULE */

#define OBST_IMP_STRINGOP
#define OBST_IMP_FORMATTED_IO
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "smg.h"
#include "cci_use.h"

#include "_tclOBST.h"

#if TCL_MAJOR_VERSION < 7
extern "C" {
#include "tclHash.h"
}
#endif
#include "_c2obst.h"


// -----------------------------------------------------------------------
// DEFINITION: Utilities
// -----------------------------------------------------------------------

LOCAL Tcl_HashTable t2o_ip_table,
		    t2o_str_table;


LOCAL sos_Object_sos_Object_Mapping t2o_blocks;

LOCAL void _t2o_delete_blocks();


// -----------------------------------------------------------------------
// DEFINITION: encoding/decoding of object handles
// -----------------------------------------------------------------------

#define SCALAR_ENCODED		32
#define SCALAR_DECODED		16
#if t2o_ALWAYS_CODE_TYPE
#  define CLASS_ENCODED		32
#  define CLASS_DECODED		16
#else
#  define CLASS_ENCODED		16
#  define CLASS_DECODED		8
#endif
#define IMPL_OFFSET     	16
#define VALUE_OFFSET		0x40


// -----------------------------------------------------------------------
// IMPLEMENTATION: Initialization / Customization
// -----------------------------------------------------------------------

/*
 * Enter variant list into string table.
 */
LOCAL void t2o_enter_variants (const t2o_OptVariants &v)
{  for (int i = 0;  i < v.size;  ++ i)
      _t2o_set_strcode (v.variants[i].name, v.variants[i].code);
}

EXPORT void t2o_init()
{  T_PROC ("t2o_init")
   TT (t2o_M, T_ENTER);

   static int initialized = 0;
   if (initialized) { TT (t2o_M, T_LEAVE); return; }
   initialized = 1;

   // Force C interface (and thereby OBST) initialization and change
   // defaults afterwards.

   _c2obst_init();
   c2obst_ctrls.check_args    	    = TRUE;
   c2obst_ctrls.catch_syserrs 	    = TRUE;
   c2obst_ctrls.set_schemas_trigger = &_t2o_delete_blocks;

   // Create temporary mapping for registering memory blocks
   t2o_blocks = sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);

   // remaining module initialization

   _t2o_active_triggers = 0;

   obst_err_sethandler ((obst_proc*) &t2o_err_handler);

   Tcl_InitHashTable (&t2o_ip_table,  TCL_ONE_WORD_KEYS);
   Tcl_InitHashTable (&t2o_str_table, TCL_STRING_KEYS);

   t2o_enter_variants (t2o_custom_opts);
   t2o_enter_variants (t2o_bool_opts);
   t2o_enter_variants (t2o_const_opts);
   t2o_enter_variants (t2o_aggDirect_opts);
   t2o_enter_variants (t2o_aggCurr_opts);
   t2o_enter_variants (t2o_trigger_opts);
   t2o_enter_variants (t2o_trgCurr_opts);

   t2o_OptVariants *ovptr = &t2o_option[t2o_LastCommand()];
   for (int cmdcode = t2o_LastCommand();  cmdcode >= 0;  -- cmdcode, -- ovptr)
   {  ovptr->size = 0;
      for (int i = t2o_FirstOption();  !t2o_InvalidOption(i);  ++ i)
	 if (_t2o_option_data[t2o_Option2Idx(i)].cmd_code == cmdcode)
	    ++ ovptr->size;

      ovptr->variants = new t2o_OptVariant[ovptr->size];

      int count = 0;
      for (i = t2o_FirstOption();  !t2o_InvalidOption(i);  ++ i)
      {  int idx = t2o_Option2Idx(i);

	 if (_t2o_option_data[idx].cmd_code == cmdcode)
         {  _t2o_set_strcode (_t2o_option_data[idx].name, i);

	    ovptr->variants[count   ].name = _t2o_option_data[idx].name;
	    ovptr->variants[count ++].code = i;
	 }
      }
   }
   TT (t2o_M, T_LEAVE);
}

/*
 * Yields the 't2o_InterpData' object associated to 'interp'.
 * The result will be NULL if no such object exists and 'enter_flag == FALSE'.
 * Otherwise, a new 't2o_InterpData' object will be created, associated to
 * 'interp', and finally returned as result.
 *
 * (Strings are duplicated since they might get deleted in t2o_customize.)
 */
EXPORT t2o_InterpData* t2o_get_ipdata(const Tcl_Interp* interp, int enter_flag)
{  static int _dummy;

   T_PROC ("t2o_get_ipdata")
   TT (t2o_L, T_ENTER);

   register
   Tcl_HashEntry* eptr = Tcl_FindHashEntry (&t2o_ip_table, (char*)interp);
   if (eptr)
   {  TT (t2o_L, T_LEAVE);
      return (t2o_InterpData*)Tcl_GetHashValue (eptr);
   }
   if (!enter_flag)
   {  err_raise (err_USE, "interpreter not initialized", 
			  "tclOBST - t2o_get_ipdata", FALSE);
      TT (t2o_L, T_LEAVE);
      return NULL;
   }
   t2o_InterpData* ipd = new t2o_InterpData;
   ipd->interp 		       = (Tcl_Interp*)interp;
   ipd->obj_prefix	       = obst_strdup (t2o_OBJ_PREFIX);
   ipd->define_obj_cmds	       = t2o_OBJ_AS_CMD;
   ipd->allow_abbrevs	       = t2o_ALLOW_ABBREVS;
   ipd->allow_defaults	       = t2o_ALLOW_DEFAULTS;
   ipd->copy_Cstrings	       = t2o_COPY_CSTRINGS;
   ipd->defs_bound	       = FALSE;
   ipd->scalar_codelen	       = (SCALAR_ENCODED + 1) + strlen(t2o_OBJ_PREFIX);
   ipd->class_codelen	       = ipd->scalar_codelen
				       + (CLASS_ENCODED - SCALAR_ENCODED);
   ipd->loop_data	       = NULL;
   ipd->cmd_name[t2o_AggCmdC ] = obst_strdup (t2o_AGG_CMD);
   ipd->cmd_name[t2o_CallCmdC] = obst_strdup (t2o_CALL_CMD);
   ipd->cmd_name[t2o_CntCmdC ] = obst_strdup (t2o_CNT_CMD);
   ipd->cmd_name[t2o_CtrlCmdC] = obst_strdup (t2o_CTRL_CMD);
   ipd->cmd_name[t2o_UtilCmdC] = obst_strdup (t2o_UTIL_CMD);

   eptr = Tcl_CreateHashEntry (&t2o_ip_table, (char*)interp, &_dummy);
   Tcl_SetHashValue (eptr, (ClientData)ipd);

   TT (t2o_L, T_LEAVE);
   return ipd;
}

/*
 * Interface for customizing the interface module per Tcl interpreter by
 * setting the resource named 'what' to a value derived from 'value'.
 * Erroneous arguments are ignored, the result is then FALSE.
 *
 * The following resources are currently customizable:
 *   t2o_ObjPrefStr   <string> .. prefix of all object encodings
 *   t2o_ObjCmdStr    <bool>   .. define object encodings as commands
 *   t2o_AllowAbbrStr <bool>   .. allow argument abbreviations
 *   t2o_AllowDfltStr <bool>   .. allow default arguments
 *   t2o_CpCstrStr    <bool>   .. copy sos_Cstring values
 *   t2o_AggCmdStr    <string> .. name of aggregate handling command
 *   t2o_CallCmdStr   <string> .. name of method call command
 *   t2o_CntCmdStr    <string> .. name of container handling command
 *   t2o_CtrlCmdStr   <string> .. name of interface control command
 *   t2o_UtilCmdStr   <string> .. name of utility command
 *
 * <string> arguments will be copied prior to storing them in the interface
 * module. <bool> arguments must be one of the strings 'TRUE', 'FALSE'.
 */
EXPORT int t2o_customize (const Tcl_Interp* interp, const char* what,
						    const char* value)
{  T_PROC ("t2o_customize")
   TT (t2o_M, T_ENTER);

   int*		   boolp = NULL;
   char**	   strp  = NULL;
   t2o_InterpData* ipd 	 = t2o_get_ipdata ((Tcl_Interp*)interp, TRUE);
   int		   whatc = _t2o_get_strcode (what);

   switch (whatc)
   {  case t2o_ObjPrefC:
      {  int scalarlen = (SCALAR_ENCODED + 1) + strlen (value);
	 if (scalarlen >= TCL_RESULT_SIZE)
	 {  smg_String msg = smg_String("\"") + value + "\" too long";
	    err_raise (err_USE, msg.make_Cstring(SMG_BORROW), "t2o_customize");

	    TT (t2o_M, T_LEAVE);
	    return FALSE;
	 }
	 else
	 {  ipd->scalar_codelen = scalarlen;
	    ipd->class_codelen  = scalarlen + (CLASS_ENCODED - SCALAR_ENCODED);
	    strp	        = &ipd->obj_prefix;
	 }
	 break;
      }
      case t2o_AllowAbbrC: boolp = & ipd->allow_abbrevs;
			   break;
      case t2o_AllowDfltC: boolp = & ipd->allow_defaults;
			   break;
      case t2o_CpCstrC	 : boolp = & ipd->copy_Cstrings;
			   break;
      case t2o_ObjCmdC   : boolp = & ipd->define_obj_cmds;
			   break;

      case t2o_AggCmdC :
      case t2o_CallCmdC:
      case t2o_CntCmdC :
      case t2o_CtrlCmdC:
      case t2o_UtilCmdC: strp = & ipd->cmd_name[whatc];
			 break;

      default:
      {  smg_String msg = smg_String("unknown resource \"") + value + "\"";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW), "t2o_customize");

	 TT (t2o_M, T_LEAVE);
	 return FALSE;
      }
   }
   if (boolp)
   {  int valc;
      if ((valc = _t2o_get_strcode (value)) == t2o_FalseC)
	 *boolp = FALSE;
      else if (valc == t2o_TrueC)
	 *boolp = TRUE;
      else
      {  smg_String msg = smg_String("\"") + value + "\" - no boolean";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW), "t2o_customize");

	 TT (t2o_M, T_LEAVE);
	 return FALSE;
      }
   }
   if (strp)
   {  if (*strp)
         delete *strp;
      *strp = obst_strdup ((char*)value);
   }
   TT (t2o_M, T_LEAVE);
   return TRUE;
}

/*
 * Binds the commands and further definitions declared in this interface to the
 * given interpreter.
 * `all_defs` specifies if just the control command, or all definitions are to
 * be bound. The complete definitions are bound just once.
 * The control command may be bound repeatedly. In this case, its old binding
 * is retracted.
 *
 * The return value indicates whether all definitions were bound in this
 * invocation (TRUE), or not.
 */
EXPORT sos_Bool _t2o_BindDefs (Tcl_Interp* interp, sos_Bool all_defs)
{  T_PROC ("_t2o_BindDefs")
   TT (t2o_M, T_ENTER);

   t2o_init();

   t2o_InterpData* ipd    = t2o_get_ipdata (interp, TRUE);
   sos_Bool	   result = FALSE;

#define CreateCmd(idx,proc) \
   Tcl_CreateCommand (interp, (char*)ipd->cmd_name[idx], (Tcl_CmdProc*)proc,\
		      (ClientData)NULL, (Tcl_CmdDeleteProc*)NULL)

      // Note: ipd->cmd_name[idx] might not be associated with t2o_*CmdC!

   if (all_defs)
   {  if (! ipd->defs_bound)
      {  CreateCmd (t2o_AggCmdC,  t2o_AggCmd);
         CreateCmd (t2o_CallCmdC, t2o_CallCmd);
         CreateCmd (t2o_CntCmdC,  t2o_CntCmd);
         CreateCmd (t2o_UtilCmdC, t2o_UtilCmd);
	 
         ipd->defs_bound = TRUE;
	 result		 = TRUE;
      }
   }
   static char* ctrl_cmd_name = NULL;

   if (ctrl_cmd_name)
   {  Tcl_DeleteCommand (interp, ctrl_cmd_name);
      delete ctrl_cmd_name;
   }
   CreateCmd (t2o_CtrlCmdC, t2o_CtrlCmd);
   ctrl_cmd_name = obst_strdup (ipd->cmd_name[t2o_CtrlCmdC]);

   TT (t2o_M, T_LEAVE);
   return result;
}


// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities -- Memory Management
// -----------------------------------------------------------------------

/*
 * Either record the given block (add == TRUE), or discard it from the 
 * registration and delete it.
 * (The mapping is logically used as a set - using the mapping is mainly for
 *  efficiency purposes.)
 */
EXPORT void t2o_manage_block (void* blk, sos_Bool add)
{  T_PROC ("t2o_manage_block")
   TT (t2o_L, T_ENTER);

   sos_Object key = make_sos_Pointer_object (blk);

   if (add)
      t2o_blocks.insert (key, sos_Object_type);
   else
   {  delete blk;
      t2o_blocks.remove (key);
   }
   TT (t2o_L, T_LEAVE);
}

/*
 * Delete all registered blocks and clear the registration.
 */
LOCAL void _t2o_delete_blocks ()
{  T_PROC ("_t2o_delete_blocks")
   TT (t2o_L, T_ENTER);

   err_assert (_t2o_active_triggers == 0,
	       "t2o_delete_blocks - schema set changed while trigger active");

   agg_iterate_association (t2o_blocks, sos_Object key, sos_Object info)
      delete make_sos_Pointer (key);
   agg_iterate_association_end (t2o_blocks, key, info);

   t2o_blocks.clear();

   TT (t2o_L, T_LEAVE);
}

// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities -- Object Encoding/Decoding
// -----------------------------------------------------------------------

/*
 * Converts the object pointed at by 'obj' into a printable string which is
 * then returned.
 * That result string will be placed in 'buffer', iff 'buffer' is not NULL.
 * Otherwise, a new buffer is allocated which must be freed by the caller.
 *
 * The type of 'obj' (class instance / scalar value) is given by 'is_scalar'.
 * This parameter does only exist, if 't2o_ALWAYS_CODE_TYPE == FALSE'.
 *
 * All object encodings of the same type (i.e. the same value of 'is_scalar')
 * will have the same length and the same initial prefix (provided the object
 * prefix is not changed via t2o_customize).
 *
 * 't2o_current_ip' must be defined.
 *
 * Two safe ways for providing a buffer are:
 *  - The static buffer of size TCL_RESULT_SIZE which is initially pointed at
 *    by a Tcl interpreter's 'result' field.
 *  - The buffer returned from a previous invocation of 't2o_obj2str' for
 *    an object of the same type.
 */
EXPORT char* _t2o_obj2str (const sos_Object *obj, char* buffer
#if !t2o_ALWAYS_CODE_TYPE
							     , int is_scalar
#endif
			 )
{  T_PROC ("_t2o_obj2str")
   TT (t2o_VL, T_ENTER);

#if c2o_ADD_DEBUG_STUFF
   err_assert (t2o_current_ip, "t2o_obj2str - t2o_current_ip not defined");
#endif

   register
   unsigned char *rptr, *ptr = (unsigned char *)_t2o_current_ip->obj_prefix,
		 *result;

   rptr   =
   result = (unsigned char *)
	    (buffer ? buffer
#if t2o_ALWAYS_CODE_TYPE
		    : new char[_t2o_current_ip->scalar_codelen]);
#else
		    : new char[is_scalar ? _t2o_current_ip->scalar_codelen
					 : _t2o_current_ip->class_codelen ]);
#endif

   while (*rptr = *(ptr ++))
      ++ rptr;
   
   ptr = (unsigned char*)obj
#if t2o_ALWAYS_CODE_TYPE
	    + (SCALAR_DECODED - 1);
#else
	    + (is_scalar ? SCALAR_DECODED-1 : CLASS_DECODED-1);
#endif
   do
   {  *(rptr ++) = (*ptr & 0xf) | VALUE_OFFSET;
      *(rptr ++) = (*ptr >> 4)  | VALUE_OFFSET;
   }
   while (-- ptr >= (unsigned char*)obj);

   *rptr = '\0';

   TT (t2o_VL, T_LEAVE);
   return (char*)result;
}

/*
 * Converts the string encoding 'str' into an sos_Object which will be
 * stored at '*obj'.
 * The function returns FALSE iff the conversion failed, i.e. if 'str' can
 * not be the result from a call to 't2o_obj2str'.
 *
 * 't2o_current_ip' must be defined.
 */
EXPORT int _t2o_str2obj (sos_Object *obj, register const char* str)
{  register unsigned char *optr;

   T_PROC ("_t2o_str2obj")
   TT (t2o_VL, T_ENTER);

#if c2o_ADD_DEBUG_STUFF
   err_assert (_t2o_current_ip, "_t2o_str2obj - _t2o_current_ip not defined");
#endif

   register const char* ptr;
   for (ptr = _t2o_current_ip->obj_prefix;  *ptr; )
      if (*(ptr ++) != *(str ++))
      {  TT (t2o_VL, T_LEAVE);
	 return FALSE;
      }
#if t2o_ALWAYS_CODE_TYPE
   optr = (unsigned char*)obj + (SCALAR_DECODED - 1);
#else
   register int is_scalar;
   if (str[CLASS_ENCODED])
   {  is_scalar = TRUE;
      optr      = (unsigned char*)obj + (SCALAR_DECODED - 1);
   }
   else
   {  is_scalar = FALSE;
      optr      = (unsigned char*)obj + (CLASS_DECODED - 1);
   }
#endif
   do
   {  if (((unsigned char)*str & 0xf0) != VALUE_OFFSET)
      {  TT (t2o_VL, T_LEAVE);
	 return FALSE;
      }
      *optr  = (unsigned char)*(str ++) & 0xf;
      *optr |= ((unsigned char)*str & 0xf) << 4;

      if (((unsigned char)*(str ++) & 0xf0) != VALUE_OFFSET)
      {  TT (t2o_VL, T_LEAVE);
	 return FALSE;
      }
   }
   while (-- optr >= (unsigned char*)obj);

   if (*str)
   {  TT (t2o_VL, T_LEAVE);
      return FALSE;
   }
#if !t2o_ALWAYS_CODE_TYPE
   if (!is_scalar)
      if (obj->offset() < ROOT_OFFSET)
      {  *obj = NO_OBJECT;

	 TT (t2o_VL, T_LEAVE);
	 return TRUE;
      }
      else
	 *(sos_Id*)((char*)obj + CLASS_DECODED)= ((sos_Id*)obj)->get_type_id();
#endif
   *(_sos_Object**)((char*)obj + IMPL_OFFSET)
      = (_sos_Object*)_cci_get_impl_obj (_sos_Object_type,
				         ( (sos_Typed_id*)obj )->get_tp());
   TT (t2o_VL, T_LEAVE);
   return TRUE;
}


// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities -- Object Encoding/Decoding: external interfaces
// -----------------------------------------------------------------------

#define t2o_CURRENT_IP_BLOCK(func,inner)	\
   T_PROC(func) TT(t2o_VL, T_ENTER);\
   t2o_InterpData* __save = _t2o_current_ip;\
   _t2o_current_ip = t2o_get_ipdata(interp, TRUE);\
   inner \
   _t2o_current_ip = __save; \
   TT(t2o_VL, T_LEAVE);

EXPORT char* t2o_obj2str (Tcl_Interp* interp, const sos_Object* objp)
{  t2o_CURRENT_IP_BLOCK("t2o_obj2str",
      char* result = t2o_OBJ2STR (objp, NULL,
				  ((sos_Object*)objp)->type().is_scalar());
   )
   return result;
}
EXPORT char* t2o_scalar2str (Tcl_Interp* interp, const sos_Object* objp)
{  t2o_CURRENT_IP_BLOCK("t2o_scalar2str",
      char* result = t2o_OBJ2STR (objp, NULL, TRUE);
   )
   return result;
}
EXPORT char* t2o_classinst2str (Tcl_Interp* interp, const sos_Object* objp)
{  t2o_CURRENT_IP_BLOCK("t2o_classinst2str",
      char* result = t2o_OBJ2STR (objp, NULL, FALSE);
   )
   return result;
}
EXPORT sos_Bool t2o_str2obj (Tcl_Interp* interp,
			     sos_Object* objp,
			     const char* str)
{  t2o_CURRENT_IP_BLOCK("t2o_str2obj",
     sos_Bool result = (sos_Bool)_t2o_str2obj (objp, str);
   )
   return result;
}


// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities -- Error Handling
// -----------------------------------------------------------------------

/*
 * Handles printing of OBST error messages: If a current interpreter is
 * defined (i.e. 't2o_current_ip != NULL'), the error is both recorded in
 * 'errorCode' and appended to the interpreter's 'result' field.
 * If multiple errors are raised, the error messages are accumulated in
 * 'result' and 'errorCode' is set to the description of the last raised error.
 * Warnings are handled as errors, too.
 *
 * The list stored in 'errorCode' consists of three elements, the first being
 * the string from 't2o_errcode' corresponding to the error class (see err(3)).
 * The second and third list element are made up by the results of
 * 'err_last_origin()' and 'err_last_raised()', respectively.
 *
 * Otherwise, the error message is logged onto 'stderr'.
 *
 * In any case, the flag 't2o_error' is set.
 */
EXPORT void t2o_err_handler ()
{  T_PROC ("t2o_err_handler")
   TT (t2o_M, T_ENTER);

   if (t2o_handle_errors)
   {  if (_t2o_current_ip)
      {  Tcl_SetErrorCode (_t2o_current_ip->interp,
			   t2o_errcode[obst_err_last_class()],
			   obst_err_last_origin(),
			   obst_err_last_raised(), (char*)NULL);
	 Tcl_AppendResult (_t2o_current_ip->interp,
			   t2o_errinfo[obst_err_last_class()],
			   obst_err_last_origin(),
			   ": ", obst_err_last_raised(), (char*)NULL);
      }
      else
	 fprintf (stderr, "%s%s: %s\n", t2o_errinfo[obst_err_last_class()],
		  obst_err_last_origin(),
		  obst_err_last_raised());
   }
   t2o_error = TRUE;

   TT (t2o_M, T_LEAVE);
}


// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities -- String Encoding
// -----------------------------------------------------------------------

/*
 * Associates the string 'str' with the code value 'code' for subsequent
 * calls of 't2o_get_strcode'.
 * 'code' must differ from 't2o_NoCode' (see below).
 */
EXPORT void _t2o_set_strcode (const char* str, int code)
{  T_PROC ("_t2o_set_strcode")
   TT (t2o_L, T_ENTER);

   static int _dummy;
   err_assert (code != t2o_NoCode, "t2o_set_strcode - invalid code");

   Tcl_SetHashValue (Tcl_CreateHashEntry (&t2o_str_table, (char*)str, &_dummy),
		     (ClientData)code);

   TT (t2o_L, T_LEAVE);
}

/*
 * Yields the code value associated to 'str', or 't2o_NoCode' if no such
 * association exists.
 */
EXPORT int _t2o_get_strcode (const char* str)
{  T_PROC ("_t2o_get_strcode")
   TT (t2o_L, T_ENTER);

   register
   Tcl_HashEntry* eptr = Tcl_FindHashEntry (&t2o_str_table, (char*)str);

   register
   int result = (eptr) ? (_t2o_StringCode)Tcl_GetHashValue (eptr)
   		       : t2o_NoCode;

   TT (t2o_L, T_LEAVE);
   return result;
}
