/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 *
 * --------------------------------------------------------------------------
 * Implementation of the Tcl --> OBST Interface: Tcl commands.
 * --------------------------------------------------------------------------
 * ORIGINAL: D. Theobald				DATE: 3/3/93
 * --------------------------------------------------------------------------
 */
/* tclOBST LIBRARY MODULE */

#define OBST_IMP_STRINGOP
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_MALLOC
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"

#include "_tclOBST.h"

// These functions are needed for the 'tclOBST trace' command -- even if
// the tclOBST library is not compiled with trace code.
// Hence, simply including "obst_trc.h" does not suffice.
extern "C" {
#if OBSTVERSIONNO >= 336
   extern void tt_init (const char* const, const int);
#  define t2o_T_INIT(name)  tt_init(name,TRUE)
#else
   extern void tt_init (const char* const);
#  define t2o_T_INIT(name)  tt_init(name)
#endif
   extern void tt_redef(const char* const);
   extern void tt_exit ();

   extern void tt_enter (const char* const);
   extern void tt_leave (const char* const);
   extern void tt_txt   (const char* const);
   extern void tt_nl	();

   extern unsigned char tt_active[];
}

// -----------------------------------------------------------------------
// DEFINITION: Triggers
// -----------------------------------------------------------------------

		// stored for trigger processing when a command gets executed
LOCAL const char** t2o_cmd_argv;	/* = NULL */
LOCAL int	   t2o_cmd_argc;
		// t2o_cmd_argv == NULL --> no command currently active


	    // must be ordered in execution order (see t2o_do_triggers)
#define t2o_BeforeIdx	0
#define t2o_InsteadIdx	1
#define t2o_AfterIdx	2

typedef struct		/* record associated with method as client data	*/
{  char*        trigger[3];/* trigger code, indices see above	*/
   cci_Fun      code;	   /* original method implementation	*/
   sos_Bool     active;
   const
   sos_Object*  obj;	  /* if active: receiver object 	*/
   char*        objstr;	  /* if active: encoding of obj, computed on demand  */
   int	        argc;	  /* if active: argc of original method invocation   */
   const char** argv;	  /* if active: argv of original method invocation   */
   char*        argvlist; /* if active: argv as tcl list, computed on demand */
} t2o_Trigger;

LOCAL  t2o_Trigger* t2o_active_trigger;	/* = NULL */
			// valid during execution of trigger action

EXPORT int	    _t2o_active_triggers; /* = 0 */


// -----------------------------------------------------------------------
// IMPLEMENTATION: Utilities
// -----------------------------------------------------------------------

LOCAL inline void DefineObjCommand (Tcl_Interp* ip, char* name)
{  Tcl_CreateCommand (ip, name, (Tcl_CmdProc*)t2o_CallCmd,
				(ClientData)1, (Tcl_CmdDeleteProc*)NULL);
}
LOCAL inline const char* CmdName(int cmdcode)
{  return _t2o_current_ip->cmd_name[cmdcode];
}

	// Some Cfronts had problems in compiling the inline version.
	// Hence a macro and no inline function.
#define IS_ERROR()	   (t2o_error ? 1\
    	      			      : (c2obst_ctrls.syserr\
					 	? (t2o_err_handler(), 1)\
		 	     			: 0))

   // Unsafe version of IS_ERROR which must only be used if it is known that
   // no err_SYS error occured since c2obst_ctrls.syserr was last checked.
   // CHK_ERROR() followed by _IS_ERROR() is safe.
#define _IS_ERROR()	   t2o_error
#define CHK_ERROR()	   {if (c2obst_ctrls.syserr && !t2o_error)\
			       t2o_err_handler();}

#define OPTION_ERR(cmd,op,vars)\
   	   t2o_option_error (cmd, op, vars, "invalid")
#define USAGE_ERR(opc,str)\
   	   t2o_usage_error (_t2o_option_data[t2o_Option2Idx(opc)], str)
#define USE_ERROR(cmd,str) err_raise (err_USE, (char*)str, (char*)CmdName(cmd))

#define PREPARE_LEAVE()    _t2o_current_ip=_ip; TT(t2o_H, T_LEAVE)

#define RETURN(res)        {PREPARE_LEAVE(); return (res);}

#define BOOL_RESULT(r)	   if (!IS_ERROR()) strcpy(interp->result, (r)?"1":"0")

#define ENTER_COMMAND()	   t2o_InterpData* _ip = _t2o_current_ip;\
   			   c2obst_ctrls.syserr = NULL;\
			   t2o_error	       = FALSE;\
			   if (!(_t2o_current_ip=t2o_get_ipdata(interp,FALSE)))\
			      RETURN (TCL_ERROR)

#define ENTER_and_CHECK(cmd,codevar1,codevar2,objvar) \
	ENTER_COMMAND(); int codevar1, codevar2; sos_Object objvar;\
	if (cmd_startup(cmd, &codevar1, &codevar2, &objvar, argc,argv)\
	    != TCL_OK)\
	   RETURN (TCL_ERROR)


/*
 * Print usage error `error` for the option `opt` given for the command
 * denoted by `cmdcode`.
 * `vars` describes the valid options.
 */
LOCAL void t2o_option_error (int	      cmdcode, char *opt,
			     t2o_OptVariants& vars,    char *error)
{  T_PROC ("t2o_option_error")
   TT (t2o_L, T_ENTER);

   smg_String msg = smg_String ("option \"") + opt + "\": " + error
		    + "\nvalid options are: ";

   t2o_OptVariant *var = vars.variants;
   for (int idx = vars.size;  idx --;  ++ var)
   {  msg += var->name;
      if (idx)
	 msg += ", ";
   }
   USE_ERROR (cmdcode, msg.make_Cstring (SMG_BORROW));

   TT (t2o_L, T_LEAVE);
}

/*
 * Print usage error `error` for the option `t2o_option_data[optidx]` as well
 * as the usage string associated with that option entry.
 */
LOCAL void t2o_usage_error (_t2o_OptionInfo& opt, char* error)
{  T_PROC ("t2o_usage_error")
   TT (t2o_L, T_ENTER);

   smg_String msg = smg_String (error) + "\nusage: "
     		    + CmdName (opt.cmd_code) + " " + opt.name + " " + opt.usage;

   USE_ERROR (opt.cmd_code, msg.make_Cstring (SMG_BORROW));

   TT (t2o_L, T_LEAVE);
}


/* Yields the encoding of the variant whose prefix equals `str`, iff there is
 * exactly one code value for `str`.
 * Otherwise, the result will be `t2o_NoCode`.
 *
 * If abbreviations are allowed within the realm of the current interpreter,
 * `str` may also be a true prefix of a variant.
 *
 * A specific error will be raised if either no match or at least two matches
 * with different encodings are found.
 */
LOCAL int t2o_get_varcode (int cmdcode, char *str, t2o_OptVariants& vars)
{  T_PROC ("t2o_get_varcode")
   TT (t2o_L, T_ENTER);

   int code   = t2o_NoCode,
       idx    = vars.size,
       length = strlen (str);

   t2o_OptVariant *var = vars.variants;
   for (idx = vars.size;  idx --;  ++ var)
   {  if (!strncmp (str, var->name, length))
         if (code == t2o_NoCode)
	    if (_t2o_current_ip->allow_abbrevs || length == strlen(var->name))
	       code = var->code;
	    else
	    {  length = -1;
	       break;
	    }
         else if (code != var->code)
	 {  code = t2o_NoCode;
	    idx  = -999;
	    break;
	 }
   }
   if (code == t2o_NoCode)
      t2o_option_error (cmdcode, str, vars,
			(idx == -999)
			   ? "ambiguous abbreviation"
		           : (length < 0) ? "no abbreviations allowed here"
			    		  : "invalid");
   TT (t2o_L, T_LEAVE);
   return code;
}


/*
 * Extracts a string list from the argument vector `argv` of `argc` elements
 * starting at index `startidx` and returns whether the extraction succeeded.
 *
 * The resulting list of `t2o_listc` elements is placed in `t2o_listv`.
 * `t2o_listv` is undefined if `t2o_list == 0`.
 * `t2o_listv` refers maybe to a part of `argv`. 
 * The values of both result variables (as well as the strings stored in
 * `t2o_listv`) remain stable until the next call of `t2o_get_list` (provided
 * they are not modified by the caller).
 *
 * The result list is empty, if `startidx` points after the last argument.
 * If `startidx` points at the last argument, then the result list is extracted
 * from this single argument.
 * Otherwise, the result list is formed from the arguments starting at
 * `startidx`.
 */
LOCAL int   t2o_listc;
LOCAL char* *t2o_listv;

LOCAL int t2o_get_list (int startidx, int argc, char* argv[])
{  T_PROC ("t2o_get_list")
   TT (t2o_L, T_ENTER);

   static int allocated = FALSE;
   if (allocated)
   {  allocated = FALSE;
      delete t2o_listv;		// sufficient (see Tcl_SplitList).
   }
   if (startidx >= argc)
   {  t2o_listc = 0;
      t2o_listv = NULL;
   }
   else if (startidx < argc-1)
   {  t2o_listv = argv + startidx;
      t2o_listc = argc - startidx;
   }
   else if (Tcl_SplitList (_t2o_current_ip->interp,
			   argv[startidx], &t2o_listc, &t2o_listv) == TCL_OK)
      allocated = TRUE;
   else
   {  t2o_listc = 0;
      t2o_listv = NULL;

      TT (t2o_L, T_LEAVE);
      return FALSE;
   }
   TT (t2o_L, T_LEAVE);
   return TRUE;
}


/*
 * Contains much of the startup code for the tclOBST Tcl commands.
 * `cmdcode` denotes the command to be processed, `argc`, `argv` the command
 * arguments as passed on by the Tcl dispatcher.
 *
 * `argv[1]` must be an encoded string which is associated to `cmdcode` via
 * the `t2o_option_data` array. The string encoding of `argv[1]` will be stored
 * in `*argv1code`. 
 *
 * `argc` must be within the bounds `t2o_option_data[*argv1code].*argc`.
 * Depending on `t2o_option_data[*argv1code].argv2_tp`, `argv[2]` will be
 * processed to:
 *  - t2o_Nop		no processing,
 *  - t2o_KnownStr	`argv[2]` must have been entered in the string table,
 *			its associated code will be stored in `*argv2code`,
 *  - t2o_List		`t2o_get_list` is used to extract a list starting at
 *			argument #2,
 *  - t2o_MapStr	maps `argv[2]` via the string table and stores the
 *			associated code in `*argv2code`.
 */
LOCAL int cmd_startup (int	      cmdcode,
		       int*	      argv1code,
		       int*	      argv2code,
		       sos_Object*    argv2obj,
		       register int   argc,
		       register char* argv[])
{  T_PROC ("cmd_startup")
   TT (t2o_L, T_ENTER);

   if (argc < 2)
   {  t2o_option_error (cmdcode, "", t2o_option[cmdcode], "missing");
      TT (t2o_L, T_LEAVE);
      return TCL_ERROR;
   }
   register _t2o_OptionInfo* opt_info;
   register int		     code1 = _t2o_get_strcode (argv[1]);

   if ((*argv1code = code1) == t2o_NoCode)
   {  *argv1code =
      code1      = t2o_get_varcode (cmdcode, argv[1], t2o_option[cmdcode]);
      if (code1 == t2o_NoCode)
      {  TT (t2o_L, T_LEAVE);
	 return TCL_ERROR;
      }
   }
   opt_info = _t2o_option_data + t2o_Option2Idx(code1);
   if (opt_info->cmd_code != cmdcode)
   {  t2o_option_error (cmdcode, argv[1], t2o_option[cmdcode], "invalid");
      TT (t2o_L, T_LEAVE);
      return TCL_ERROR;
   }
   if (argc < opt_info->min_argc)
   {  t2o_usage_error (*opt_info, "too few arguments");
      TT (t2o_L, T_LEAVE);
      return TCL_ERROR;
   }
   if (argc > opt_info->max_argc)
   {  t2o_usage_error (*opt_info, "too many arguments");
      TT (t2o_L, T_LEAVE);
      return TCL_ERROR;
   }
   switch (opt_info->argv2_tp)
   {  case t2o_ObjCode:
      {  if (!_t2o_str2obj (argv2obj, argv[2]))
	 {  t2o_usage_error (*opt_info, "arg#2 - encoded object expected");
	    TT (t2o_L, T_LEAVE);
	    return TCL_ERROR;
	 }
	 break;
      }
      case t2o_MapStr:
      {  *argv2code = _t2o_get_strcode (argv[2]);
	 break;
      }
      case t2o_List:
      {  if (!t2o_get_list (2, argc, argv))
	 {  TT (t2o_L, T_LEAVE);
	    return TCL_ERROR;
	 }
      }
      default: ;
   }
   TT (t2o_L, T_LEAVE);
   return TCL_OK;
}

/*
 * Stores the encoding of 'obj' as the result of the command currently
 * executed by 'interp'.
 * 'tpcode' encodes the result type as specified for the field 'resulttp' of
 * the type 'obst_mInfo'.
 * Nothing happens if 't2o_error' is set. The error status can be queried by
 * '_IS_ERROR()' afterwards.
 */
LOCAL void t2o_return_obj (Tcl_Interp* interp,
			   OBST_PARDECL(sos_Object) obj, char* tpcode)
{  T_PROC ("t2o_return_obj")
   TT (t2o_L, T_ENTER);

   if (IS_ERROR())
   {  TT (t2o_L, T_LEAVE);
      return;
   }
#if c2o_ADD_DEBUG_STUFF
   err_assert(!*interp->result, "t2o_return_obj - result field must be empty");
#endif
   switch ((long) tpcode)
   {  case obst_VOID:
	 break;
      case obst_CLASS:
      {  t2o_OBJ2STR (&obj, interp->result, FALSE);
	 if (_t2o_current_ip->define_obj_cmds && obj != NO_OBJECT)
	    DefineObjCommand (interp, interp->result);
	 break;
      }
      case obst_CLASS_OR_SCALAR:  
      {  register int not_nil = (obj != NO_OBJECT);
#if !t2o_ALWAYS_CODE_TYPE
         register int scalar  = (not_nil && obj.type().is_scalar());
#endif
	 t2o_OBJ2STR (&obj, interp->result, scalar);
	 if (_t2o_current_ip->define_obj_cmds && not_nil)
	    DefineObjCommand (interp, interp->result);
	 break;
      }
      default:
      {  if (obj != NO_OBJECT)
	 {  Tcl_SetResult (interp, obst_object2string (&obj), TCL_DYNAMIC);
	    CHK_ERROR();
	 }
      }
   }
   TT (t2o_L, T_LEAVE);
}

// -----------------------------------------------------------------------
// IMPLEMENTATION: Tcl Commands
//
// (Note: OBST errors set t2o_error via t2o_err_handler, which will also
//	  store the error message in interp->result.)
// -----------------------------------------------------------------------

/*
 * Method call interface for OBST methods.
 * Class methods are called according to '<cmd> <class>::<method> <arg>...',
 * the call format for instance methods is '<cmd> <obj> <method> <arg>...'.
 * Both <class> and <method> are names (strings), <obj> is an encoded object.
 *
 * This command can be installed in two ways for instance methods:
 *  - If the client data is NULL, then a fixed global command name is used.
 *  - Otherwise, the command must be installed under the name of the encoding
 *    of <obj>, the call format will then be '<obj> <method> <arg>...'.
 *
 * Arguments are converted according to the parameter types of the called
 * method. They will be type-checked:
 *  - In case of reference parameters, the corresponding <arg> must name a
 *    Tcl variable whose value will then be taken. Additionally, the value
 *    after executing the method will be stored in the variable.
 *    Note, that this isn't 'call by reference' but 'call by value and result'.
 *  - Arguments for parameters of scalar types must be given in the string
 *    representation of that scalar type. They will then be converted.
 *  - Arguments for parameters of class types must be encoded objects.
 *  - If a parameter can take as well class instances as scalars (e.g.
 *    type sos_Object), the object representation must be used for scalars.
 *  - Inverse conversions will be applied to result values (i.e. either the
 *    method result, or reference parameters).
 *  - Methods with result type 'void' yield the empty string.
 *  - The result of converting NO_OBJECT from the object to the string
 *    representation of a scalar type is the empty string, too.
 *
 * Errors are handled according to Tcl conventions. If errors are raised while
 * executing OBST code, the OBST error messages are accumulated in a format
 * similar to the default printing format of the err(3) module.
 * Additionally, 'errorCode' will be set to the last raised OBST error (see
 * 't2o_err_handler').
 */
EXPORT int t2o_CallCmd (const ClientData cdata,
		        Tcl_Interp       *interp, int argc, char *argv[])
{  static const char* t2o_call_usage
		      = " (<class>::<method> | <obj> <method>) <arg>...";

   T_PROC ("t2o_CallCmd")
   TT (t2o_H, T_ENTER);

   ENTER_COMMAND();
   if (argc < 2)
   {  USE_ERROR (t2o_CallCmdC, t2o_call_usage);
      RETURN (TCL_ERROR)
   }
   sos_Object   call_obj;
   int	        call_argc;
   const char** call_argv;

			// Either this argument is convertible to a sos_Object,
			// or a class method is called.
   if (_t2o_str2obj (&call_obj, argv[ cdata ? 0 : 1 ]))
   {  if (cdata)
      {  call_argv = argv + 2;   call_argc = argc - 2;	// command == object
      }
      else
      {  call_argv = argv + 3;   call_argc = argc - 3;
	 if (argc < 3)
	 {  USE_ERROR (t2o_CallCmdC, t2o_call_usage);
	    RETURN (TCL_ERROR)
	 }
      }
      obst_mcall_prep (&call_obj, *(call_argv - 1), call_argc);
   }
   else		// Handle calls of class methods / parse "<class>::<method>"
   {  register
      int idx = 0;			// <class>  == argv[1][0 .. cnend-1]
      int mnstart = 0, cnend = 0;	// <method> == argv[1][mnstart .. ]
      for (register char* ptr = (char*)argv[1];  *ptr;  ++ ptr, ++ idx)
      {  if (*ptr == ':')
	 {  if (cnend)
	    {  if (mnstart)
	       {  argv[1][cnend] = ':';
		  USE_ERROR (t2o_CallCmdC, t2o_call_usage);
		  RETURN (TCL_ERROR)
	       }
	       else
	          mnstart = idx + 1;
	    }
	    else
	    {  *ptr = '\0';  cnend = idx;
	    }
	 }
      }
      call_argv = argv + 2;
      ptr       = argv[1];

      if (!mnstart)
      {  USE_ERROR (t2o_CallCmdC, t2o_call_usage);
	 RETURN (TCL_ERROR)
      }
      obst_cmcall_prep (ptr, ptr + mnstart, call_argc = argc - 2);

      ptr[cnend] = ':';
   }
   // No difference between calls of instance / class methods from here on.

   if (IS_ERROR())
      RETURN (TCL_ERROR)

   sos_Object* oargv;	// used argument vector

   static int call_level;  /* = 0 */

   // hide all internal names in this block
   {  static sos_Object** _oargv;   /* = NULL */      // argument vector pool
      static int*	  _vsize;   /* = NULL */      // size of _oargv[i]
      static int	  maxlevel; /* = 0 */	      // max. call level so far

      if (call_level == maxlevel)
      {  ++ call_level;

	 if (maxlevel)
	 {  _vsize  = (int*)	     REALLOC(_vsize, call_level*sizeof(int));
	    _oargv  = (sos_Object**) REALLOC(_oargv,
					     call_level*sizeof(sos_Object*));
	 }
	 else
	 {  _oargv = new (sos_Object*);
	    _vsize = new int;
	 }
	 _vsize[maxlevel] = call_argc + 1;	// want only REALLOC case below
	 _oargv[maxlevel] =
	 oargv		  = new sos_Object [call_argc + 1];

	 maxlevel = call_level;
      }
      else if (_vsize[call_level] < call_argc)
      {  _vsize[call_level] = call_argc;
	 _oargv[call_level] =
	 oargv		    = (sos_Object*)
	 		      REALLOC (_oargv[call_level],
				       call_argc * sizeof(sos_Object));
	 ++ call_level;
      }
      else
	 oargv = _oargv[call_level ++];
   }

   if (call_argc)			// Fill up argument vector.
   {  static   int _dummy = -1;
      register int *refp_idx, idx = 0;

      if (!( refp_idx = obst_prepared->refparams ))	// Either NULL, or -1
         refp_idx = &_dummy;				// terminated vector of
							// indices.
      register char**	   p_tp  = obst_prepared->paramtps;
      register char*	   *_arg = (char**)call_argv;
      register sos_Object* oarg  = oargv;

      while (1)			// Convert Tcl arguments to sos_Objects.
      {  register char* arg;
	 if (idx++ != *refp_idx)
	    arg = *_arg;
	 else			// reference parameter: *_arg is variable name.
	 {  if (arg = Tcl_GetVar (interp, *_arg, 0))
	       ++ refp_idx;
	    else
	    {  smg_String msg = smg_String("arg #") + idx
				   + " - variable name expected";
	       USE_ERROR (t2o_CallCmdC, msg.make_Cstring(SMG_BORROW));
	       -- call_level;
	       RETURN (TCL_ERROR)
	    }
	 }
	 switch ((long) *p_tp)	      // ignoring case obst_VOID for parameters
	 {  case (long)obst_CLASS:
	    case (long)obst_CLASS_OR_SCALAR:
	    {  t2o_error = !_t2o_str2obj (oarg, arg);
	       break;
	    }
	    default:
	    {  *oarg = obst_string2object (*p_tp, arg);
	       CHK_ERROR();
	    }
	 }
	 // err_SYS errors which are caught can only occur in obst_string2object

	 if (_IS_ERROR())
	 {  smg_String msg = smg_String("arg #") + (idx+1)
				+ " - conversion to sos_Object failed";
	    USE_ERROR (t2o_CallCmdC, msg.make_Cstring(SMG_BORROW));
	    -- call_level;
	    RETURN (TCL_ERROR)
	 }
	 if (idx == call_argc)
	    break;
	 ++ p_tp;   ++ _arg;   ++ oarg;
      } // while
   } // if (argc)

   	// Make some switches specific for each interpreter.
   c2obst_ctrls.allow_defaults = (sos_Bool)_t2o_current_ip->allow_defaults;
   c2obst_ctrls.copy_Cstrings  = (sos_Bool)_t2o_current_ip->copy_Cstrings;

   	// Record arguments for trigger processing
   t2o_cmd_argv = call_argv;
   t2o_cmd_argc = call_argc;

        // Note: obst_prepared must not be used beyond this point!
   obst_mInfo* prepared_info = obst_prepared;

   sos_Object _result = obst_call_prepared (oargv);	// Actual call!

   t2o_cmd_argv = NULL;
   -- call_level;

   if (IS_ERROR())
      RETURN (TCL_ERROR)

   register int* refp_idx;
   if (refp_idx = prepared_info->refparams)   // Copy back reference paramters.
   {
      	     // Do not copy back iff there is a trigger with an instead action.
      t2o_Trigger* tdata = (t2o_Trigger*)( prepared_info->md.user_data );
      if (tdata && tdata->trigger[t2o_InsteadIdx])
      {  do
	 {  register char*	 refp_val;
	    register sos_Object* oarg = oargv + *refp_idx;
	    register int         def_cmd = _t2o_current_ip->define_obj_cmds;
	    
	    switch ((long) prepared_info->paramtps[*refp_idx])
	    {  case obst_CLASS:
	       {  if (def_cmd && *oarg == NO_OBJECT)
		     def_cmd = FALSE;
		  refp_val = t2o_OBJ2STR (oarg, NULL, FALSE);
		  break;
	       }
	       case obst_CLASS_OR_SCALAR: 
	       {  if (def_cmd && *oarg == NO_OBJECT)
		     def_cmd = FALSE;

		  refp_val = t2o_OBJ2STR(oarg,NULL,*oarg != NO_OBJECT
					           && oarg->type().is_scalar());
		  break;
	       }
	       default:
	       {  def_cmd  = FALSE;
		  if (*oarg == NO_OBJECT)
		     refp_val = "";
		  else
		     refp_val = obst_object2string (oarg);
	       }
	    }
	    if (! Tcl_SetVar (interp, (char*)call_argv[*refp_idx], refp_val, 0))
	    {  smg_String msg = smg_String("unable to store in $<argv>[")
	       + (*refp_idx + 1) + "]";
	       USE_ERROR (t2o_CallCmdC, msg.make_Cstring(SMG_BORROW));
	       RETURN (TCL_ERROR)
	    }
	    if (def_cmd)
	       DefineObjCommand (interp, refp_val);

	    delete refp_val;	// Tcl_SetVar stores a copy
	    // (It would make sense to use the interpreter's return field for
	    //  the obst_CLASS* cases.)
         }
	 while (*(++ refp_idx) >= 0);
      }
   }
   t2o_return_obj (interp, _result, prepared_info->resulttp);

   RETURN (_IS_ERROR() ? TCL_ERROR : TCL_OK)
}

// --------------------------------------------------------------------------

struct t2o_LoopData
{  char  *aggcode,		// encoding of this aggregate
         *curr1code, *curr1tp,	// encodings of the current scan elements
	 *curr2code, *curr2tp;
   void* iterdescr;		// descriptor for associated scan
};

struct t2o_LoopStack
{  t2o_LoopData* stack;		// stack for loop data
   int           size;		// size of the 'stack' array
   int		 top;		// index of stacktop
};

/*
 * Provides Tcl commands for processing OBST aggregates.
 * The following options are implemented:
 *
 * - loop [forward|reverse] <agg> <body>
 *   Loops over <agg> in either forward or reverse direction, executing <body>
 *   on each iteration.
 *   <agg> must be a subclass of either 'Collection<X>', or 'Association<X,Y'.
 *   'forward' ('reverse') may be abbreviated as 'fwd' ('rev'), the
 *   default direction is 'forward'.
 *   The ordering of aggregate elements is system-defined in some cases, i.e.
 *   the distinction between interation orders makes not much sense in these
 *   cases.
 *   Removal of aggregate elements within <body> is erroneous, if cursor
 *   stability is not guaranteed for <agg> (see OBST manual pages).
 *   <agg> is evaluated just once upon entering the loop for the first time.
 *
 * - current [agg|role1|role2] [<level>]
 *   Yields the current element of a (dynamically) enclosing aggregate loop.
 *   The second argument must be ommitted for collections. In case of
 *   associations, it specifies one of the access method 'get_role1', resp.
 *   'get_role2' defined for each association.
 *   'agg' as second argument refers to the aggregate itself.
 *
 *   <level> specifies the loop 'current' refers to. It defaults to '0' which
 *   denotes the innermost dynamically enclosing loop. Outer loops are denoted
 *   by negative numbers (i.e. -1, -2, ...).
 *
 * - card <agg>
 *   Yields the cardinality of the given aggregate.
 */
EXPORT int t2o_AggCmd (const ClientData,
		       Tcl_Interp *interp, int argc, char *argv[])
{  T_PROC ("t2o_AggCmd")
   TT (t2o_H, T_ENTER);

   ENTER_and_CHECK (t2o_AggCmdC, option_code, argv2_idx, obj);

   t2o_LoopStack* stackp = (t2o_LoopStack *)_t2o_current_ip->loop_data;

   /*
    * Each option must ensure that the return status can be queried by
    * _IS_ERROR() afterwards.
    */
   switch (option_code)
   {  case t2o_CardC:				// option "card"
      {  int card = (obst_identical (&obj, &NO_OBJECT)) ? 0
							: obst_card (&obj);
	 if (!IS_ERROR())
	    sprintf (interp->result, "%d", card);
	 break;
      }
      case t2o_CurrentC:			// option "current"
      {  int levelarg = (argv2_idx == t2o_NoCode) ? 2 : 3;
	 int level    = 0;
	 if (levelarg < argc
	     && Tcl_GetInt (interp, argv[levelarg], &level) != TCL_OK)
	    RETURN (TCL_ERROR)

	 if (!stackp  ||  level > 0  ||  level + stackp->top < 0)
	 {  smg_String s = smg_String("invalid reference to loop at level ")
	    		   + level;
	    USE_ERROR (t2o_AggCmdC, s.make_Cstring (SMG_BORROW));
	    RETURN (TCL_ERROR)
	 }
	 t2o_LoopData* ldata = &stackp->stack[stackp->top + level];

	 argv2_idx = (argc < 3 ) ? 0 : _t2o_get_strcode(argv[2]);
	 switch (argv2_idx)
	 {  case t2o_AggC:
	    {  strcpy (interp->result, ldata->aggcode);
	       if (_t2o_current_ip->define_obj_cmds)
		  DefineObjCommand (interp, interp->result);
	       break;
	    }
	    case 0:
	    case t2o_Role1C:
            {  if (ldata->curr1code)
		  strcpy (interp->result, ldata->curr1code);
	       else
	       {  if (!ldata->curr1tp)
		     ldata->curr1tp = obst_iter_elemtp(ldata->iterdescr, 1);
		  
		  sos_Object cobj = (argv2_idx)
		     			? obst_iter_role1(ldata->iterdescr)
					: obst_iter_get  (ldata->iterdescr);
		  
		  t2o_return_obj (interp, cobj, ldata->curr1tp);
		  ldata->curr1code = obst_strdup (interp->result);
		  
		  if (_IS_ERROR())
		     RETURN (TCL_ERROR)
	       }
	       break;
	    }
	    case t2o_Role2C:
            {  if (ldata->curr2code)
		  strcpy (interp->result, ldata->curr2code);
	       else
	       {  if (!ldata->curr2tp)
		     ldata->curr2tp = obst_iter_elemtp(ldata->iterdescr, 2);
		  
		  sos_Object cobj = obst_iter_role2 (ldata->iterdescr);
		  
		  t2o_return_obj (interp, cobj, ldata->curr2tp);
		  ldata->curr2code = obst_strdup (interp->result);
		  
		  if (_IS_ERROR())
		     RETURN (TCL_ERROR)
	       }
	       break;
	    }
	    default:
            {  argv2_idx = t2o_get_varcode (t2o_AggCmdC, argv[2],
					    t2o_aggCurr_opts);
	       if (argv2_idx == t2o_NoCode)
		  RETURN (TCL_ERROR)
	    }
         }
	 break;
      }
      case t2o_LoopC:				// option "loop"
      {  int	  agg_idx;
	 sos_Bool forward;
         
	 if (_t2o_str2obj (&obj, argv[agg_idx = 2]))
	    forward = TRUE;
	 else
	 {  if (!_t2o_str2obj (&obj, argv[agg_idx = 3]))
	    {  USE_ERROR (t2o_AggCmdC, "encoded aggregate object expected");
	       RETURN (TCL_ERROR)
	    }
	    if ((argv2_idx = _t2o_get_strcode(argv[2])) == t2o_NoCode)
	    {  argv2_idx = t2o_get_varcode (t2o_AggCmdC, argv[2],
					    t2o_aggDirect_opts);
	       if (argv2_idx == t2o_NoCode)
	          RETURN (TCL_ERROR)
	    }
	    forward = (sos_Bool)(argv2_idx == t2o_ForwardC);
	 }
	 if (agg_idx + 1 >= argc)
	 {  USE_ERROR (t2o_AggCmdC, "too few arguments");
	    RETURN (TCL_ERROR)
	 }
	 if (!stackp)
         {  _t2o_current_ip->loop_data =
	    stackp   		      = new t2o_LoopStack;
	    stackp->stack 	      = new t2o_LoopData;
	    stackp->size  	      = 1;
	    stackp->top   	      = 0;
	 }
	 else if (++stackp->top >= stackp->size)
	    stackp->stack = (t2o_LoopData*)
			    REALLOC (stackp->stack,
				     ++stackp->size * sizeof(t2o_LoopData));
 
	 t2o_LoopData* ldata = stackp->stack + stackp->top;
	 ldata->aggcode      = obst_strdup (argv[agg_idx]);
	 ldata->curr1code    = NULL;  ldata->curr1tp = NULL;
	 ldata->curr2code    = NULL;  ldata->curr2tp = NULL;

	 char* body = argv[agg_idx + 1];
	 int   result;
	 if (ldata->iterdescr = obst_iter_start (&obj, forward)) 
	 {  do
	    {  result = TCL_EVAL (interp, body);

	    		// note: stackp->stack might have been re-allocated!
	       ldata  = stackp->stack + stackp->top;

	       if (ldata->curr1code)
	       {  delete ldata->curr1code;
		  ldata->curr1code = NULL;
	       }
	       if (ldata->curr2code)
	       {  delete ldata->curr2code;
		  ldata->curr2code = NULL;
	       }
	    }
	    while (   (result == TCL_OK || result == TCL_CONTINUE)
		   && (ldata->iterdescr = obst_iter_next (ldata->iterdescr)));
	 }
	 else
	    result = TCL_OK;

	 obst_iter_finish (ldata->iterdescr);
	 CHK_ERROR();

	 if (result == TCL_ERROR)
	 {  smg_String msg = smg_String("\n    (\"") + argv[1] 
			     + "\" body line " + interp->errorLine + ")";
	    Tcl_AddErrorInfo (interp, msg.make_Cstring (SMG_BORROW));
	    t2o_error = TRUE;
	 }  
	 delete ldata->aggcode;
	 -- stackp->top;
	 break;
      }
   }
   RETURN (_IS_ERROR() ? TCL_ERROR : TCL_OK)
}

// --------------------------------------------------------------------------

/*
 * Converts the list stored in `t2o_listv`/`t2o_listc` into either a new(!)
 * container set, or a single container.
 * The function returns `TRUE` if the conversion was successful as a whole.
 *
 * The last converted container will be stored in `t2o_ct`.
 * `t2o_cset` will not be created, if it is to contain at most one container.
 *
 * This default behaviour changes if `want_set == TRUE`: `t2o_cset` will then
 * be created in any case and it is up to the caller to destroy `t2o_cset`
 * after use.
 */
LOCAL sos_Container      t2o_ct;
LOCAL obst_Container_set t2o_cset;

LOCAL int t2o_list2cset (int want_set)
{  T_PROC ("t2o_list2cset")
   TT (t2o_M, T_ENTER);

   static int allocated = FALSE;
   if (allocated)
   {  obst_cset_destroy (t2o_cset);
      allocated = FALSE;
   }
   if (want_set)
      t2o_cset = obst_cset_create();

   for (int idx = 0;  idx < t2o_listc; )
   {  if (idx == 1 && !want_set)
      {  t2o_cset  = obst_cset_insert (obst_cset_create(), t2o_ct);
	 allocated = TRUE;
      }
      obst_string2scalar ("sos_Container", t2o_listv[idx], &t2o_ct);
      if (IS_ERROR())
      {  TT (t2o_M, T_LEAVE);
	 return FALSE;
      }
      if (idx ++ || want_set)
         obst_cset_insert (t2o_cset, t2o_ct);
   }
   TT (t2o_M, T_LEAVE);
   return TRUE;
}

/*
 * Implements operations on OBST containers and container sets.
 * Container sets are represented as lists at the Tcl interface. This allows
 * to unify operations on container sets and on single containers by treating
 * containers as singleton container sets.
 *
 * If an operation has a single container set parameter, it will be the last
 * one and may be specified either by a variable length argument list or by
 * a single argument holding a Tcl list.
 * This required some changes in operation signatures by shifting the container
 * (set) parameter to the end of the parameter list.
 * Operations on empty sets are NOPs.
 *
 * Containers as well as values of the enumeration types sos_Access_mode and
 * sos_Sync_mode are specified in their string representation.
 *
 * The following operations are implemented, where <cnt>* stands for a
 * container list (see above), <cnt>*` for a container list with at most one
 * element, <cnt>*! for a true container list (i.e. a list stored in a single
 * argv[]), <mode> a value of type sos_Access_mode, <sync> for a value of type
 * sos_Sync_mode, <state> for a value of type sos_Container_status, <off> for
 * a value of type sos_Offset, and <int> for a value of type sos_Int,
 * respectively:
 *	create
 *	clear	 	<cnt>*`
 *	close	 	<cnt>*
 *	commit	 	<cnt>*
 *	compress	<cnt>*`
 *	destroy	 	<cnt>*`
 *	exists	 	<cnt>*`
 *	modified 	<cnt>*`
 *	occupied 	<cnt>*`
 *	object_exists 	<off> <int> <cnt>*`
 *	reset		<cnt>*
 *	root_object  	<cnt>*`
 *	squeeze	 	<cnt>*`
 *	status	 	<cnt>*`
 *	update	 	<cnt>*`
 *	access	 	<mode> <sync> <cnt>*`
 *	open	 	<mode> <sync> <cnt>*
 *	open_containers	<state>
 *	open_sets	<sync> <cnt>*! <cnt>*! <cnt>*!
 */
EXPORT int t2o_CntCmd (const ClientData,
		       Tcl_Interp *interp, int argc, char *argv[])
{  T_PROC ("t2o_CntCmd")
   TT (t2o_H, T_ENTER);

   ENTER_and_CHECK (t2o_CntCmdC, option_code, argv2_idx, obj);

   int optidx  = t2o_Option2Idx(option_code);
   int listidx = _t2o_option_data[optidx].data1;// `data1` gives the index
   if (listidx)					// of the container list
   {  if (!t2o_get_list (listidx, argc, argv))  // argument
	 RETURN (TCL_ERROR)
      if (t2o_listc == 0)			// empty list --> NOP
	 RETURN (TCL_OK)

      if (t2o_listc > _t2o_option_data[optidx].data2)
      {  smg_String msg = smg_String(argv[1])
	 		  + " - only applicable to single containers";
	 USE_ERROR (t2o_CntCmdC, msg.make_Cstring (SMG_BORROW));
	 RETURN (TCL_ERROR)
      }
      if (!t2o_list2cset (FALSE))
	 RETURN(TCL_ERROR)
   }
   sos_Access_mode amode;
   sos_Sync_mode   smode;
   sos_Offset	   offset;
   sos_Int	   intval;
   int		   enumval;
   char*	   enumtp = NULL;

   switch (option_code)
   {  case t2o_AccessC:
      {  obst_string2scalar ("sos_Access_mode", argv[2], &amode);
	 obst_string2scalar ("sos_Sync_mode",   argv[3], &smode);
	 if (!IS_ERROR())
         {  enumval = obst_cnt_access (t2o_ct, amode, smode);
	    enumtp  = "sos_Open_result";
	 }
	 break;
      }
      case t2o_ClearC:
      {  obst_cnt_clear (t2o_ct);
         break;
      }
      case t2o_CloseC:
      {  if (t2o_listc > 1)
	    obst_cset_close (t2o_cset);
         else
	    obst_cnt_close (t2o_ct);
         break;
      }
      case t2o_CommitC:
      {  if (t2o_listc > 1)
	    obst_cset_commit (t2o_cset);
         else
	    obst_cnt_commit (t2o_ct);
         break;
      }
      case t2o_CompressC:
      {  obst_cnt_compress (t2o_ct);
	 break;
      }
      case t2o_CreateC:
      {  sos_Container ct = obst_cnt_create();
	 if (!IS_ERROR())
	 {  char* ctstr;
	    if (ctstr = obst_scalar2string ("sos_Container", &ct))
	       Tcl_SetResult (_t2o_current_ip->interp, ctstr, TCL_DYNAMIC);
	 }
         break;
      }
      case t2o_DestroyC:
      {  obst_cnt_destroy (t2o_ct);
	 break;
      }
      case t2o_ExistsC:
      {  int result = obst_cnt_exists (t2o_ct);
	 BOOL_RESULT (result);
	 break;
      }
      case t2o_ModifiedC:
      {  int result = obst_cnt_modified (t2o_ct);
	 BOOL_RESULT (result);
	 break;
      }
      case t2o_ObjectExistsC:
      {  obst_string2scalar ("sos_Offset", argv[2], &offset);
	 obst_string2scalar ("sos_Int",    argv[3], &intval);
	 if (!IS_ERROR())
         {  enumtp  = "sos_Existing_status";
	    enumval = obst_cnt_object_exists (t2o_ct, offset, intval);
	 }
	 break;
      }
      case t2o_OccupiedC:
      {  int result = obst_cnt_occupied (t2o_ct);
	 if (!IS_ERROR())
	    sprintf (interp->result, "%d", result);
	 break;
      }
      case t2o_OpenC:
      {  obst_string2scalar ("sos_Access_mode", argv[2], &amode);
	 obst_string2scalar ("sos_Sync_mode",   argv[3], &smode);
	 if (!IS_ERROR())
         {  enumtp  = "sos_Open_result";
	    enumval = (t2o_listc > 1) ? obst_cset_open(t2o_cset, amode, smode)
	       			      : obst_cnt_open (t2o_ct,   amode, smode);
	 }
	 break;
      }
      case t2o_OpenCntC:
      {  sos_Container_status	  ctstatus;
	 sos_Container_mod_status ctmstatus;
	 sos_Bool		  is_mod_status = TRUE;
	 obst_Container_set	  cset;

	 t2o_handle_errors = FALSE;
	 obst_string2scalar ("sos_Container_mod_status", argv[2], &ctmstatus);
	 if (IS_ERROR())
	 {  t2o_error	  = FALSE;
	    is_mod_status = FALSE;
	    obst_string2scalar ("sos_Container_status", argv[2], &ctstatus);
	 }
	 t2o_handle_errors = TRUE;
	 if (IS_ERROR())
	 {  USE_ERROR (t2o_CntCmdC, "arg#2 of type <sos_Container_status> or <sos_Container_mod_status> expected");
	    break;
	 }
	 cset = (is_mod_status) ? obst_cset_open_containers_mod (ctmstatus)
	    			: obst_cset_open_containers (ctstatus);
	 if (IS_ERROR())
	    break;

	 int csetc;
	 if ((csetc = obst_cset_card(cset)) > 0)
	 {  char** csetv = new char* [csetc];
	    for (int i = 0;  i < csetc && !t2o_error;  ++ i)
	    {  sos_Container ct = obst_cset_at (cset, i);
	       csetv[i] = obst_scalar2string("sos_Container", &ct);
	    }
	    if (!IS_ERROR())
	       Tcl_SetResult (interp, Tcl_Merge(csetc, csetv), TCL_DYNAMIC);
	    delete csetv;
	 }
	 obst_cset_destroy (cset);
	 break;
      }
      case t2o_OpenSetsC:
      {  obst_string2scalar ("sos_Sync_mode", argv[2], &smode);

	 if (!t2o_get_list (3, 4, argv))
	    RETURN (TCL_ERROR)
         if (!t2o_list2cset (TRUE) || !t2o_get_list (4, 5, argv))
	 {  obst_cset_destroy (t2o_cset);
	    RETURN (TCL_ERROR)
	 }
	 obst_Container_set cset1 = t2o_cset;

         if (!t2o_list2cset (TRUE) || !t2o_get_list (5, 6, argv))
	 {  obst_cset_destroy (t2o_cset);
	    obst_cset_destroy (cset1);
	    RETURN (TCL_ERROR)
	 }
	 obst_Container_set cset2 = t2o_cset;

	 if (t2o_list2cset (TRUE))
	    obst_cset_open_sets (cset1, cset2, t2o_cset, smode);

	 obst_cset_destroy (cset1);
	 obst_cset_destroy (cset2);
	 obst_cset_destroy (t2o_cset);

	 break;
      }
      case t2o_ResetC:
      {  if (t2o_listc > 1)
	    obst_cset_reset (t2o_cset);
	 else
	    obst_cnt_reset (t2o_ct);
	 break;
      }
      case t2o_RootObjectC:
      {  sos_Object result = obst_cnt_root_object (t2o_ct);
	 t2o_return_obj (interp, result, (char*)obst_CLASS_OR_SCALAR);
	 break;
      }
      case t2o_SqueezeC:
      {  obst_cnt_squeeze (t2o_ct);
	 break;
      }
      case t2o_StatusC:
      {  enumval = obst_cnt_status (t2o_ct);
	 enumtp  = "sos_Container_status";
	 break;
      }
      case t2o_UpdateC:
      {  obst_cnt_update (t2o_ct);
	 break;
      }
      default:
   	 err_raise (err_USE, err_NOT_IMPLEMENTED, "t2o_CntCmd", FALSE);
   }
   CHK_ERROR();
   if (enumtp && !_IS_ERROR())
   {  char* enumstr;
      if (enumstr = obst_scalar2string (enumtp, &enumval))
	 Tcl_SetResult (interp, enumstr, TCL_DYNAMIC);

      CHK_ERROR();
   }
   RETURN (_IS_ERROR() ? TCL_ERROR : TCL_OK)
}

// --------------------------------------------------------------------------

/*
 * Trigger handler.
 */
LOCAL sos_Object t2o_do_triggers (const sos_Object&	  o,
				  const sos_Object_Array& p)
{  T_PROC ("t2o_do_triggers")
   TT (t2o_L, T_ENTER);
   err_assert (t2o_cmd_argv, "t2o_do_triggers - no command handler active");

   t2o_Trigger* tdata = (t2o_Trigger*)( obst_prepared->md.user_data );
   sos_Object   result;

   if (tdata->active)
   {  result = (*tdata->code)(o, p);

      TT (t2o_L, T_LEAVE);
      return result;
   }
   tdata->active = TRUE;
   ++ _t2o_active_triggers;

   // Store original execution environment in tdata structure.
   tdata->obj	   = &o;
   tdata->argc     = t2o_cmd_argc;
   tdata->argv     = t2o_cmd_argv;
   tdata->argvlist = NULL;		// computed on demand
   tdata->objstr   = NULL;

   obst_mInfo* mi  = obst_prepared;

   // obst_prepared, t2o_cmd_argc, t2o_cmd_argv must not be used beyond this
   // point. However, *tdata and the current contents of t2o_cmd_argc,
   // t2o_cmd_argv will remain stable in this procedure invocation.

   for (int idx = t2o_BeforeIdx;  idx <= t2o_AfterIdx;  ++ idx)
   {  if (tdata->trigger[idx])
      {  t2o_Trigger* _tdata = t2o_active_trigger;
	 t2o_active_trigger  = tdata;

	 int ret = Tcl_Eval (_t2o_current_ip->interp, tdata->trigger[idx]);

	 t2o_active_trigger = _tdata;

	 if (ret != TCL_OK && ret != TCL_RETURN)
	 {  smg_String txt = smg_String(" illegal result status ") + ret + ")";
	    Tcl_AppendResult (_t2o_current_ip->interp,
			      txt.make_Cstring (SMG_BORROW), (char*) NULL);
	    t2o_error = TRUE;
	 }
	 else
	 {  if (idx == t2o_InsteadIdx)
	    {  char* rtp = obst_prepared->resulttp;
	       if ((long)rtp == obst_VOID)
		  result = NO_OBJECT;
	       else if (   (long)rtp == obst_CLASS
			|| (long)rtp == obst_CLASS_OR_SCALAR)
	       {  if (!_t2o_str2obj (&result, _t2o_current_ip->interp->result))
		  {  Tcl_AppendResult (_t2o_current_ip->interp,
				       " - conversion to sos_Object failed",
				       (char*)NULL);
		     t2o_error = TRUE;
		  }
	       }
	       else
	       {  result = obst_string2object (rtp,
					       _t2o_current_ip->interp->result);
		  if (result == NO_OBJECT)
		  {  Tcl_AppendResult (_t2o_current_ip->interp,
				       " - conversion to ", rtp, " failed",
				       (char*)NULL);
		     t2o_error = TRUE;
		  }
	       }
	    }
	    if (!t2o_error)
	       Tcl_ResetResult (_t2o_current_ip->interp);
	 }
      }
      else if (idx == t2o_InsteadIdx)
      {  err_block
	    result = (*tdata->code)(o, p);
	 err_exception
	    result = NO_OBJECT;
	    break;			// must clean up properly
	 err_block_end
      }
      if (t2o_error)
      {  if (tdata->trigger[idx])
	 {  smg_String txt = smg_String(" {trigger '")
	    		     + t2o_trigger_opts.variants[idx].name
	    		     + "' on method " + mi->mname
			     + "/" + mi->param_no + "}";
	    Tcl_AppendResult (_t2o_current_ip->interp,
			      txt.make_Cstring (SMG_BORROW), (char*)NULL);
	 }
	 result = NO_OBJECT;
	 break;
      }
   }
   if (tdata->argvlist)
      delete tdata->argvlist;
   if (tdata->objstr)
      delete tdata->objstr;

   tdata->active = FALSE;
   -- _t2o_active_triggers;

   TT (t2o_L, T_LEAVE);
   return result;
}

/*
 * Command for controlling as well the Tcl and the underlying C interface to
 * OBST as the OBST system itself.
 *
 *  - bind
 *	Employs t2o_BindDefs to bind all tclOBST definitions to the current
 *	interpreter and yields 1 (0) if the binding took place (was already
 *	established).
 *  - bind <name>
 *	Binds the command call handler to the given name.
 *
 *  - customize <resource> [<value>]
 *	Calls t2o_customize with the given parameters. If <value> is omitted,
 *	the current value is of the given <resource> is returned.
 *
 *  - schemas <namelist>
 *    schemas <name>...
 *	Queries/defines the schemas of interest for this process.
 *	The OBST core schemata will be automatically added to a definition.
 *	If no arguments are given, the currently defined schemata are returned.
 *	If the single argument is an empty list, all currently known schemas
 *	will then be searched.
 *
 *  - statistics
 *	Yields various statistics on the interfaces to and OBST itself as
 *	a formatted result.
 *
 *  - trace [<file> [<trclevel>]]
 *	Either closes the current output file (no args), defines a trace
 *	output file, and/or sets the current OBST trace level.
 *  - trace <switch> ( (enter|leave) [<proc>] | <text> )
 *      Based on the given trace switch record either entering/leaving <proc>,
 *      or write the given <text>, respectively.
 *
 *  - trigger (after|before|instead) <class> <method> <arity> [<code>] -> <code>
 *	Queries/sets the named trigger kind for the named method.
 *
 *  - trigger current (receiver | argc | argv [<no>]) -> string
 *	Queries the execution context of an active trigger.
 *
 *  - _test <arg>...
 *	Test command echoing its arguments to stderr.
 */
EXPORT int t2o_CtrlCmd (const ClientData,
		        Tcl_Interp *interp, int argc, char *argv[])
{  T_PROC ("t2o_CtrlCmd")
   TT (t2o_H, T_ENTER);

   ENTER_and_CHECK (t2o_CtrlCmdC, option_code, argv2_idx, obj);

   switch (option_code)
   {  case t2o_BindC:
      {  if (argc == 3)
	    DefineObjCommand (interp, argv[2]);
	 else
	 {  int res = t2o_BindDefs (interp, TRUE);
	    BOOL_RESULT (res);
	 }
	 break;
      }
      case t2o_CustomizeC:
      {  argv2_idx = t2o_get_varcode (t2o_CtrlCmdC, argv[2], t2o_custom_opts);
	 if (argv2_idx == t2o_NoCode)
	    RETURN (TCL_ERROR)

	 if (argc == 3)
	 {  switch (argv2_idx)
	    {  case t2o_AggCmdC:   case t2o_CallCmdC:
	       case t2o_CntCmdC:   case t2o_CtrlCmdC:
	       case t2o_UtilCmdC:
	          strcpy (interp->result, _t2o_current_ip->cmd_name[argv2_idx]);
	          break;
	       case t2o_ObjPrefC:
	          strcpy (interp->result, _t2o_current_ip->obj_prefix);
	          break;
	       case t2o_ObjCmdC:
		  BOOL_RESULT (_t2o_current_ip->define_obj_cmds);
	          break;
	       case t2o_AllowAbbrC:
		  BOOL_RESULT (_t2o_current_ip->allow_abbrevs);
	          break;
	       case t2o_AllowDfltC:
		  BOOL_RESULT (_t2o_current_ip->allow_defaults);
	          break;
	       case t2o_CpCstrC:
		  BOOL_RESULT (_t2o_current_ip->copy_Cstrings);
	          break;
	    }
	 }
	 else
	 {  char *arg3str;

	    if (argv2_idx < t2o_AllowAbbrC && argv2_idx != t2o_ObjCmdC)
	       arg3str = argv[3];
	    else
	    {  int code = t2o_get_varcode(t2o_CtrlCmdC,argv[3], t2o_bool_opts);
	       if (code == t2o_NoCode)
	          RETURN (TCL_ERROR)

	       arg3str = (code == t2o_TrueC) ? "TRUE" : "FALSE";
	    }
            t2o_customize (interp,
			   t2o_custom_opts.variants[argv2_idx].name, arg3str);
	 }
	 break;
      }
      case t2o_SchemasC:
      {  if (_t2o_active_triggers && (t2o_listc || argc == 3))
	 {  USE_ERROR (t2o_CtrlCmdC, "there are active triggers");
	    break;
	 }
	 if (t2o_listc)
	 {  char* *schemav = new char* [t2o_listc + 3];
	    memcpy (schemav, t2o_listv, t2o_listc * sizeof(char*));
	    schemav[t2o_listc    ] = "cci";
	    schemav[t2o_listc + 1] = "trans";
	    schemav[t2o_listc + 2] = NULL;
	    
	    obst_set_schemas (schemav);
	    delete schemav;
	 }
	 else if (argc == 3)
	    obst_set_schemas (NULL);
	 else
	 {  char* *schemav = obst_get_schemas();
	    if (schemav)
	    {  for (char** nm_ptr = schemav;  *nm_ptr;  ++ nm_ptr)
	       {  TCL_APPEND_ELEMENT (interp, *nm_ptr);
		  delete *nm_ptr;
	       }
	       delete schemav;
	    }
         }
 	 break;
      }
      case t2o_StatisticsC:
      {  smg_String res = smg_String (obst_stats())
	 		+ "\nactive triggers: " + _t2o_active_triggers + "\n";
	 if (!IS_ERROR())
	    Tcl_SetResult (interp, res.make_Cstring(SMG_TRANSFER), TCL_DYNAMIC);
	 break;
      }
      case t2o_TestC:
      {  for (int i = 0;  i < argc;  ++ i)
	    fprintf (stderr, "argv(%d): \"%s\"\n", i, argv[i]);
	 break;
      }
      case t2o_TraceC:
      {  if (argc == 2)
	    tt_exit();
	 else
	 {  char* ptr;
	    long  trc_switch = strtol(argv[2], &ptr, 10);

	    if (ptr == argv[2])
	    {  if (argc > 4)
	       {  USAGE_ERR (option_code, "too many arguments");
		  break;
	       }
	       if (*argv[2])
		  t2o_T_INIT (argv[2]);
	       if (argc > 3)
		  tt_redef (argv[3]);

	       if (*argv[2])
		  TT (t2o_H, T_ENTER);	// ensure proper indentation
	    }
	    else
	    {  if (trc_switch < 0 || trc_switch > 8192)
	       {  USAGE_ERR (option_code, "invalid trace switch number");
		  break;
	       }
	       if (tt_active[trc_switch] && argc > 3)
	       {  if (   argc == 5
		      || streql(argv[3], "enter") || streql(argv[3], "leave"))
		  {  static char* procname /* = NULL */;

		     if (argc > 4)
		     {  if (procname)
			   delete procname;
			procname = obst_strdup (argv[4]);
		     }
		     char* pname = (procname) ? procname
					      : "<unknown>";
		     if (*argv[3] == 'e')
			tt_enter (pname);
		     else
			tt_leave (pname);
		  }
		  else if (argc > 4)
		     USAGE_ERR (option_code, "too many arguments");
		  else
		     tt_txt (argv[3]);
	       }
	       BOOL_RESULT (tt_active[trc_switch]);
	    }
	 }
	 break;
      }
      case t2o_TriggerC:
      {  if (argv2_idx == t2o_CurrentC)
	    argv2_idx = -1;
	 else
	 {  switch (t2o_get_varcode (t2o_CtrlCmdC, argv[2], t2o_trigger_opts))
	    {  case t2o_AfterC:   argv2_idx = t2o_AfterIdx;	break;
	       case t2o_BeforeC:  argv2_idx = t2o_BeforeIdx;	break;
	       case t2o_InsteadC: argv2_idx = t2o_InsteadIdx;	break;
	       case t2o_CurrentC: argv2_idx = -1;		break;
	       default:	       	  RETURN (TCL_ERROR)
	    }
	 }
	 if (argv2_idx >= 0)	// 'trigger before|instead|after' ...
	 {  if (argc < 6)
	    {  USAGE_ERR (option_code, "too few arguments");
	       break;
	    }
	    int arity;
	    obst_string2scalar ("sos_Int", argv[5], &arity);
	    if (IS_ERROR())
	       RETURN (TCL_ERROR)

	    obst_mData md;
	    obst_method_data (argv[3], argv[4], arity, &md, FALSE);
	    if (IS_ERROR())
	       RETURN (TCL_ERROR)

	    t2o_Trigger* tptr = (t2o_Trigger*)md.user_data;

	    char* result = NULL;

	    if (argc <= 6)
	    {  if (tptr)
		  result = tptr->trigger[argv2_idx];
	    }
	    else
	    {  if (!tptr)
	       {  tptr = new t2o_Trigger;
		  memset (tptr, 0, sizeof(t2o_Trigger));

		  tptr->code   = md.code;
		  md.code      = &t2o_do_triggers;
		  md.user_data = tptr;
	       }
	       else if (tptr->active)
	       {  USE_ERROR (t2o_CtrlCmdC, "attempt to modify active trigger");
		  break;
	       }
	       if (tptr->trigger[argv2_idx])
		  t2o_manage_block (tptr->trigger[argv2_idx], FALSE);

	       if (*argv[6])
	       {  t2o_manage_block (result = obst_strdup (argv[6]), TRUE);
		  tptr->trigger[argv2_idx] = result;
	       }
	       else
	       {  tptr->trigger[argv2_idx] = NULL;
		  if (! (   tptr->trigger[0]
			 || tptr->trigger[1] || tptr->trigger[2]))
		  {  md.code      = tptr->code;
		     md.user_data = NULL;
		     delete tptr;
		  }
	       }
	       obst_method_data (argv[3], argv[4], arity, &md, TRUE);
	    }
	    if (!IS_ERROR() && result)
	       strcpy (interp->result, result);
	 }
	 else	// 'trigger current ...'
	 {  int opt = _t2o_get_strcode(argv[3]);
	    if (opt == t2o_NoCode)
	    {  opt = t2o_get_varcode (t2o_CtrlCmdC, argv[3], t2o_trgCurr_opts);
	       if (opt == t2o_NoCode)
		  RETURN (TCL_ERROR)
	    }
	    if (argc > ((opt == t2o_ArgvC) ? 5 : 4))
	    {  USAGE_ERR (option_code, "too many arguments");
	       break;
	    }
	    if (!(t2o_active_trigger && t2o_active_trigger->active))
	    {  USE_ERROR (t2o_CtrlCmdC, "no active trigger");
	       break;
	    }
	    switch (opt)
	    {  case t2o_ReceiverC:
	       {  if (!t2o_active_trigger->objstr)
		     t2o_active_trigger->objstr
		     = t2o_OBJ2STR (t2o_active_trigger->obj, NULL, FALSE);
		  strcpy (interp->result, t2o_active_trigger->objstr);
		  break;
	       }
	       case t2o_ArgcC:
	       {  sprintf (interp->result, "%d", t2o_active_trigger->argc);
		  break;
	       }
	       case t2o_ArgvC:
	       {  if (argc == 4)
		  {  if (!t2o_active_trigger->argvlist)
			t2o_active_trigger->argvlist
			= (t2o_active_trigger->argc)
			     ? Tcl_Merge (t2o_active_trigger->argc,
					  t2o_active_trigger->argv)
			     : obst_strdup("");
		     strcpy (interp->result, t2o_active_trigger->argvlist);
		  }
		  else
		  {  int no;
		     obst_string2scalar ("sos_Int", argv[4], &no);
		     if (IS_ERROR())
			RETURN (TCL_ERROR)
		     if (no < 0 || no >= t2o_active_trigger->argc)
		     {  smg_String txt = smg_String(" invalid arg# ") + no
				       + " [0 .. "
				       + (t2o_active_trigger->argc - 1) + "]";
			USE_ERROR(t2o_CtrlCmdC, txt.make_Cstring (SMG_BORROW));
		     }
		     strcpy (interp->result, t2o_active_trigger->argv[no]);
		  }
		  break;
	       }
	       default:
	       {  OPTION_ERR (t2o_CtrlCmdC, argv[3], t2o_trgCurr_opts);
	       }
	    }
	 }
	 break;
      }
      case t2o_VersionC:
      {  sprintf (interp->result,
		  "tclOBST version %s / c2obst version %s / OBST version %s",
		  t2o_version, c2obst_version, obst_version);
	 break;
      }
   }
   RETURN (IS_ERROR() ? TCL_ERROR : TCL_OK)
}

// --------------------------------------------------------------------------

/*
 * Command for general utilities related to the Tcl -> OBST interface:
 *
 *  - const <name> --> <?>
 *	Yields the value associated with the given OBST constant.
 *	Valid constant names are NO_OBJECT, ROOT_CONTAINER, SYNC_CONTAINER,
 *	TEMP_CONTAINER, TRANSACTION, and UNUSED_CONTAINER.
 *
 *  - (has_type | isa | is_some) <obj> <tp_name> --> 0/1
 *	Checks if the given type relation holds between <obj> (encoded form)
 *	and the type named <tp_name>.
 *
 *  - is_scalar <obj> --> 0/1
 *	Yields 0/1 depending on <obj> being the encoded form of a scalar value.
 *
 *  - obj2scalar <obj>		    --> <string>
 *    scalar2obj <string> <tp_name> --> <obj>
 *	Convert between the string (<string>) and the object representation
 *	(<obj>) of scalars. <tp_name> is the name of the scalar type.
 *
 *  - objsize <obj> --> <int>
 *	Yields the size of the given object in bytes.
 *
 *  - tmpstr <string> --> <str_obj>
 *	Yields a newly created temporary string object with content <string>.
 *
 *  - type <obj>     --> <tp_obj>
 *    type <tp_name> --> <tp_obj>
 *	Yields the type object associated either to the encoded object <obj>,
 *	or to the type name <tp_name>.
 */
EXPORT int t2o_UtilCmd (const ClientData,
		        Tcl_Interp *interp, int argc, char *argv[])
{  T_PROC ("t2o_UtilCmd")
   TT (t2o_H, T_ENTER);

   ENTER_and_CHECK (t2o_UtilCmdC, option_code, argv2_idx, obj);

   switch (option_code)
   {  case t2o_ConstC:	// const <name> --> value of constant
      {  do
	 {  switch (argv2_idx)
	    {  case t2o_NOOBJECT_C:
	       {  t2o_return_obj (interp, NO_OBJECT, (char*)obst_CLASS);
	          break;
	       }
	       case t2o_ROOTcntC:
	       {  sprintf (interp->result, "%d", (int)ROOT_CONTAINER);
	          break;
	       }
	       case t2o_SYNCcntC:
	       {  sprintf (interp->result, "%d", (int)SYNC_CONTAINER);
	          break;
	       }
	       case t2o_TEMPcntC:
	       {  sprintf (interp->result, "%d", (int)TEMP_CONTAINER);
	          break;
	       }
#ifdef OBST_HAVE_JOYCE
	       case t2o_TransactionC:
	       {  t2o_return_obj (interp, OBST_CAST(sos_Object,TRANSACTION),
				  	  (char*)obst_CLASS);
	          break;
	       }
#endif
	       case t2o_UNUSEDcntC:
	       {  sprintf (interp->result, "%d", (int)UNUSED_CONTAINER);
		  break;
	       }
	       default:
	       {  argv2_idx = t2o_get_varcode (t2o_AggCmdC, argv[2],
					       t2o_const_opts);
		  if (argv2_idx == t2o_NoCode)
		     RETURN (TCL_ERROR)

		  continue;
	       }
	    }
	    break;
	 }
	 while (1);
	 break;
      }
      case t2o_HasTypeC:
      {  int res = obst_has_type (&obj, argv[3]);
	 BOOL_RESULT (res);
	 break;
      }
      case t2o_IsaC:
      {  int res = obst_isa (&obj, argv[3]);
	 BOOL_RESULT (res);
	 break;
      }
      case t2o_IsScalarC:
      {  int res = obst_is_scalar (&obj);
	 BOOL_RESULT (res);
	 break;
      }
      case t2o_IsSomeC:
      {  int res = obst_is_some (&obj, argv[3]);
	 BOOL_RESULT (res);
	 break;
      }
      case t2o_ObjectSizeC:
      {  int res = obst_objsize (&obj);
	 if (!IS_ERROR())
	    sprintf (interp->result, "%d", (int)res);
	 break;
      }
      case t2o_Obj2ScalarC:
      {  char* res = obst_object2string (&obj);
	 if (!IS_ERROR())
	    Tcl_SetResult (interp, res, TCL_DYNAMIC);
	 break;
      }
      case t2o_Scalar2ObjC:
      {  obj = obst_string2object (argv[3], argv[2]);
	 t2o_return_obj (interp, obj, (char*)obst_CLASS_OR_SCALAR);
	 break;
      }
      case t2o_TmpStrC:
      {  obj = sos_String::create (TEMP_CONTAINER, argv[2]);
	 t2o_return_obj (interp, obj, (char*)obst_CLASS);
	 break;
      }
      case t2o_TypeC:
      {  obj = (_t2o_str2obj (&obj, argv[2])) ? obst_type (&obj)
					      : obst_lookup_type (argv[2]);
	 t2o_return_obj (interp, obj, (char*)obst_CLASS);
	 break;
      }
   }
   RETURN (IS_ERROR() ? TCL_ERROR : TCL_OK)
}
