/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 *
 * --------------------------------------------------------------------------
 * Implementation of the C --> OBST Interface.
 * --------------------------------------------------------------------------
 * ORIGINAL: D. Theobald				DATE: 14/2/93
 * 	This version is not reentrant and hence single threaded.
 * --------------------------------------------------------------------------
 */
/* tclOBST LIBRARY MODULE */

#define OBST_IMP_STRINGOP
#define OBST_IMP_STDARG
#define OBST_IMP_MALLOC
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "obst.h"
#include "obst_stats.h"
#include "smg.h"
#include "cci_use.h"

#include "_c2obst.h"
#include "_t2oconf.h"


// -----------------------------------------------------------------------
// DEFINITION: External global variables
// -----------------------------------------------------------------------

EXPORT const char*  c2obst_version;
EXPORT c2obst_Ctrls c2obst_ctrls;

// -----------------------------------------------------------------------
// DEFINITION: local stuff
// -----------------------------------------------------------------------

LOCAL  err_msg	    err_NO_OBJECT;
LOCAL  int	    _dummy = (_c2obst_init(), 0);

#if c2o_ADD_DEBUG_STUFF
LOCAL struct
{  int	str2Cstr_no, Cstr2str_no,   // counter (invocations, sizes) for
	str2Cstr_sz, Cstr2str_sz;   // specialized sos_Cstring conversions
} c2o_localstats;
#endif


// -----------------------------------------------------------------------
// DEFINITION: Meta Schema Interface
// -----------------------------------------------------------------------

typedef void cvt_byterep (void*,void*);  // - bcopy_... converter


LOCAL inline sos_Bool derived_from_class (sos_Type tp)
{  return (sos_Bool)(tp == sos_Class_type_type);
}
LOCAL inline sos_Bool derived_from_enum (sos_Type tp)
{  return (sos_Bool)(tp == sos_Enum_type_type);
}
LOCAL inline sos_Bool derived_from_extern (sos_Type tp)
{  return (sos_Bool)(tp == sos_Extern_type_type);
}
LOCAL inline sos_Bool basetype_of_scalars (sos_Type tp)
{  return (sos_Bool)(tp == sos_Object_type || tp == sos_Scalar_object_type);
}

LOCAL inline sos_Bool isa_class (sos_Object o)
{  return (sos_Bool)(NOT o.type().is_scalar());
}

LOCAL void c2o_set_schemas_trigger();


/* ------------------------------------------------------------------------
 * Hash bucket entry holding information on enumeration literals.
 */
class LiteralInfo
{ public:
    LiteralInfo (const char*, int);
   ~LiteralInfo ();

   LiteralInfo* next;		// hash bucket chain
   char*	literal;	// enumeration literal
   int		lit_no;		// associated encoding
};

/* ------------------------------------------------------------------------
 * Hash bucket entry holding information on bound methods.
 */
class MethodInfo
{ public:
    MethodInfo (const char*, int, const sos_Method&);
   ~MethodInfo ();

   MethodInfo*  next;		// hash bucket chain
   obst_mInfo   mi;		// (mi.md.code may be NULL till first call)
   int		min_param_no;	// minimal acceptable parameter number
   int		fixed_params;	// no. of non-default parameters, <min_param_no
   				// if >=1 default can not evaluated
   sos_Object   *defargs;	// NULL, or
				// defaultargs 'fixed_params+1 .. mi.param_no'.
   sos_Type     *ptypes;	// parameter types
   sos_Type     restype;	// result type
   sos_Method	descr;		// corresponding meta schema object

   cci_Fun	lookup_code (sos_Bool);
};
EXPORT obst_mInfo* obst_prepared /* = NULL */;

/* ------------------------------------------------------------------------
 * Hash bucket entries holding information on OBST types.
 */
enum TypeKind { CLASS_TP, EXTERN_TP, ENUM_TP, UNION_TP }; // wuerg!
enum CvtIdx   { Str2Obj, Obj2Str, Ext2Int, Int2Ext };

#define I_HTAB_SIZE		 16
#define I_HASH_INT(hashval)	 ((hashval) & 0xF)

class TypeInfo
{ public:
    TypeInfo (const sos_Type&);
   ~TypeInfo ();

   TypeInfo*    next;			// hash bucket chain (OID hash table)
   char*	name;			// type name (no typedef name)
   sos_Type	descr;			// corresponding meta schema object
   TypeKind	tp_kind;		// type encoding
   void*  	itable[I_HTAB_SIZE];	//    (name, #params) --> MethodInfo*
					// or literal --> enumeration value
   char*	*literal;		// enumeration literals
   obst_proc*   cvt_op[4];		// converters for scalar types

   MethodInfo*  lookup_method  (const char*, int);
   int		lookup_literal (const char*);
   sos_Bool	eval_default   (sos_Expr, sos_Object&);
};

 	// Note, that the mapping 'name --> type' is not 1:1 due to typedefs!
	// Hence, the following type can't be merged with type TypeInfo.
class NamedType
{ public:
    NamedType (TypeInfo&);
   ~NamedType ();

   NamedType* next;		// hash bucket chain
   char*      name;		// type name
   TypeInfo*  named_type;	// type associated to this name
};

/* ------------------------------------------------------------------------
 * Abstract data object caching data on OBST types.
 */
#define PREP_STRADDR(a)		((unsigned long)(a)>>4)
#define TP_OBJ_HTAB_SIZE 	1024
#define TP_HASH_OBJ(ct,os)	(((ct) ^ (os)>>4) & 0x3FF)
#define TP_NAME_HTAB_SIZE	512
#define TP_HASH_STRADDR(a)	(PREP_STRADDR(a) & 0x1FF)


class TypeTable
{ public:
   static TypeInfo*	lookup (const char*);
   static TypeInfo*	lookup (const sos_Type&);
   static obst_proc*	lookup_converter (const char*, obst_cvt_ops);

   static void		set_schemas (const char*[]);
   static char**	get_schemas ();
   static void		init ();

   static void		collect_stats (obst_htabstat*);

  private:
   static sos_Object_List smlist;		   	 // search types here
   static NamedType*	  tp_by_name [TP_NAME_HTAB_SIZE];// char* --> TypeInfo*
   static TypeInfo*	  tp_by_ID [TP_OBJ_HTAB_SIZE];   // OID   --> TypeInfo*
};
NamedType*	TypeTable::tp_by_name [TP_NAME_HTAB_SIZE];
TypeInfo*	TypeTable::tp_by_ID [TP_OBJ_HTAB_SIZE];
sos_Object_List TypeTable::smlist;


// -----------------------------------------------------------------------
// IMPLEMENTATION: Meta Schema Interface
// -----------------------------------------------------------------------

// ----- Meta Schema Interface - LiteralInfo -----------------------------

LiteralInfo::LiteralInfo (const char* l, int no)
{  T_PROC ("LiteralInfo::LiteralInfo")
   TT (c2o_VL, T_ENTER);

   next    = NULL;
   literal = (char*)l;
   lit_no  = no;

   TT (c2o_VL, T_LEAVE);
}

LiteralInfo::~LiteralInfo ()
{  T_PROC ("LiteralInfo::~LiteralInfo")
   TT (c2o_VL, T_ENTER);

   if (next)
      delete next;

   TT (c2o_VL, T_LEAVE);
}


// ----- Meta Schema Interface - MethodInfo ------------------------------

EXPORT char* c2obst_encode_type (const sos_Type& tp)
{  T_PROC ("c2obst_encode_type")
   TT (c2o_VL, T_ENTER);

   if (tp == void_type)
      return (char*)obst_VOID;

   else if (tp.is_scalar())
      return TypeTable::lookup (tp)->name;

   else if (tp.has_type (sos_Union_type_type))
   {  sos_Type_List tl = sos_Union_type::make (tp).get_united();
      agg_iterate (tl, sos_Type utp)
	 if (utp.is_scalar() || basetype_of_scalars (utp))
	    return (char*)obst_CLASS_OR_SCALAR;
      agg_iterate_end (tl, utp);
      return (char*)obst_CLASS;
   }
   else
      return basetype_of_scalars (tp) ? (char*)obst_CLASS_OR_SCALAR
				      : (char*)obst_CLASS;
   TT (c2o_VL, T_LEAVE);
}

MethodInfo::MethodInfo (const char* mn, int p_no, const sos_Method& desc)
{  T_PROC ("MethodInfo::MethodInfo")
   TT (c2o_VL, T_ENTER);

   next	           = NULL;
   mi.md.code      = NULL;
   mi.md.user_data = NULL;
   mi.mname        = (char*)mn;
   mi.param_no     = p_no;
   fixed_params    = p_no;
   min_param_no    = p_no;
   defargs	   = NULL;
   descr           = desc;
   restype         = descr.get_result_type().make_type();
   mi.resulttp     = c2obst_encode_type (restype);

   if (!p_no)
   {  mi.refparams = NULL;
      ptypes       = NULL;
      mi.paramtps  = NULL;
   }
   else
   {  ptypes	  = new sos_Type[p_no];
      mi.paramtps = new char* [p_no];

      int rparam_no     = 0;
      sos_Param_List pl = descr.get_params();

      p_no = 0;
      agg_iterate (pl, sos_Param p)
         sos_Type ptp = p.get_type().make_type();
	 
	 mi.paramtps[p_no] = c2obst_encode_type(ptypes[p_no] = ptp);
	 
	 if (p.get_is_ref())
	    ++ rparam_no;
	 
	 // Defaults: trailing part of parameter list, precompute and
	 // adjust min_param_no if evaluation not possible.
	 if (defargs || p.get_default_expr() != NO_OBJECT)
	 {  if (!defargs)
	    {  defargs      = new sos_Object[mi.param_no - p_no];
	       fixed_params = p_no;
	       min_param_no = p_no;
	    }
	    if (! TypeTable::lookup (ptp)
		->eval_default (p.get_default_expr(),
				defargs[p_no - fixed_params]))
	       min_param_no = p_no;
	 }
	 ++ p_no;
      agg_iterate_end (pl, p);

      if (rparam_no)
      {  mi.refparams 	         = new int [rparam_no + 1];
	 mi.refparams[rparam_no] = -1;		// mark list end

	 if (pl != NO_OBJECT)
	 {  int rparam_no = p_no = 0;
	    agg_iterate (pl, sos_Param p)
	       if (p.get_is_ref())
	          mi.refparams[rparam_no ++] = p_no;
	       ++ p_no;
	    agg_iterate_end (pl, p);
	 }
      }
      else
	 mi.refparams = NULL;
   }
   lookup_code (FALSE);		// just try

   TT (c2o_VL, T_LEAVE);
}

MethodInfo::~MethodInfo ()
{  T_PROC ("MethodInfo::~MethodInfo")
   TT (c2o_VL, T_ENTER);

   if (ptypes)
   {  delete ptypes;
      delete mi.paramtps;
      if (mi.refparams)
	 delete mi.refparams;
   }
   if (mi.md.user_data)
      delete mi.md.user_data;

   if (next)
      delete next;

   // note: Must not delete names, since they were entered via uniqueString.

   TT (c2o_VL, T_LEAVE);
}

cci_Fun MethodInfo::lookup_code (sos_Bool must_be_found)
{  T_PROC ("MethodInfo::lookup_code")
   TT (c2o_VL, T_ENTER);

   if (!mi.md.code)
   {  		// The implementation of a method is either the method itself,
      		// or an original method of the root class.
      sos_Method impl_method = descr.get_generated_from();
      if (   impl_method EQ NO_OBJECT
	  OR impl_method.get_defined_in().is_generic_class())
	 impl_method = descr;

      sos_Method_impl_List impls = impl_method.get_impls();
      if (impls != NO_OBJECT)
	 agg_iterate (impls, sos_Method_impl impl)
	    if (impl.isa (cci_Method_impl_type))
	    {  mi.md.code = cci_Schema_impl::get_fun (impl._self_id());
	       break;
	    }
	 agg_iterate_end (impls, impl);

      if (!mi.md.code && must_be_found)
      {  smg_String msg = smg_String("no implementation found for method ")
			  + mi.mname + "/" + mi.param_no;
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			     "c2obst - MethodInfo::lookup_code");
      }
   }
   TT (c2o_VL, T_LEAVE);
   return mi.md.code;
}


// ----- Meta Schema Interface - NamedType -------------------------------

NamedType::NamedType (TypeInfo& tpinfo)
{  T_PROC ("NamedType::NamedType")
   TT (c2o_VL, T_ENTER);

   next	      = NULL;
   name	      = tpinfo.name;
   named_type = &tpinfo;

   TT (c2o_VL, T_LEAVE);
}

NamedType::~NamedType ()
{  T_PROC ("NamedType::~NamedType")
   TT (c2o_VL, T_ENTER);

   if (next)
      delete next;

   TT (c2o_VL, T_LEAVE);
}


// ----- Meta Schema Interface - TypeInfo --------------------------------

TypeInfo::TypeInfo (const sos_Type& tp)
{  T_PROC ("TypeInfo::TypeInfo")
   TT (c2o_VL, T_ENTER);

   void** iptr = itable + I_HTAB_SIZE;
   while (--iptr >= itable)
      *iptr = NULL;

   literal = NULL;
   next    = NULL;
   descr   = tp;

   char* tname = ((sos_Type&)tp).get_name().make_Cstring();
   name = obst_uniqueString(tname);
   delete tname;

   sos_Type tp_type = descr.type();

   tp_kind = derived_from_class(tp_type)
		? CLASS_TP
		: derived_from_enum(tp_type)
		     ? ENUM_TP
		     : derived_from_extern(tp_type)
			  ? EXTERN_TP
			  : UNION_TP;
   
   cvt_op[Ext2Int] = NULL;
   cvt_op[Int2Ext] = NULL;
   cvt_op[Str2Obj] = NULL;
   cvt_op[Obj2Str] = NULL;

   if (tp_kind == ENUM_TP)
   {  sos_String_List sl  = sos_Enum_type::make (descr).get_literals();
      int             idx = 0;

      literal = new char* [sl.card()];
      
      agg_iterate (sl, smg_String s)
	 literal[idx ++] = obst_uniqueString (s.make_Cstring (SMG_BORROW));
      agg_iterate_end (sl, s);
   }

   else if (tp_kind == EXTERN_TP && ((sos_Type&)tp).get_object_size() > 0)
   {  sos_Id key = descr._self_id();

      cvt_op[Ext2Int]
	 = (obst_proc*)cci_Schema_impl::get_fun (_cci_ext2obj_key(key));
      cvt_op[Int2Ext]
	 = (obst_proc*)cci_Schema_impl::get_fun (_cci_obj2ext_key(key));

      cvt_op[Obj2Str]
	 = (obst_proc*)cci_Schema_impl::get_fun (_cci_obj2str_key(key));
      cvt_op[Str2Obj]
	 = (obst_proc*)cci_Schema_impl::get_fun (_cci_str2obj_key(key));
   }
   TT (c2o_VL, T_LEAVE);
}

TypeInfo::~TypeInfo ()
{  T_PROC ("TypeInfo::~TypeInfo")
   TT (c2o_VL, T_ENTER);

   if (tp_kind == CLASS_TP || tp_kind == ENUM_TP)
   {  void** iptr = itable + I_HTAB_SIZE;
      while (--iptr >= itable)
      {  if (*iptr)
	    if (tp_kind == CLASS_TP)
	       delete (MethodInfo*)*iptr;
      	    else
	       delete (LiteralInfo*)*iptr;
      }
   }
   if (next)
      delete next;
   if (literal)
      delete literal;

   TT (c2o_VL, T_LEAVE);
}

/*
 * Yields the method with the given name and number of parameters of the
 * type associated with *this, in case of a class type.
 * In case of an external or enumeration type, the query is re-routed to the
 * (class) type sos_Scalar_object.
 * The result is NULL if no such method can be found.
 *
 * The code field of the resulting method info may be undefined (NULL).
 */
MethodInfo* TypeInfo::lookup_method (const char* mnm, int param_no)
{  T_PROC ("TypeInfo::lookup_method")
   TT (c2o_VL, T_ENTER);

   if (tp_kind == EXTERN_TP || tp_kind == ENUM_TP)
   {  TypeInfo* tiptr = TypeTable::lookup (sos_Scalar_object_type);

      return (tiptr) ? tiptr->lookup_method (mnm, param_no)
	 	     : NULL;
   }
   else if (tp_kind != CLASS_TP)
   {  smg_String msg = smg_String("can't lookup method for that kind of type ")
		       + descr.get_name();
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - TypeInfo::lookup_method");
      return NULL;
   }
   // The hash bucket must not be chosen based on param_no, since a MethodInfo
   // might than have to be inserted in an unbounded number of buckets due to
   // the existence of default parameters.
   char*           mname = obst_uniqueString(mnm);
   register void** miptr = &itable[I_HASH_INT(PREP_STRADDR(mname))];
   while (*miptr)
   {  if (   ((MethodInfo*)*miptr)->mi.mname     == mname
	  && ((MethodInfo*)*miptr)->mi.param_no  >= param_no
	  && ((MethodInfo*)*miptr)->fixed_params <= param_no)
	 return (MethodInfo*)*miptr;

      miptr = (void**)&((MethodInfo*)*miptr)->next;
   }
   static sos_String mstr = sos_String::create (TEMP_CONTAINER);

   mstr.assign_Cstring (mname);

   sos_Class_type  ctp       = sos_Class_type::make (descr);
   sos_Method_List ml        = ctp.get_methods()[mstr];
   sos_Method      result    = sos_Method::make (NO_OBJECT);
   int		   res_pcard = 1000000;

   if (ml != NO_OBJECT)		// list of overloaded methods
      agg_iterate (ml, sos_Method m)
	 sos_Param_List pl    = m.get_params();
   	 int		pcard = (pl == NO_OBJECT) ? 0 : pl.card();

   	 if (pcard >= param_no && pcard < res_pcard)
	 {  result    = m;
	    res_pcard = pcard;
	    if (pcard == param_no)
	       break;
	 }
      agg_iterate_end (ml, m);

   MethodInfo* new_mi = NULL;

   if (result != NO_OBJECT)
   {  new_mi = new MethodInfo (mname, res_pcard, result);
      if (new_mi->fixed_params > param_no)
	 result = sos_Method::make (NO_OBJECT);
   }
   if (result == NO_OBJECT)
   {  smg_String msg = smg_String("no method ")
			  + ctp.get_name() + "::" + mname + "/" + param_no;
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - TypeInfo::lookup_method");
   }
   else if (result.get_kind() != sos_PUBLIC)
   {  smg_String msg = smg_String("method ")
			  + ctp.get_name() + "::" + mname
			  + "/" + new_mi->mi.param_no
			  + " is not PUBLIC";
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - TypeInfo::lookup_method");

      result = sos_Method::make (NO_OBJECT);
   }
   if (result != NO_OBJECT)
      *(MethodInfo**)miptr = new_mi;
   else if (new_mi)
   {  delete new_mi;
      new_mi = NULL;
   }
   TT (c2o_VL, T_LEAVE);
   return new_mi;
}

/*
 * Yields the integer encoding of the given literal of this type, which must
 * be an enumeration type.
 * In case of error, the return value is -1.
 *
 * BUGS: Maybe we should better use the methods of class sos_Enum_type, but
 *	 these do not provide caching.
 *	 Anyway, the two stage mapping (char* --> type, type X char* --> int)
 *	 might be reduced to a single stage.
 */
int TypeInfo::lookup_literal (const char* literal)
{  T_PROC ("TypeInfo::lookup_literal")
   TT (c2o_VL, T_ENTER);

   if (tp_kind != ENUM_TP)
   {  smg_String msg = smg_String("literal searched for non-enumeration type ")
		       + descr.get_name();
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - TypeInfo::lookup_literal");
      return -1;
   }
   register
   char*           lname = obst_uniqueString(literal);
   register void** liptr = &itable [I_HASH_INT( (int)lname )];
   while (*liptr)
   {  if (((LiteralInfo*)*liptr)->literal == lname)
	 return ((LiteralInfo*)*liptr)->lit_no;

      liptr = (void**) &((LiteralInfo*)*liptr)->next;
   }
   sos_String_List sl = sos_Enum_type::make (descr).get_literals();
   int             no = 0;
   smg_String      s;

   agg_iterate (sl, s)
      if (s.equal (literal))
	 break;
      ++ no;
   agg_iterate_end (sl, s);

   if (no >= sl.card())
   {  no = -1;
      s  = smg_String("no literal ") + lname + " in enumeration type "
				     + descr.get_name();
      err_raise (err_USE, s.make_Cstring (SMG_BORROW),
			  "c2obst - TypeInfo::lookup_literal");
   }
   else
      *(LiteralInfo**)liptr = new LiteralInfo (lname, no);
   
   TT (c2o_VL, T_LEAVE);
   return no;
}

/*
 * Evaluates the the given expression and returns its value via 'expr_result'.
 * This object represents the expected result type of 'expr'.
 *
 * The method returns TRUE, iff 'expr' could be successfully evaluated.
 * Otherwise, the value of 'expr_result' is undefined.
 *
 * (The method should probably be part of the OBST meta schema interface.)
 */
sos_Bool TypeInfo::eval_default (sos_Expr expr, sos_Object& expr_result)
{  T_PROC ("TypeInfo::eval_default")
   TT (c2o_VL, T_ENTER);

   sos_Bool   result = FALSE,
   	      is_int = expr.has_type (sos_Int_expr_type);

   int        ival;
   smg_String sval;

   if (is_int)
      ival = sos_Int_expr::make (expr).get_value();
   else
      sval = sos_Identifier::make (expr).get_id();

   if (tp_kind == CLASS_TP || tp_kind == UNION_TP)
   {  if (!is_int && sval.equal ("NO_OBJECT"))
      {  result	     = TRUE;
	 expr_result = NO_OBJECT;
      }
   }
   else							// scalar type
   {  if (!is_int)
	 if (streql (name, "sos_Container"))
	 {  if (sval.equal ("ROOT_CONTAINER"))
	       expr_result = make_sos_Container_object (ROOT_CONTAINER);
	    else if (sval.equal ("SYNC_CONTAINER"))
	       expr_result = make_sos_Container_object (SYNC_CONTAINER);
	    else if (sval.equal ("TEMP_CONTAINER"))
	       expr_result = make_sos_Container_object (TEMP_CONTAINER);
	    else if (sval.equal ("UNUSED_CONTAINER"))
	       expr_result = make_sos_Container_object (UNUSED_CONTAINER);
	    else
	       expr_result = NO_OBJECT;
	 }
         else
            expr_result = obst_string2object (name,
					      sval.make_Cstring (SMG_BORROW));
      else if (tp_kind == ENUM_TP)
	 expr_result = obst_enum2object (name, ival);

      else if (streql (name, "sos_Int"))
	 expr_result = make_sos_Int_object (ival);
      else
	 expr_result = NO_OBJECT;

      result = (expr_result != NO_OBJECT);
   }
   TT (c2o_VL, T_LEAVE);
   return result;
}


// ----- Meta Schema Interface - TypeTable -------------------------------

/*
 * Initialization of ADO TypeTable - must be called before using the ADO.
 */
void TypeTable::init ()
{  T_PROC ("TypeTable::init")
   TT (c2o_L, T_ENTER);

#if t2o_FORCE_SET_SCHEMAS
   set_schemas (NULL);
#else
   smlist = sos_Object_List::make (NO_OBJECT);
#endif
   //set_schemas requires that ROOT_CONTAINER is defined now (needed in
   //sos_Schema_module::schema_dir()). This can only be assured starting
   //from release OBST3-3.4.
   //Otherwise, the call will eventually be performed in lookup(char*), if
   //not done so explicitly before.

   // type_by_ID and type_by_name are statically initialized.

   TT (c2o_L, T_LEAVE);
}

/*
 * Defines the 'name --> type' binding by listing the schemas which will be
 * searched for a type.
 * 
 * The schemas are specified by a string list whose end is marked by either
 * an empty string or a NULL pointer.
 * The named schemas must be known at call time.
 *
 * A type will be looked up in the specified schemas using the ordering
 * defined by the argument array. 
 * When searching a schema, (transitively) imported schemas are searched, too.
 * The first found type with the given name is then returned.
 *
 * If no schemas are named (i.e. schemas == NULL), all currently known schemas
 * are searched in a system defined order.
 *
 * BUGS: see TypeTable::lookup(char*).
 */
void TypeTable::set_schemas (const char* schemas[])
{  T_PROC ("TypeTable::set_schemas")
   TT (c2o_VL, T_ENTER);

   static sos_Schema_module_Directory smdir;
   static sos_String		      str;
   static sos_Bool do_init = TRUE;
   if (do_init)
   {  do_init = FALSE;
      str     = sos_String::create (TEMP_CONTAINER);
      smlist  = sos_Object_List::create (TEMP_CONTAINER);
      smdir   = sos_Schema_module::schema_dir();
   }
   else
   {  smlist.clear();
      /*
       * Must drop speedup caches of this module and of the meta schema, too.
       * (It would be sufficient to drop the name-->type mapping plus all
       *  scalar converters/method implementations entered via
       *  obst_scalar_converter, obst_method_data - but this optimization is
       *  not yet implemented.)
       */
      sos_Schema_module knlsm
	 = sos_Schema_module::retrieve (sos_Object_type.container());
      knlsm.lookup_type (sos_String::make(NO_OBJECT), sos_IMP_NONE);

      TypeInfo** tpiptr;
      for (tpiptr = tp_by_ID + TP_OBJ_HTAB_SIZE;  --tpiptr >= tp_by_ID; )
	 if (*tpiptr)
	 {  delete *tpiptr;
	    *tpiptr = NULL;
	 }
      NamedType** ntpptr;
      for (ntpptr = tp_by_name + TP_NAME_HTAB_SIZE;  --ntpptr >= tp_by_name; )
	 if (*ntpptr)
	 {  delete *ntpptr;
	    *ntpptr = NULL;
	 }
   }
   if (schemas && *schemas)
      for (char** sname = (char**)schemas;  *sname && **sname;  ++ sname)
      {  str.assign_Cstring (*sname);
	 sos_Schema_module sm = smdir[str];
	 if (sm == NO_OBJECT)
	 {  smg_String msg = smg_String("unknown schema ") + *sname;
	    err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
				"c2obst - TypeTable::set_schemas");
	 }
	 else
	    smlist.append (sm);			// ignoring duplicates
      }
   else
      agg_iterate_association (smdir, sos_String sn, sos_Schema_module sm)
	 smlist.append (sm);
      agg_iterate_association_end (smdir, sn, sm);

   // This is the place to insert any table modifications which are to be
   // stable w.r.t. changes in the type mapping. An example of such a
   // modification is the definition of specific conversion operations (using
   // obst_scalar_converter) which override the standard conversions.

   c2o_set_schemas_trigger();

   if (c2obst_ctrls.set_schemas_trigger)
      (*c2obst_ctrls.set_schemas_trigger)();

   TT (c2o_VL, T_LEAVE);
}

/*
 * Yields the currently searched schemas in the same format as used by
 * TypeTable::set_schemas to define the schemas of interest.
 *
 * It is up to the caller to destroy the resulting string array as well as the
 * strings contained therein.
 */
char** TypeTable::get_schemas ()
{  T_PROC ("TypeTable::lookup")
   TT (c2o_VL, T_ENTER);

   char** snames = NULL;

   if (smlist != NO_OBJECT && smlist.card() > 0)
   {  snames = new char* [smlist.card() + 1];

      int idx = 0;
      agg_iterate (smlist, sos_Object obj)
	 snames[idx ++]
	    = sos_Schema_module::make(obj).get_name().make_Cstring();
      agg_iterate_end (smlist, obj);
      snames[idx] = NULL;
   }
   TT (c2o_VL, T_LEAVE);
   return snames;
}

/*
 * Look up the data belonging to the given type.
 * Multiple lookups are sped up by caching the type data.
 */
TypeInfo* TypeTable::lookup (const sos_Type& tp)
{  T_PROC ("TypeTable::lookup")
   TT (c2o_VL, T_ENTER);

   register unsigned   tp_os  = tp.offset();
   register unsigned   tp_ct  = tp.container();
   register TypeInfo** tpiptr = &tp_by_ID [TP_HASH_OBJ (tp_ct, tp_os)];

   while (*tpiptr)
   {  if (   (*tpiptr)->descr.offset()    == tp_os
	  && (*tpiptr)->descr.container() == tp_ct)
	 return *tpiptr;

      tpiptr = &(*tpiptr)->next;
   }
   *tpiptr = new TypeInfo (tp);

   TT (c2o_VL, T_LEAVE);
   return *tpiptr;
}

/*
 * Look up the data belonging to the type with the given name.
 * If the type is not yet known, look it up as described for
 * TypeTable::set_schemas.
 * The result is NULL, if no such type can be found.
 *
 * Multiple lookups are sped up by caching the type data.
 *
 * The cache for the 'name --> type data' binding is assumed to be less
 * important than the 'OID --> type data' cache.
 * Therefore, any data looked up via this lookup method is inserted in the
 * 'OID --> type data' cache, too - but not vice versa.
 *
 * BUGS: The bindings 'name --> type' and 'object --> type' may be different if
 *	 there are two types with the same name (which must then be defined in
 *	 two schemas not connected via an (transitive) import relation).
 *	 Currently, this case is not checked.
 */
TypeInfo* TypeTable::lookup (const char* tn)
{  T_PROC ("TypeTable::lookup")
   TT (c2o_VL, T_ENTER);

   register char*       tname = obst_uniqueString (tn);
   register NamedType** ntptr = &tp_by_name [TP_HASH_STRADDR (tname)];

   while (*ntptr)
   {  if ((*ntptr)->name == tname)
	 return (*ntptr)->named_type;

      ntptr = &(*ntptr)->next;
   }
#if !t2o_FORCE_SET_SCHEMAS
   if (smlist == NO_OBJECT)
      set_schemas (NULL);
#endif

   static sos_String tnstr = sos_String::create (TEMP_CONTAINER);
   tnstr.assign_Cstring (tname);

   TypeInfo* tpinfo = NULL;

   agg_iterate (smlist, sos_Object _sm)
      sos_Type_descr td = sos_Schema_module::make(_sm).lookup_type (tnstr);
      if (td != NO_OBJECT && (tpinfo = TypeTable::lookup (td.make_type())))
      {  *ntptr = new NamedType (*tpinfo);
	 break;
      }
   agg_iterate_end (smlist, _sm);
   if (!tpinfo)
   {  smg_String msg = smg_String("unknown type ") + tname;
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - TypeTable::lookup");
   }
   TT (c2o_VL, T_LEAVE);
   return tpinfo;
}

/*
 * Extend *ht by statistics about the local caches.
 */
void TypeTable::collect_stats (obst_htabstat* ht)
{  T_PROC ("TypeTable::collect_stats")
   TT (c2o_VL, T_ENTER);

   obst_htabinfo *htinfo_tp, *htinfo_class, *htinfo_enum;

   int i, j, size_tp, size_other,
       count_tpinfo = 0,
       count_linfo  = 0,
       count_minfo  = 0,
       count_enum   = 0,
       count_class  = 0;

   htinfo_tp 	      = obst_add_htabinfo (ht);
   htinfo_tp->buckets = TP_NAME_HTAB_SIZE;
   
   for (i = TP_NAME_HTAB_SIZE;  i --; )
   {  NamedType* ntp = TypeTable::tp_by_name[i];
      for (size_tp = 0;  ntp;  ntp = ntp->next)
      {  ++ size_tp;
	 ++ count_tpinfo;
      }
      obst_add_histdata (&htinfo_tp->histogram, &htinfo_tp->histsize, size_tp);
   }
   smg_String descr = smg_String("c2obst: type name --> type data (")
      		      + (sizeof(NamedType*) * TP_NAME_HTAB_SIZE
			 + sizeof(NamedType) * count_tpinfo)
		      + " bytes)";
   htinfo_tp->descr = descr.make_Cstring (SMG_TRANSFER);

   i = ht->size;
   obst_add_htabinfo (ht);		// Pointers into ht->htabs will in
   obst_add_htabinfo (ht);		// general become invalid by adding
   obst_add_htabinfo (ht);		// new htabinfo's.

   count_tpinfo = 0;
   htinfo_tp 	= ht->htabs + i;	// Hence, the pointers are extracted
   htinfo_class	= ht->htabs + i + 1;	// after all additions are performed.
   htinfo_enum	= ht->htabs + i + 2;

   for (i = TP_OBJ_HTAB_SIZE;  i --; )
   {  TypeInfo* tp = TypeTable::tp_by_ID[i];
      for (size_tp = 0;  tp;  tp = tp->next)
      {  ++ size_tp;
	 ++ count_tpinfo;

	 if (tp->tp_kind == CLASS_TP)
 	 {  ++ count_class;
	    for (j = I_HTAB_SIZE;  j --; )
	    {  MethodInfo* mip = (MethodInfo*)tp->itable[j];
	       for (size_other = 0;  mip;  mip = mip->next)
	       {  ++ size_other;
		  ++ count_minfo;
	       }
	       obst_add_histdata (&htinfo_class->histogram,
			          &htinfo_class->histsize, size_other);
	    }
	 }
	 else if (tp->tp_kind == ENUM_TP)
	 {  ++ count_enum;
	    for (j = I_HTAB_SIZE;  j --; )
	    {  LiteralInfo* mip = (LiteralInfo*)tp->itable[j];
	       for (size_other = 0;  mip;  mip = mip->next)
	       {  ++ size_other;
	          ++ count_linfo;
	       }
	       obst_add_histdata (&htinfo_enum->histogram,
			          &htinfo_enum->histsize, size_other);
	    }
	 }
      }
      obst_add_histdata (&htinfo_tp->histogram, &htinfo_tp->histsize, size_tp);
   }
   descr = smg_String("c2obst: type ID --> type data (")
           + (sizeof(TypeInfo) * count_tpinfo
	      + sizeof(TypeInfo*) * TP_OBJ_HTAB_SIZE)
	   + " bytes)";
   htinfo_tp->descr   = descr.make_Cstring (SMG_TRANSFER);
   htinfo_tp->buckets = TP_OBJ_HTAB_SIZE;

   descr = smg_String("c2obst: method name --> method data (")
      	   	      + count_class + " tables; "
		      + (sizeof(MethodInfo) * count_minfo)
		      + " bytes)";
   htinfo_class->descr   = descr.make_Cstring (SMG_TRANSFER);
   htinfo_class->buckets = count_class * I_HTAB_SIZE;

   descr = smg_String("c2obst: enumeration literal --> value (")
      		      + count_enum + " tables; "
		      + (sizeof(LiteralInfo) * count_linfo)
		      + " bytes)";

   htinfo_enum->descr   = descr.make_Cstring (SMG_TRANSFER);
   htinfo_enum->buckets = count_enum * I_HTAB_SIZE;

   TT (c2o_VL, T_LEAVE);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: Method Calls
// --------------------------------------------------------------------

// Requirements for reentrant calls:
// - data from current_call must not be used after the actual call is done.
// - current_call.obj must point to stable storage.
// - obst_set_schemas must not be called while a method is active.

struct t2o_cc_t
{  MethodInfo* minfo;
   int	       argc;
   sos_Object* obj;
};

LOCAL t2o_cc_t current_call;
LOCAL int      call_level /* = 0 */;

// Must store call_level outside of call_method to allow jumps out of that
// method.
#ifdef ERR_NO_ASSERT
#  define ENTER_CALL()
#  define LEAVE_CALL()
#else
#  define ENTER_CALL()	int __lvl = call_level
#  define LEAVE_CALL()	call_level = __lvl
#endif


/* Definition of `efficient' code for setting the elements of an OBST Array
 * of type sos_Object_Array by a scan starting at the first array element.
 * Run-time efficiency is gained by breaking the encapsulation of the
 * Array<sos_Object> encapsulation. Thereby, intermediary method calls, index
 * checks, and address calculations can be avoided.
 *
 * While 'c2o_BREAK_ENCAPSULATION == 0' should be rather safe,
 * 'c2o_BREAK_ENCAPSULATION == 1' requires additionally:
 *    . sos_Object::_self has the relative offset 0 wrt. an sos_Object object
 *    . the bcopy operations for sos_Offset and for sos_Pointer make the same
 *	conversions
 *
 * The scan loop must not depend on any computation performed by elaborating
 * the `idx' argument of c2o_ARRAY_SET_nTH.
 */

#if c2o_BREAK_ENCAPSULATION
#  define c2o_ARRAY_ADDR_RELPOS 18
		// offset of `Array::address' component within an Array object
		// **** will probably change in OBST versions after 3-4.0

#  define c2o_ARRAY_DEFS()	char* __addr

#  if c2o_BREAK_ENCAPSULATION > 1
#  define c2o_ARRAY_INIT_SCAN(array) \
	{ sos_Offset off, _off;\
	  memcpy (&_off, (void*)((array).offset() + c2o_ARRAY_ADDR_RELPOS),\
			 SOS_OFFSET_SIZE);\
	  bcopy_to_sos_Offset (&__addr, &_off); }

#     define c2o_ARRAY_SET_nTH(array, idx, val)\
	{ bcopy_from_sos_Typed_id (&(val), __addr);\
	  __addr += SOS_TYPED_ID_SIZE; }
#  else
#     define c2o_ARRAY_INIT_SCAN(array) \
	{ sos_Offset off, _off;\
	  memcpy (&_off, (void*)((array).offset() + c2o_ARRAY_ADDR_RELPOS),\
			 SOS_OFFSET_SIZE);\
	  bcopy_to_sos_Offset (&off, &_off);\
	  __addr = (char*)off; }

#     define c2o_ARRAY_SET_nTH(array, idx, val)\
	{ sos_Typed_id __tpid = (val)._typed_id();\
	  bcopy_from_sos_Typed_id (&__tpid, __addr);\
	  __addr += SOS_TYPED_ID_SIZE; }
#  endif
#else
#  define c2o_ARRAY_DEFS()
#  define c2o_ARRAY_INIT_SCAN(array)
#  define c2o_ARRAY_RESET_SCAN(array)
#  define c2o_ARRAY_ADVANCE_SCAN(array,mv)
#  define c2o_ARRAY_SET_nTH(array,idx,val)	(array).set_nth(idx, val)
#endif

/*
 * Calls the method specified by 'current_call.minfo'.
 * Instance methods are called on '*current_call.obj', otherwise this field is
 * ignored.
 * The arguments of the method call are given via the array 'args'. This 
 * argument is ignored if 'current_call.argc == 0'.
 *
 * If obst_check_args == TRUE, a check is performed that each argument is
 * either NO_OBJECT or belongs to a subtype of the parameter type.
 *
 * The result of a method with result type 'void' is NO_OBJECT.
 * The result values of reference parameters will be transfered back to the
 * argument array 'args'.
 *
 * This function is not reentrant.
 * The function is left immediately if '!current_call.minfo == NULL' since this
 * means that a type lookup failed and this error has already been reported.
 */
LOCAL sos_Object call_method (sos_Object args[])
{  T_PROC ("call_method")
   TT (c2o_L, T_ENTER);

   if (!current_call.minfo)
      return NO_OBJECT;

   int      pno = current_call.minfo->mi.param_no;
   sos_Bool uses_defargs
	    = (sos_Bool)(current_call.argc < current_call.minfo->mi.param_no);

   if (uses_defargs
       && (   !c2obst_ctrls.allow_defaults
	   || current_call.argc < current_call.minfo->min_param_no))
   {  smg_String mname = smg_String (current_call.minfo->mi.mname) + "/" + pno,
	 	 msg;
      if (!c2obst_ctrls.allow_defaults)
	 msg = smg_String("default arguments not allowed (") + mname + ")";
      else
	 msg = mname + " takes at least " + current_call.minfo->min_param_no
	    	     + " argument(s)";

      err_raise (err_USE, msg.make_Cstring(SMG_BORROW),"c2obst - call_method");
      return NO_OBJECT;
   }
   sos_Object_Array* oargv;	// pointer to argument array

   {  static sos_Object_Array* _oargv;			// argument array pool
      static int*	       _oargc   /* = 0 */;	// size of _oargv[i]
      static int	       maxlevel /* = 0 */;	// max. level so far

      if (call_level == maxlevel)
      {  ++ call_level;

	 if (maxlevel)
	 {  _oargc = (int*)REALLOC(_oargc, call_level*sizeof(int));
	    _oargv = (sos_Object_Array*)
			   REALLOC(_oargv, call_level*sizeof(sos_Object_Array));
	 }
	 else
	 {  _oargc = new int;
	    _oargv = new sos_Object_Array;
	 }
	 oargv		  = _oargv + maxlevel;
	 *oargv		  = sos_Object_Array::create (TEMP_CONTAINER, 0, pno);
	 _oargc[maxlevel] = pno + 1;

	 maxlevel = call_level;
      }
      else
      {  oargv = _oargv + call_level;
	 
	 if (_oargc[call_level] < pno)
	 {  (*oargv).destroy();

	    *oargv	       = sos_Object_Array::create (TEMP_CONTAINER,
							   0, pno);
	    _oargc[call_level] = pno + 1;
	 }
	 ++ call_level;
      }
   }

   sos_Object result;

   if (current_call.minfo->mi.md.code || current_call.minfo->lookup_code (TRUE))
   {  c2o_ARRAY_DEFS();

      register int idx = 0;

      if (c2obst_ctrls.check_args)
      {  if (current_call.argc)
	 {  c2o_ARRAY_INIT_SCAN (*oargv);

      	    register sos_Object *arg_i = (sos_Object*)args;
	    register sos_Type   *ptp_i = current_call.minfo->ptypes;

	    for (;  idx < current_call.argc;  ++idx, ++arg_i, ++ptp_i)
	    {  if (*arg_i == NO_OBJECT || arg_i->is_some (*ptp_i))
	       {  c2o_ARRAY_SET_nTH (*oargv, idx, *arg_i);
	       }
	       else
	       {  smg_String msg = smg_String (current_call.minfo->mi.mname)
 				   + "/" + pno + ", arg #" + (idx+1)
 				   + " - type " + ptp_i->get_name()
 				   + " expected";
		  err_raise (err_USE, msg.make_Cstring(SMG_BORROW),
 				      "c2obst - call_method");
      		  return NO_OBJECT;
	       }
	    }
	 }
      }
      else if (current_call.argc)
      {  c2o_ARRAY_INIT_SCAN (*oargv);

      	 register sos_Object *arg_i = (sos_Object*)args;
	 register int 	     stop   = current_call.argc;
	 do
	 {  c2o_ARRAY_SET_nTH (*oargv, idx, *(arg_i ++));
	 }
	 while (++ idx < stop);
      }
#if c2o_ADD_DEBUG_STUFF
      err_assert (idx == current_call.argc, "c2obst - call_method");
#endif
      if (uses_defargs)
      {  if (!current_call.argc)
	    c2o_ARRAY_INIT_SCAN (*oargv);

	 register sos_Object *arg_i = current_call.minfo->defargs
				      + (idx-current_call.minfo->fixed_params);
	 register int	     stop   = pno;
	 do
	 {  c2o_ARRAY_SET_nTH (*oargv, idx, *(arg_i ++));
	 }
	 while (++ idx < stop);
      }
      obst_mInfo* mi = &current_call.minfo->mi;

      result = (* (mi->md.code)) (*current_call.obj, *oargv);

      if (mi->refparams)
      {  register int *idxptr = mi->refparams;
	 do
	    args[*idxptr] = (*oargv)[*idxptr];
	 while (*(++ idxptr) >= 0);
      }
   }
   else
      result = NO_OBJECT;

   TT (c2o_L, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------

LOCAL sos_Object varargs_buffer[c2o_MAX_VARARGS];

/*
 * Move the list of 'argc' sos_Object's into varargs_buffer.
 * It is an error to pass at least MAXARGS arguments.
 * 
 * (Simple implementation which uses a static buffer - reentrant calls must
 *  be made via the non-varargs interfaces.)
 */
LOCAL void get_varargs (register int argc, register va_list ap)
{  T_PROC ("get_varargs")
   TT (c2o_L, T_ENTER);

#ifndef ERR_NO_ASSERT
   if (call_level != 0)
   {  TT (c2o_L, T_LEAVE);
      err_raise (err_SYS, "reentrant call", "c2obst - get_varargs", FALSE);
   }
   if (argc >= c2o_MAX_VARARGS)
   {  TT (c2o_L, T_LEAVE);
      err_raise (err_SYS, "too many arguments passed to method", 
			  "c2obst - get_varargs", FALSE);
   }
#endif
   register
   sos_Object* arg = varargs_buffer;

   while (argc--)
      *(arg ++) = va_arg (ap, sos_Object);

   TT (c2o_L, T_LEAVE);
}


// --------------------------------------------------------------------

/*
 * Externally visible wrapper around 'call_method' for calling the
 * instance method 'mname' on object 'obj' with arguments 'args'.
 */
EXPORT sos_Object obst_call_method (const sos_Object* obj,
				    const char*       mname,
				    int argc, sos_Object args[])
{  T_PROC ("obst_call_method")
   TT (c2o_H, T_ENTER);

   ENTER_CALL();
   current_call.argc = argc;

   sos_Object result;
   if (*(current_call.obj = (sos_Object*)obj) == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_call_method", FALSE);
      result = NO_OBJECT;
   }
   else
   {  TypeInfo* tiptr;
      ERR_HANDLING_WRAPPER (
	 current_call.minfo
	    = (tiptr = TypeTable::lookup (current_call.obj->type()))
		 ? tiptr->lookup_method (mname, argc)
		 : NULL;
	 result = call_method (args);,

	 result = NO_OBJECT;)
   }
   LEAVE_CALL();

   TT (c2o_H, T_LEAVE);
   return result;
}

/*
 * Externally visible wrapper around 'call_method' for calling the
 * class method 'mname' of class 'cname' with arguments 'args'.
 */
EXPORT sos_Object obst_call_classmethod (const char* cname,
					 const char* mname,
					 int argc, sos_Object args[])
{  T_PROC ("obst_call_classmethod")
   TT (c2o_H, T_ENTER);

   ENTER_CALL();
   current_call.obj  = &NO_OBJECT;
   current_call.argc = argc;

   TypeInfo*  tiptr;
   sos_Object result;
   ERR_HANDLING_WRAPPER (
      current_call.minfo
       = (tiptr = TypeTable::lookup (cname)) ? tiptr->lookup_method(mname,argc)
					     : NULL;
      result = call_method (args);,

      result = NO_OBJECT;)
   LEAVE_CALL();

   TT (c2o_H, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------

/*
 * <stdarg.h> wrapper around 'call_method' for calling the instance method
 * 'mname' on object 'obj'. The arguments are given as a NULL terminated
 * list.
 */
EXPORT sos_Object obst_vcall_method (const sos_Object* obj,
				     const char*       mname, int argc ...)
{  T_PROC ("obst_vcall_method")
   TT (c2o_H, T_ENTER);

   ENTER_CALL();
   va_list ap;
   va_start (ap, argc);

   sos_Object result;

   current_call.argc = argc;
   if (*(current_call.obj = (sos_Object*)obj) == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_vcall_method", FALSE);
      result = NO_OBJECT;
   }
   else
   {  TypeInfo*  tiptr;
      ERR_HANDLING_WRAPPER (
	 get_varargs (argc, ap);
	 current_call.minfo
		= (tiptr = TypeTable::lookup (current_call.obj->type()))
		     ? tiptr->lookup_method (mname, argc)
		     : NULL;
	 result = call_method (varargs_buffer);,

	 result = NO_OBJECT;)
   }
   va_end (ap);
   LEAVE_CALL();

   TT (c2o_H, T_LEAVE);
   return result;
}

/*
 * <stdarg.h> wrapper around 'call_method' for calling the class method
 * 'mname' of class 'cname'. The arguments are given as a NULL terminated list.
 */
EXPORT sos_Object obst_vcall_classmethod (const char* cname,
					  const char* mname, int argc ...)
{  T_PROC ("obst_vcall_classmethod")
   TT (c2o_H, T_ENTER);

   ENTER_CALL();
   va_list ap;
   va_start (ap, argc);

   current_call.obj  = &NO_OBJECT;
   current_call.argc = argc;

   TypeInfo*  tiptr;
   sos_Object result;
   ERR_HANDLING_WRAPPER (
      get_varargs (argc, ap);
      current_call.minfo
	     = (tiptr = TypeTable::lookup (cname))
		  ? tiptr->lookup_method (mname, argc)
		  : NULL;
      result = call_method (varargs_buffer);,

      result = NO_OBJECT;
      );
   va_end (ap);
   LEAVE_CALL();

   TT (c2o_H, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------

/*
 * Prepares for a call of the instance method specified via 'mname' and 'argc'
 * on 'obj' by looking up the respective meta data.
 * Some meta data can afterwards be accessed via 'obst_prepared'.
 *
 * The (boolean) result indicates the success of the lookup.
 */
EXPORT sos_Bool obst_mcall_prep (const sos_Object* obj,
				 const char*	   mname, int argc)
{  T_PROC ("obst_mcall_prep")
   TT (c2o_H, T_ENTER);

   current_call.argc  = argc;
   current_call.minfo = NULL;
   obst_prepared      = NULL;

   sos_Bool result    = FALSE;

   if (*(current_call.obj = (sos_Object*)obj) == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_mcall_prep", FALSE);
   else
   {  TypeInfo* tiptr;
      ERR_HANDLING_WRAPPER (
	 if (tiptr = TypeTable::lookup (current_call.obj->type()))
	 {  if (current_call.minfo = tiptr->lookup_method (mname, argc))
	    {  obst_prepared = &current_call.minfo->mi;
	       result = TRUE;
	    }
	 }, ;)
   }
   TT (c2o_H, T_LEAVE);
   return result;
}

/*
 * Prepares for a call of the class method specified via 'cname', 'mname' and
 * 'argc' by looking up the respective meta data.
 * Some meta data can afterwards be accessed via 'obst_prepared'.
 *
 * The (boolean) result indicates the success of the lookup.
 */
EXPORT sos_Bool obst_cmcall_prep (const char* cname,
				  const char* mname, int argc)
{  T_PROC ("obst_cmcall_prep")
   TT (c2o_H, T_ENTER);

   current_call.obj   = &NO_OBJECT;
   current_call.argc  = argc;
   current_call.minfo = NULL;
   obst_prepared      = NULL;

   sos_Bool  result   = FALSE;
   TypeInfo* tiptr;
   ERR_HANDLING_WRAPPER (
      if (tiptr = TypeTable::lookup (cname))
      {  if (current_call.minfo = tiptr->lookup_method (mname, argc))
	 {  obst_prepared = &current_call.minfo->mi;
	    result = TRUE;
	 }
      }, ;)

   TT (c2o_H, T_LEAVE);
   return result;
}

/*
 * Wrapper around 'call_method' to execute a previously prepared method call.
 */
EXPORT sos_Object obst_call_prepared (sos_Object args[])
{  T_PROC ("obst_call_prepared")
   TT (c2o_H, T_ENTER);

   ENTER_CALL();

   sos_Object result;
   if (obst_prepared)
      ERR_HANDLING_WRAPPER (result = call_method (args);,
			    result = NO_OBJECT;)
   else
   {  err_raise (err_USE, "no call prepared",
			  "c2obst - obst_call_prepared", FALSE);
      result = NO_OBJECT;
   }
   LEAVE_CALL();

   TT (c2o_H, T_LEAVE);
   return result;
}

/*
 * Function for retrieving information about the current method: the meta
 * object describing this method will be assigned to `*descr'.
 * Furthermore, if `args != NULL', the `args' argument array will be check
 * w.r.t. the current method.
 * The function yields
 *  . -2 if there is no current method,
 *  . the number of the first offending argument (starting from 0) if arguments
 *    are checked and there is an argument type error, or
 *  . -1, otherwise.
 */
EXPORT int obst_prepared_info (sos_Object* descr, sos_Object* args)
{  T_PROC ("obst_prepared_info")
   TT (c2o_H, T_ENTER);

   int result = -1;

   if (obst_prepared)
   {  *descr = current_call.minfo->descr;

      if (args != NULL && current_call.argc)
      {  sos_Object *arg_i = (sos_Object*)args;
	 sos_Type   *ptp_i = current_call.minfo->ptypes;

	 ERR_HANDLING_WRAPPER (
      	     for (result = 0;
		  result < current_call.argc;
		  ++result, ++arg_i, ++ptp_i)
	     {  if (*arg_i != NO_OBJECT && !arg_i->is_some (*ptp_i))
		   break;
	     },

	     ;)
	 if (result == current_call.argc)
	    result = -1;
      }
   }
   else
   {  err_raise (err_USE, "no call prepared",
			  "c2obst - obst_call_prepared", FALSE);
      result = -2;
   }
   TT (c2o_H, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------
// IMPLEMENTATION: Type Handling
// --------------------------------------------------------------------

/*
 * Wrapper around TypeTable::get_/set_schemas for external access.
 */
EXPORT void obst_set_schemas (const char* schemas[])
{  T_PROC ("obst_set_schemas")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (TypeTable::set_schemas (schemas);, ;);

   TT (c2o_M, T_LEAVE);
}
EXPORT char** obst_get_schemas ()
{  T_PROC ("obst_get_schemas")
   TT (c2o_M, T_ENTER);

   char** result;
   ERR_HANDLING_WRAPPER (result = TypeTable::get_schemas();, result = NULL;);

   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Yields the meta database object of the named type, or NO_OBJECT if no
 * such type is found.
 */
EXPORT sos_Object obst_lookup_type (const char* tname)
{  T_PROC ("obst_lookup_type")
   TT (c2o_M, T_ENTER);

   TypeInfo   *tiptr;
   sos_Object result;
   ERR_HANDLING_WRAPPER (
      if (tiptr = TypeTable::lookup (tname))
	 result = tiptr->descr;
      else
	 result = NO_OBJECT;,

      result = NO_OBJECT;)

   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Enters the conversion operator of kind 'which', pointed at by 'op_ptr', into
 * the type data of the scalar type 'stp'.
 */
EXPORT void obst_scalar_converter (const char*  stp,
				   obst_cvt_ops which, obst_proc* op_ptr)
{  T_PROC ("obst_scalar_converter")
   TT (c2o_M, T_ENTER);

   register TypeInfo* tiptr;

   ERR_HANDLING_WRAPPER (tiptr = TypeTable::lookup(stp);,
			 tiptr = NULL;)

   if (!tiptr  ||  tiptr->tp_kind != ENUM_TP && tiptr->tp_kind != EXTERN_TP)
   {  smg_String msg = smg_String((char*)stp) + " is not a scalar type";
      err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			  "c2obst - obst_scalar_converter");
   }
   else
      tiptr->cvt_op[which] = op_ptr;

   TT (c2o_M, T_LEAVE);
}

EXPORT sos_Bool obst_method_data (const char* cname,
				  const char* mname, int      argc,
				  obst_mData* md,    sos_Bool do_set)
{  T_PROC ("obst_method_data")
   TT (c2o_M, T_ENTER);

   sos_Bool    result = FALSE;

   TypeInfo*   tiptr;
   MethodInfo* miptr;

   ERR_HANDLING_WRAPPER (
      if (tiptr = TypeTable::lookup (cname))
      {  if (miptr = tiptr->lookup_method (mname, argc))
	 {  if (do_set)
	       miptr->mi.md = *md;
	    else
	       *md = miptr->mi.md;

	    result = TRUE;
	 }
      },
   )

   TT (c2o_M, T_LEAVE);
   return result;
}


// --------------------------------------------------------------------
// IMPLEMENTATION: Scalar Types
// --------------------------------------------------------------------

/*
 * Creates an object representation of the scalar value pointed at by
 * sptr. stp is the name of the scalar type.
 * The real type of *sptr must be int for enumeration types.
 *
 * The result is NO_OBJECT in case of error.
 */
LOCAL sos_Object _obst_scalar2object (const char* stp, void* sptr)
{  T_PROC ("_obst_scalar2object")
   TT (c2o_L, T_ENTER);

   register TypeInfo* tiptr;
   if (tiptr = TypeTable::lookup(stp))
   {  if (tiptr->tp_kind == ENUM_TP || tiptr->tp_kind == EXTERN_TP)
      {  if (tiptr->cvt_op[Ext2Int])
	 {  TT (c2o_L, T_LEAVE);
   	    return _obst_object_from_extern
		      (sptr,
		       (cvt_byterep*)tiptr->cvt_op[Ext2Int],
		       tiptr->descr._self_id());
	 }
	 else if (tiptr->tp_kind == ENUM_TP)
	 {  TT (c2o_L, T_LEAVE);
   	    return _obst_enum_from_sos_Int (*(int*)sptr, tiptr->descr);
	 }
	 else
	 {  smg_String msg = smg_String((char*)stp) +" no conversion operator";
	    err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
				"c2obst - obst_scalar2object");
	 }
      }
      else
      {  smg_String msg = smg_String((char*)stp) + " is no scalar type";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			     "c2obst - obst_scalar2object");
      }
   }
   TT (c2o_L, T_LEAVE);
   return NO_OBJECT;
}
EXPORT sos_Object obst_scalar2object (const char* stp, void* sptr)
{  T_PROC ("obst_scalar2object")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER (result = _obst_scalar2object (stp, sptr);,
			 result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Converts the object representation pointed at by sobj to an external
 * representation which is stored at *sptr.
 * The real type of *sptr must be int for enumeration types.
 *
 * The value of *sptr is undefined if an error occurs.
 */
LOCAL void _obst_object2scalar (const sos_Object* sobj, void* sptr)
{  T_PROC ("_obst_object2scalar")
   TT (c2o_L, T_ENTER);

   if (*(sos_Object*)sobj == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_object2scalar", FALSE);
      TT (c2o_L, T_LEAVE);
      return;
   }
   register TypeInfo* tiptr;
   if (tiptr = TypeTable::lookup(((sos_Object*)sobj)->type()))
   {  if (tiptr->tp_kind == ENUM_TP || tiptr->tp_kind == EXTERN_TP)
      {  if (tiptr->cvt_op[Int2Ext])
	    _obst_extern_from_object
	       (sptr,
		*(sos_Object*)sobj,
		(cvt_byterep*)tiptr->cvt_op[Int2Ext],
	        tiptr->descr._self_id());

	 else if (tiptr->tp_kind == ENUM_TP)
	    *(int*)sptr = _obst_Int_from_enum(*(sos_Object*)sobj,tiptr->descr);
	 else
	 {  smg_String msg = smg_String(tiptr->descr.get_name())
			     + " no conversion operator";
	    err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
				"c2obst - obst_object2scalar");
	 }
      }
      else
      {  smg_String msg = smg_String(tiptr->descr.get_name())
			  + " is no scalar type";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			     "c2obst - obst_object2scalar");
      }
   }
   TT (c2o_L, T_LEAVE);
}
EXPORT void obst_object2scalar (const sos_Object* sobj, void* sptr)
{  T_PROC ("obst_object2scalar")
   TT (c2o_M, T_ENTER);

   ERR_HANDLING_WRAPPER (_obst_object2scalar(sobj, sptr);, ;);

   TT (c2o_M, T_LEAVE);
}

/*
 * Creates the object representation of the enumeration value val of the
 * enumeration type with name etp.
 * The result is NO_OBJECT in case of error.
 */
EXPORT sos_Object obst_enum2object (const char* etp, int val)
{  T_PROC ("obst_enum2object")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   TypeInfo*  tiptr;
   ERR_HANDLING_WRAPPER (
      if (tiptr = TypeTable::lookup(etp))
	 result = _obst_enum_from_sos_Int (val, tiptr->descr);
      else
	 result = NO_OBJECT;,

      result = NO_OBJECT;)

   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Yields the enumeration value from its object representation eobj.
 * The result is -1 in case of error.
 */
EXPORT int obst_object2enum (const sos_Object* eobj)
{  T_PROC ("obst_object2enum")
   TT (c2o_M, T_ENTER);

   int result;
   ERR_HANDLING_WRAPPER (
      result = _obst_Int_from_enum (*(sos_Object*)eobj,
				    ((sos_Object*)eobj)->type());,
      result = -1;)

   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Yields a scalar in its object representation from the string 'str'. The
 * name of the scalar type is given by 'stp'.
 * The result is NO_OBJECT in case of error.
 */
LOCAL sos_Object _obst_string2object(const char* stp, const char* str)
{  T_PROC ("_obst_string2object")
   TT (c2o_L, T_ENTER);

   register TypeInfo* tiptr;
   if (tiptr = TypeTable::lookup(stp))
   {  if (tiptr->tp_kind == ENUM_TP || tiptr->tp_kind == EXTERN_TP)
      {  if (tiptr->cvt_op[Str2Obj])
	 {  TT (c2o_L, T_LEAVE);
	    return ( *(cci_str2obj_fun)tiptr->cvt_op[Str2Obj] )((char*)str);
	 }
	 else if (tiptr->tp_kind == ENUM_TP)
	 {  register int enumval;
	    if ((enumval = tiptr->lookup_literal(str)) >= 0)
	    {  TT (c2o_L, T_LEAVE);
	       return _obst_enum_from_sos_Int (enumval, tiptr->descr);
	    }
	 }
	 else
	 {  smg_String msg = smg_String((char*)stp) +" no conversion operator";
	    err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
		       "c2obst - obst_string2object");
	 }
      }
      else
      {  smg_String msg = smg_String((char*)stp) + " is no scalar type";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
		    "c2obst - obst_string2object");
      }
   }
   TT (c2o_L, T_LEAVE);
   return NO_OBJECT;
}
EXPORT sos_Object obst_string2object (const char* stp, const char* str)
{  T_PROC ("obst_string2object")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER (result = _obst_string2object (stp, str);,
			 result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Converts the given scalar object in its string representation.
 * The resulting string must be destroyed by the caller. It is NULL in case
 * of error.
 */
LOCAL char* _obst_object2string (const sos_Object* sobj)
{  T_PROC ("_obst_object2string")
   TT (c2o_L, T_ENTER);

   if (*(sos_Object*)sobj == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_object2string", FALSE);
      TT (c2o_L, T_LEAVE);
      return "";
   }
   register TypeInfo* tiptr;
   if (tiptr = TypeTable::lookup(((sos_Object*)sobj)->type()))
   {  if (tiptr->tp_kind == ENUM_TP || tiptr->tp_kind == EXTERN_TP)
      {  char* result;
	 if (tiptr->cvt_op[Obj2Str])
	    result = ( *(cci_obj2str_fun)tiptr->cvt_op[Obj2Str] )
		          ( *(sos_Object*)sobj );

	 else if (tiptr->tp_kind == ENUM_TP)
	    result
	       = obst_strdup (
		   tiptr->literal
		     [_obst_Int_from_enum (*(sos_Object*)sobj, tiptr->descr)]);
	 else
	 {  smg_String msg = smg_String(tiptr->descr.get_name())
			     + " no conversion operator";
	    err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
				"c2obst - obst_object2string");
	    result = NULL;
	 }
	 TT (c2o_L, T_LEAVE);
	 return result;
      }
      else
      {  smg_String msg = smg_String(tiptr->descr.get_name())
			  + " is no scalar type";
	 err_raise (err_USE, msg.make_Cstring (SMG_BORROW),
			     "c2obst - obst_object2string");
      }
   }
   TT (c2o_L, T_LEAVE);
   return NULL;
}
EXPORT char* obst_object2string (const sos_Object* sobj)
{  T_PROC ("obst_object2string")
   TT (c2o_M, T_ENTER);

   char* result;
   ERR_HANDLING_WRAPPER (result = _obst_object2string(sobj);,
			 result = NULL;)
   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Wrapper around _obst_scalar2object and _obst_object2string which converts
 * a scalar value from its external to its string representation.
 * The result string must be destroyed by the caller. It is NULL in case of
 * failure.
 */
EXPORT char* obst_scalar2string (const char* stp, void* sptr)
{  T_PROC ("obst_scalar2string")
   TT (c2o_M, T_ENTER);

   sos_Object scalarobj;
   char*      result = NULL;
   ERR_HANDLING_WRAPPER (
      if ((scalarobj = _obst_scalar2object(stp, sptr)) != NO_OBJECT)
	 result = _obst_object2string (&scalarobj);,
      ;)
   TT (c2o_M, T_LEAVE);
   return result;
}

/*
 * Wrapper around _obst_object2scalar and _obst_string2object which converts
 * a scalar value from its string to its external representation.
 * The result is undefined in case of failure.
 */
EXPORT void obst_string2scalar (const char* stp, const char* str, void* sptr)
{  T_PROC ("obst_string2scalar")
   TT (c2o_M, T_ENTER);

   sos_Object scalarobj;
   ERR_HANDLING_WRAPPER (
      if ((scalarobj = _obst_string2object(stp, str)) != NO_OBJECT)
	 _obst_object2scalar (&scalarobj, sptr);
      else
	 err_raise (err_USE, str, "c2obst - obst_string2scalar");
      ,;)
   TT (c2o_M, T_LEAVE);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: method call shortcuts
// --------------------------------------------------------------------

EXPORT sos_Bool obst_identical (const sos_Object* obj1, const sos_Object* obj2)
{  T_PROC ("obst_identical")
   TT (c2o_M, T_ENTER);

   sos_Bool result;
   ERR_HANDLING_WRAPPER (
      result = (sos_Bool)( *(sos_Object*)obj1 == *(sos_Object*)obj2 );,
      result = FALSE;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Object obst_type (const sos_Object* obj)
{  T_PROC ("obst_type")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   if (*(sos_Object*)obj == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_type", FALSE);
      result = NO_OBJECT;
   }
   else
      ERR_HANDLING_WRAPPER (result = ((sos_Object*)obj)->type();,
			    result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_is_scalar (const sos_Object* obj)
{  T_PROC ("obst_is_scalar")
   TT (c2o_M, T_ENTER);

   sos_Bool result;
   if (*(sos_Object*)obj == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_is_scalar", FALSE);
      result = FALSE;
   }
   else
      ERR_HANDLING_WRAPPER (
	 result = sos_Type::make(*(sos_Object*)obj).is_scalar();,
	 result = FALSE;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_has_type (const sos_Object* obj, const char* tname)
{  T_PROC ("obst_has_type")
   TT (c2o_M, T_ENTER);

   sos_Bool result = FALSE;

   if (*(sos_Object*)obj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_has_type", FALSE);
   else
   {  TypeInfo *tiptr;
      ERR_HANDLING_WRAPPER (
	 if (tiptr = TypeTable::lookup (tname))
	    result = ((sos_Object*)obj)->has_type (tiptr->descr);, ;)
   }
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_isa (const sos_Object* obj, const char* tname)
{  T_PROC ("obst_isa")
   TT (c2o_M, T_ENTER);

   sos_Bool result = FALSE;

   if (*(sos_Object*)obj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_isa", FALSE);
   else
   {  TypeInfo *tiptr;
      ERR_HANDLING_WRAPPER (
	 if (tiptr = TypeTable::lookup (tname))
	    result = ((sos_Object*)obj)->isa (tiptr->descr);, ;)
   }
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Bool obst_is_some (const sos_Object* obj, const char* tname)
{  T_PROC ("obst_is_some")
   TT (c2o_M, T_ENTER);

   sos_Bool result = FALSE;

   if (*(sos_Object*)obj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_is_some", FALSE);
   else
   {  TypeInfo *tiptr;
      ERR_HANDLING_WRAPPER (
	 if (tiptr = TypeTable::lookup (tname))
	    result = ((sos_Object*)obj)->is_some (tiptr->descr);, ;)
   }
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT int obst_card (const sos_Object* obj)
{  T_PROC ("obst_card")
   TT (c2o_M, T_ENTER);

   int result = 0;

   if (*(sos_Object*)obj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_card", FALSE);
   else
      ERR_HANDLING_WRAPPER (
	 result = sos_Aggregate::make(*(sos_Object*)obj).card();, ;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT int obst_objsize (const sos_Object* obj)
{  T_PROC ("obst_objsize")
   TT (c2o_M, T_ENTER);

   int result = 0;

   if (*(sos_Object*)obj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_objsize", FALSE);
   else
      ERR_HANDLING_WRAPPER (
	 result = ROOT_CONTAINER.realsize( (*(sos_Object*)obj)
					      .type().get_object_size() );, ;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT char* obst_obj2str (const sos_Object* strobj)
{  T_PROC ("obst_obj2str")
   TT (c2o_M, T_ENTER);

   char* result = NULL;

   if (*(sos_Object*)strobj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_obj2str", FALSE);
   else
      ERR_HANDLING_WRAPPER (result = sos_String::make(*strobj).make_Cstring();,
			    ;)

   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Object obst_str2obj (const char* str)
{  T_PROC ("obst_str2obj")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER(result = sos_String::create(TEMP_CONTAINER,(char*)str);,
		        result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Container obst_obj2cnt (const sos_Object* ctobj)
{  T_PROC ("obst_obj2cnt")
   TT (c2o_M, T_ENTER);

   sos_Container result;

   if (*(sos_Object*)ctobj == NO_OBJECT)
   {  err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_obj2cnt", FALSE);
      result = UNUSED_CONTAINER;
   }
   else
      ERR_HANDLING_WRAPPER (result = make_sos_Container(*ctobj);,
			    result = UNUSED_CONTAINER;)
   TT (c2o_M, T_LEAVE);
   return result;
}
EXPORT sos_Object obst_cnt2obj (sos_Container ct)
{  T_PROC ("obst_cnt2obj")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER (result = make_sos_Container_object (ct);,
			 result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}

EXPORT sos_Int obst_obj2int (const sos_Object* intobj)
{  T_PROC ("obst_obj2int")
   TT (c2o_M, T_ENTER);

   sos_Int result = 0;

   if (*(sos_Object*)intobj == NO_OBJECT)
      err_raise (err_USE, err_NO_OBJECT, "c2obst - obst_obj2int", FALSE);
   else
      ERR_HANDLING_WRAPPER (result = make_sos_Int(*intobj);, ;)

   TT (c2o_M, T_LEAVE);
   return result;
}
EXPORT sos_Object obst_int2obj (sos_Int i)
{  T_PROC ("obst_int2obj")
   TT (c2o_M, T_ENTER);

   sos_Object result;
   ERR_HANDLING_WRAPPER (result = make_sos_Int_object (i);,
			 result = NO_OBJECT;)
   TT (c2o_M, T_LEAVE);
   return result;
}


// --------------------------------------------------------------------
// IMPLEMENTATION: statistics
// --------------------------------------------------------------------

EXPORT char* obst_stats ()
{  T_PROC ("obst_stats")
   TT (c2o_M, T_ENTER);

   obst_htabstat  statbuf;

   obst_collect_stats (&statbuf);
   TypeTable::collect_stats (&statbuf);

   smg_String descr;

#if OBSTVERSIONNO >= 336
   descr = smg_String (obst_print_htabstats (&statbuf, TRUE), SMG_TRANSFER);

#else
   smg_String sep = "--------------------------------------------------------------------\n";

   for (int i = 0;  i < statbuf.size;  ++ i)
   {  descr += sep
	    +  "TABLE: " + statbuf.htabs[i].descr
	    +  "(" + statbuf.htabs[i].buckets + " buckets)\n";

      for (int j = 0;  j < statbuf.htabs[i].histsize;  ++ j)
      {  obst_histentry* histo = &statbuf.htabs[i].histogram[j];
         descr += smg_String("\tbucket size: ") + histo->data
               +  "\toccurences: " + histo->occurences + "\n";
      }
   }
   descr += sep;

   obst_free_htabstat (&statbuf);

#endif

#if c2o_ADD_DEBUG_STUFF
   descr += smg_String("\nstr  --> Cstr: [# ") + c2o_localstats.str2Cstr_no
         +  " - " + c2o_localstats.str2Cstr_sz + " bytes]"
	 +  "\nCstr --> str : [# " + c2o_localstats.Cstr2str_no
	 +  " - " + c2o_localstats.Cstr2str_sz + " bytes]\n";
#endif

   TT (c2o_M, T_LEAVE);
   return descr.make_Cstring (SMG_TRANSFER);
}


// --------------------------------------------------------------------
// IMPLEMENTATION: Initialization
// --------------------------------------------------------------------

LOCAL sos_Cstring c2o_string_from_Cstring (const sos_Object& s)
{  T_PROC ("c2o_string_from_sos_Cstring")
   TT (c2o_VL, T_ENTER);

   register
   sos_Cstring result = make_sos_Cstring (s);
   if (c2obst_ctrls.copy_Cstrings)
      result = obst_strdup (result);

#if c2o_ADD_DEBUG_STUFF
   ++ c2o_localstats.Cstr2str_no;
   c2o_localstats.Cstr2str_sz += strlen(result);
#endif

   TT (c2o_VL, T_LEAVE);
   return result;
}

LOCAL sos_Object c2o_Cstring_from_string (const sos_Cstring _s)
{  T_PROC ("c2o_sos_Cstring_from_string")
   TT (c2o_VL, T_ENTER);

   register
   sos_Cstring s = (c2obst_ctrls.copy_Cstrings) ? obst_strdup (_s)
						: _s;
   sos_Object  r = make_sos_Cstring_object (s);

#if c2o_ADD_DEBUG_STUFF
   ++ c2o_localstats.str2Cstr_no;
   c2o_localstats.str2Cstr_sz += strlen(s);
#endif

   TT (c2o_VL, T_LEAVE);
   return r;
}


LOCAL void c2o_set_schemas_trigger()
{  T_PROC ("c2o_set_schemas_trigger")
   TT (c2o_L, T_ENTER);

   obst_scalar_converter ("sos_Cstring", cvt_String2Object,
			  (obst_proc*)c2o_Cstring_from_string);
   obst_scalar_converter ("sos_Cstring", cvt_Object2String,
			  (obst_proc*)c2o_string_from_Cstring);
   TT (c2o_L, T_LEAVE);
}

// --------------------------------------------------------------------

#define check_enum(etp, ename) \
   if (sizeof(etp) != sizeof(int)) \
      err_raise (err_SYS, "storage layout of enum type incompatible with C", \
			  ename, FALSE)

EXPORT void _c2obst_init()
{  T_PROC ("_c2obst_init")
   TT (c2o_L, T_ENTER);

   static int initialized = 0;
   if (initialized) { TT (c2o_L, T_LEAVE); return; }
   initialized = 1;

   _obst_initcore();

   if (sizeof (sos_Object) != OBST_OBJ_SIZE)
      err_raise (err_SYS, "adjust OBST_OBJ_SIZE", "c2obst_init");

		// The number of literals of other enumeration types lies
		// in between the numbers of these two types.
   check_enum (sos_Bool, 	     "sos_Bool");
   check_enum (sos_Container_status, "sos_Container_status");

   err_assert (   (int)Str2Obj == (int)cvt_String2Object
	       && (int)Obj2Str == (int)cvt_Object2String,
	       "c2obst_init - type obst_cvt_ops not reflected internally");

   // explicit initialization for external globals

   err_NO_OBJECT		    = "applied to NO_OBJECT";
   c2obst_version		    = C2OBST_VERSION;
   
   c2obst_ctrls.allow_defaults	    = c2o_ALLOW_DEFAULTS;
   c2obst_ctrls.check_args	    = c2o_CHECK_ARGS;
   c2obst_ctrls.catch_syserrs	    = c2o_CATCH_SYSERRS;
   c2obst_ctrls.syserr		    = NULL;
   c2obst_ctrls.set_schemas_trigger = NULL;
   c2obst_ctrls.copy_Cstrings	    = c2o_COPY_CSTRINGS;

   TypeTable::init();
   _c2obst_Cinit();

   TT (c2o_L, T_LEAVE);
}
