/* --------------------------------------------------------------------------
 * Copyright 1993-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* --------------------------------------------------------------------------
 * Arrows support routines
 *
 * ORIGINAL: Ch. Popp					Date: 2/6/93
 * --------------------------------------------------------------------------
 */
#include <stdio.h>
#include <math.h>

#include "tcl.h"
#include "arrows_tcl.h"


static double intersect (edgeX,edgeY,vectorX,vectorY,centerX,centerY)

   double edgeX,edgeY,vectorX,vectorY,centerX,centerY;

{
 double a1 = (edgeX - centerX) / vectorX,
        a2 = (edgeY - centerY) / vectorY;

 if (a1 < 0) a1 = -a1;
 if (a2 < 0) a2 = -a2;

 return (a1 > a2) ? a2 : a1;
}


/* -------
 * Compute coordinates for 'continuous' movement mode and 'rect' node shape.
 */
static edge_coords_continuous_rect (cd, interp, argc, argv)

   ClientData	cd;
   Tcl_Interp	*interp;
   int		argc;
   char		*argv[];
 
{ 
 double startX  = atoi(argv[1]),
 	startY  = atoi(argv[2]),
 	widthS  = atoi(argv[3]),
 	heightS = atoi(argv[4]),
	destX   = atoi(argv[5]),
 	destY   = atoi(argv[6]);

 int    edgestartX, edgestartY;

 double deltaY  = startY - destY,
 	deltaX  = startX - destX,
 	a, edgeX, edgeY;
 char*  position;

 if (deltaX == 0) {
    edgestartX = startX;

    if (deltaY >= 0) {
       edgestartY = startY - (heightS/2);
       position   = "NE";
    } else {
       edgestartY = startY + (heightS/2);
       position   = "SE";
    }

 } else if (deltaY == 0) {
    edgestartY = startY;

    if (deltaX >= 0) {
       edgestartX = startX - (widthS/2);
       position   = "SW";
     } else {
       edgestartX = startX + (widthS/2);
       position   = "NW";
     }

 } else if (deltaX < 0) {
    if (deltaY >= 0) {
       /* 1. quadrant */

       edgeX      = startX + (widthS/2);
       edgeY      = startY - (heightS/2);
       a          = intersect(edgeX,edgeY,deltaX,deltaY,startX,startY);
       edgestartX = startX - (a * deltaX);
       edgestartY = startY - (a * deltaY);

       position  = "NE";

    } else {
       /* 4. quadrant */

       edgeX      = startX + (widthS/2);
       edgeY      = startY + (heightS/2);
       a          = intersect(edgeX,edgeY,deltaX,deltaY,startX,startY);
       edgestartX = startX - (a * deltaX);
       edgestartY = startY - (a * deltaY);

       position = "SE";
    }
 } else if (deltaY < 0) {
    /* 3. quadrant */

    edgeX      = startX - (widthS/2);
    edgeY      = startY + (heightS/2);
    a          = intersect(edgeX,edgeY,deltaX,deltaY,startX,startY);
    edgestartX = startX - (a * deltaX);
    edgestartY = startY - (a * deltaY);

    position = "SW";

 } else {
    /* 2. quadrant */

    edgeX      = startX - (widthS/2);
    edgeY      = startY - (heightS/2);
    a          = intersect(edgeX,edgeY,deltaX,deltaY,startX,startY);
    edgestartX = startX - (a * deltaX);
    edgestartY = startY - (a * deltaY);

    position = "NW";
 }
	/* we have at least 200 bytes - and that's enough */
 sprintf (interp->result, "%s %d %d", position, edgestartX, edgestartY);

 return TCL_OK;
}


/* ----------------------------------------------------------------------------
 * Compute coordinates for 'gridded' movement mode and 'rect' node shape.
 */
static edge_coords_gridded_rect (cd, interp, argc, argv)

   ClientData	cd;
   Tcl_Interp	*interp;
   int		argc;
   char		*argv[];
 
{ 
 int    startX  = atoi(argv[1]),
 	startY  = atoi(argv[2]),
 	widthS  = atoi(argv[3]),
 	heightS = atoi(argv[4]),
 	destX   = atoi(argv[5]),
 	destY   = atoi(argv[6]),
 	widthD  = atoi(argv[7]),
 	heightD = atoi(argv[8]);

 int    edgestartX, edgestartY;
 char*  position;

 int    start_HighY, start_LowY, start_RightX;

 if ((start_LowY = startY + (heightS/2))
     		<= destY  - (heightD/2)) {		/* dest_HighY */
    edgestartY = start_LowY;
    edgestartX = startX;
    position   = "S";

 } else if ((start_HighY = startY - (heightS/2))
	    		>= destY  + (heightD/2)) {	/* dest_LowY */
    edgestartY = start_HighY;
    edgestartX = startX;
    position   = "N";

 } else if ((start_RightX = startX + (widthS/2))
	    		  < destX  - (widthD/2)) {	/* dest_LeftX */
    edgestartY = startY;
    edgestartX = start_RightX;
    position   = "E";

 } else {
    edgestartY = startY;
    edgestartX = startX - (widthS/2);	/* start_LeftX */
    position   = "W";
 }
	/* we have at least 200 bytes - and that's enough */
 sprintf (interp->result, "%s %d %d", position, edgestartX, edgestartY);

 return TCL_OK;
}

/* ----------------------------------------------------------------------------
 * Compute coordinates for 'continuous' movement mode and 'oval' node shape.
 */
static edge_coords_continuous_oval (cd, interp, argc, argv)

   ClientData	cd;
   Tcl_Interp	*interp;
   int		argc;
   char		*argv[];
 
{ 
 double startX  = atoi(argv[1]),
 	startY  = atoi(argv[2]),
 	widthS  = atoi(argv[3]),
 	heightS = atoi(argv[4]),
 	destX   = atoi(argv[5]),
 	destY   = atoi(argv[6]);

 double deltaY  = destY - startY,
 	deltaX  = destX - startX,
 	f       = 2 * sqrt(deltaX*deltaX + deltaY*deltaY);

 int    edgeX   = startX + (widthS  * deltaX / f),
	edgeY   = startY + (heightS * deltaY / f); 

 char*  position;

 if (deltaX >= 0) {
    position = (deltaY <= 0) ? "NE" : "SE";
 } else {
    position = (deltaY <= 0) ? "NW" : "SW";
 }
	/* we have at least 200 bytes - and that's enough */
 sprintf (interp->result, "%s %d %d", position, edgeX, edgeY);

 return TCL_OK;
}


/* -------
 * module initialization
 */
void arrow_init (interp)

   Tcl_Interp* interp;

{
  Tcl_CreateCommand(interp,
		    "arrow_continuous_rect_C",
		    (Tcl_CmdProc*)edge_coords_continuous_rect,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,
		    "arrow_continuous_oval_C",
		    (Tcl_CmdProc*)edge_coords_continuous_oval,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,
		    "arrow_gridded_rect_C",
		    (Tcl_CmdProc*)edge_coords_gridded_rect,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
  Tcl_CreateCommand(interp,
		    "arrow_gridded_oval_C",
		    (Tcl_CmdProc*)edge_coords_gridded_rect,
		    (ClientData)NULL,(Tcl_CmdDeleteProc*)NULL);
}
