# --------------------------------------------------------------------------
# Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
#
# You can use and distribute this software under the terms of the license
# version 1 you should have received along with this software.
# If not or if you want additional information, write to
# Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
# D-76131 Karlsruhe, Germany.
# --------------------------------------------------------------------------
# Module: arrows.tcl
# Tcl version: 7.0 (Tcl/Tk/XF)
# Tk version: 3.3
# XF version: 2.2
#

# module contents
global moduleList
global autoLoadList
set moduleList(arrows.tcl) { arrow_doc arrow_init arrow_coordfunc arrow_set_coordfuncs arrow_option Node_add Node_move Node_drag Node_remove Node_list Node_neighbours Edge_create Edge_remove Edge_ID Edge_type Edge_coords arrow_set_nodeattrs arrow_gridded_rect_Tcl arrow_continuous_rect_Tcl arrow_continuous_oval_Tcl arrow_isct edge_create_bidir edge_move_bidir edge_remove_bidir edge_create_unidir edge_move_unidir edge_remove_unidir edge_create_nodir edge_move_nodir edge_remove_nodir}
set autoLoadList(arrows.tcl) {0}

# procedures to show toplevel windows


# User defined procedures


# ---------
 proc arrow_doc {} {
#
# arrow_:
#  (Internal) array holding the state of the arrows module.
#
#  The following fields hold display parameters:
#    AttachMode
#	Builtin modes are either "continuous" (the default) or "gridded"
#	denoting edges which smoothly circulate the bounding box of a node when
#	moved, or edges which are connected only to a few points along the box,
#	respectively.
#    NodeShape
#	Default node shape. Builtin shapes are "rect" and "oval".
#	It defaults to "rect".
#    EdgeType
#	Default edge type. It defaults to "unidir".
#
#  The following fields record coordinate computation procedures:
#    C,<mode>,<shape>
#	Procedure for the given movement mode and node shape. There will always
#	be entries with indices 'C,<mode>,rect', and 'C,gridded,<shape>',
#	respectively.
#    Modes
#	Set (tcl list) with edge attachment modes seen so far.
#    Shapes
#	Set (tcl list) with node shapes seen so far.
#
#  For each given $nodeID, there are the following fields <f> stored at the
#  indices arrow_($nodeID,<f>):
#    Nodes:
#	List of IDs of adjacent nodes which are connected via an edge to this
#	node. 
#    X, Y, W, H:
#	Boundary box of the node in the canvas coordinate system. (X,Y) is the
#	center of the boundary box.
#    S: Node shape.
#    M: Edge attachment mode for this node, "" if global mode is to be used.
#    Mfield:
#	Field of arrow_ which holds the edge attachment mode for this node:
#	either '$nodeID,M', or 'AttachMode'.
#
#  For each edge with ID $edgeID, there	are the following fields stored at
#  the indices arrow_($edgeID,<f>):
#    S: edge style as suitable for invoking a drawing procedure.
#
#  The existence of a node/edge may be tested by testing if one of the
#  corresponding array entries exists.
# --------------------------------------------------------------------------
# proc arrow_init {}
#
# in/out: void
#
# Initializes the arrow module by setting storing the default values for
# all display options into the arrow_ array and by computing coordinate
# functions for all builtin movement modes and builtin node shapes.
# --------------------------------------------------------------------------
# proc arrow_coordfunc { mode shape}
#
# in : edge attachment mode and node shape
# out: coordinate function for the given parameters
#
# This function does hide how replacements for missing coordinate functions
# are made.
# --------------------------------------------------------------------------
# proc arrow_set_coordfuncs { mode shape}
#
# in : edge attachment mode and node shape, exactly one must be undefined ("")
# out: void
#
# Computes the coordinate functions for the defined argument and all currently
# known values of the other argument (i.e. arrow_(Modes), or arrow_(Shapes)).
# The defined argument is finally stored in arrow_(Modes), or arrow_(Shapes)
# if it is not yet contained there.
# --------------------------------------------------------------------------
# proc arrow_option { option value}
#
# out: option value
#
# Procedure to set a display parameter to the given value: parameters are the
# edge attachment mode ("attach", see below), the default edge type ("type"),
# edge width ("width"), edge color ("color"), and stipple pattern ("stipple").
#
# If no value is given (the default), the current parameter setting is just
# returned. The value "" (re)sets the named display parameter to its default
# value (which may be "", too).
# The function yields the current/new parameter setting. This will be "" if
# the display parameter is so far undefined and no new value is given.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc Node_add { window nodeID mode shape mode x y width height}
#
# out: void
#
# Initializes a Tk item as a node. The item is identified by nodeID in the Tk
# window with pathname window. If no such item exists, nothing will happen.
# Mode must be either "new" or "refresh":
#  - "new" creates a new node for an item if nodeID is so far unknown.
#  - "refresh" can be used to re-initialize the parameters of an existing
#    node, if the coordinates, height or width of the item refering to this
#    node have changed.
# Shape defines the internal node shape, i.e. where edges start/end. It
# defaults to the current default node shape.
# Mode defines the edge attachment mode for this node. It default to the
# current global edge attachment mode.
# X, y set the center of the rectangle region, surrounding the item, where the 
# edges are to start and end. Height and width set the size of this region.
# If all these coordinates are undefined, a default bounding box is computed.
# --------------------------------------------------------------------------
# proc Node_move { window nodeID goX goY}
#
# out: void
#
# Moves the edges of the Tk item with tag nodeID in the Tk window with
# pathname window relative to the former position by (goX,goY).
# An item which is moved by "$window move $nodeID $goX $goY" must afterwards
# by moved by Node_add with the same parameters in order to move the edges of
# the item correctly.
# If the node is not moved (i.e. goX == 0, goY == 0), the edges will just be
# refreshed. This may be useful e.g. after changing the node size.
# --------------------------------------------------------------------------
# proc Node_drag { mode window nodeID x y}
#
# out: void
#
# Standard drag handler which does move the node itself as well as any edges
# connected to that node. Mode indicates if dragging is to be initialized
# ("init") or if the node is in motion ("move").
# If mouse button 1 is to be used for dragging, event bindings should be set
# up as follows:
#   $window bind $nodeID <1>	     "Node_drag init %W $nodeID %x %y"
#   $window bind $nodeID <B1-Motion> "Node_drag move %W $nodeID %x %y"
# --------------------------------------------------------------------------
# proc Node_remove { window nodeID}
#
# out: void
#
# Removes the node corresponding to the Tk item with tag nodeID in the Tk
# window with pathname window.
# All edges starting or ending at this item are deleted.
# --------------------------------------------------------------------------
# Node_list { window}
#
# out: List of IDs of all nodes in the given Tk window. The resulting list
#      might be empty.
# --------------------------------------------------------------------------
# Node_neighbours { window nodeID}
#
# out: List of IDs of all nodes directly connected to $nodeID,
#      or an empty list if either $nodeID is not initialized as a node or if
#      there are no such edges.
# --------------------------------------------------------------------------
# Edge_create { window startID destID type}
#
# out: ID of newly created edge, or "" if no edge was created
#
# Creates an edge between two Tk items in the Tk window with pathname window
# and the tags startID and destID. Both items must have been initialized as
# nodes with the command Node_add.
# Nothing will happen if there exists already an edge between the two items or
# if startID == destID, or if any of the nodes is no proper node.
# Type sets the type of the edge. If omitted, the default edge type is used.
# The three standard types are: lines with no direction (mode nodir),
# unidirectional lines (mode unidir) and bidirectional lines (mode bidir).
# Further edge types may be used provided their implementation is present.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc Edge_remove { window ID_1 ID_2}
#
# out: void
#
# Removes an existing edge between two Tk items in the Tk window with 
# pathname window and the tags ID_1 and ID_2.
# Nothing will happen, if there exists no edge between the two items or if
# at least one of the items is not initialized as a node.
# --------------------------------------------------------------------------
# proc Edge_ID { window ID_1 ID_2}
#
# out: ID of the edge between the two given Tk items with tags ID_1 and ID_2,
#      or "" if there is no such edge.
# --------------------------------------------------------------------------
# proc Edge_type { window ID_1 ID_2 type}
#
# out: "" if type is undefined or "" if there is no such edge and the current
#      edge type otherwise.
#
# Queries or modifies the type of the edge connecting the two given nodes.
# --------------------------------------------------------------------------
# proc Edge_coords { startID destID}
#
# Returns a list of five elements, or an empty list if at least one of the
# given items is not initialized as a node:
#
# [0] = The positon of destID relatively to the position of startID.
#	This position can be N,S,E,W in move mode "gridded" or NE,NW,SW,SE in
#	mode "continuous".
#	It can be used e.g. to make the look of an edge depend on the relative
#	position of two items.
# [1] = The x-coordinate of the begin of an edge starting at the bounding box
#	of startID.
# [2] = The y-coordinate of the begin of an edge starting at the bounding box
#	of startID.
# [3] = The x-coordinate of the end of an edge ending at the bounding box of
#	destID.
# [4] = The y-coordinate of the end of an edge ending at the bounding box of
#	destID.
# --------------------------------------------------------------------------
# proc arrow_set_nodeattrs { window nodeID shape mode x y width height}
#
# out: void
#
# Internal procedure which registers the coordinates for the given node.
# If all coordinates are undefined (i.e. empty strings), a default boundary
# box is computed.
# It does also set the node shape and edge attachment mode for this node.
# Defaults are the current default node shape, and the current (at the time of
# usage) default attachment mode.
# --------------------------------------------------------------------------
# proc arrow_gridded_rect_Tcl { sx sy widthS heightS dx dy widthD heightD}
#
# For each item, there are four positions, where an edge can start from:
# N, S, E, and W.
# arrow_gridded_rect_Tcl selects the position such, that the edge
# does not enter the boundary box of one of the connected items.
# --------------------------------------------------------------------------
# arrow_continuous_rect_Tcl {sx sy widthS heightS dx dy widthD heightD}
#
# Tcl version of the C function arrow_continuous_rect_C.
# --------------------------------------------------------------------------
# arrow_continuous_oval_Tcl {sx sy widthS heightS dx dy widthD heightD}
#
# Tcl version of the C function arrow_continuous_oval_C.
# --------------------------------------------------------------------------
# proc arrow_isct { edgeX edgeY vectorX vectorY centerX centerY}
#
# Auxiliary procedure of arrow_edge_coords_Tcl that computes the intersection
# of two vectors (or lines).
# Result is the factor that is used to get to the point of the intersection by
# multiplying this factor and one of the vectors.
# --------------------------------------------------------------------------
# --------------------------------------------------------------------------
# proc edge_create_bidir { window startID destID edgeID}
#
# out: void
#
# Creates a bidirectional line which is tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_move_bidir { window startID destID edgeID}
#
# out: void
#
# Moves bidirectional lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_remove_bidir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_create_unidir { window startID destID edgeID}
#
# out: void
#
# Creates an unidirectional line which is tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_move_unidir { window startID destID edgeID}
#
# out: void
#
# Moves unidirectional lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_remove_unidir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_create_nodir { window startID destID edgeID}
#
# out: void
#
# Creates an undirected line which is tagged with $edgeID of the current
# edge color and edge width. If any of the latter display parameters is
# undefined, Tk's defaults are used instead.
# --------------------------------------------------------------------------
# proc edge_move_nodir { window startID destID edgeID}
#
# out: void
#
# Moves lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
# proc edge_remove_nodir { window edgeID}
#
# out: void
#
# Removes lines which are tagged with $edgeID.
# --------------------------------------------------------------------------
}


# ---------
proc arrow_init { } {
 global arrow_

 set arrow_(Modes)  {}
 set arrow_(Shapes) {}

 arrow_set_coordfuncs gridded    ""
 arrow_set_coordfuncs continuous ""
 arrow_set_coordfuncs ""	 rect

 arrow_option color   ""
 arrow_option attach  ""
 arrow_option shape   ""
 arrow_option stipple ""
 arrow_option type    ""
 arrow_option width   ""
}

# ---------
 proc arrow_coordfunc { mode shape} {

 foreach func [list arrow_${mode}_${shape}_C arrow_${mode}_${shape}_Tcl \
		    arrow_${mode}_rect_C     arrow_${mode}_rect_Tcl \
		    arrow_continous_rect_C] {
    if {[info commands $func] != ""} {
       return $func
    }
 }
 return arrow_continous_rect_Tcl
}

# ---------
 proc arrow_set_coordfuncs { mode shape} {
 global arrow_

 if {$mode != ""} {
    foreach shape $arrow_(Shapes) {
       set arrow_(C,$mode,$shape) [arrow_coordfunc $mode $shape]
    }
    if {[lsearch -exact $arrow_(Modes) $mode] == -1} {
       lappend arrow_(Modes) $mode
    }
 } else {
    foreach mode $arrow_(Modes) {
       set arrow_(C,$mode,$shape) [arrow_coordfunc $mode $shape]
    }
    if {[lsearch -exact $arrow_(Shapes) $shape] == -1} {
       lappend arrow_(Shapes) $shape
    }
 }
}

# ---------
proc arrow_option { option {value _%_}} {
 global arrow_

 case $option in {
    color   { set field EdgeColor   
    	      set dflt  ""
    	    }
    attach  { set field AttachMode
    	      set dflt  continuous
    	    }
    shape   { set field NodeShape
    	      set dflt  rect
    	    }
    stipple { set field EdgeStipple 
    	      set dflt  ""
    	    }
    type    { set field EdgeType    
    	      set dflt  unidir
    	    }
    width   { set field EdgeWidth   
    	      set dflt  ""
    	    }
    default { error "arrow_option: unknown display parameter '$option'" }
 }
 case $value in {
    "_%_" {
       if {[info exists arrow_($field)]} {
	  return $arrow_($field)
       } else {
	  return ""
       }
    }
    "" {
       set value $dflt
    }
 }
 if {$option == "attach"} {
    if {![info exists arrow_(C,$value,rect)]} {
       arrow_set_coordfuncs $value ""
    }
 }
 return [set arrow_($field) $value]
}


# ---------
proc Node_add { window nodeID {mode new} {shape ""} {attach ""}
					 {x ""} {y ""} {width ""} {height ""}} {
 global arrow_
 
 case $mode in {
    new     { if {![info exists arrow_($nodeID,X)]} {
		 set arrow_($nodeID,Nodes) {}
       		 arrow_set_nodeattrs $window $nodeID $shape $attach $x $y $width $height
	    }}
    refresh { if {[info exists arrow_($nodeID,X)]} {
		arrow_set_nodeattrs $window $nodeID $shape $attach $x $y $width $height
	    }}
    default { error "Node_add: unkown mode '$mode'" }
 }
}


# ---------
proc Node_move { window nodeID {goX 0} {goY 0}} {
 global arrow_
        
 if {[info exists arrow_($nodeID,X)]} then {

    incr arrow_($nodeID,X) $goX
    incr arrow_($nodeID,Y) $goY

    foreach otherID $arrow_($nodeID,Nodes) {
       set edge1 "${otherID}_$nodeID"
       set edge2 "${nodeID}_$otherID"

       if {[info exists arrow_($edge1,S)]} {
	  "edge_move_$arrow_($edge1,S)" $window $otherID $nodeID $edge1
       } elseif {[info exists arrow_($edge2,S)]} {
	  "edge_move_$arrow_($edge2,S)" $window $nodeID $otherID $edge2
       }
 }}
}


# ---------
proc Node_drag { mode window nodeID x y} {
   global arrow_

   set x [$window canvasx $x]
   set y [$window canvasy $y]

   case $mode in {
      move {
	 set xmove [expr $x-$arrow_(dragX)]
	 set ymove [expr $y-$arrow_(dragY)]

	 $window move $nodeID $xmove $ymove
	 Node_move $window $nodeID $xmove $ymove
      }
   }
   set arrow_(dragX) $x
   set arrow_(dragY) $y
}


# ---------
proc Node_remove { window nodeID} {
 global arrow_

 if {[info exists arrow_($nodeID,X)]} then {
    foreach otherID $arrow_($nodeID,Nodes) {
       set idx		          [lsearch  $arrow_($otherID,Nodes) $nodeID]
       set arrow_($otherID,Nodes) [lreplace $arrow_($otherID,Nodes) $idx $idx]

       set edge1 "${otherID}_$nodeID"
       set edge2 "${nodeID}_$otherID"

       if {[info exists arrow_($edge1,S)]} {
	  "edge_remove_$arrow_($edge1,S)" $window $edge1
	  unset arrow_($edge1,S)

       } elseif {[info exists arrow_($edge2,S)]} {
	  "edge_remove_$arrow_($edge2,S)" $window $edge2
	  unset arrow_($edge2,S)
       }
    }
    unset arrow_($nodeID,Nodes)\
          arrow_($nodeID,X) arrow_($nodeID,Y)\
          arrow_($nodeID,W) arrow_($nodeID,H)\
          arrow_($nodeID,S) arrow_($nodeID,M)\
          arrow_($nodeID,Mfield)
 }
}


# ---------
proc Node_list { window} {
 global arrow_

 set resultlist {}

 foreach item [$window find all] {
    if {[info exists arrow_($item,X)]} {
       lappend resultlist $item
    }
    foreach tag [$window gettags $item] {
      if {[info exists arrow_($tag,X)] && [lsearch $resultlist $tag] == -1} {
	 lappend resultlist $tag
      }}}
 return $resultlist
}


# ---------
proc Node_neighbours { window nodeID} {
 global arrow_

 if {[info exists arrow_($nodeID,Nodes)]} {
    return $arrow_($nodeID,Nodes)
 }
 return {}
}


# ---------
proc Edge_create { window startID destID {type ""}} {
 global arrow_
       
 set edgeID "${startID}_$destID" 

 if {   [info exists arrow_($startID,X)]
     && [info exists arrow_($destID,X)]
     && $startID != $destID
     && ![info exists arrow_($edgeID,S)]
     && ![info exists arrow_(${destID}_$startID,S)]} {

    if {$type == ""} {
       set type $arrow_(EdgeType)
    }
    lappend arrow_($startID,Nodes) $destID
    lappend arrow_($destID,Nodes)  $startID
    set     arrow_($edgeID,S)	   $type

    "edge_create_$type" $window $startID $destID $edgeID
    return $edgeID
 }
 return ""
}


# ---------
proc Edge_remove { window ID_1 ID_2} {
 global arrow_

 if {[info exists arrow_($ID_1,X)] && [info exists arrow_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"

    if {![info exists arrow_($edgeID,S)]} {
       set edgeID "${ID_2}_$ID_1"

       if {![info exists arrow_($edgeID,S)]} return
    }
    "edge_remove_$arrow_($edgeID,S)" $window $edgeID

    set idx		    [lsearch  $arrow_($ID_1,Nodes) $ID_2]
    set arrow_($ID_1,Nodes) [lreplace $arrow_($ID_1,Nodes) $idx $idx]

    set idx		    [lsearch  $arrow_($ID_2,Nodes) $ID_1]
    set arrow_($ID_2,Nodes) [lreplace $arrow_($ID_2,Nodes) $idx $idx]

    unset arrow_($edgeID,S)
 }
}


# ---------
proc Edge_ID { window ID_1 ID_2} {
 global arrow_

 if {[info exists arrow_($ID_1,X)] && [info exists arrow_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"
    if {[info exists arrow_($edgeID,S)]} { return $edgeID }

    set edgeID "${ID_2}_$ID_1"
    if {[info exists arrow_($edgeID,S)]} { return $edgeID }
 }
 return ""
}


# ---------
proc Edge_type { window ID_1 ID_2 {type ""}} {
 global arrow_

 if {[info exists arrow_($ID_1,X)] && [info exists arrow_($ID_2,X)]} {
    set edgeID "${ID_1}_$ID_2"
    if {![info exists arrow_([set edgeID "${ID_1}_$ID_2"],S)]} {
       if {![info exists arrow_([set edgeID "${ID_2}_$ID_1"],S)]} { return "" }

       set _id	  $ID_1
       set ID_1	  $ID_2
       set ID_2	  $_id
    }
    if {$type == ""} {
       return $arrow_($edgeID,S)

    } elseif {$type != $arrow_($edgeID,S)} {
       "edge_remove_$arrow_($edgeID,S)" $window $edgeID
       "edge_create_[set arrow_($edgeID,S) $type]" $window $ID_1 $ID_2 $edgeID
 }}
 return ""
}


# ---------
proc Edge_coords { startID destID} {
 global arrow_

 if {[info exists arrow_($startID,X)] && [info exists arrow_($destID,X)]} {
    set argS "$arrow_($startID,X) $arrow_($startID,Y)\
	      $arrow_($startID,W) $arrow_($startID,H)"
    set argD "$arrow_($destID,X)  $arrow_($destID,Y)\
	      $arrow_($destID,W) $arrow_($destID,H)"

    return "[eval "$arrow_(C,$arrow_($arrow_($startID,Mfield)),$arrow_($startID,S)) $argS $argD"] [lrange\
	    	   [eval "$arrow_(C,$arrow_($arrow_($destID,Mfield)),$arrow_($destID,S)) $argD $argS"]\
		   1 2]"
 }
 return {}
}


# ---------
 proc arrow_set_nodeattrs { window nodeID shape mode x y width height} {
 global arrow_

 if {"$height$width$x$y" == ""} {
    set coords [$window bbox $nodeID]
    
    set x1 [lindex $coords 0]
    set y1 [lindex $coords 1]
    set x2 [lindex $coords 2]
    set y2 [lindex $coords 3]

    set x      [expr {$x1 + ($x2-$x1)/2}]
    set width  [expr {$x2-$x1}]
    set y      [expr {$y1 + ($y2-$y1)/2}]
    set height [expr {$y2-$y1}]
 }
 set arrow_($nodeID,X)	    $x
 set arrow_($nodeID,Y)	    $y
 set arrow_($nodeID,H)	    $height
 set arrow_($nodeID,W)	    $width
 set arrow_($nodeID,M)	    $mode
 set arrow_($nodeID,Mfield) [expr {$mode == "" ? "AttachMode" : "$nodeID,M"}]

 if {$shape == ""} {
    set shape $arrow_(NodeShape)
 }
 set arrow_($nodeID,S) $shape

 if {![info exists arrow_(C,continuous,$shape)]} {
    arrow_set_coordfuncs "" $shape
 }
}


# ---------
 proc arrow_gridded_rect_Tcl { sx sy widthS heightS dx dy widthD heightD} {

 if {[set start_LowY [expr {$sy + ($heightS/2)}]]
		  <= [expr {$dy - ($heightD/2)}]} { # dest_HighY
    set startY   $start_LowY		
    set startX   $sx	   
    set pos      S

 } elseif {[set start_HighY [expr {$sy - ($heightS/2)}]]
	   		 >= [expr {$dy + ($heightD/2)}]} { # dest_LowY
    set startY   $start_HighY 
    set startX   $sx
    set pos      N

 } elseif {[set start_RightX [expr {$sx + ($widthS/2)}]]
			   < [expr {$dx - ($widthD/2)}]} { # dest_LeftX
    set startY   $sy 
    set startX   $start_RightX
    set pos      E

 } else {
    set startY   $sy 
    set startX   [expr {$sx - ($widthS/2)}]
    set pos      W
 }

 return [list $pos $startX $startY]
}


# ---------
 proc arrow_continuous_rect_Tcl {sx sy widthS heightS dx dy widthD heightD} {

 set deltaY [expr {$sy - $dy}]
 set deltaX [expr {$sx - $dx}]

 if {$deltaX == 0} {
    set edgestartX $sx

    if {$deltaY >= 0} {
       set edgestartY [expr {$sy - ($heightS/2)}]
       set edgepos    NE
    } else {
       set edgestartY [expr {$sy + ($heightS/2)}]
       set edgepos    SE
    }

 } elseif {$deltaY == 0} {
    set edgestartY $sy

    if {$deltaX >= 0} then {
       set edgestartX [expr {$sx - ($widthS/2)}]
       set edgepos    SW 
    } else {
       set edgestartX [expr {$sx + ($widthS/2)}]
       set edgepos    NW
    }

 } elseif {$deltaX < 0} {
    if {$deltaY >= 0} {
       #-- 1. quadrant :
       set edgeX      [expr {$sx + ($widthS/2)}]
       set edgeY      [expr {$sy - ($heightS/2)}]
       set a	      [arrow_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
       set edgestartX [expr {round($sx - ($a * $deltaX))}]
       set edgestartY [expr {round($sy - ($a * $deltaY))}]
       
       set edgepos   NE

    } else {
       #-- 4. quadrant :
       set edgeX      [expr {$sx + ($widthS/2)}]
       set edgeY      [expr {$sy + ($heightS/2)}]
       set a	      [arrow_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
       set edgestartX [expr {round($sx - ($a * $deltaX))}]
       set edgestartY [expr {round($sy - ($a * $deltaY))}]
       
       set edgepos   SE
    }
 } elseif {$deltaY < 0} {
    #-- 3. quadrant :
    set edgeX	   [expr {$sx - ($widthS/2)}]
    set edgeY	   [expr {$sy + ($heightS/2)}]
    set a	   [arrow_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
    set edgestartX [expr {round($sx - ($a * $deltaX))}]
    set edgestartY [expr {round($sy - ($a * $deltaY))}]

    set edgepos   SW

 } else {
    #-- 2. quadrant :
    set edgeX	   [expr {$sx - ($widthS/2)}]
    set edgeY	   [expr {$sy - ($heightS/2)}]
    set a	   [arrow_isct $edgeX $edgeY $deltaX $deltaY $sx $sy]
    set edgestartX [expr {round($sx - ($a * $deltaX))}]
    set edgestartY [expr {round($sy - ($a * $deltaY))}]

    set edgepos   NW
 }
 return [list $edgepos $edgestartX $edgestartY]
}


# ---------
 proc arrow_continuous_oval_Tcl {sx sy widthS heightS dx dy widthD heightD} {

 set deltaX [expr {$dx - $sx}]
 set deltaY [expr {$dy - $sy}]
 set f	    [expr {2 * sqrt($deltaX*$deltaX + $deltaY*$deltaY)}]

 set edgeX  [expr {$sx + round($widthS  * $deltaX / $f)}]
 set edgeY  [expr {$sy + round($heightS * $deltaY / $f)}]

 if {$deltaX >= 0} {
    set edgepos [expr {($deltaY <= 0) ? "NE" : "SE"}]
 } else {
    set edgepos [expr {($deltaY <= 0) ? "NW" : "SW"}]
 }

 return [list $edgepos $edgeX $edgeY]
}


# ---------
 proc arrow_isct { edgeX edgeY vectorX vectorY centerX centerY} {

 set a1 [expr {abs(double(($edgeX - $centerX)) / $vectorX)}]
 set a2 [expr {abs(double(($edgeY - $centerY)) / $vectorY)}]

 if {$a1 > $a2} {return $a2}
 return $a1
}


# ---------
 proc edge_create_bidir { window startID destID edgeID} {

 edge_create_nodir $window $startID $destID $edgeID
 $window itemconfigure $edgeID -arrow both
}


# ---------
 proc edge_move_bidir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc edge_remove_bidir { window edgeID} {

 $window delete $edgeID
}


# ---------
 proc edge_create_unidir { window startID destID edgeID} {

 edge_create_nodir $window $startID $destID $edgeID
 $window itemconfigure $edgeID -arrow last
}


# ---------
 proc edge_move_unidir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc edge_remove_unidir { window edgeID} {

 $window delete $edgeID
}


# ---------
 proc edge_create_nodir { window startID destID edgeID} {

 global arrow_

 set opts ""
 if {$arrow_(EdgeColor) != ""} {
    append opts " -fill $arrow_(EdgeColor)"
 }
 if {$arrow_(EdgeWidth) != ""} {
    append opts " -width $arrow_(EdgeWidth)"
 }
 if {$arrow_(EdgeStipple) != ""} {
    append opts " -stipple $arrow_(EdgeStipple)"
 }

 eval "$window create line [lrange [Edge_coords $startID $destID] 1 4]\
			   -tags $edgeID -arrow none $opts"
}


# ---------
 proc edge_move_nodir { window startID destID edgeID} {

 eval "$window coords $edgeID [lrange [Edge_coords $startID $destID] 1 4]"
}


# ---------
 proc edge_remove_nodir { window edgeID} {

 $window delete $edgeID
}

arrow_init

# Internal procedures

# eof
#

