/* =================================================================== */
/*       CLASS - IMPLEMENTATION                                    === */
/* =================================================================== */
/*                                                                     */
/* CLASS-NAME: add_Name_Space                 Class_Id: NSP            */
/*                                                                     */
/* IMPLEMENTATION ISSUES:                                              */
/*                                                                     */
/* =================================================================== */
#define NAME_SPACE_CONCEPT 1

/* =================================================================== */
/* Substitution of C-symbols :                                         */
#include "predef.h"

/* =================================================================== */
/* Trace switches:                                                     */
#include "trc_add.h"

/* =================================================================== */
// Export File of this concept
#include "Name_Space_obst.h"

// indicates that a name_space is assigned
// warnings about corrected names
// should be supressed during this phase.
sos_Bool nsp_no_corrected_syntax_warnings = FALSE;

add_Name_Space lost_and_found_nsp;

Void nsp_test_init2 ()
  // initializies the runtime data structures of this process
  // for a stand alone test of the name space concept.
  // This function should be called only by main();
{
  T_PROC("nsp_test_init2"); TT(nsp_H, T_ENTER);
  sos_String tmp2_string = sos_String::create(TEMP_CONTAINER);
  tmp2_string.assign_Cstring("lost_and_found");
  lost_and_found_nsp = add_Name_Space::create(TEMP_CONTAINER, tmp2_string, 
					      NSP_NO_NSP);
  TT(nsp_H, T_LEAVE); return;
}



// ---- Methods of this Class ---- //

add_Name_Space_Element_Set add_Name_Space::ls()
   //
   // PURPOSE:
   // returns a set of the contained Name_Space_Elements.
   // the returned set is only a copy (snapshot) of the contents of
   // this name space. The set is a buffer which will be overwritten
   // by the next execution of this method. Therefore the set
   // should be copied by the invoking method in case the
   // retrieved information should be preserved.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     none
{
  T_PROC("add_Name_Space::ls()"); TT(nsp_H, T_ENTER);

  Map_str_to_nspel elements_m = self.get_elements();  
  TT(nsp_H, TI( elements_m.card()));
  add_Name_Space_Element_Set buffer_for_element_set =
      self.get_buffer_for_element_set();
  buffer_for_element_set.clear();
  sos_Cursor cursor = elements_m.open_cursor();
  if (elements_m.to_first(cursor)) 
    {
      buffer_for_element_set.insert(elements_m.get_info(cursor));
      while (elements_m.to_succ(cursor)) 
	buffer_for_element_set.insert(elements_m.get_info(cursor));
    }
  elements_m.close_cursor(cursor);
  TT(nsp_H, TI( buffer_for_element_set.card()));
  TT(nsp_H, T_LEAVE); return (buffer_for_element_set);
}


add_Name_Space_Element add_Name_Space::element(sos_Cstring path)
   //
   // PURPOSE:
   // retrieves a Name_Space_Element by its absolute pathname.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   // the error_state() may return
   //     NSP_WRONG_PATHNAME  if the pathname is not correct.
   // If no element can be found this function returns NSPEL_NO_NSPEL.
{
  T_PROC("add_Name_Space::element(sos_Cstring path)"); 
  TT(nsp_H, T_ENTER; TS( path));

  char copy_of_pathname[NSPEL_MAX_PATH_LENGTH];
  strcpy(copy_of_pathname, path);

  add_Name_Space_Element nsp_elem = NSPEL_NO_NSPEL;

  self.clear_error();

  if (streql(copy_of_pathname,""))
      {
	char tmp_str [ERR_MAX_ERR_MESSAGE_LENGTH]; 
	sprintf(tmp_str, "%s%s\n does not contain ""\n", 
			  self.error_message(NSP_WRONG_PATHNAME), self.pathname());
	self.flag_error(NSP_WRONG_PATHNAME, tmp_str);
	TT(nsp_H, TXT("not found"); T_LEAVE); return(NSPEL_NO_NSPEL); 
      }
  
  if (copy_of_pathname[0] == '/') // it should be an absolute pathname
    {                          
      add_Name_Space nsp_root = self.root();
      sos_Cstring searched_name = strtok(REF(copy_of_pathname[1]),"/");
      TT(nsp_H, TS( searched_name));
      if (streql(searched_name, nsp_root.name()))
	{
	  if (streql(searched_name, path)) {
	    TT(nsp_H, T_LEAVE); return (nsp_root); 
	  }
	  else
	    {
	      TT(nsp_M, TXT("rest_of_pathname: "); 
		 TS(REF(path[strlen(searched_name)+2])));
	      nsp_elem = 
		nsp_root.element(REF(path[strlen(searched_name)+2]));
	      self.flag_error(nsp_root.error_state(), 
			      nsp_root.latest_error_message());
	      TT(nsp_H, T_LEAVE); return(nsp_elem);
	    }
	}
      else
	{
	  char tmp_str [ERR_MAX_ERR_MESSAGE_LENGTH]; 
	  sprintf(tmp_str, "%s%s\n does not contain %s\n", 
		  self.error_message(NSP_WRONG_PATHNAME), 
		  self.name(), searched_name);
	  self.flag_error(NSP_WRONG_PATHNAME, tmp_str);
	  TT(nsp_H, TXT("not found"); T_LEAVE); return(NSPEL_NO_NSPEL); 
	}
    }

  // retrieve name of element
  sos_Cstring searched_name = strtok(copy_of_pathname,"/");
  TT(nsp_H, TS( searched_name));
  TT(nsp_M, TXT("rest_of_pathname: "); TS(REF(path[strlen(searched_name)])));

  if (streql(searched_name,"."))
    nsp_elem = self;
  if (streql(searched_name,".."))
    nsp_elem = self.get_name_space(); 
  else
    {
      Map_str_to_nspel map = self.get_elements();
      sos_String tmp_string = sos_String::create(TEMP_CONTAINER, searched_name);
      nsp_elem = map[tmp_string];
      tmp_string.destroy();
    }

  if (nsp_elem == NO_OBJECT)
    {
      char tmp_str [ERR_MAX_ERR_MESSAGE_LENGTH]; 
      sprintf(tmp_str, "%s%s\n does not contain %s\n", 
	      self.error_message(NSP_WRONG_PATHNAME), 
	      self.name(), searched_name);
      self.flag_error(NSP_WRONG_PATHNAME, tmp_str);
      TT(nsp_H, TXT("not found"); T_LEAVE); return(NSPEL_NO_NSPEL); 
    }
  TT(nsp_H, TS(nsp_elem.pathname()));
  
  if(streql(searched_name, path))  // pathname does not contain any '/'
    {
      TT(nsp_M, TXT("element found: "); TS(nsp_elem.pathname()));
      TT(nsp_H, T_LEAVE); return(nsp_elem);
    }
  if (NOT nsp_elem.isa(add_Name_Space_type))
    {
      sos_Cstring b_string = self.error_message(NSP_WRONG_PATHNAME);
      sos_Cstring a_string = self.pathname();
      sos_Cstring c_string = strnew(b_string, a_string);
      sos_Cstring d_string = strnew("\n ", searched_name);
      sos_Cstring e_string = strnew(c_string, d_string);
      sos_Cstring f_string = strnew(e_string, " is not a name space. \n");
      self.flag_error(NSP_WRONG_PATHNAME, f_string);
      delete c_string;
      delete d_string;
      delete e_string;
      delete f_string;
      TT(nsp_H, T_LEAVE); return (NSPEL_NO_NSPEL); 
    }
  else
    {
      strcpy(copy_of_pathname, REF(path[strlen(searched_name)+1]));
      TT(nsp_M, TXT("looking for: "); TS(copy_of_pathname));
      add_Name_Space nsp = add_Name_Space::make(nsp_elem);
      add_Name_Space_Element result = nsp.element(copy_of_pathname);
      self.flag_error(nsp.error_state(), 
		      nsp.latest_error_message());
      TT(nsp_H, T_LEAVE); return(result);
    }
}


sos_Bool add_Name_Space::ready_for_destruction()
   //
   // PURPOSE:
   // returns TRUE if this object can be destroyed.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     none
{
  T_PROC("add_Name_Space::ready_for_destruction()"); TT(nsp_H, T_ENTER);

  sos_Bool retvalue = self.add_Name_Space_Element::ready_for_destruction();

  Map_str_to_nspel nspel_map = self.get_elements();
  add_Name_Space_Element nspel;
  sos_Cursor cursor = nspel_map.open_cursor();
  nspel_map.to_first(cursor);
  while (nspel_map.is_valid(cursor))
    {
      nspel = nspel_map.get_info(cursor);
      retvalue = retvalue AND nspel.ready_for_destruction();
      nspel_map.to_succ(cursor);      
    }
  nspel_map.close_cursor(cursor);

  TT(nsp_H, TB( retvalue));
  TT(nsp_H, T_LEAVE); return (retvalue);
} 


sos_Bool short_info = TRUE;
static sos_Bool nsp_first_level = TRUE;

sos_Cstring add_Name_Space::description()
   //
   // PURPOSE:
   //     returns a description of this object.
   //     Contained name spaces are described in a 
   //     non recursive way.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     none
{
  T_PROC("add_Name_Space::description()"); TT(nsp_H, T_ENTER);

  sos_Cstring tmp = self.add_Name_Space_Element::description();
  sos_Cstring string = strnew(tmp, " ");
  delete tmp;

  if ((NOT short_info) AND (nsp_first_level))
    {
      nsp_first_level = FALSE;
      tmp = string;
      string = strnew(tmp, ": \n");
      delete tmp;            // deallocate space
      Map_str_to_nspel nspel_map = self.get_elements();
      agg_iterate_association (nspel_map, sos_String key, 
			       add_Name_Space_Element nspel)
	{  
	  tmp = string;
	  string = strnew(tmp, nspel.description());
	  delete tmp;            // deallocate space
	}
      agg_iterate_association_end (nspel_map, key, nspel);
      nsp_first_level = TRUE;
    }

  TT(nsp_H, TS( string));
  TT(nsp_H, T_LEAVE); return (string);
}


sos_Bool add_Name_Space::insert_permitted(add_Name_Space_Element nspel)
   //
   // PURPOSE:
   //     returns TRUE in case nspel can be inserted into this name space.
{
  return (TRUE);
}

sos_Bool add_Name_Space::destruction_permitted(add_Name_Space_Element nspel)
   //
   // PURPOSE:
   //     returns TRUE in case nspel can be removed from this name space.
{
  return (TRUE);
}


sos_Bool add_Name_Space::assignment_permitted(add_Name_Space_Element source,
					      add_Name_Space_Element destination)
   //
   // PURPOSE:
   //     returns TRUE in case nspel can be removed from this
   //     name space.
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                  
   //     none
{
  // no check necessary because local_assign does not assign the name of source to destination.
  return (self.destruction_permitted(destination));

/*
  T_PROC("add_Name_Space::assignment_permitted(add_Name_Space_Element  source, add_Name_Space_Element destination)"); TT(nsp_H, T_ENTER);
  sos_Bool retvalue;
//  char new_name [NSPEL_MAX_NAME_LENGTH]; HUK
//  strcpy (new_name, source.name());
//  if((NOT streql(new_name, destination.name())) AND 

  sos_Cstring source_name = source.name();

//  if((NOT source.get_name().equal(destination.get_name())) AND    // Better ??
  if((NOT streql(source_name, destination.name())) AND 
     self.get_name_space().is_existing_name(source_name))
//     (self.containing_name_space().is_existing_name(source_name))) 
    retvalue = FALSE;
  else
    retvalue = self.destruction_permitted(destination);

  TT(nsp_H, TB( retvalue));
  TT(nsp_H, T_LEAVE); return (retvalue);
*/
}


sos_Bool add_Name_Space::rename_permitted
  (add_Name_Space_Element nsp_elem, sos_Cstring new_name)
   //
   // PURPOSE:
   //     called by add_Name_Space_Element::new_name(new_name)
   //     and ::local_assign(...);
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   // in case of success nsp_elem is accessible via new_name.
   // error_state() may return
   //     NSP_RENAME_NOT_PERMITTED   if it is not permitted to
   //			             rename nsp_elem.
   //     
{
  sos_Bool retvalue = TRUE;
  T_PROC("add_Name_Space::rename_permitted(add_Name_Space_Element, sos_Cstring new_name)");
  TT(nsp_H, T_ENTER; 
     sos_Cstring copy_of_new_name = strnew(new_name,"");
     TS( nsp_elem.name()); 
     TS( copy_of_new_name);

     if ((NOT streql(nsp_elem.name(),copy_of_new_name)) AND 
	 (self.is_existing_name(copy_of_new_name)))
     {
       sos_Cstring x_string = self.error_message(NSP_RENAME_NOT_PERMITTED);
       sos_Cstring a_string = strnew(copy_of_new_name,self.error_message( NSP_EXISTING_NAME));
       sos_Cstring b_string = strnew(self.pathname(), "\n");
       sos_Cstring c_string = strnew(x_string, a_string);
       sos_Cstring d_string = strnew(c_string, b_string);
       self.flag_error (NSP_RENAME_NOT_PERMITTED, d_string);
       delete a_string;
       delete b_string;
       delete c_string;
       delete d_string;
       retvalue = FALSE;
     }
     else
     {
       retvalue = TRUE;
       //      self.clear_error();
     }

     delete copy_of_new_name;
     T_LEAVE; TB( retvalue);
     ); 
  return (retvalue);
}


sos_Bool add_Name_Space::is_existing_name(sos_Cstring new_name)
   //
   // PURPOSE:
   //     called only by add_Name_Space_Element::new_name and
   //     add_Name_Space::note_insert.
   //     returns TRUE in case the new_name exists in this
   //     name space.
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                  
   //     none
{
  T_PROC("add_Name_Space::is_existing_name(sos_Cstring new_name)"); 
  TT(nsp_H, T_ENTER; 
     char tmp_str [2*NSPEL_MAX_NAME_LENGTH + 20]; 
     sprintf(tmp_str,"Does %s exist in %s ?", new_name, self.name());
     TXT(tmp_str);
     );

  sos_String tmp_string = sos_String::create(TEMP_CONTAINER, new_name);
  Map_str_to_nspel nspel_map = self.get_elements();
  sos_Bool found = nspel_map.is_key(tmp_string);
  tmp_string.destroy();

  TT(nsp_H, T_LEAVE; TB( found)); return (found);
}


sos_Cstring add_Name_Space::generate_name()
   //
   // PURPOSE:
   // generates a new name valid in this name_space
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                  
   //     none
{
  T_PROC("add_Name_Space::generate_name()"); TT(nsp_H, T_ENTER);

  TT(nsp_H, T_LEAVE; TS( "untitled")); return ("untitled");
}


add_Name_Space add_Name_Space::lost_and_found()
   //
   // PURPOSE:
   // returns a name_space where every name_space_element
   // of this type can be inserted.
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                  
   //     none
{
  return(lost_and_found_nsp);
}


Void add_Name_Space::note_insert(add_Name_Space_Element nsp_elem)
   //
   // PURPOSE:
   //     called only by add_Name_Space_Element::init and ::move_to.
   //     inserts the new element into the set of elements.
   //     returns FALSE in case insertion is not permitted.
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                  
   // error_state() may return
   //     NSP_INSERT_NOT_PERMITTED   if it is not permitted to
   //			      insert nsp_elem in this name space.
   //     NSP_EXISTING_NAME          if an element with the same name
   //                         allready exists in this name space 
   // In both cases nsp_elem is not inserted into this name space.
{
  T_PROC 
    ("add_Name_Space::note_insert(add_Name_Space_Element nsp_elem)");
  TT(nsp_H, T_ENTER; TS( nsp_elem.name());
     if (self.is_existing_name(nsp_elem.name()))
     {
       sos_Cstring a_string = nsp_elem.name();
       sos_Cstring b_string = strnew(a_string, 
				     self.error_message(NSP_EXISTING_NAME));
       sos_Cstring c_string = strnew(b_string, self.pathname());
       self.flag_error(NSP_EXISTING_NAME, c_string);
       self.signal_error(NSP_EXISTING_NAME, c_string);
       delete b_string;
       delete c_string;
       TT(nsp_H, T_LEAVE); return;
     }
     );
  Map_str_to_nspel nspel_map = self.get_elements();
  nspel_map.insert(nsp_elem.get_sos_name(), nsp_elem); // HUK 22.03.94

  self.clear_error();
  TT(nsp_H, T_LEAVE); return;
}


Void add_Name_Space::note_remove(add_Name_Space_Element nsp_elem)
   //
   // PURPOSE:
   //     called only by add_Name_Space_Element::destroy and ::move_to.
   //     removes the nsp_element from the set of elements.
   //     returns FALSE in case remove is not permitted.
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     
{
  T_PROC 
    ("add_Name_Space::note_remove(add_Name_Space_Element nsp_elem)");
  TT(nsp_H, T_ENTER; TS( nsp_elem.name()));

  Map_str_to_nspel nspel_map = self.get_elements();

  // Destroy key string
  sos_Cursor cursor = nspel_map.open_cursor();
  nspel_map.move_cursor(cursor, nsp_elem.get_sos_name());
// HUK 22.03.94  sos_String key = nspel_map.get_key(cursor);
  // Destroy mapping entry
  TT(nsp_H, TI( nspel_map.card()));
  nspel_map.remove_at(cursor);
// HUK 22.03.94  key.destroy();
  nspel_map.close_cursor(cursor);
  TT(nsp_H, TI( nspel_map.card());
     if (nspel_map.is_key(nsp_elem.get_sos_name()))
     { TT(serr_URGENT, 
	  TXT("add_Name_Space::note_remove(): name is still a key of map"));
     }
     );

  TT(nsp_H, T_LEAVE); return;
}


Void add_Name_Space::note_rename(add_Name_Space_Element nsp_elem, sos_Cstring new_name)
   //
   // PURPOSE:
   //     changes the name of nsp_elem too (!!)  HUK 22.03.94
   //     called only by add_Name_Space_Element::new_name(new_name)
   //
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     
{
  T_PROC 
    ("add_Name_Space::note_rename(add_Name_Space_Element nsp_elem, sos_Cstring new_name)");
  TT(nsp_H, T_ENTER; TS( nsp_elem.name()); TS( new_name));

  Map_str_to_nspel nspel_map = self.get_elements();
  sos_Cursor cursor = nspel_map.open_cursor();
  nspel_map.move_cursor(cursor, nsp_elem.get_sos_name());
  sos_String key_string = nsp_elem.get_sos_name(); // HUK 22.03.94
// HUK 22.03.94  sos_String key_string = nspel_map.get_key(cursor);
  add_Name_Space_Element info_nspel = nspel_map.get_info(cursor);
  nspel_map.close_cursor(cursor);
  if(nsp_elem.identical(info_nspel))
    {
      nspel_map.remove(key_string);
      key_string.assign_Cstring(new_name);
      nspel_map.insert(key_string, info_nspel);
    }
  else
    self.signal_error(ERR_INCONSISTENT, 
		      "add_Name_Space::note_rename(): assertion failed");

  TT(nsp_H, T_LEAVE); return;
}


sos_Cstring add_Name_Space::error_message(int err_value)
   //
   // PURPOSE:
   // returns the error message belonging to the error value.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   // If no message is found this method returns
   // "!! unexpected error: err_value"
{
   sos_Cstring retvalue;

   T_PROC(" add_Name_Space::error_message(int err_value)"); 
   TT(nsp_H, T_ENTER; TI( err_value));

   switch (err_value)
   {
   case NSP_EXISTING_NAME:
     retvalue = " allready exists in \n";
     break;
   case NSP_WRONG_PATHNAME:
     retvalue = " wrong pathname:\n";
     break;
   case NSP_RENAME_NOT_PERMITTED:
     retvalue = " rename is not permitted: \n";
     break;
   case NSP_DEL_NOT_PERMITTED:
     retvalue = " is not permitted to be removed from \n";
     break;
   case NSP_INSERT_NOT_PERMITTED:
     retvalue = " is not permitted to be inserted into \n";
     break;
   default:
     retvalue = self.add_Name_Space_Element::error_message(err_value);
   }

   TT(nsp_H, T_LEAVE); return(retvalue);
}     
  

Void add_Name_Space::note_note_new_ancestors()
   //
   // PURPOSE:
   // This function is called by the containing Name_Space if its
   // pathname has changed.
   // -rename pathname on account of the new containing name_space
   // -call note_note_new_ancestors on all of its name_space_elements     
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     none
{
  T_PROC("add_Name_Space::note_new_ancestors()"); TT(nsp_H, T_ENTER);

  self.add_Name_Space_Element::note_new_ancestors();

  // Notify all elements
  Map_str_to_nspel nspel_map = self.get_elements();
  agg_iterate_association (nspel_map, sos_String key, 
			   add_Name_Space_Element nspel)
    {  
      nspel.note_new_ancestors();
    }
  agg_iterate_association_end (nspel_map, key, nspel);

  TT(nsp_H, T_LEAVE); return;
} 


Void add_Name_Space::local_initialize(add_Name_Space this_one)
   //
   // PURPOSE:
   // initializes (i.e. creates) the set of elements.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     
{
  T_PROC("add_Name_Space::local_initialize(add_Name_Space)"); TT(nsp_H, T_ENTER);

  this_one.set_elements(Map_str_to_nspel::create(this_one.container(), FALSE,
						 TRUE, FALSE, 
						 AGG_HASH));  // AGG_LIST AGG_HASH
//  this_one.set_elements(Map_str_to_nspel::create(this_one.container(),FALSE ,TRUE));
//  this_one.set_elements(Map_str_to_nspel::create(this_one.container(),TRUE ,TRUE));
  this_one.set_buffer_for_element_set(add_Name_Space_Element_Set::create(this_one.container(), FALSE));

  TT(nsp_H, T_LEAVE); return;
}


Void add_Name_Space::local_finalize(add_Name_Space this_one)
   //
   // PURPOSE:
   //     destroys all contained Name_Space_Elements
   //     and the set of references pointing to them.
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     
{
  T_PROC("add_Name_Space::local_finalize(add_Name_Space this_one)"); 
  TT(nsp_H, T_ENTER; TS( this_one.pathname()));

  Map_str_to_nspel nspel_map = this_one.get_elements();
  sos_Int no_of_elements = nspel_map.card();
  for(int i = 0; (i < no_of_elements); i++)
    {
      sos_Cursor cursor = nspel_map.open_cursor(TEMP_CONTAINER);
      if (nspel_map.to_first(cursor)) 
	{ 
	  nspel_map.get_info(cursor).destruction();  
	  // nspel_map.get_key(cursor) is destroyed by self.note_remove(...).
	  nspel_map.close_cursor(cursor);
    	}
      else
	{
	  nspel_map.close_cursor(cursor);
	}
    }
  nspel_map.destroy();
  this_one.get_buffer_for_element_set().destroy();

  TT(nsp_H, T_LEAVE); return;
}


Void add_Name_Space::local_assign(add_Name_Space destination, 
				  sos_Object source_obj)
   //
   // PURPOSE:
  // assign elements of source to their counterparts in destination.
  // clone all other elements of source and insert them into destination:
   //                                                         
   // PRECONDITION:                                           
   //     none
   //                                                         
   // POSTCONDITION, ERRORS:                                  
   //     no effect.
{
  T_PROC("add_Name_Space::local_assign(add_Name_Space source, sos_Object dest)"); 
  TT(nsp_H, T_ENTER);
  add_Name_Space source = add_Name_Space::make(source_obj);
  TT(nsp_H, TS( source.pathname()));
  TT(nsp_H, TS( destination.pathname()));

  // destroy the elements of destination if they have no counterparts in source:
  Map_str_to_nspel nspel_map_dest = destination.get_elements();
  Map_str_to_nspel nspel_map_source = source.get_elements();
  int dest_card = nspel_map_dest.card();
  for(int i = 0; i < dest_card; i++)
      {
	  sos_Cursor cursor = nspel_map_dest.open_cursor();
	  if (nspel_map_dest.to_first(cursor)) 
	      { 
		  agg_iterate_association (nspel_map_dest, sos_String key, 
					   add_Name_Space_Element nspel_dest)
		      { 
			  // does not destroy elements of destination
			  // which have counterparts in source
			  // in order to assign them later.
			  if (NOT (nspel_map_source.is_key(key) AND
				   (nspel_map_source[key].type() == nspel_dest.type())))
			      {
				  nspel_dest.destruction();
				  break;
			      }
		      }
		  agg_iterate_association_end (nspel_map_dest, key, nspel);  
		  nspel_map_dest.close_cursor(cursor);
	      }
	  else
	      {
		  nspel_map_dest.close_cursor(cursor);
		  break;
	      }
      }

  if (nspel_map_dest.card() != 0)
    {
      // assign elements of source to their counterparts in destination.
      // clone all other elements of source and insert them into destination:
      agg_iterate_association (nspel_map_source, sos_String key, 
			       add_Name_Space_Element nspel)
	{ 
	  add_Name_Space_Element counterpart_in_destination = nspel_map_dest[key];
	  if (counterpart_in_destination == NO_OBJECT)
	    {
	      // create a clone in the destination name space.
	      add_Name_Space parent_of_nspel = nspel.get_name_space();
	      nspel.set_name_space(destination);  // in order to place the clone there
	      add_Name_Space_Element nspel_clone = 
		add_Name_Space_Element::clone (nspel, destination.container()); 
	      nspel.set_name_space(parent_of_nspel);
	    }
	  else
	    nspel.assign_to(counterpart_in_destination);
	}
      agg_iterate_association_end (nspel_map_source, key, nspel);
    }
  else
    {
      // faster (e.g. used each time a name space is cloned):
      agg_iterate_association (nspel_map_source, sos_String key, 
			       add_Name_Space_Element nspel)
	{ 
	  // create a clone in the destination name space.
	  add_Name_Space parent_of_nspel = nspel.get_name_space();
	  nspel.set_name_space(destination);  // in order to place the clone there
	  add_Name_Space_Element nspel_clone = 
	    add_Name_Space_Element::clone (nspel, destination.container()); 
	  nspel.set_name_space(parent_of_nspel);
	}
      agg_iterate_association_end (nspel_map_source, key, nspel);
    }

  TT(nsp_H, T_LEAVE); return;
}


Void print_nsp_recursive (add_Name_Space_Element nspel)
  // prints a description of a Name_Space hierarchy to cout.
{
  T_PROC("print_nsp_recursive"); TT(nsp_H, T_ENTER);
  sos_Cstring descr = nspel.description();
  cout << descr;
  delete descr;

  if (nspel.isa(add_Name_Space_type))
    {
      add_Name_Space nsp = add_Name_Space::make(nspel);
      add_Name_Space_Element_Set act_set = nsp.ls();
      agg_iterate(act_set, add_Name_Space_Element tmp)
	{
	  print_nsp_recursive(tmp);
	} 
      agg_iterate_end (act_set, tmp);
      cout << "\n";
    }

  TT(nsp_H, T_LEAVE); return;
}
