#line 1 "./src/sync/synccont.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : SyncCont (root object of SYNC_CONTAINER)                         */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Claus Engel                         DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_ERROR
#define OBST_IMP_STRINGOP
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_FILE
#define OBST_IMP_PROCESS
#define OBST_IMP_SOCKETS
#include "obst_stdinc.h"

#include "smg.h"

#include "obst.h"
#include "obst_progstd.h"
#include "_obst_config.h"

#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"
#include "trc_tsy.h"
#include "psm_util.h"

//-----------------------------------------------------------------------------

void _SyncCont::local_initialize (OBST_PARDECL(SyncCont) ct)
{
  ct.set_ts (1);                 // Erste moegliche Zeitmarke
  ct.set_host_id(1);             // Erste moegliche Host-Id
  ct.set_sync_tab(sos_SyncTab::create (SYNC_CONTAINER));
  ct.set_ta_tab  (sos_TaTab::create (SYNC_CONTAINER));
  ct.set_opta_tab(sos_TaTab::create(SYNC_CONTAINER));
  ct.set_host_tab(sos_HostTab::create(SYNC_CONTAINER,FALSE,TRUE));
}

//-----------------------------------------------------------------------------

void _SyncCont::local_finalize (OBST_PARDECL(SyncCont) ct)
{
    ct.get_sync_tab().destroy ();
    ct.get_ta_tab().destroy ();
}


/* ================================================================ */
/* Method: SyncCont::get_root()                                     */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

SyncCont _SyncCont::get_root()
{
  T_PROC("SyncCont::get_root()");
  TT(tsy_H, T_ENTER);

  SyncCont syct;


  syct = SyncCont::make(SYNC_CONTAINER.root_object());

  if INVALID(syct) 
     err_raise(err_SYS, err_SYNC_NO_SYCT_ROOT, ERR, FALSE);
  else
    syct.set_error (SYNC_OK);
 
  TT(tsy_H, T_LEAVE);
  return syct;
}


/* ================================================================ */
/* Method: SyncCont::exists_TA()                                    */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Bool _SyncCont::exists_TA (const sos_Typed_id&_OBSThis,sos_HostProc_Id hpid)
const{
  T_PROC("SyncCont::exists_TA");
  TT(tsy_H, T_ENTER);
  TT(tsy_VL, TI(hpid));

  sos_Bool okay;


  okay = get_ta_tab(_OBSThis).is_key(hpid);
  set_error(_OBSThis,SYNC_OK);

  TT(tsy_H, TB(okay); T_LEAVE);
  return okay;
}

/* ================================================================ */
/* Method: SyncCont::insert_TA                                      */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ALREADY_EXISTS                           */
/* ================================================================ */

void _SyncCont::insert_TA (const sos_Typed_id&_OBSThis,OBST_PARDECL(Transaction) ta)
const{
  T_PROC("SyncCont::insertTA");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(ta.get_ts()));
  
  int hpid;
  sos_TaTab   ta_tab;

  hpid = get_HP(_OBSThis);

  if (ta.get_type() != OPTIMISTIC)
     ta_tab = get_ta_tab(_OBSThis);
  else
     ta_tab = get_opta_tab(_OBSThis);

 
  if (ta_tab.is_key (hpid))
    { // this process has already started a transaction
      set_error(_OBSThis,SYNC_TA_ALREADY_EXISTS); 
      TT(tsy_H, TXT("SYNC_TA_ALREADY_EXISTS")); }  
    else
    { ta_tab.insert(hpid, ta);
      set_error(_OBSThis,SYNC_OK);
    }
 
  TT(tsy_H, T_LEAVE);
}


/* ================================================================ */
/* Method: SyncCont::remove_TA                                      */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_NO_EXISTS, SYNC_TA_NOT_OWN               */
/* ================================================================ */

void _SyncCont::remove_TA (const sos_Typed_id&_OBSThis,OBST_PARDECL(Transaction) ta) 
const{
  T_PROC("SyncCont::remove_TA");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(ta.get_ts()));

  sos_TaTab   ta_tab;
  Transaction mta;


  if (ta.get_type() != OPTIMISTIC)
     ta_tab = get_ta_tab(_OBSThis);
  else
     ta_tab = get_opta_tab(_OBSThis);

  mta = ta_tab[ta.get_hostproc_id()];
  TT(tsy_VL, TOBJ(mta));
  
  if (mta == ta)
    { // process removes it's own transaction
      ta_tab.remove(ta.get_hostproc_id());
      set_error(_OBSThis,SYNC_OK); }
   else if INVALID(mta) 
    { // ta not in mapping
      set_error(_OBSThis,SYNC_TA_NOT_EXISTS);
      TT(tsy_H, TXT("SYNC_TA_NOT_EXISTS")); }
   else
    { // process not owner of transaction
      set_error(_OBSThis,SYNC_TA_NOT_OWN);
      TT(tsy_H, TXT("SYNC_TA_NOT_OWN")); }

  TT(tsy_H, T_LEAVE);
}
 
 
/* ================================================================ */
/* Method: SyncCont::get_SO                                         */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

SyncObj _SyncCont::get_SO (const sos_Typed_id&_OBSThis,sos_Container ct)
const{
  T_PROC("SyncCont::getSO");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(ct));

  // if there is a synchronization object belonging to 'ct' then return it.
  // Otherwise create a new one and return it.
 
  SyncObj so;

  so  = get_sync_tab(_OBSThis)[ct];

  if INVALID(so) 
    {
      so = SyncObj::create (SYNC_CONTAINER, ct);

      if INVALID(so) 
        { // it's not possible to create a new SO
          TT(tsy_H, TXT("SYNC_NO_CREATE_SYNCOBJ"); T_LEAVE);
          err_raise(err_SYS, err_SYNC_NO_CREATE_SYNCOBJ,ERR, FALSE);
        }

      get_sync_tab(_OBSThis).insert (ct, so);
    }

  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return so;
}

/* ================================================================ */
/* Method: SyncCont::release_SO                                     */
/*                                                                  */
/* Error: SYNC_OK, SYNC_SYNCOBJ_NOT_EXISTS                          */
/* ================================================================ */

void _SyncCont::release_SO (const sos_Typed_id&_OBSThis,sos_Container ct)
const{
  T_PROC("SyncCont::release_SO");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(ct));
 
  SyncObj so;


  so = (get_sync_tab(_OBSThis))[ct];
  if INVALID(so) 
    { // SyncObj does not exist
      set_error(_OBSThis,SYNC_SYNCOBJ_NOT_EXISTS);
      TT(tsy_H, TXT("SYNC_SYNCOBJ_NOT_EXISTS")); }
   else
    { // remove & destroy SyncObj
      get_sync_tab(_OBSThis).remove(ct);
      so.destroy();
      set_error(_OBSThis,SYNC_OK); }
  
  TT(tsy_H, T_LEAVE);
}

/* ================================================================ */
/* Method: SyncCont::get_HP                                         */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_HostProc_Id _SyncCont::get_HP(const sos_Typed_id&_OBSThis)
const{
  T_PROC("SyncCont::get_HP");
  TT(tsy_H, T_ENTER);

  struct utsname node;
  smg_String     hostname;
  sos_String     tmphname;
  sos_String     perhname;
  unsigned       hpid;
  char           hid;

  uname(&node);
  hostname = node.nodename;
  tmphname = hostname.make_String(TEMP_CONTAINER);
 
  if (get_host_tab(_OBSThis).is_key(tmphname)) 
    {  hid = get_host_tab(_OBSThis)[tmphname]; } 
  else
    { hid = get_host_id(_OBSThis);
      if (hid == 127 ) 
         err_raise(err_SYS, err_SYNC_TOO_MANY_HOST, ERR, FALSE); 
      else
         set_host_id(_OBSThis,(sos_Char)(hid+1));
      perhname = hostname.make_String(SYNC_CONTAINER); 
      get_host_tab(_OBSThis).insert(perhname, hid);
      TT(tsy_L, TXT("create :"); TS(hostname.make_Cstring()); TC(hid));
    }

  hpid = getpid();

  TT(tsy_VL, TXT("Pid: "); TI(hpid); TXT("Hid: "); TC(hid));
  memcpy((char *) &hpid, &hid, 1);
  TT(tsy_VL, TXT("HPid:"); TI(hpid));

  TT(tsy_H, T_LEAVE);
  return hpid;
}

/* ================================================================ */
/* Method: SyncCont::get_TS                                         */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Timestamp _SyncCont::get_TS (const sos_Typed_id&_OBSThis) 
const{
  T_PROC("SyncCont::get_TS");
  TT(tsy_H, T_ENTER);

  // get the newest timestamp
 
  sos_Timestamp ts;


  ts = get_ts(_OBSThis);
  if (ts == 32767)
    ts = 0;
  set_ts(_OBSThis,ts+1);

  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, TI(ts);T_LEAVE);
  return ts;
}

/* ================================================================ */
/* Method: SyncCont::process_check                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

sos_Bool _SyncCont::process_check (const sos_Typed_id&_OBSThis,sos_HostProc_Id hpid, sos_Bool run_check)
const{
  T_PROC("SyncCont::process_check");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(hpid));

  // <af>error falls eigener HostProc
  // run_check = TRUE:  True is returned if the process still locks the
  //                    pid_file. This means the process is still running.
  // run_check = FALSE: True is returned if the process still locks the
  //                    pid_file or if there exists neither a pid_file nor
  //                    a transaction belonging to the process.
  //                    This is the default parameter.

  char*         env_path = obst_getenv(ENVVAR_CONTAINER);
  int           fd, lockresult;
  char          path[MAXPATHLEN];


  // lookup Pid-File
   // get path 
   if (env_path != 0)
     // 20 > length(process_id) + length('/process/')
     if (strlen(env_path) < MAXPATHLEN-20)
       {
        strcpy(path, env_path);
        strcat(path,"/process/");
        sprintf(path + strlen(path),"%d",(int) hpid);
       }
      else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
    else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE);
 
   // open file 
   fd = ::open(path,O_RDWR);
   if (fd == -1)
    { if (errno == ENOENT)
        { // pid-file does not exist. That means that the process has
          // not been aborted during a transaction, because the pid-
          // file is deleted in local_finalize. So there is nothing left 
          // of transaction in the Sync_Container, beside if the process
          // was aborted after the pid-file was deleted in Transaction::
          // local_finalize and before the Sync_Container was closed.
          // In this case there is still data left of the transaction 
          // in the Sync_Container.
          if (!get_ta_tab(_OBSThis).is_key(hpid))  
            { // the Sync_Container was committed before the process
              // ended or was aborted
              if (run_check)
                { TT(tsy_H, TXT("FALSE"); T_LEAVE);
                  return FALSE;
                }
               else
                { TT(tsy_H, TXT("TRUE"); T_LEAVE);
                  return TRUE;
                }
            }
          // else --> FALSE
        } 
       else
        { // should not happen
          err_raise(err_SYS, err_SYNC_PIDFILE_OPEN, ERR, FALSE);
        }
    }
   else
    { // try to lock the process-file (exclusive, testing)
      // flock (fd, LOCK_EX | LOCK_NB);
      lockresult = psm_lock_a_file(fd, WRITING, TESTING, psm_LOCK, FALSE);
      if (!lockresult)
        { // the process still employs a transaction. The Pid-File is 
          // still locked, so that the process cannot be aborted.
          ::close(fd);
          TT(tsy_H, TXT("TRUE"); T_LEAVE);
          return TRUE;
        }

      // unlock, close pid_file again
      // flock(fd, LOCK_UN | LOCK_NB);
      // Reading & Testing = dummy
      psm_lock_a_file(fd, READING, TESTING, psm_UNLOCK, FALSE); 
      ::close(fd);
    } 

  // So this means that the process is aborted and that there was a
  // transaction , invoked by the process, active at the time of
  // the abort. Therefore there are still data in the mappings of
  // the Sync_Container which are useless now. They have to be 
  // removed via SyncCont::process_remove(hpid).
    
  // end 
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, TXT("FALSE"); T_LEAVE);
  return FALSE; 
}
 

/* ================================================================ */
/* Method: SyncCont::process_remove                                 */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */
 
void _SyncCont::process_remove (const sos_Typed_id&_OBSThis,sos_HostProc_Id hpid)
const{
  T_PROC("SyncCont::process_remove");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TI(hpid));

  // precondition is that process_check has returned FALSE
 
  Transaction   ta;
  char*         env_path = obst_getenv(ENVVAR_CONTAINER);
  char          path[MAXPATHLEN];


  // lookup Pid-File
   // get path
   if (env_path != 0)
     // 20 > length(process_id) + length('/process/')
     if (strlen(env_path) < MAXPATHLEN-20)
       {
        strcpy(path, env_path);
        strcat(path,"/process/");
        sprintf(path + strlen(path),"%d",(int) hpid);
       }
      else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
    else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE);

  // That process_check has returned FALSE
  // means that the process is aborted and that there was a
  // transaction , invoked by the process, active at the time of
  // the abort. Therefore there are still data in the mappings of
  // the Sync_Container which are useless now. They have to be
  // removed.
  
  // get transaction object
  ta = get_ta_tab(_OBSThis)[hpid];
  TTA(ta);
  if INVALID(ta) 
    err_raise(err_SYS, err_SYNC_NOTA_PROCESS, ERR, FALSE);
 
  // remove the remains of the transaction
  ta.extern_abort();
 
  // destroy ta
  // (there is nothing to do about the pid-file in Transaction::local_finalize
  // and therefore ta.proc_fd is set to -1, so that the part about the
  // pid-file is skipped. )
  ta.set_proc_fd(-1);
  ta.destroy();
 
  // remove pid_file
  unlink(path);
 
  // end
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}
 

