#line 1 "./src/sync/sync.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : syncronization (Main part)                                       */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_STRINGOP
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_FILE
#define OBST_IMP_PROCESS
#define OBST_IMP_SOCKETS
#include "obst_stdinc.h"

#include "_obst_config.h"
 
#include "obst.h"
#include "obst_progstd.h"

#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"

#include "trc_tsy.h"

#include "psm_util.h"


/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: Transaction                                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: Transaction::local_initialize()                          */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void _Transaction::local_initialize (OBST_PARDECL(Transaction) ta)

{
  T_PROC("Transaction::local_initialize");
  TT(tsy_H, T_ENTER);
  
  SyncCont       syct;                  // root_object of Sync_Container
  int            hpid;
  char*          env_path = obst_getenv(ENVVAR_CONTAINER);
  int            fd;
  char           path[MAXPATHLEN];
  struct utsname node;

  // get root_obj of SYNC_CONTAINER;
  syct = _SyncCont::get_root();
  hpid = syct.get_HP();

  // create file to recognize process 
   
   // get path ENVVAR_CONTAINER/process to place the file
   if (env_path != 0)
     // 20 > length(process_id) + length('/process/')
     if (strlen(env_path) < MAXPATHLEN-20)
       { 
        strcpy(path, env_path);
        strcat(path,"/process/");
        sprintf(path + strlen(path),"%d",(int) hpid);
       }
      else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
    else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE); 

   // create file ( owner:rw, group:rw others:r )
   fd = ::open(path,O_RDWR | O_CREAT | O_EXCL,0664);
   if (fd == -1)
      err_raise(err_SYS, err_SYNC_PIDFILE_CREATE, ERR, FALSE);

   // lock the process-file (exclusive, testing)
   // -> flock(fd, LOCK_EX | LOCK_NB)
   if (!psm_lock_a_file(fd, WRITING, TESTING, psm_LOCK, FALSE))
     err_raise (err_SYS, err_SYNC_PIDFILE_LOCK, ERR, FALSE);

     
  // Create sets
  ta.set_openset  (sos_SOSet::create(SYNC_CONTAINER));
  ta.set_readset  (sos_SOSet::create(SYNC_CONTAINER));
  ta.set_writeset (sos_SOSet::create(SYNC_CONTAINER));

  ta.set_hostname (sos_String::create(SYNC_CONTAINER));
  
  // get hostname
  uname(&node);
  ta.get_hostname().assign_Cstring(node.nodename);

  // set variables
  ta.set_implicit     (FALSE);
  ta.set_hostproc_id  (hpid);
  ta.set_proc_fd      (fd);
  ta.set_user_id      (getuid());
  ta.set_status       (ACTIVE);
  ta.set_socket_port  (-1);
  ta.set_socket_seq_nr(0);
  TT(tsy_VL, TI(ta.get_hostproc_id()); TI(ta.get_user_id());
	     TI(ta.get_proc_fd()); TS(path);
             TS(ta.get_hostname().make_Cstring());
             TB(ta.get_implicit())); 
 

  // set timestamp
  ta.set_ts (syct.get_TS());
  TT(tsy_H, TI(ta.get_ts()));


  // put ta in mapping
  syct.insert_TA(ta);
  if (syct.get_error() != SYNC_OK)
    { ta.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 


  // end local_initialize
  ta.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::local_finalize()                            */
/*                                                                  */
/* Error: SYNC_OK,                                                  */
/* ================================================================ */

void _Transaction::local_finalize (OBST_PARDECL(Transaction) ta)

{ 
  T_PROC("Transaction::local_finalize");
  TT(tsy_H, T_ENTER);
  TTA(ta);

  SyncCont      syct;                  // root_object of Sync_Container
  int           fd       = ta.get_proc_fd();
  char*         env_path = obst_getenv(ENVVAR_CONTAINER);
  char          path[MAXPATHLEN];
 

  // destroy sets
  ta.get_openset().destroy();
  ta.get_readset().destroy();
  ta.get_writeset().destroy();
  
 
  // get root_obj of SYNC_CONTAINER;
  syct = _SyncCont::get_root();

  // remove ta from mapping
  syct.remove_TA(ta);
  if (syct.get_error() != SYNC_OK)
    { TT(tsy_H, T_LEAVE);
      return; } 

  
  // remove file, which was created in local_initialize
  if (ta.get_proc_fd() != -1)
    { // if the process, which started the transaction, is still running
      // ta.proc_fd is unequal -1. Because if the process is aborted,
      // ta.destroy is called from SyncCont::process_remove, where fd is
      // set to -1.
       // calculate path
       if (env_path != 0)
         // 20 > length(process_id) + length('/process/')
         if (strlen(env_path) < MAXPATHLEN-20)
           { strcpy(path, env_path);
             strcat(path,"/process/");
             sprintf(path + strlen(path),"%d", ta.get_hostproc_id());
           }
          else err_raise(err_SYS, err_SYNC_PATH_TOO_LONG,ERR, FALSE);
        else err_raise(err_SYS, err_SYNC_NO_CONTAINER_PATH,ERR, FALSE);

       // unlock,destroy, close  pid_file
       TT(tsy_VL, TI(fd); TS(path));
       // flock(fd, LOCK_UN | LOCK_NB);
       // Reading & Testing = dummy
       psm_lock_a_file(fd, READING, TESTING, psm_UNLOCK, FALSE); 
       unlink(path);
       ::close(fd);
    } 
  
  // end local_finalize
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::is_accessed()                               */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

ta_Access_mode _Transaction::is_accessed (const sos_Typed_id&_OBSThis,OBST_PARDECL(SyncObj) syob)
 
const{ 
  T_PROC("Transaction::is_accessed");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(Transaction::make(_OBSThis,this));

  set_error(_OBSThis,SYNC_OK);

  // search readset
  if (get_readset(_OBSThis).is_element(syob))
    { TT(tsy_H, TXT("READ_AC"); T_LEAVE);
     return READ_AC; }


  // search writeset
  if (get_writeset(_OBSThis).is_element(syob))
    { TT(tsy_H, TXT("WRITE_AC"); T_LEAVE);
      return WRITE_AC; }


  // container not in the sets
  TT(tsy_H, TXT("NO_AC"); T_LEAVE);
  return NO_AC;
}

/* ================================================================ */
/* Method: Transaction::is_released()                               */
/*                                                                  */
/* Error: SYNC_OK, SYNC_CONT_NO_FOUND                               */
/* ================================================================ */

sos_Bool _Transaction::is_released (const sos_Typed_id&_OBSThis,OBST_PARDECL(SyncObj) syob)

const{
  T_PROC("Transaction::is_released");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(Transaction::make(_OBSThis,this));

  // is syob in openset ?
  if (get_openset(_OBSThis).is_element(syob))
    { // syob found 
      set_error(_OBSThis,SYNC_OK);
      TT(tsy_H, TXT("FALSE"); T_LEAVE);
      return FALSE; }


  // does syncobj exists ?       
  if ((get_readset(_OBSThis).is_element(syob)) ||
      (get_writeset(_OBSThis).is_element(syob)))
    { // syncobj exists
      set_error(_OBSThis,SYNC_OK); 
      TT(tsy_H, TXT("TRUE"); T_LEAVE);
      return TRUE;}
  
  
  // syob not in sets 
  set_error(_OBSThis,SYNC_CONT_NO_FOUND);
  TT(tsy_H, TXT("SYNC_CONT_NO_FOUND"); T_LEAVE);
  return TRUE;
}

/* ================================================================ */
/* Method: Transaction::get_SO()                                    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_WRONG_ACCESS                                */
/* ================================================================ */

void _Transaction::get_SO (const sos_Typed_id&_OBSThis,OBST_PARDECL(SyncObj) syob, sos_Access_mode am)

const{
  T_PROC("Transaction::get_SO");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TAM(am));
  TSOBJ(syob);
  TTA(Transaction::make(_OBSThis,this));

  sos_Bool         readfound;                // entry in readset ?
  sos_Bool         writefound;               // entry in writeset ?
  

  // controll parameter
  if ((am != READING) && (am != WRITING))
    { set_error(_OBSThis,SYNC_WRONG_ACCESS);
      TT(tsy_H, TXT("SYNC_WRONG_ACCESS"); T_LEAVE); 
      return; }


  // search readset / writeset
  readfound  = get_readset(_OBSThis).is_element(syob);  // readset
  writefound = get_writeset(_OBSThis).is_element(syob); // writeset
  TT(tsy_VL, TB(readfound); TB(writefound));

  // READING / WRITING without previous access
  if ((!readfound) && (!writefound))
    { 
      // insert in sets
      if (am == READING)
        { TT(tsy_VL, TXT("insert in readset"));
          get_readset(_OBSThis).insert(syob); }
       else
        { TT(tsy_VL, TXT("insert in writeset"));
          get_writeset(_OBSThis).insert(syob); }


      // insert in openset
      TT(tsy_VL, TXT("insert in openset"));
      get_openset(_OBSThis).insert(syob);


      // end it
      set_error(_OBSThis,SYNC_OK);
      TT(tsy_H, T_LEAVE);
      return;
    }
    


  // READING changed to WRITING
  if ((readfound) && (am == WRITING))
     {
       // remove from readset
       TT(tsy_VL, TXT("remove from readset"));
       get_readset(_OBSThis).remove(syob);

       // insert in writeset
       TT(tsy_VL, TXT("insert in writeset"));
       get_writeset(_OBSThis).insert(syob);
     }
   

  // (re-) open container 
  if (!get_openset(_OBSThis).is_element(syob))
    { TT(tsy_VL, TXT("insert in openset"));
      get_openset(_OBSThis).insert(syob); }


  // end get_SO
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: Transaction::release_SO()                                */
/*                                                                  */
/* Error: SYNC_OK, SYNC_CONT_NO_FOUND                               */
/* ================================================================ */

void _Transaction::release_SO (const sos_Typed_id&_OBSThis,OBST_PARDECL(SyncObj) syob)

const{
  T_PROC("Transaction::release_SO");
  TT(tsy_H, T_ENTER);
  TSOBJ(syob);
  TTA(Transaction::make(_OBSThis,this));

  // search openset 
  if (get_openset(_OBSThis).is_element(syob))
    { 
      // remove found SyncObj
      TT(tsy_VL, TXT("remove from openset"));
      get_openset(_OBSThis).remove(syob);
      set_error(_OBSThis,SYNC_OK);
    }
   else
    { 
      // container not in sets
      set_error(_OBSThis,SYNC_CONT_NO_FOUND);
      TT(tsy_H, TXT("SYNC_CONT_NO_FOUND"));
    }

  
  // end release_SO
  TT(tsy_H, T_LEAVE);
  return;
}
    


/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: SyncObj (independent parts)                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Axel Freyberg                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */


/* ================================================================ */
/* Method: SyncObj::local_initialize()                              */
/*                                                                  */
/* Error: SYNC_OK (syct.get_SO)                                     */
/* ================================================================ */

void _SyncObj::local_initialize (OBST_PARDECL(SyncObj) syob)

{
  // SyncCont        syct;

  T_PROC("SyncObj::local_initialize");
  TT(tsy_H, T_ENTER);

  // get root_obj of SYNC_CONTAINER;
  /* syct = SyncCont::get_root();

  // insert syob into mapping
  syct.get_SO (syob);
  if (syct.get_error() != SYNC_OK)
    { syob.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 
   $? */

  // end local_initialize
  syob.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: SyncObj::local_finalize()                                */
/*                                                                  */
/* Error: SYNC_OK, ??                                               */
/* ================================================================ */

void _SyncObj::local_finalize (OBST_PARDECL(SyncObj) syob)

{
  // SyncCont        syct;

  T_PROC("SyncObj::local_finalize");
  TT(tsy_H, T_ENTER);

  // get root_obj of SYNC_CONTAINER;
  /* syct = SyncCont::get_root();

  // remove syob from mapping
  syct.release_SO(syob);
  if (syct.get_error() != SYNC_OK)
    { syob.set_error(syct.get_error());
      TT(tsy_H, T_LEAVE);
      return; } 
  $? */

  // end local_initialize
  syob.set_error(SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: SyncObj::get                                             */
/*                                                                  */
/* Error: SYNC_OK, SYNC_WRONG_ACCESS, SYNC_TA_ABORTED,              */
/*        SYNC_IMPLTA_ABORTED                                       */
/* ================================================================ */

sos_Bool _SyncObj::get (const sos_Typed_id&_OBSThis,sos_Access_mode am, sos_Sync_mode sm)

const{ 
  T_PROC("SyncObj::get");
  TT(tsy_H, T_ENTER);
  TT(tsy_L, TAM(am); TSM(sm));
  TSOBJ(SyncObj::make(_OBSThis,this));

  SyncCont         syct;
  Transaction      ta; 
  LockingTA        tplta;
  sos_HostProc_Id  hpid;
  sos_HostProc_Id  tapid;
  sos_Open_result  result;
  sos_Bool         access_delay = TRUE;
  sos_Bool         okay;
  sos_Bool         impl_firstcont = FALSE;
  ta_Access_mode   ct_am;

  // control parameter 
  if ((am != READING) && (am != WRITING))
    { set_error(_OBSThis,SYNC_WRONG_ACCESS);
      TT(tsy_H, TXT("SYNC_WRONG_ACCESS"); T_LEAVE);
      return FALSE; }

  while (access_delay)
    {     
      access_delay = FALSE; 
       
      syct = _SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // is transaction intern aborted ?
      if (VALID(ta) && (ta.get_status() == ABORTED))
        { // nothing to do
          set_error(_OBSThis,SYNC_OK);
          TT(tsy_H, TXT("nothing to do -> TRUE"); T_LEAVE);
          return TRUE; }
 

      if ((get_awaking(_OBSThis)) &&
          (INVALID(ta) || (ta.get_type() != OPTIMISTIC)))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
          // awake. The other ta, which has accessed the SyncObj before has 
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj, 
          // under the assumption that the process was not aborted.
          
          if (!get_queue(_OBSThis).is_empty())
            {    
              tapid = get_queue(_OBSThis).get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay) 
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
                
                SYNC_CONTAINER.close();
      
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);

                access_delay = TRUE;
              } 
            }
            else
              set_awaking(_OBSThis,FALSE); 
        } 
    }  

  // if no transaction, create one implicit
  if INVALID(ta) 
    { TT(tsy_L, TXT("create implicit TA"));
      ta = LockingTA::create(SYNC_CONTAINER);
      if INVALID(ta) 
        { TT(tsy_H, TXT("SYNC_NO_CREATE_IMPLICIT_TA -> FALSE"); T_LEAVE);
          err_raise(err_SYS, err_SYNC_NO_CREATE_IMPLICIT_TA, ERR, FALSE); 
        }
      ta.set_implicit(TRUE);
      impl_firstcont = TRUE;
      TTA(ta);
    }
 

  // container already in set of transaction ?
  ct_am = ta.is_accessed (SyncObj::make(_OBSThis,this));
  TT(tsy_VL, TXT("CT:"); TTM(ct_am); TXT("AM:"); TAM(am));

  // change to higher access
  if ((ct_am == READ_AC) && (am == WRITING))
    {
      if (ta.get_type() == LOCKING)
        { okay = tpl_writelock(_OBSThis,LockingTA::make(ta), sm);
          if (!okay)
            { TT(tsy_H, TXT("FALSE"); T_LEAVE); 
              return FALSE; } 
        }     

      // move from readset to writeset
      ta.get_SO (SyncObj::make(_OBSThis,this), am);
      set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TB(get_error(_OBSThis)==SYNC_OK); T_LEAVE);
      return sos_Bool(get_error(_OBSThis) == SYNC_OK);
    }

      
  // first access to container by transaction
  if ((ct_am == NO_AC) && ((am == READING) || (am == WRITING)))
    { 
       if (am == READING) 
	 okay = tpl_readlock(_OBSThis,LockingTA::make(ta), sm);
       else
	 okay = tpl_writelock(_OBSThis,LockingTA::make(ta), sm); 


      // if read failed, return
      if (!okay) 
        { // implict transaction objects have to be destroyed if the 
          // transaction was intern aborted or if the implict transaction just
          // has been started at the beginning of this method and the SyncObj,
          // which should be opened TESTING, failed to open.
          if ((ta.get_implicit()) && 
             ((impl_firstcont) || (get_error(_OBSThis) == SYNC_IMPLTA_ABORTED)))
            { TT(tsy_L, TXT("Implicit TA intern aborted -> destroy"));
              tplta = LockingTA::make(ta);
              tplta.destroy(); }
          TT(tsy_H, TXT("FALSE"); T_LEAVE);
          return FALSE; } 

      ta.get_SO (SyncObj::make(_OBSThis,this), am);
      set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TB(get_error(_OBSThis) == SYNC_OK); T_LEAVE);
      return sos_Bool(get_error(_OBSThis) == SYNC_OK);       
    }      



  // the other cases: ct_am = READ_AC && am == READING
  //                  ct_am = WRITE_AC && (am == READING || am == WRITING)

  // reopen if already released
  ta.get_SO(SyncObj::make(_OBSThis,this), am);
  if (ta.get_error() != SYNC_OK)
    { set_error(_OBSThis,ta.get_error());
      TT(tsy_H, TXT("FALSE"); T_LEAVE);
      return FALSE; }   


  // access is already given
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return TRUE;      
}   

/* ================================================================ */
/* Method: SyncObj::release()                                       */
/*                                                                  */
/* Error: SYNC_OK, SYNC_NO_GET, SYNC_TA_ABORTED (ts),               */
/* ================================================================ */

void _SyncObj::release (const sos_Typed_id&_OBSThis)

const{ 
  T_PROC("SyncObj::release");
  TT(tsy_H, T_ENTER);
  TSOBJ(SyncObj::make(_OBSThis,this));

  SyncCont        syct;
  Transaction     ta;
  LockingTA       tplta; 
  sos_HostProc_Id hpid;
  sos_HostProc_Id tapid;
  sos_Open_result result;
  sos_Bool        access_delay = TRUE;
  sos_Bool        okay;
  ta_Access_mode  ct_am;


  while (access_delay)
    { 
      access_delay = FALSE;

      syct = _SyncCont::get_root();

      // look for transaction 
      hpid = syct.get_HP();
      ta  = syct.get_ta_tab()[hpid];  
      TT(tsy_VL, TI(hpid));
      TTA(ta);

      // if no transaction, no container open
      if INVALID(ta) 
        { set_error(_OBSThis,SYNC_NOT_GET);
          TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
          return; }


      // is transaction intern aborted ?
      if (ta.get_status() == ABORTED)
        { // nothing to do
          set_error(_OBSThis,SYNC_OK);
          TT(tsy_H, TXT("intern aborted - nothing to do"); T_LEAVE);
          return; }


      if ((get_awaking(_OBSThis)) &&
          (ta.get_type() != OPTIMISTIC))
        { // another ta was waiting to access this SyncObj and to this ta a
          // message was sent that it can access the SyncObj. But as long
          // as awaking is set, the ta has not accessed the SyncObj, it has not
         // awake. The other ta, which has accessed the SyncObj before has
          // probably released all locks, so that to keep evrything in order
          // we have to wait, till the other ta has accessed the Obj,
          // under the assumption that the process was not aborted.
         
          if (!get_queue(_OBSThis).is_empty())
            {
              tapid = get_queue(_OBSThis).get_nth(1).get_ta().get_hostproc_id();
              okay  = syct.process_check(tapid);
              if (!okay)
                 syct.process_remove(tapid);
              else
              { // close Sync_Container and open it to give the awaking ta
                // the possibility to access the Sync_Container
 
                SYNC_CONTAINER.close();
 
                result=SYNC_CONTAINER.open(WRITING, WAITING);
                if (result != OPENED)
                   err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE);
 
                access_delay = TRUE;
              }
            }
            else
              set_awaking(_OBSThis,FALSE);
        }
    } 

  // get access mode
  ct_am = ta.is_accessed(SyncObj::make(_OBSThis,this));
  TT(tsy_L, TXT("CT:"); TTM(ct_am));
  if (ct_am == NO_AC)
    { set_error(_OBSThis,SYNC_NOT_GET);
      TT(tsy_H, TXT("SYNC_NOT_GET"); T_LEAVE);
      return; }

  // remove from openset
  ta.release_SO(SyncObj::make(_OBSThis,this));
  if (ta.get_error() != SYNC_OK)
    { set_error(_OBSThis,ta.get_error());
      TT(tsy_H, T_LEAVE);
      return; }


  // commit implicit transactions
  // !! beware of  _TA aborted implicitly somewhere before 
  TT(tsy_VL, TI(ta.get_openset().card()));
  if (ta.get_implicit())
    { if (ta.get_openset().is_empty())
	{ tplta = LockingTA::make(ta);
	  tplta.commit();
	  tplta.destroy();
	}
      else
      if (!ta.get_writeset().is_empty())
	{ sos_SOSet helpset = sos_SOSet::create(TEMP_CONTAINER);
          helpset += ta.get_openset();
          helpset *= ta.get_writeset();
          if (helpset.is_empty())
            { tplta = LockingTA::make(ta);
              tplta.commit();
            } 
        }
    }
  

  // end release
  set_error(_OBSThis,SYNC_OK);
  TT(tsy_H, T_LEAVE);
  return;
}  

