/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: Queued                                                        */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* VERSION: none                                                             */
/* ========================================================================= */

#define OBST_IMP_MEMOP
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_SIGNAL
#define OBST_IMP_TIMER
#define OBST_IMP_SOCKETS
#define OBST_IMP_PROCESS
#include "obst_stdinc.h"


// the rest
#include "obst_config.h"
#include "obst_progstd.h"
#include "obst.h"
#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "sync_trc.h"
#include "trc_tsy.h"

#define SOS_BLCK_TIMEOUT 10 
				/* seconds */
#define SOS_MAX_FLOW     10 
				/* packets */

// global variables (global due to the signal handler, because I don't know
// whether signal handling procedures can be parameterized)
LOCAL  sos_Bool	      blocked;
EXPORT sos_Bool	      sync_deadlock /* = FALSE */;
LOCAL  Transaction    blckd_ta;
LOCAL  Queued         sync_obj;
LOCAL  struct sigaction * ign_hdl;

/* ========================================================================= */
/* Everything about sockets                                                  */
/* ========================================================================= */

/* ================================================================ */
/* Help-Method: sock_init()                                         */
/*                                                                  */
/* ================================================================ */

LOCAL int sock_init( Transaction ta)
{
  T_PROC("sock_init");
  TT(tsy_M, T_ENTER);

  int    sd;
  int    addrlen;
  struct sockaddr addr;
  struct sockaddr_in name;
  struct sockaddr_in *ptr;

  if ((sd = socket(AF_INET, SOCK_DGRAM,0)) <0)
    { TT(tsy_M, TXT("create socket failed"); TI(-1); T_LEAVE);
      return (-1); } 
 
  name.sin_family      = AF_INET;
  name.sin_port        = htons(0);
  name.sin_addr.s_addr = INADDR_ANY;
 
  if (bind(sd, (sockaddr *)&name, sizeof(name))<0)
    { TT(tsy_M, TXT("bind failed"); TI(-1); T_LEAVE);
      return (-1); }
 
  addrlen = sizeof(addr);
 
  if (getsockname(sd, &addr, &addrlen) < 0)
    { TT(tsy_M, TXT("getsockname failed"); TI(-1); T_LEAVE);
      return (-1); } 
 
  ptr = (struct sockaddr_in *) &addr;
  ta.set_socket_port(htons(ptr->sin_port));

  TT(tsy_M, T_LEAVE);

  return sd; 
}

/* ================================================================ */
/* Help-Method: sock_sleep()                                        */
/*                                                                  */
/* ================================================================ */

LOCAL sos_Bool sock_sleep( int sd, int seq_nr)
{
  T_PROC("sock_sleep");
  TT(tsy_M, T_ENTER);

  char buf;
  int  rc;

  rc = recv(sd, &buf, sizeof(buf), 0);
 
  TT(tsy_M, T_LEAVE);
 
  return (sos_Bool)((rc != -1) && (atoi(&buf) == seq_nr));
}
   

/* ================================================================ */
/* Help-Method: sock_wakeup()                                       */
/*                                                                  */
/* ================================================================ */
 
LOCAL sos_Bool sock_wakeup( Transaction ta)
{
  T_PROC("sock_wakeup");
  TT(tsy_M, T_ENTER);

  int                rsd;
  int                rc;
  char               buf[8];  
  struct sockaddr_in name;
  struct hostent    *hptr;

  if ((rsd = socket(AF_INET, SOCK_DGRAM,0)) <0)
   { TT(tsy_M, TXT("socket failed"); TXT("FALSE"); T_LEAVE);
     return FALSE; }

  name.sin_family      = AF_INET;
  name.sin_port        = ta.get_socket_port();

  hptr = gethostbyname(ta.get_hostname().make_Cstring());
  memcpy(hptr->h_addr,(char *) &name.sin_addr, hptr->h_length); 

  if (connect(rsd, (struct sockaddr *) &name, sizeof(name)) <0)
    { TT(tsy_M, TXT("connect failed"); TXT("FALSE"); T_LEAVE);
      return FALSE; }

  TT(tsy_L, TXT(ta.get_hostname().make_Cstring()); TI(ta.get_socket_port()));
  sprintf(buf,"%d",ta.get_socket_seq_nr());
  rc  = send(rsd, buf, sizeof(buf),0);
  TT(tsy_L, TI(rc));

  close(rsd);

  TT(tsy_M, TB(rc!=-1); T_LEAVE);
  return (sos_Bool)(rc != -1);
} 


/* ========================================================================= */
/* The original Methods of Queued                                            */
/* ========================================================================= */

void Queued::local_initialize(Queued q)
{
 T_PROC("Queued::local_initialize");
 TT(tsy_H, T_ENTER); 

 q.set_awaking  (FALSE); 
 q.set_accesses (sos_AccessMap::create(SYNC_CONTAINER));
 q.set_queue    (sos_BlockList::create(SYNC_CONTAINER));
 q.set_lock     (NOLOCK);
 
 TT(tsy_H, T_LEAVE);
}


/* ================================================================ */
/* Help-Method: wakeup()                                            */
/*                                                                  */
/* Purpose: Operation is called when wakeup call arrives via the    */
/*          socket.                                                 */
/*          First it resets the timer to prevent sync_deadlock detection */
/*          then it opens the SYNC_CONTAINER                        */
/* ================================================================ */

LOCAL void wakeup()
{ 
   T_PROC ("Queued wakeup");
   TT(tsy_M, T_ENTER);

   typedef struct itimerval interval;     // defined in sys/time.h

   interval                *toutval_ptr;  // data for interval timer
   sos_Open_result          result; 


   // set timeout value
   toutval_ptr = new interval;   
   toutval_ptr->it_interval.tv_sec  = 0; // disables timer after timout
   toutval_ptr->it_interval.tv_usec = 0; // disables timer after timout
   toutval_ptr->it_value.tv_sec     = 0; // resets timer immediately
   toutval_ptr->it_value.tv_usec    = 0;


   // reset timer, otherwise a sync_deadlock detection cycle will be started
   if ( setitimer(ITIMER_REAL, toutval_ptr, (interval *)0) == -1)
   { // error handling: Timeout could not be resetted.
      printf("error in resetting setitimer.\n");
   }


   // disable timeout signal
   sigaction(SIGALRM, ign_hdl, (struct sigaction *)0);


   // reopen SYNC_CONTAINER
   result=SYNC_CONTAINER.open(WRITING, WAITING);
   if (result != OPENED)
     err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); 
 
   TT(tsy_M, T_LEAVE);
}


/* ================================================================ */
/* Help-Method: detect                                              */
/*                                                                  */
/* Purpose: recursively check for deadlocks                         */
/* ================================================================ */

LOCAL sos_Bool detect(sos_TransactionSet ta_set, sos_TransactionSet checked_set,
            	      sos_SOSet obj_set)
{
   T_PROC("Queued detect");
   TT(tsy_M, T_ENTER);


   sos_TransactionSet   new_set;    // blocked transactions found in this cycle
   SyncObj              waits_for;  // Object a transaction waits for
   sos_AccessMap        accesses;   // transactions currently accessing an 
                                    // object
   sos_Bool		deadl;      // sync_deadlock found?


   new_set = sos_TransactionSet::create(TEMP_CONTAINER);


   // check for transactions blocked by current (or others)
   agg_iterate(ta_set, Transaction ta)
   {
      waits_for = BlockableTA::make(ta).get_waits_for();
      if INVALID(waits_for)
         // the transaction is not blocked
         checked_set.insert(ta);
      else if ( !obj_set.is_element(waits_for) ) 
	      // doesn't wait for objecked locked by detecting ta?
      {
         // transaction is blocked by others, get them for next cycle
         accesses = waits_for.get_accesses();
         agg_iterate_association(accesses, sos_Timestamp ts, BlockedTA bta)
         {
            new_set.insert(bta.get_ta());
         }
         agg_iterate_association_end(accesses, ts, bta);
         checked_set.insert(ta);
      }
      else
      {
	 // does wait for objecked locked by detecting ta!
         // transaction is blocked by current --> sync_deadlock
         new_set.destroy();
         ta_set.destroy();
         checked_set.destroy();
         TT(tsy_M, TXT("Deadlock found"); TXT("TRUE"); T_LEAVE);
         return(TRUE);
      }
   }
   agg_iterate_end(ta_set, ta)

	
   // check whether newly found blocked transactions have been visited already
   new_set -= checked_set;

   if (new_set.is_empty())
   {
      new_set.destroy();
      ta_set.destroy();
      checked_set.destroy();
      TT(tsy_M, TXT("No sync_deadlock found"); TXT("FALSE"); T_LEAVE);
      return(FALSE);
   }

   ta_set.destroy();
   deadl = detect(new_set, checked_set, obj_set);

   TT(tsy_M, TB(deadl); T_LEAVE);
   return(deadl);
} // detect


/* ================================================================ */
/* Help-Method: dl_detect                                           */
/*                                                                  */
/* Purpose: called at timeout to detect deadlocks                   */
/*          gets all transactions that access the object where the  */
/*          checking transaction ('blckd_ta') is blocked and        */
/*          checks whether they are blocked at an object 'blocked_  */
/*          ta' holds.                                              */
/* ================================================================ */

LOCAL void dl_detect()
{
   T_PROC("Queued dl_detect");
   TT(tsy_M, T_ENTER);

   Transaction          ta;         // accessing transaction
   sos_TransactionSet   new_set;    // blocking transactions
   sos_TransactionSet   checked_set;  // transactions not involved in sync_deadlock
   sos_SOSet            obj_set;    // objects locked by 'blckd_ta'
   sos_AccessMap        accesses;   // transactions currently accessing object
   sos_PidSet           pid_set;    // set of nonexisting processes
   sos_HostProc_Id      hpid;
   sos_HostProc_Id      tpid;
   SyncCont             syct;

   new_set     = sos_TransactionSet::create(TEMP_CONTAINER);
   checked_set = sos_TransactionSet::create(TEMP_CONTAINER);
   obj_set     = sos_SOSet::create(TEMP_CONTAINER);
   pid_set     = sos_PidSet::create(TEMP_CONTAINER);

   syct = SyncCont::get_root();

   accesses = sync_obj.get_accesses();
   agg_iterate_association(accesses, sos_Timestamp ts, BlockedTA bta)
   {
      ta = bta.get_ta();
      if (ta != blckd_ta)
        // The testing transaction itself is not inserted in the new_set, 
        // because it might have a lock on the SyncObj (readlock) and might
        // wait for a lock on the same SyncObj (writelock) at the same time
        // and that is no sync_deadlock
        { hpid = ta.get_hostproc_id(); 
          if (syct.process_check(hpid)) 
             // process is still alive
             new_set.insert(ta);
           else
             // process is aborted
             pid_set.insert(hpid);
        }
   }
   agg_iterate_association_end(accesses, ts, bta);


   // remove aborted process
   agg_iterate ( pid_set, sos_Int pid )
   {
      syct.process_remove(pid);
   }
   agg_iterate_end (pid_set, pid);
   pid_set.clear();
 

   // of course, 'blckd_ta' must not be checked since it can't block itself
   checked_set.insert(blckd_ta);

   if (new_set.is_empty())
   {  // In this case there is no transaction left to wait for
      // So there cannot be a sync_deadlock. So the transaction can be
      // continued.
      new_set.destroy();
      checked_set.destroy();
      obj_set.destroy();
      TT(tsy_L, TXT("No ta accesses syncobj !"));
      hpid  = syct.get_HP(); 
      tpid  = sync_obj.get_queue().get_nth(1).get_ta().get_hostproc_id();
      while ((tpid != hpid) && (!syct.process_check(tpid)))
       { syct.process_remove(tpid);
         tpid = sync_obj.get_queue().get_nth(1).get_ta().get_hostproc_id();
       }
      if (tpid != hpid) 
        { // testing ta is not in head of queue
          TT(tsy_L, TXT(" not in queue head -> no awake, awake of head"));
          blocked = TRUE;
        }
       else
        { //testing ta represents head of queue
          TT(tsy_L, TXT("queue head -> awake"));
        }
      TT(tsy_M, T_LEAVE);
      return;
   }

   // collect objects locked by detecting ta ('blckd_ta'), to check whether other
   // ta's wait for them
   obj_set += blckd_ta.get_writeset();
   obj_set += blckd_ta.get_readset();

   sync_deadlock = detect(new_set, checked_set, obj_set);
   obj_set.destroy();

   if (!sync_deadlock) blocked = TRUE;

   TT(tsy_M, T_LEAVE);

} // dl_detect



/* ================================================================ */
/* Help-Method: timeout()                                           */
/*                                                                  */
/* Purpose: This is the signal handler routine for timeouts.        */
/*          Each time a timeout occurs, a sync_deadlock detection cycle  */
/*          is started. Afterwards the global variable 'blocked' is */
/*          assigned TRUE, which causes the process to pause again. */
/* ================================================================ */

LOCAL void timeout()
{
   T_PROC("Queued timeout");
   TT(tsy_M, T_ENTER);

   typedef struct itimerval interval;    // defined in sys/time.h

   interval                *toutval_ptr; // data for interval timer
   sos_Open_result          result;


   // set timeout value
   toutval_ptr = new interval;
   toutval_ptr->it_interval.tv_sec  = 0; // disables timer after timout
   toutval_ptr->it_interval.tv_usec = 0; // disables timer after timout
   toutval_ptr->it_value.tv_sec     = 0; // resets timer immediately
   toutval_ptr->it_value.tv_usec    = 0;


   // reset timer, otherwise a sync_deadlock detection cycle will be started
   if ( setitimer(ITIMER_REAL, toutval_ptr, (interval *)0) == -1)
   { // Fehlerbehandlung: Timeout konnte nicht zurueckgesetzt werden.
      printf("error in resetting setitimer.\n");
   }
  
 
   // reopen SYNC_CONTAINER
   result=SYNC_CONTAINER.open(WRITING, WAITING);
   if (result != OPENED) 
     err_raise(err_SYS, err_SYNC_NO_SYNC_CONTAINER, ERR, FALSE); 

   // sync_deadlock detection
   blocked = FALSE;
   dl_detect();   


   // disable wakeup signal if ta not blocked
   if (!blocked)
      sigaction(SIGUSR1, ign_hdl, (struct sigaction *)0);


   TT(tsy_M, T_LEAVE);
}

/* ================================================================ */
/* Method: Queued::enqueue()                                        */
/*                                                                  */
/* Purpose: (1) Enque blocked_ta at the specified position (default */
/*              is end of queue) and block the process belonging to */
/*              blocked_ta.                                         */
/*          (2) Before blocking the process the SyncContainer must  */
/*              be released. Also, a socket for the wakeup call     */
/*              and the start for the sync_deadlock                      */
/*              detection cycle (SIGALRM) has to be installed.      */
/*          (3) Blocking is coupled with a timeout. If the process  */
/*              is activated because of a timeout, sync_deadlock         */
/*              detection is performed.                             */
/*          (4) -                                                   */
/*          (5) If no sync_deadlock was detected, it is checked whether  */
/*              the blocking reasons still exist, if so the process */
/*              is blocked again, else                              */
/*          (6) the process tries to activate further blocked       */
/*              processes and continues normally                    */
/*          (7) The process might be woken up, but it is still not  */
/*              allowed to access the container. To be fair it      */
/*              has to be inserted at the same position as it was   */
/*              when it was dequeued. Therefore the actual position */
/*              in the queue is returned.                           */
/*                                                                  */
/* Error: SYNC_OK,                                                  */
/* ================================================================ */

Index Queued::enqueue(Transaction blocked_ta, sos_Access_mode am, 
		     Index before)
{
   T_PROC("Queue::enqueue");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TOBJ(blocked_ta); TI(am); TI(before)); 


   typedef struct sigaction sighdl;   // defined in signal.h
   typedef struct itimerval interval; // defined in sys/time.h 


   static int    sd  = -1;
   sos_Bool      sreceive;
   int           seq_nr;
   Index         takeout_pos;
   BlockedTA	 bta = BlockedTA::create(SYNC_CONTAINER, blocked_ta, am);
   interval     *toutval_ptr;               // data for interval timer
   sighdl       *tout_hdl, *oldalrm_hdl;    // data for signal handling

  
  // preparation of socket 
   if (sd == -1) 
     { sd = sock_init( blocked_ta);
       if (sd == -1)    
          err_raise(err_SYS, err_SYNC_SOCKET, ERR, FALSE);  
     }
   seq_nr = (blocked_ta.get_socket_seq_nr()+1) % SOS_MAX_FLOW;
   blocked_ta.set_socket_seq_nr(seq_nr); 


   // definition of signal handlers 
   // set timeout value
   toutval_ptr = new interval;
   toutval_ptr->it_interval.tv_sec  = 0; // disables timer after timout
   toutval_ptr->it_interval.tv_usec = 0; // disables timer after timout
   toutval_ptr->it_value.tv_sec     = SOS_BLCK_TIMEOUT;
   toutval_ptr->it_value.tv_usec    = 0;

   // initialization of signal handlers
   oldalrm_hdl = new sighdl;
   tout_hdl    = new sighdl;
   ign_hdl     = new sighdl;

   tout_hdl->sa_handler  = (sighandler_t *)timeout;
                                        // sighandler_t from obst_stdinc

   sigemptyset(&(tout_hdl->sa_mask)); // init signal set
   sigaddset(&(tout_hdl->sa_mask), SIGUSR1); // block wakeup
#ifdef HAVE_NO_INTERRUPT
   tout_hdl->sa_flags     = 0;                // no flags
#else
   tout_hdl->sa_flags     = SV_INTERRUPT;     // don't restart system calls
#endif
 
   ign_hdl->sa_handler    = (sighandler_t *)SIG_IGN;  // signal ignore handler
   sigemptyset(&(ign_hdl->sa_mask));                // no blocked signals
   ign_hdl->sa_flags      = 0;                      // no flags


   // handle queue 
   // insert appends if specified index == 0   (1)
   self.get_queue().insert(before, bta);


   // init global variables for sync_deadlock detection
   sync_deadlock = FALSE;
   blckd_ta = blocked_ta;
   sync_obj = self;


   // install signal handlers and save old ones
   TT(tsy_L, TXT("Blocking TA"));
   sigaction(SIGALRM, tout_hdl, oldalrm_hdl);


   // wait till you can continue  
   blocked  = TRUE;
   sreceive = FALSE;
   while ((blocked) && (!sreceive))
   {

      // (3)
      if ( setitimer(ITIMER_REAL, toutval_ptr, (interval *)0) == -1)
	 { // Fehlerbehandlung: Timeout konnte nicht gesetzt werden $E$
	    printf("error in setting setitimer.\n");
	 }

      // (2)
      // Closing the SYNC_CONTAINER has to be done every time this loop is
      // executed, because it is opened immediately after the transaction is
      // activated. After activation by timeout the sync_deadlock detection starts
      // and the transaction *must not* close the SYNC_CONTAINER until it is
      // asured that no sync_deadlock occured or the sync_deadlock is resolved. 
      // Otherwise another transaction may find the same sync_deadlock and resolve
      // it additionally !!
      SYNC_CONTAINER.close();

      sreceive = sock_sleep(sd, seq_nr);
   }
   
   TT(tsy_VL, TB(blocked); TB(sreceive));

   if (blocked) 
     { // socket has received a message with the right sequence number 
       wakeup();
     }

 
   // reinstall old signal handler 
   sigaction(SIGALRM, oldalrm_hdl, (sighdl *)0);

   // awake, other processes are allowed to access the SyncObj again
   self.set_awaking(FALSE);  
  
   // save position in queue for re-enqueue if necessary
   takeout_pos = self.q_takeout(bta);
   bta.destroy();

   // free signal handler
   delete toutval_ptr;
   delete tout_hdl;
   delete ign_hdl;
   delete oldalrm_hdl;

    // (5)
   if (sync_deadlock)
      self.set_error(SYNC_DEADLOCK); 
   else
      self.set_error(SYNC_OK); 

   TT(tsy_L, TI(takeout_pos));
   TT(tsy_H, T_LEAVE);
   return takeout_pos;
}

/* ================================================================ */
/* Method: Queued::dequeue()                                        */
/*                                                                  */
/* Purpose: Activates first processes in Queue, which is still      */
/*          alive, and cleans up the remains of the transactions,   */
/*          where the process does not exist anymore.               */
/*                                                                  */
/* Error: SYNC_OK,                                                  */
/* ================================================================ */

void Queued::dequeue()
{

   T_PROC("Queued::dequeue");
   TT(tsy_H, T_ENTER);

   Transaction   ta;
   int           ta_pid;
   int           own_pid;
   sos_Bool      result;
   sos_Bool      success = FALSE;
   sos_BlockList queue   = self.get_queue();
   SyncCont      syct;
 
   self.set_error(SYNC_OK);

   syct    = SyncCont::get_root();
   own_pid = syct.get_HP();
 
   while ((!queue.is_empty()) && (!success))
    { 
      ta     = queue.get_nth(1).get_ta();
      ta_pid = ta.get_hostproc_id();

      if (ta_pid != own_pid)
 
        // no signal it send to running process. ta_pid == own_pid
        // is possible, if dequeue is called during process_remove

        if (!sock_wakeup(ta))
          {
            // the signal send to the process failed. Probably the
            // process was aborted and the data of the transaction
            // is still in the SYNC_CONTAINER. Therefore it has to
            // be cleaned up. If result is true, the process is
            // still alive and something else is wrong.
 
            TT(tsy_L, TXT("process in queue was aborted"));
 
            result = syct.process_check(ta_pid);
            if (result)
               err_raise(err_SYS, err_SYNC_ACTIVATE_ERR, ERR, FALSE);
              else
               syct.process_remove(ta_pid);
            // <af> semantisch waere self.q_takeout besser
            // queue.remove(1);
          }
         else 
          { TT(tsy_L, TXT("dequeued"); TI(ta_pid));
            self.set_awaking(TRUE);
            success = TRUE;
          }
       else
        { TT(tsy_L, TXT("self_dequeue"); TI(ta_pid));
          success = TRUE;
        }
    }
   
   TT(tsy_H, T_LEAVE);
}

/* ================================================================ */
/* Method: Queued::q_takeout()                                      */
/*                                                                  */
/* Purpose: Searches the queue for the specified transaction and    */
/*          takes it out of the queue                               */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */
Index Queued::q_takeout(Transaction ta, sos_Bool report)
{

   T_PROC("Queued::q_takeout()");
   TT(tsy_H, T_ENTER);
   TTA(ta);TT(tsy_L, TB(report));
 

   sos_BlockList   queue = self.get_queue();
   Index           pos   = 1;
 
 
   agg_iterate(queue, BlockedTA qta)
   {
      if ( qta.get_ta() == ta )
         break;
      else
         pos++;
   }agg_iterate_end(queue, qta);

 
   if ((pos > queue.card()) && (report))
     err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE);
   else
   { TT(tsy_L, TXT("remove from pos "); TI(pos));
     queue.remove(pos);
     if (pos == 1) 
        self.set_awaking(FALSE);
     self.set_error(SYNC_OK); }


   TT(tsy_L, TI(pos)); 
   TT(tsy_H, T_LEAVE);
   return pos;
}


Index Queued::q_takeout(BlockedTA bta)
{
   T_PROC("Queued::q_takeout()");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TOBJ(bta));


   sos_BlockList   queue = self.get_queue();
   Index           pos   = 1;


   agg_iterate(queue, BlockedTA qta)
   {
      if ( qta == bta )
	 break;
      else
	 pos++;
   }agg_iterate_end(queue, qta);


   if ( pos > queue.card() )
     err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE);
   else
   { queue.remove(pos);
     if (pos == 1)
        self.set_awaking(FALSE);
     self.set_error(SYNC_OK); }

   TT(tsy_L, TI(pos));
   TT(tsy_H, T_LEAVE);
   return pos;
}
