/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : Locking Synchronization & Transaktion                            */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* PROBLEMS: A transaction might be awoken and woke up                       */
/*           another transacation and then failed.                           */
/*           The other transaction will read and maybe                       */
/*           misinterpret the first ones error code.                         */
/*                                                                           */
/* VERSION: 1.1                                                              */
/* ========================================================================= */

#include "obst_progstd.h"
#include "sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "trc_tsy.h"
#include "sync_trc.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: LockingSync                                                   */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: LockingSync::grant()                                     */
/*                                                                  */
/* Purpose: This method is called whenever a transaction gains      */
/*          access to data not accessed in that mode before.        */
/*          It marks the data accessed with the accessing           */
/*          transaction and notes the accessed data in the          */
/*          appropriate access sets of the transaction.             */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingSync::grant (LockingTA ta, Lock_Mode lm)
{
   T_PROC("LockingSync::grant"); 
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Lock_Mode"); TI(lm);
             TXT("Container"); TI(self.get_soscontainer()));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(self); TTA(ta);

   BlockedTA entry;
   BlockedTA bta;


   self.set_lock(lm);

   entry = self.get_accesses()[ta.get_ts()];
   if VALID(entry) 
   {
      // Upgrade _TA's lock.
      // It's assumed that grant is called only if _TA doesn't
      // hold a lock on this data item or holds a weaker lock.
      // Presently this can only be a shared lock.
      entry.set_am(WRITING);

      // Update transaction's data.
      ta.get_writeset().insert(SyncObj::make(self));
      ta.get_readset().remove(SyncObj::make(self)); 
      ta.get_openset().insert(SyncObj::make(self));
   }
   else
   {
      // Create new entry using BlockedTA type.
      
      if (lm == READLOCK)
      {
	 bta = BlockedTA::create(SYNC_CONTAINER, ta, READING);
	 ta.get_readset().insert(SyncObj::make(self));
      }
      else
      {
	 bta = BlockedTA::create(SYNC_CONTAINER, ta, WRITING);
	 ta.get_writeset().insert(SyncObj::make(self));
      }

      self.get_accesses().insert(ta.get_ts(), bta);
      ta.get_openset().insert(SyncObj::make(self));
   }

   self.set_error(SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingSync::tpl_readlock()                              */
/*                                                                  */
/* Purpose: Check for lock conflict and                             */
/*          (1) either grant lock or                                */
/*          (2) block transaction or                                */
/*          (3) return error to non-waiting transactions.           */
/*                                                                  */
/*              The check is done within a loop to recheck after    */
/*              the transaction has been blocked. This is necessary */
/*              because of the wake-up mechanism of timestamp _TA.  */
/*                                                                  */
/*          At present, only two lock modes (shared, exclusive)     */
/*          are supported. Thus, if data is already locked in a     */
/*          lockmode different to READLOCK, a lock conflict may     */
/*          have occured.                                           */
/*                                                                  */
/*          (4)                                                     */
/*          After waiting for a lock, a transaction has to recheck  */
/*          that the lock may be granted. This is because a         */
/*          successfully woke up transaction tries to wake up the   */
/*          next in queue. Wakeup may be done before lock           */
/*          aquisition is finished because the woke up transaction  */
/*          has to wait for the SyncContainer.                      */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK, SYNC_DEADLOCK                                    */
/* ================================================================ */

sos_Bool LockingSync::tpl_readlock (LockingTA ta, sos_Sync_mode sm)
{

   T_PROC("LockingSync::tpl_readlock");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Sync_mode"); TI(sm));
   TT(tsy_L, TXT("Container"); TI(self.get_soscontainer()));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(self); TTA(ta);

   sos_Bool waited = FALSE;
   Index    pos    = 0;

   while (TRUE)
   {
      if (self.get_lock() == NOLOCK)
      {
	 // (4)
	 if ( (waited) && (!self.get_queue().is_empty()) )
	    self.dequeue();

	 // (1)
	 self.grant(ta, READLOCK);
         
         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      }

      if (self.get_lock() != READLOCK)
      {
	 // does transaction hold write lock itself?
	 if  (self.get_accesses().is_key(ta.get_ts()) )
	 {
	    // Yes, accesses holds only one writing transaction at
	    // a time and this is the accessing _TA.
	    // (1)
	    self.set_error(SYNC_OK);
            TT(tsy_H, TXT("TRUE"); T_LEAVE);
	    return(TRUE);
	 }
	 else if (sm == TESTING)
	 {
	    // Inform transaction of failed access.
	    // (3)
	    self.set_error(SYNC_OK);
            TT(tsy_H, TXT("TESTING->FALSE"); T_LEAVE);
	    return(FALSE);
	 }
	 else
	 {
	    // Wait until lock is available.
            // When the enqueue call returns, the old position in the 
            // queue is stored in the variable pos, so that it is possible to
            // insert the ta again, if the access is still not possible. 
	    // (2)
	    waited = TRUE;
            ta.set_waits_for(SyncObj::make(self));
	    pos = self.enqueue(ta, READING, pos);
            ta.set_waits_for(SyncObj::make(NO_OBJECT));
	    if (self.get_error() == SYNC_DEADLOCK)
	      { TT(tsy_H, TXT("FALSE"); T_LEAVE);
                return(FALSE); // for example deadlock
              }
	 }
      }
      else
      {
	 // The data is locked in shared mode. 
	 // Maybe _TA holds read lock already.
	 if ( ! ( self.get_accesses().is_key(ta.get_ts()) ) )
	    // No. (1)
	    self.grant(ta, READLOCK);

	 // (4)
	 if ( (waited) && (!self.get_queue().is_empty()) )
	    self.dequeue();

	 // (1)
	 self.set_error(SYNC_OK);
         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      } 
   } 
} 

/* ================================================================ */
/* Method: LockingSync::tpl_writelock()                             */
/*                                                                  */
/* Purpose: see LockingSync::tpl_readlock()                         */
/*                                                                  */
/* Error: SYNC_OK, SYNC_DEADLOCK                                    */
/* ================================================================ */

sos_Bool LockingSync::tpl_writelock (LockingTA ta, sos_Sync_mode sm)
{

   T_PROC("LockingSync::tpl_writelock");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Sync_mode"); TI(sm));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(self); TTA(ta);

   sos_Bool waited = FALSE;
   Index    pos    = 0;

   while (TRUE)
   {
      if ((self.get_lock() == NOLOCK) && (self.get_accesses().card() ==0))
      {
	 // (1)
	 self.grant(ta, WRITELOCK);

         TT(tsy_H, TXT("TRUE"); T_LEAVE);
	 return(TRUE);
      }

      // Someone holds a lock on data item; _TA itself?
      if  (self.get_accesses().is_key(ta.get_ts()) )
      {
	 // Does _TA hold sole lock on data?
	 if (self.get_accesses().card() == 1)
	 {
	    // (1)
	    // holds already write lock?
	    if (self.get_lock() == WRITELOCK)
	    {
	       self.set_error(SYNC_OK);
	       TT(tsy_H, TXT("TRUE"); T_LEAVE);
               return(TRUE);
	    }
	    else
	    {
	       self.grant(ta, WRITELOCK);
 	       TT(tsy_H, TXT("TRUE"); T_LEAVE);
               return(TRUE);
            }
	 }
      }

      // Somebody else holds a lock (, too).
      if (sm == TESTING)
      {
	 // Inform transaction of failed access.
	 // (3)
	 self.set_error(SYNC_OK);
         TT(tsy_H, TXT("TESTING->FALSE"); T_LEAVE);
	 return(FALSE);
      }
      else
      {
	 // Wait until lock is available.
         // When the enqueue call returns, the old position in the
         // queue is stored in the variable pos, so that it is possible to
         // insert the ta again, if the access is still not possible.
	 // (2)
	 waited = TRUE;
         ta.set_waits_for(SyncObj::make(self));
	 pos = self.enqueue(ta, WRITING, pos);
         ta.set_waits_for(SyncObj::make(NO_OBJECT));
	 if (self.get_error() == SYNC_DEADLOCK)
           { TT(tsy_H, TXT("FALSE"); T_LEAVE);
	     return(FALSE); // for example deadlock
           }
      }
   } 
}

/* ================================================================ */
/* Method: LockingSync::tpl_release_n_wakeup()                      */
/*                                                                  */
/* Purpose: The transaction is removed from the list of accessing   */
/*          transactions for this data item. If some other trans-   */
/*          actions are waiting for the data item, the first of     */
/*          them is activated.                                      */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingSync::tpl_release_n_wakeup (LockingTA ta)
{

   T_PROC("LockingSync::tpl_release_n_wakeup");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Container"); TI(self.get_soscontainer()));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(self); TTA(ta);

   sos_AccessMap access = self.get_accesses();
   BlockedTA     bta;
   sos_Timestamp ts;
   sos_Bool      found;

   // remove ta from accesses
   access.remove(ta.get_ts());

   // test for existing locks
   if (self.get_lock() == READLOCK)
     {
       found = FALSE;
       agg_iterate_association (access, ts, bta) 
         { if ((bta.get_am() == READING) &&
               (bta.get_ta().get_type() == LOCKING))
 // <af> break      
             found = TRUE; }
       agg_iterate_association_end (access, ts , bta);
       
       if (!found)
         self.set_lock(NOLOCK);   // last Readlock released
        else
         if ((access.card() == 1) && (!self.get_queue().is_empty())) 
           { // If a transaction tries to upgrade its lock and it waits
             // in the head of the queue and its lock is the only lock
             // on the syncobj, the transaction can be dequeued 
             ts = self.get_queue().get_nth(1).get_ta().get_ts();
             if (access.is_key(ts))
               self.dequeue(); 
           }
     }
    else
         self.set_lock(NOLOCK);   // Writelock released

       
   // Release blocked transactions.
   if (self.get_lock() == NOLOCK)
      self.dequeue();
   
   self.set_error(SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
} 


/* ================================================================ */
/* Method: LockingSync::tpl_commit()                                */
/*                                                                  */
/* Purpose: Tells the data item that a locking transaction has      */
/*          committed, which means, its lock can be released.       */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingSync::tpl_commit (LockingTA ta)
{
   T_PROC("LockingSync::tpl_commit");
   TT(tsy_H, T_ENTER);
   TSOBJ(self); TTA(ta);

   sos_Container ct;


   // *all* transactions are listed by their timestamps
   if ( self.get_accesses().is_key(ta.get_ts()) )
   {
      ct = self.get_soscontainer();

      if (self.get_lock() == WRITELOCK)
	ct.write_close(); 
      else
	ct.read_close(); 

      self.tpl_release_n_wakeup(ta);
      self.set_error(SYNC_OK);
   }
   else
      // _TA doesn't exist.
      err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE);

   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingSync::tpl_abort()                                 */
/*                                                                  */
/* Purpose: Tells the data item that a locking transaction has      */
/*          aborted, which means, its lock can be released.         */
/*                                                                  */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingSync::tpl_abort (LockingTA ta)
{

   T_PROC("LockingSync::tpl_abort");
   TT(tsy_H, T_ENTER);
   TT(tsy_L, TXT("Container"); TI(self.get_soscontainer()));
   TT(tsy_VL, TXT("HostProcID"); TI(ta.get_hostproc_id()));
   TSOBJ(self); TTA(ta);

   sos_Container ct;


   // *all* transactions are listed by their timestamps
   if ( self.get_accesses().is_key(ta.get_ts()) )
   {
      ct = self.get_soscontainer();

      ct.read_close();  

      self.tpl_release_n_wakeup(ta);
      self.set_error(SYNC_OK);
   }
   else
      // _TA doesn't exist.
      err_raise(err_SYS, err_SYNC_TA_NOT_EXISTS, ERR, FALSE); 

   TT(tsy_H, T_LEAVE);
   return;
}

/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: LockingTA                                                     */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

void LockingTA::local_initialize (LockingTA ta)

{
  T_PROC("LockingTA::local_initialize");
  TT(tsy_H, T_ENTER);

  // set type of ta
  ta.set_type(LOCKING);

  ta.set_waits_for(SyncObj::make(NO_OBJECT));

  TT(tsy_H, T_LEAVE);
  return;
}

/* ================================================================ */
/* Method: LockingTA::commit                                        */
/*                                                                  */
/* Purpose: Closes all opended containers, if ta not implicit.      */
/*          Calls commit method on every accessed data item,        */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingTA::commit()
{
   T_PROC("LockingTA::commit");
   TT(tsy_H, T_ENTER);
   TTA(self);

   sos_SOSet       opset;
   sos_SOSet       clset;
   sos_SOSet       soset;
   SyncObj         so;
   sos_Bool        implta = self.get_implicit();
   sos_Bool        pcom;        // partial commit ? 


   // is ta aborted internally ?
   if (self.get_status() == ABORTED)
     { self.set_error(SYNC_OK);
       TT(tsy_H, TXT("Intern aborted"); T_LEAVE);
       return; }

   self.set_error(SYNC_OK);

   if (!implta)
     { // close used containers of explicit started transactions
       opset  = sos_SOSet::clone(self.get_openset(), TEMP_CONTAINER); 
       agg_iterate(opset, SyncObj so)
       {
         so.release();
         if (so.get_error() != SYNC_OK)
           { self.set_error(so.get_error());
             TT(tsy_H, TXT("SyncObj-Error: "); TI(so.get_error()));
             break; }
       }
       agg_iterate_end(opset, so)
       if (self.get_error() != SYNC_OK)
         { TT(tsy_H, T_LEAVE);
           return; }

       opset.clear();
     }

   if ((!implta) ||
      (( implta) && (self.get_openset().is_empty())))
     { // commit explicit started transactions or
       // commit implicit started transactions with all containers closed

       soset = self.get_readset(); 
       agg_iterate(soset, so)
       {
          LockingSync::make(so).tpl_commit(self);
       }
       agg_iterate_end(soset, so)
    
       soset.clear();


       soset = self.get_writeset();
       agg_iterate(soset, so)
       {
          LockingSync::make(so).tpl_commit(self);
       }
       agg_iterate_end(soset, so)

       pcom = FALSE;
     }

    else
     { // partial commit implicit started transactions with all
       // containers opened writing closed
       // all closed containers are committed

       clset = sos_SOSet::create(TEMP_CONTAINER);
       clset += self.get_writeset();
       clset += self.get_readset();
       clset -= self.get_openset();

       agg_iterate(clset, so)
       {
          LockingSync::make(so).tpl_commit(self);
       }
       agg_iterate_end(clset, so)

       // only open container remain in readset
       self.get_readset() *= self.get_openset();
    
       pcom = TRUE;
     }

   if (!self.get_writeset().is_empty())
      self.get_writeset().clear();

   // set status
   if (!pcom)
     { self.set_status(COMMITTED);
       TT(tsy_M, TXT("Status: COMMITTED"));
     }
    else
     { self.set_status(ACTIVE);
       TT(tsy_M, TXT("Status: ACTIVE"));
     }

   // end commit
   self.set_error(SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingTA::abort()                                       */
/*                                                                  */
/* Purpose: Resets all opended containers.                          */
/*          Calls abort method on every accessed data item,         */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingTA::abort()
{
   T_PROC("LockingTA::abort");
   TT(tsy_H, T_ENTER);
   TTA(self);

   self.set_error(SYNC_OK);

   if (self.get_status() != ABORTED)
      self.intern_abort();

   // end abort
   TT(tsy_H, T_LEAVE);
   return;
}

/* ================================================================ */
/* Method: LockingTA::intern_abort()                                */
/*                                                                  */
/* Purpose: Calls abort method on every accessed data item,         */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void LockingTA::intern_abort()
{
   T_PROC("LockingTA::intern_abort");
   TT(tsy_H, T_ENTER);
   TTA(self);

   sos_SOSet    helpset;                       // read/writeset of _TA
   sos_Cursor   cursor;                        // to iterate the sets
   SyncObj      sobj;                          // entry in set
   int          i;                             // counter
   sos_Bool     okay;


   // abort containers and clear sets
   for (i=0; i<=1; i++)
    {
      if (i==0)
        helpset = self.get_readset();
       else
        helpset = self.get_writeset();
 
      TSOSET(helpset);
 
      if (!helpset.is_empty())
       {
         // iterate set
         TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset":"writeset")));
         okay   = TRUE;
         cursor = helpset.open_cursor();
         while ((helpset.is_valid(cursor)) && (okay))
           { sobj = helpset.get(cursor);
             TSOBJ(sobj);
 
             // abort container
             LockingSync::make(sobj).tpl_abort(self);
 
             okay = helpset.to_succ(cursor);
           }
         helpset.close_cursor(cursor);
         helpset.clear();
      }
    }

   // clear openset
   self.get_openset().clear();

   // set status
   self.set_status(ABORTED);
   TT(tsy_M, TXT("Status: ABORTED"));
 
   // end intern_abort
   self.set_error(SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
}


/* ================================================================ */
/* Method: LockingTA::extern_abort()                                */
/*                                                                  */
/* Purpose: Calls tpl_release_n_wakeup on every accessed data item, */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */
 
void LockingTA::extern_abort()
{
   T_PROC("LockingTA::extern_abort");
   TT(tsy_H, T_ENTER);
   TTA(self);

   sos_SOSet    helpset;                       // read/writeset of _TA
   sos_Cursor   cursor;                        // to iterate the sets
   SyncObj      sobj;                          // entry in set
   int          i;                             // counter
   int          pos;                           // takeout position
   sos_Bool     okay;
 
 
   // abort containers and clear sets
   for (i=0; i<=1; i++)
    {
      if (i==0)
        helpset = self.get_readset();
       else
        helpset = self.get_writeset();
 
      TSOSET(helpset);
 
      if (!helpset.is_empty())
       {
         // iterate set
         TT(tsy_L, TXT("iterate set:"); TXT((i==0 ? "readset":"writeset")));
         okay   = TRUE;
         cursor = helpset.open_cursor();
         while ((helpset.is_valid(cursor)) && (okay))
          { sobj = helpset.get(cursor);
            TSOBJ(sobj);
 
            // release_n_wakeup container
            if (sobj.get_accesses().is_key(self.get_ts()))
              LockingSync::make(sobj).tpl_release_n_wakeup(self);
 
            okay = helpset.to_succ(cursor);
          }
         helpset.close_cursor(cursor);
         helpset.clear();
       }
    }
 
   // clear sets
   self.get_openset().clear();

   // remove from queue of SyncObj 
   if VALID(self.get_waits_for())
     { // remove from queue
       pos = Queued::make(self.get_waits_for()).q_takeout(self, FALSE);
       if (pos == 1)
          // awake next, if ta was at head of the queue, because 
          // it might be possible for it to access the SyncObj
          Queued::make(self.get_waits_for()).dequeue();  
     }
   
   // end extern_abort
   self.set_error(SYNC_OK);
   TT(tsy_H, T_LEAVE);
   return;
} 



/**/
/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: SyncObj (Locking Part)                                        */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */

/* ================================================================ */
/* Method: SyncObj::tpl_readlock()                                  */
/*                                                                  */
/* Purpose: This method mainly calls LockingSync::tpl_readlock and  */
/*          additionally asures deadlocks abort the transaction.    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED, SYNC_IMPLTA_ABORTED             */
/* ================================================================ */

sos_Bool SyncObj::tpl_readlock(LockingTA ta, sos_Sync_mode sm) 
{
   T_PROC("SyncObj::tpl_readlock");
   TT(tsy_H, T_ENTER);
   TSOBJ(self); TTA(ta); TT(tsy_L, TSM(sm));

   sos_Bool granted; //states whether access is granted or not

   granted = self.LockingSync::tpl_readlock(ta,sm);
   if (! granted)
     { if ((self.get_error() == SYNC_DEADLOCK) || (sm == WAITING))
       { // transaction has to be aborted
         ta.intern_abort();
         if (ta.get_implicit())
           { self.set_error(SYNC_IMPLTA_ABORTED);
             TT(tsy_H, TXT("SYNC_IMPLTA_ABORTED -> FALSE")); }
          else
           { self.set_error(SYNC_TA_ABORTED);
             TT(tsy_H, TXT("SYNC_TA_ABORTED -> FALSE")); }
       }
      else
	 self.set_error(SYNC_OK); // just tell user that access not possible
     }
    else
       self.set_error(SYNC_OK); 

   TT(tsy_H, TB(granted); T_LEAVE);
   return(granted);
} 

/* ================================================================ */
/* Method: SyncObj::tpl_writelock()                                 */
/*                                                                  */
/* Purpose: This method mainly calls LockingSync::tpl_writelock and */
/*          additionally asures deadlocks abort the transaction.    */
/*                                                                  */
/* Error: SYNC_OK, SYNC_TA_ABORTED, SYNC_IMPLTA_ABORTED             */
/* ================================================================ */

sos_Bool SyncObj::tpl_writelock(LockingTA ta, sos_Sync_mode sm) 
{
   T_PROC("SyncObj::tpl_writelock");
   TT(tsy_H, T_ENTER);
   TSOBJ(self); TTA(ta); TT(tsy_L, TSM(sm));

   sos_Bool granted; //states whether access is granted or not


   granted = self.LockingSync::tpl_writelock(ta,sm);
   if (! granted)
     { if ((self.get_error() == SYNC_DEADLOCK) || (sm == WAITING))
       { // transaktion has to be aborted
         ta.intern_abort();
         if (ta.get_implicit())
           { self.set_error(SYNC_IMPLTA_ABORTED);
             TT(tsy_H, TXT("SYNC_IMPLTA_ABORTED -> FALSE")); }
          else
           { self.set_error(SYNC_TA_ABORTED);
             TT(tsy_H, TXT("SYNC_TA_ABORTED -> FALSE")); }
       }
      else
         self.set_error(SYNC_OK); // just tell user that access not possible
     }
    else
       self.set_error(SYNC_OK);

   TT(tsy_H, TB(granted); T_LEAVE);
   return(granted);

} 

