/* -----------------------------------------------------------------------------
* Copyright 1992 by Forschungszentrum Informatik(FZI)
*
* You can use and distribute this software under the terms of the license
* you should have received along with this software.
* If not or if you want additional information, write to
* Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
* D-76131 Karlsruhe, Germany.
* ------------------------------------------------------------------------------
*/
/* OBST LIBRARY MODULE */

// *****************************************************************************
// Module psm_err              Emil Sekerinski, Oliver Spatscheck, Walter Zimmer
//
// *****************************************************************************
//  persistent storage manager: monitor 
// *****************************************************************************



// *****************************************************************************
// INCLUDE FILES 
// *****************************************************************************

#define OBST_IMP_STDCONST
#define OBST_IMP_XLIB
#define OBST_IMP_TIME
#include "obst_stdinc.h"
#include "_obst_config.h"	// CTTABLE_SIZE
#include "knl_use.h"            // sos_Bool in psm_err.h

#include "psm_mon.h"
#include "psm_err.h"        // err_raise, err_SYS

// *****************************************************************************
// PRIVATE DECLARATIONS
// *****************************************************************************

// ================= enums =====================================================

enum Box {black, white, grey};
enum Border {thick, thin, no};

// ================= function prototyps ========================================

static void paint_border(Window win, Border b); 
static void paint_box(Window win, int number, Box b); 
static void paint_grid(Window win); 
static void write_title(Window win, char* title); 

// ================= classes & structures ======================================

struct win_info 
{
   int cid;           // cid=0 if unused
   Window win; 
};

// ================= constant declarations =====================================

#define uwait		15
#define notice		500000 	
				// microsec
#define max_windows	CTTABLE_SIZE

#define box_sz		9	
				// size of each box in pixels
#define border_sz	10	
				// size of border in pixels

#define number_of_boxes	400	
				// number of black/white/grey boxes per window
#define dim		20	
				// number of boxes in each dimension

#define length_of_win	(dim*(box_sz+1) + 1)
#define height_of_win	(length_of_win + 2*border_sz)
				// height of window contents in pixels
#define width_of_win	(length_of_win + 2*border_sz)
				// width of window contents in pixels


// ================= data declarations =========================================

static const char* ERR;
static char*       grey_bg;
static win_info    win_tbl[max_windows];

static Display*    disp;
static GC 	   gc;


// *****************************************************************************
// PUBLIC FUNCTION DEFINITIONS 
// *****************************************************************************

void mon_black(mon_win win, int number) 
{
   if (win < max_windows) 
      paint_box(win_tbl[win].win, number, black); 
}

//------------------------------------------------------------------------------

void mon_close(mon_win win) 
{
   if (win < max_windows) 
      paint_border(win_tbl[win].win, thick); 
}

//------------------------------------------------------------------------------

mon_win mon_create(int cid, int last) 
{
   // check if window of cid already displayed and find free slot in win_tbl
   int n=0; 
   int free_position=max_windows;
   
   while (n<max_windows && win_tbl[n].cid != cid) 
   {
      if (win_tbl[n].cid==0) 
         free_position=n;

      n++; 
   }
  
   if (n==max_windows) 
   {
      // monitor of cid not yet displayed
      if (free_position < max_windows) 
      { 
         // window table not exhausted, create new window
         const win_dx = free_position * 135;
         const win_dy = 20;
         win_info& win = win_tbl[free_position];
         win.cid = cid;
         win.win = XCreateSimpleWindow(disp, RootWindow(disp, 0),
                                       win_dx, win_dy, width_of_win,
                                       height_of_win, 4,
                                       BlackPixel(disp, 0), 
                                       WhitePixel(disp, 0));
         XSetWindowAttributes attr;
         attr.backing_store=Always;  // enables automatic repainting
         // also makes waiting for first expose event obsolete !
         XChangeWindowAttributes(disp, win.win, CWBackingStore, &attr);
         XMapRaised(disp, win.win); // map window to make it visible
         write_title(win.win, 0);
         paint_grid(win.win);
         for (int number=0; number<last; number++) 
            paint_box(win.win, number, white);

         usleep(notice); 
      } 
   }
   else  
   { 
      // monitor already displayed
      free_position=n; 
   }

   return free_position; 
}

//------------------------------------------------------------------------------

void mon_destroy(mon_win win) 
{
   if (win < max_windows) 
      XDestroyWindow(disp, win_tbl[win].win);
      win_tbl[win].cid = 0; 
}

//------------------------------------------------------------------------------

void mon_grey(mon_win win, int number) 
{
   if (win < max_windows)
      paint_box(win_tbl[win].win, number, grey); 
}

//------------------------------------------------------------------------------

void mon_initialize() 
{  static int initialized;
   if (initialized) return;
   ++ initialized;

   ERR     = "from monitor of container manager";
   grey_bg = "\125\252\125\252\125\252\125\252";

   disp = XOpenDisplay(0);
   if (disp == 0)
      err_raise(err_SYS, err_MON_SERVER_CONNECTION_FAILED, NULL, FALSE);

   XrmInitialize();
   gc = XCreateGC(disp, RootWindow(disp, 0), 0, 0);

   Pixmap tile = XCreatePixmapFromBitmapData(disp, RootWindow(disp, 0),
                 grey_bg, 8, 8, BlackPixel(disp, 0), WhitePixel(disp, 0)
               , DefaultDepth(disp, 0));
   if (tile == 0)
      err_raise(err_SYS, err_MON_PIXMAP_CREATION_FAILED, NULL, FALSE);

   XSetTile(disp, gc, tile);
   XSynchronize(disp, TRUE);
    
   for (int i=0; i<max_windows; i++) 
      win_tbl[i].cid=0; 
}

//------------------------------------------------------------------------------

void mon_open(mon_win win, int rw, char* title) 
{
   if (win < max_windows) 
   {
      paint_border(win_tbl[win].win, (rw) ? no : thin);

      write_title(win_tbl[win].win, title); 
   } 
}

//------------------------------------------------------------------------------

void mon_white(mon_win win, int number) 
{
   if (win < max_windows) 
      paint_box(win_tbl[win].win, number, white); 
}

// *****************************************************************************
// PRIVATE FUNCTION DEFINITIONS 
// *****************************************************************************

static void paint_border(Window win, Border b) 
{
   switch (b) 
   {
      case thick: 
         XSetForeground(disp, gc, BlackPixel(disp, 0));
         XSetBackground(disp, gc, WhitePixel(disp, 0));
         XSetLineAttributes(disp, gc, 5, LineSolid, CapRound, JoinRound);
         break;
      case thin: 
         paint_border(win, no);
         XSetForeground(disp, gc, BlackPixel(disp, 0));
         XSetLineAttributes(disp, gc, 2, LineSolid, CapRound, JoinRound);
         break;
      case no: 
         XSetForeground(disp, gc, WhitePixel(disp, 0));
         XSetBackground(disp, gc, BlackPixel(disp, 0));
         XSetLineAttributes(disp, gc, 6, LineSolid, CapRound, JoinRound);
   }
   XDrawRectangle(disp, win, gc, border_sz/2, border_sz/2, 
                  border_sz+length_of_win, border_sz+length_of_win);
   usleep(notice); 
}

//------------------------------------------------------------------------------

static void paint_box(Window win, int number, Box b) 
{
   if (number < number_of_boxes) 
   {
      const y_coordinat = border_sz + 1 +(number / dim) *(box_sz + 1);
      const x_coordinat = border_sz + 1 +(number % dim) *(box_sz + 1);

      switch (b) 
      {
         case black: 
            XSetForeground(disp, gc, BlackPixel(disp, 0));
            XSetFillStyle(disp, gc, FillSolid);
            break;
         case white: 
            XSetForeground(disp, gc, WhitePixel(disp, 0));
            XSetFillStyle(disp, gc, FillSolid);
            break;
         case grey:  
            XSetFillStyle(disp, gc, FillTiled);
      }
      XFillRectangle(disp, win, gc, x_coordinat, y_coordinat, box_sz, box_sz);
      usleep(uwait); 
   } 
}

//------------------------------------------------------------------------------

static void paint_grid(Window win) 
{
   XSetForeground(disp, gc, BlackPixel(disp, 0));
   XSetBackground(disp, gc, WhitePixel(disp, 0));
   XSetFillStyle  (disp, gc, FillTiled);
   XFillRectangle(disp, win, gc, border_sz, border_sz, length_of_win,
                  length_of_win);

   XSetFillStyle  (disp, gc, FillSolid);
   XSetLineAttributes(disp, gc, 1, LineSolid, CapRound, JoinRound);
   for (int y_offset=0; y_offset<length_of_win; y_offset+=box_sz+1)
   {
      XDrawLine(disp, win, gc, border_sz, border_sz + y_offset, 
                border_sz + length_of_win - 1, border_sz + y_offset);
   }

   for (int x_offset=0; x_offset<length_of_win; x_offset+=box_sz+1)
   {
      XDrawLine(disp, win, gc, border_sz + x_offset, border_sz,
                border_sz + x_offset, border_sz + length_of_win - 1); 
   }
}

//------------------------------------------------------------------------------

static void write_title(Window win, char* title) 
{
   if (title != 0) 
      XStoreName(disp, win, title);
   else
      XStoreName(disp, win, ""); 
}
