#line 1 "./src/prot/prot_class.C"
/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module prot                   13-Aug-1993	       Bernhard Schiefer (bs)
// **************************************************************************
// implements methods of all classes declared in schema "prot"
// **************************************************************************
 
#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "mta_obst.h"
#include "trc_prot.h"

#include "prot_obst.h"
#include "prot_err.h"

// -------------------------------------------------------------------------
// sos_ClassModif
// -------------------------------------------------------------------------
// *************************************************************************
void _sos_ClassModif::local_initialize(OBST_PARDECL(sos_ClassModif) cm)
// *************************************************************************
{
   T_PROC("sos_ClassModif::local_initialize");
   TT(prot_M, T_ENTER);

   sos_Container cnt = cm.container();

   cm.set_change_create_params(sos_ParamModif_List::create(cnt));

   cm.set_added_friends         (sos_Type_Set::create(cnt));
   cm.set_deleted_friends       (sos_Type_Set::create(cnt));
  
   cm.set_change_superclasses   (sos_SuperClassModif_List::create(cnt));

   cm.set_change_comps          (sos_CompModif_List::create(cnt));

   cm.set_old_comp_offset(sos_Comp_descr_sos_Offset_Mapping::create(cnt));
   cm.set_old_spcl_offset(sos_Type_descr_sos_Offset_Mapping::create(cnt));
 
   cm.set_old_local_size(-1);
   cm.set_old_object_size(-1);
 
   cm.set_added_methods         (sos_Method_Set::create(cnt));
   cm.set_deleted_methods       (sos_Method_Set::create(cnt));
   cm.set_modified_methods      (sos_MethodModif_List::create(cnt));

   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::local_finalize(OBST_PARDECL(sos_ClassModif) cm)
// *************************************************************************
{
   T_PROC("sos_ClassModif::local_finalize");
   TT(prot_M, T_ENTER);

   sos_ParamModif_List             pm_list;
   sos_ParamModif                  pm;
   sos_SuperClassModif_List        scm_list;
   sos_SuperClassModif             scm;
   sos_CompModif_List              cpm_list;
   sos_CompModif                   cpm;
   sos_MethodModif_List            mm_list;
   sos_MethodModif                 mm;
   

   cm.deinstall();

   pm_list = cm.get_change_create_params();
   agg_iterate(pm_list, pm)
   {
      pm.destroy();
   } agg_iterate_end(pm_list, pm);
   pm_list.destroy();

   cm.get_added_friends().destroy();
   cm.get_deleted_friends().destroy();

   scm_list = cm.get_change_superclasses();
   agg_iterate(scm_list, scm)
   {
      scm.destroy();
   } agg_iterate_end(scm_list, scm);
   scm_list.destroy();

   cpm_list = cm.get_change_comps();
   agg_iterate(cpm_list, cpm)
   {
      cpm.destroy();
   } agg_iterate_end(cpm_list, cpm);
   cpm_list.destroy();

   cm.get_added_methods().destroy();
   cm.get_deleted_methods().destroy();
  
   mm_list = cm.get_modified_methods();
   agg_iterate(mm_list, mm)
   {
      mm.destroy();
   } agg_iterate_end(mm_list, mm);
   mm_list.destroy();

   cm.get_old_comp_offset().clear();
   cm.get_old_comp_offset().destroy();
   cm.get_old_spcl_offset().clear();
   cm.get_old_spcl_offset().destroy();

   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::install(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Schema_module) sm)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::install");
   TT(prot_M, T_ENTER);

   sos_SchemaHistory sh = _sos_ModifHistory::modif_history().get_history()[sm];
   sos_Class_type    ct;
   sos_VCM_List      vcm;

   ct  = sos_Class_type::make(get_modified_type(_OBSThis)); 
   vcm = sh.get_class_modifs()[ct];
   
   if (INVALID(vcm))
   {  vcm = sos_VCM_List::create(_OBSThis.container());
      sh.get_class_modifs().insert(ct, vcm);
   }

   vcm.insert(1, sos_ClassModif::make(_OBSThis,this));

   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::deinstall(const sos_Typed_id&_OBSThis)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::deinstall");
   TT(prot_M, T_ENTER);

   sos_SchemaHistory sh;
   sos_Schema_module sc;
   sos_Class_type    ct;
   Index             pos;
   sos_VCM_List      vl;
 
   ct = sos_Class_type::make(get_modified_type(_OBSThis));
   sc = _sos_Schema_module::retrieve(ct.container()); 
   if (VALID(sc))
   {
     sh = _sos_ModifHistory::modif_history().get_history()[sc]; 
     if (VALID(sh))
     {  vl = sh.get_class_modifs()[ct];
        if (VALID(vl))
        {  pos = vl.find(sos_ClassModif::make(_OBSThis,this));
           vl.remove(pos);
           if (vl.is_empty()) 
             sh.get_class_modifs().remove(ct);
        }
     }
   }
   TT(prot_M, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::save_offsets(const sos_Typed_id&_OBSThis,sos_Bool only_spcl)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::save_offsets");
   TT(prot_M, T_ENTER);

   sos_Schema_module    sc;
   sos_SchemaModif      sm;
   sos_Class_type_List  subclasses;
   sos_Class_type       sub;
   sos_ClassModif       subcm;	
   sos_Super_class_List superclasses;
   sos_Super_class      super;
   sos_Comp_descr_List  components;
   sos_Comp_descr	comp;
   sos_Class_type       ct;

   ct         = sos_Class_type::make(get_modified_type(_OBSThis));
   subclasses = ct.get_subclasses();

   // save the superclasses offsets for all subclasses and the class itself,
   // if they have not already been saved
   agg_iterate(subclasses, sub)
   {
      if (sub != ct)
      {  sc = _sos_Schema_module::retrieve(sub.container());
         sm = _sos_SchemaModif::mod_schema(sc); 
 
         subcm = sos_ClassModif::make(sm.mod_type(sub));
      }
      else
         subcm = sos_ClassModif::make(_OBSThis,this);

      if (VALID(subcm) && subcm.get_old_spcl_offset().is_empty())
      {  superclasses = sub.get_super_closure();
    
         agg_iterate(superclasses, super) 
         {  
            subcm.get_old_spcl_offset().insert(super.get_super_class(),
					       super.get_offset());
         } agg_iterate_end(superclasses, super);
  
         subcm.set_old_local_size (sub.get_local_size());
         subcm.set_old_object_size(sub.get_object_size()); 
      }

   } agg_iterate_end(subclasses, sub);
  

   // save the relative component offsets of the class
   // if they have not already been saved
   if (!only_spcl && get_old_comp_offset(_OBSThis).is_empty())
   {
      components = ct.get_components();
 
      agg_iterate(components, comp)
      {
         get_old_comp_offset(_OBSThis).insert(comp, comp.get_offset());
      } agg_iterate_end(components, comp);
   }
   TT(prot_M, T_LEAVE);
} 

// *************************************************************************
void _sos_ClassModif::add_create_par(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Param) param, sos_Int position)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::add_create_par");
   TT(prot_H, T_ENTER);

   sos_ParamModif pm;
 
   pm = sos_ParamModif::create(_OBSThis.container());
   pm.set_modified_param(param);
   pm.set_position      (position);
   pm.set_change        (PROT_ADD);
 
   get_change_create_params(_OBSThis).append(pm);

   set_change(_OBSThis,PROT_MOD);

   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::del_create_par(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Param) param)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::del_create_par");
   TT(prot_H, T_ENTER);

   sos_ParamModif       pm;
   sos_Class_type       ct;

   ct = sos_Class_type::make(get_modified_type(_OBSThis)); 

   pm = sos_ParamModif::create(_OBSThis.container());
   pm.set_modified_param(param);
   pm.set_position      (ct.get_create_params().find(param));
   pm.set_change        (PROT_DEL);
 
   get_change_create_params(_OBSThis).append(pm);

   set_change(_OBSThis,PROT_MOD);

   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::mod_create_par(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Param) param)
// *************************************************************************
const{
   T_PROC("os_ClassModif::mod_create_par");
   TT(prot_H, T_ENTER);

   sos_Container        cnt;
   sos_ParamModif_List  parammodifs;
   sos_ParamModif       pmod;
   sos_ParamModif       pm;
   sos_Class_type       ct;
   sos_Bool             found = FALSE;
 
   parammodifs = get_change_create_params(_OBSThis);
 
   agg_iterate(parammodifs, pmod)
   {
      if (pmod.get_modified_param() == param)
      {  found = TRUE;
         pm    = pmod;
      }
   } agg_iterate_end(parammodifs, pmod);
 
 
   if (found)
   {  if (pm.get_change() == PROT_DEL)
      {  // param is deleted
         err_raise(err_SYS, err_PROT_PARAM_DEL, prot_ERR, FALSE);
      }
   }
   else
   {
      ct   = sos_Class_type::make(get_modified_type(_OBSThis));
      cnt  = _OBSThis.container();
      pmod = sos_ParamModif::create(cnt);
      pmod.set_modified_param(param);
      pmod.set_position      (ct.get_create_params().find(param));
      pmod.set_change        (PROT_MOD);
 
      pmod.set_old_name(sos_String::clone(param.get_name(), cnt));
      pmod.set_old_type      (param.get_type());
      pmod.set_old_default   (param.get_default_expr());
      pmod.set_old_ref       (param.get_is_ref());
 
      parammodifs.append(pmod);

      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::add_super(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Super_class) spcl)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::add_super");
   TT(prot_H, T_ENTER);

   sos_SuperClassModif_List supermodifs;  
   sos_SuperClassModif      spmod;
   sos_SuperClassModif      spm;
   sos_Super_class          super;
   sos_Bool                 found = FALSE;

   supermodifs = get_change_superclasses(_OBSThis);


   // search whether superclass has been touched before 
   agg_iterate(supermodifs, spmod)
   {
      super = spmod.get_super_class();
      if (!found &&
          (super == spcl  ||  super.make_type() == spcl.make_type()))
      {  // corresponding superclass is found 
         found      = TRUE;
         spm        = spmod;
      }
   } agg_iterate_end(supermodifs, spmod);


   // evaluate search
   if (found)   
   {  if (spm.get_change() == PROT_DEL)
      {  // superclass deleted: mark it as modified and insert new spcl-OID as 
         // new_super_class, just in case the OID of the superclass has changed
         spm.set_change(PROT_MOD);
         spm.set_new_super_class(spcl); 
      }
      else
      {  // superclass modified: impossible, error in MTA 
         // superclass added   : impossible, error in MTA
         err_raise(err_SYS, err_PROT_SUPERCLASS_ADD, prot_ERR, FALSE); 
      }
   }
   else
   {  // superclass not found: insert added superclass 
      spmod = sos_SuperClassModif::create(_OBSThis.container());
      spmod.set_super_class    (spcl);
      spmod.set_change         (PROT_ADD);
      spmod.set_new_super_class(sos_Super_class::make(NO_OBJECT));

      supermodifs.append(spmod);

      // if found true, the offsets have been saved before
      save_offsets(_OBSThis,TRUE);

      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::del_super(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Super_class) spcl)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::del_super");
   TT(prot_H, T_ENTER);

   sos_SuperClassModif_List supermodifs;
   sos_SuperClassModif      spmod;
   sos_SuperClassModif      spm;
   sos_Bool                 found  = FALSE;
   sos_Int                  index  = 0;
   sos_Int                  spm_index;
 
   supermodifs = get_change_superclasses(_OBSThis);
 

   // search whether superclass was touched before
   agg_iterate(supermodifs, spmod)      
   {
      index ++;
 
      if (!found) 
      {  if (   spmod.get_super_class()      == spcl
	     || spmod.get_new_super_class()  == spcl)
         {  found      = TRUE;
            spm        = spmod; 
            spm_index  = index;
         }
      }
   } agg_iterate_end(supermodifs, spmod);


 
   // evaluate search 
   if (found)
   {  if (spm.get_change() == PROT_ADD)
      {  // superclass has been added before: remove modif
         supermodifs.remove(spm_index);
         spm.destroy();
      }
      else if (spm.get_change() == PROT_MOD)
      {  // superclass has been deleted and added before: change modif
         spm.set_change(PROT_DEL);
         spm.set_new_super_class(sos_Super_class::make(NO_OBJECT));
      }
      else 
      {  // superclass has been deleted before
         err_raise(err_SYS, err_PROT_SUPERCLASS_DEL, prot_ERR, FALSE); 
      }
   } 
   else
   {  // superclass not found: insert deleted superclass    
      spmod = sos_SuperClassModif::create(_OBSThis.container()); 
      spmod.set_super_class    (spcl);
      spmod.set_change         (PROT_DEL);
      spmod.set_new_super_class(sos_Super_class::make(NO_OBJECT));
     
      supermodifs.append(spmod);
 
      // if found true, the offsets have been saved before
      save_offsets(_OBSThis,TRUE);
    
      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::add_friend(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type) fd) 
// *************************************************************************
const{
   T_PROC("sos_ClassModif::add_friend");
   TT(prot_H, T_ENTER);

   if (! fd.has_type(sos_Class_type_type))
   {  // friend is not a class
      err_raise(err_SYS, err_PROT_FRIEND_NO_CLASS, prot_ERR, FALSE);
   }

   if (get_deleted_friends(_OBSThis).is_element(fd))
      get_deleted_friends(_OBSThis).remove(fd);
   else
   {  get_added_friends(_OBSThis).insert(fd);
      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::del_friend(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Type) fd) 
// *************************************************************************
const{
   T_PROC("sos_ClassModif::del_friend");
   TT(prot_H, T_ENTER);

   if (!fd.has_type(sos_Class_type_type))
   {  // friend is not a class
      err_raise(err_SYS, err_PROT_FRIEND_NO_CLASS, prot_ERR, FALSE); 
   }
 
   if (get_added_friends(_OBSThis).is_element(fd)) 
      get_added_friends(_OBSThis).remove(fd);
   else
   {  get_deleted_friends(_OBSThis).insert(fd);
      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::add_comp(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Comp_descr) comp)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::add_comp");
   TT(prot_H, T_ENTER);

   sos_CompModif       cpad; 
   sos_Class_type      ct;

   // create new sos_CompModif and append it to change_comps
   // it is not checked, whether the same component was deleted or
   // modified before. This connection will be made after the version
   // is complete and the adaption functions are created
   // but: err_raise, if component was modified before 
   ct   = sos_Class_type::make(get_modified_type(_OBSThis)); 
   cpad = sos_CompModif::create(_OBSThis.container());
   cpad.set_modified_component(comp);
   cpad.set_change(PROT_ADD);

   get_change_comps(_OBSThis).append(cpad); 

   save_offsets(_OBSThis,FALSE);

   set_change(_OBSThis,PROT_MOD);

   TT(prot_H, T_LEAVE);
}   

// *************************************************************************
void _sos_ClassModif::del_comp(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Comp_descr) comp)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::del_comp");
   TT(prot_H, T_ENTER);

   sos_CompModif_List  compmodifs;
   sos_CompModif       cpmod;
   sos_CompModif       cpm;
   sos_Bool            found = FALSE;
   sos_Int             index = 0;
   sos_Int             cpm_pos;
  

   // check whether the component was added or modified during this version
   // if it was added, the sos_CompModif can be deleted from the list
   // if it was modified, change of that sos_CompModif is set to
   // PROT_DEL. By doing this, it is garantieed that the modification
   // executed before the deletion are not lost if the Komponet is
   // added again (undo)
   compmodifs = get_change_comps(_OBSThis);
 
   agg_iterate(compmodifs, cpmod) 
   {
      index ++;

      if (cpmod.get_modified_component() == comp)
      {  // remember this CompModif 
         found   = TRUE;
         cpm     = cpmod;       
         cpm_pos = index;
      }
   } agg_iterate_end(compmodifs, cpmod);


   if (found)
   {  if (cpm.get_change() == PROT_ADD)
      {  // component was added during this version
         compmodifs.remove(cpm_pos);
         cpm.destroy();
      }
      else if (cpm.get_change() == PROT_MOD)
      {  // component was modified during this version
         cpm.set_change(PROT_DEL);
         save_offsets(_OBSThis,TRUE);
      }
      else
      {  // component was deleted during this version
         err_raise(err_SYS, err_PROT_COMP_DEL, prot_ERR, FALSE);
      }
   }
   else
   {  // component was never touched during this version
      cpmod = sos_CompModif::create(_OBSThis.container());
      cpmod.set_modified_component(comp);
      cpmod.set_change(PROT_DEL);

      compmodifs.append(cpmod);
 
      save_offsets(_OBSThis,FALSE);
 
      set_change(_OBSThis,PROT_MOD);
   }      
   TT(prot_H, T_LEAVE);
}


// *************************************************************************
void _sos_ClassModif::mod_comp(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Comp_descr) comp)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::mod_comp");
   TT(prot_H, T_ENTER);

   sos_Container       cnt;
   sos_CompModif_List  compmodifs;
   sos_CompModif       cpmod;
   sos_CompModif       cpm;
   sos_Bool            found = FALSE;


   // check whether the component was touched before 
   // if it was added, the modification has not to be logged. If the addition
   // was an undo of a previous deletion, the old values have already 
   // been saved at deletion time.  
   // if it was modified, nothing has to be done 
   // if it was deleted, error in MTA 
   compmodifs = get_change_comps(_OBSThis);
 
   agg_iterate(compmodifs, cpmod)
   {
      if (cpmod.get_modified_component() == comp)
      {  // remember this CompModif
         found   = TRUE;
         cpm     = cpmod;
      }
   } agg_iterate_end(compmodifs, cpmod);
 
 
   if (found)
   {  if (cpm.get_change() != PROT_ADD &&
	  cpm.get_change() != PROT_MOD) 
      {  // component is deleted 
         err_raise(err_SYS, err_PROT_COMP_DEL, prot_ERR, FALSE);
      }
      else 
        save_offsets(_OBSThis,FALSE);
   }
   else
   {  // the component has neither been added, deleted nor modified during
      // this version. So log the modification

      cnt   = _OBSThis.container();
      cpmod = sos_CompModif::create(cnt);
      cpmod.set_modified_component(comp);
      cpmod.set_change(PROT_MOD);

      cpmod.set_old_name       (sos_String::clone(comp.get_name(), cnt));
      cpmod.set_old_type       (comp.get_get_method().get_result_type());
      cpmod.set_old_init       (comp.get_init_expr());
      cpmod.set_old_local      (comp.get_is_local());
      cpmod.set_old_static     (FALSE);  // <af>
      cpmod.set_old_value      (comp.get_is_value());
      cpmod.set_old_get_scope  (comp.get_get_method().get_kind());
      cpmod.set_old_set_scope  (comp.get_set_method().get_kind());
       
      compmodifs.append(cpmod);

      save_offsets(_OBSThis,FALSE);           

      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}
   
// *************************************************************************
void _sos_ClassModif::add_meth(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Method) meth)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::add_meth");
   TT(prot_H, T_ENTER);

   // it is assumed that the method was not modified before

   if (get_deleted_methods(_OBSThis).is_element(meth))
      get_deleted_methods(_OBSThis).remove(meth);
   else
   {  get_added_methods(_OBSThis).insert(meth);
      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
void _sos_ClassModif::del_meth(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Method) meth)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::del_meth");
   TT(prot_H, T_ENTER);

   if (get_added_methods(_OBSThis).is_element(meth))
      get_added_methods(_OBSThis).remove(meth);
   else
   {  // if the method was modified before, the modification log stays 
      // in modified_methods to allow an undo of the deletion
      // (not implemented yet)
      // but there is no connection established between deleted and 
      // added methods because it don't matter whether a method is
      // readded or not
      get_deleted_methods(_OBSThis).insert(meth);

      set_change(_OBSThis,PROT_MOD);
   }
   TT(prot_H, T_LEAVE);
}

// *************************************************************************
sos_MethodModif _sos_ClassModif::mod_meth(const sos_Typed_id&_OBSThis,OBST_PARDECL(sos_Method) meth)
// *************************************************************************
const{
   T_PROC("sos_ClassModif::mod_meth");
   TT(prot_H, T_ENTER);

   sos_Container        cnt = _OBSThis.container();
   sos_MethodModif_List mm_list;
   sos_MethodModif      mm  = sos_MethodModif::make(NO_OBJECT);
   sos_MethodModif      act_mm;

   // method is new or deleted, nothing has to be done
   if (get_added_methods(_OBSThis).is_element(meth) ||
       get_deleted_methods(_OBSThis).is_element(meth))
   {  TT(prot_H, T_LEAVE);
      return sos_MethodModif::make(NO_OBJECT);
   }

   // search whether method was modified before, nothing has to be done
   mm_list = get_modified_methods(_OBSThis);
   agg_iterate(mm_list, act_mm)
   {  if (act_mm.get_modified_method() == meth)  
         mm = act_mm;
   } agg_iterate_end(mm_list, act_mm);

   if (INVALID(mm))
   {
      // first modification of method in this version 
      mm = sos_MethodModif::create(cnt);
      mm.set_modified_method(meth);
      get_modified_methods(_OBSThis).append(mm);

      mm.set_old_name        (sos_String::clone(meth.get_name(), cnt));
      mm.set_old_kind        (meth.get_kind());
      mm.set_old_result_type (meth.get_result_type());
      mm.set_old_abstract    (meth.get_is_abstract());
      mm.set_old_definite    (meth.get_is_definite());
      mm.set_old_static      (meth.get_is_static()); 
 
      set_change(_OBSThis,PROT_MOD);
   } 

   TT(prot_H, T_LEAVE);
   return mm;
} 
    

// *************************************************************************
void _sos_CompModif::local_finalize(OBST_PARDECL(sos_CompModif) cpm) 
// *************************************************************************
{
   T_PROC("sos_CompModif::local_finalize");
   TT(prot_M, T_ENTER);

   if (VALID(cpm.get_old_name()))
      cpm.get_old_name().destroy();

   TT(prot_M, T_LEAVE);
} 
