 /* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// ********************************************************************
// msh_main
// ********************************************************************

// Shall I catch the error messages from OBST evolution methods, or
// shall I enable debugging with a kept error stack ?
#define MSH_DEBUG

#include "obst_err.h"
#include "msh_err.h"
#include "mta.h"
#include "msh.h"		  // includes msh_err.h

#define OBST_IMP_FILE
#define OBST_IMP_STRINGOP
#define OBST_IMP_STREAM
#define OBST_IMP_STDCONST
#include "obst_stdinc.h"

LOCAL  FILE*    old_stdin = stdin;	 	// old stream handle of stdin
LOCAL  sos_Bool msh_stdio = FALSE;       	// input from file or cin 
EXPORT sos_Bool msh_echo  = FALSE;       	// display command lines 
EXPORT sos_Schema_module_Directory msh_sd;	// Schema-Directory

EXPORT void msh_list_subclasses (const sos_Class_type &ct)
{
   sos_Class_type_List tl = ct.get_subclasses();
   agg_iterate (tl, sos_Class_type t)
      cout << sos_Schema_module::retrieve (t.container()).get_name() 
	   << ":"
	   << MTA_GET_NAME_DECLARATION(t) << "\n";
   agg_iterate_end (tl, t)
}

EXPORT void msh_list_schemas()
{
   sos_Schema_module_Directory schema_dir = sos_Schema_module::schema_dir();
   agg_iterate_association (schema_dir, 
			    sos_String name, sos_Schema_module sm)
      cout << name << " ";
   agg_iterate_association_end (schema_dir, name, sm)
   cout << "\n";cout.flush();
}

EXPORT sos_Method msh_lookup_method (const sos_Class_type& ct,
				     const sos_String&     name,
				     sos_Int		   num_par /* = -1 */)
{  
   if (NOT ct.get_methods().is_key(name))
      msh_error (err_USE, err_MSH_INVALID_METHOD_NAME);
   else
   {
      sos_Method_List ml = 
	 sos_Method_List::copy(ct.get_methods()[name], TEMP_CONTAINER);

	 // remove methods that are not defined in ct 
      for (sos_Cursor c = ml.open_cursor();  ml.is_valid(c); )
      {  if (ml.get(c).get_defined_in() != ct)
	    ml.remove_at (c);
	 else
	    ml.to_succ(c);
      }

      if (ml.card() == 0)
	 msh_error (err_USE, err_MSH_INVALID_METHOD_NAME);
      else if (ml.card() == 1)
      {  sos_Method m = ml.get_nth(1);
	 ml.destroy();
	 return m;
      }
      else if (num_par == -1)
	 msh_error (err_USE, err_MSH_OVERLOADED_METHOD);
      else
      {  agg_iterate (ml, sos_Method m)
         {  if (m.get_params().card() == num_par)
	    {  ml.close_cursor (agg_current_cursor());
	       ml.destroy();
	       return m;
	    }
	 }
	 agg_iterate_end (ml, m)
	 msh_error (err_USE, err_MSH_OVERLOADED_NOT_FOUND);
      }
      ml.destroy();
   }
   return sos_Method::make (NO_OBJECT); // suppress warning 
} // ** msh_lookup_method ***

EXPORT void help_schema()
{
   cout << "Schema Commands:'schema' followed by:\n"
	<< "   .print\n"
	<< "   .create([(""name"")]\n"
	<< "   .remove((""name""[,RECURSIVE])\n\n"
	<< "'schema.(""name"")' followed by:\n"
	<< "   .name.set(""newname"")\n"
	<< "   .with_extern.[re]set\n"
	<< "   .generate[(""language"")]\n"
	<< "   .import.insert([(""import"")]\n"
	<< "   .import.remove([(""name"")]\n"
	<< "   .import.update([(""name"")]\n"
	<< "   .print\n";
}

EXPORT void help_type()
{
   cout << "General Type Commands: enum, union, class, extern, typedef\n"
	<< "'schema(""name"").enum/union/class/extern/typedef' followed by:\n"
	<< "   .create[(""name"")]\n"
	<< "   .remove(""name"")\n"
	<< "   (""name"").name.set(""name"")\n"
	<< "   (""name"").print\n";
}

EXPORT void help_extern()
{
   cout << "Extern Type Commands:'schema(""name"").extern(""name"")' followed by:\n"
	<< "   .size.set(<no>)\n";
}

EXPORT void help_typedef()
{
   cout << "Typedef Commands:'schema(""name"").typedef(""name"")' followed by:\n"
	<< "   .type.set(""name"")\n";
}

EXPORT void help_enum()
{
   cout << "Enumeration Type Commands:'schema(""name"").enum(""name"")' followed by:\n"
	<< "   .literal.insert(""name"")\n"
	<< "   .literal.remove(""name"")\n";
}

EXPORT void help_class()
{
   cout << "Class Type Commands:'schema(""name"").class(""name"")' followed by ...\n"
	<< "   .subclass.print\n"
	<< "   .friend.remove (""Name"")\n"
	<< "   .friend.insert (""Name"")\n"
	<< "   .super.insert (""Name""[<genpar1,genpar2..>])\n"
	<< "   .super(""Name"")....\n"
	<< "   .comp.create(""Name""[,""Type""][,<Init>][,local]\n"
	<< "                [,<getscope>],[<setscope>])\n"
	<< "   .method.create(""Name""[,""Type""][,<scope>])\n"
	<< "   .instantiation.create(""schema"",<genpar1,genpar2..>)\n";
}

EXPORT void help_method()
{
   cout << "Method Commands:'...class(""name"").method' followed by\n"
	<< "   .create(<pos>,""name""[,""type""][,<init>]\n"
	<< "           [,public/protected/private][,static][,abstract][,definite])\n"
	<< "   .remove(""name"")\n"
	<< "   (""name""[,ParamNo]).static.set/reset\n"
	<< "   (""name""[,ParamNo]).abstract.set/reset\n"
	<< "   (""name""[,ParamNo]).definite.set/reset\n"
	<< "   (""name""[,ParamNo]).name.set(""name"")\n"
	<< "   (""name""[,ParamNo]).result.set(""name"")\n"
	<< "   (""name""[,ParamNo]).scope.set(public/private/protected)\n"
	<< "Method param Commands:'...method(""name""[,<ParamNo>].param' followed by:\n"
	<< "   .create(<Pos>,""name""[,""typename""][,""default-expr""]\n"
	<< "   .remove(<Pos>)\n"
	<< "   (<pos>).modify(""name""[,""typename""][,""default-expr""]\n\n";
}

EXPORT void help_comp()
{
   cout << "Component Commands:'schema(""name"").class(""name"").comp' followed by:\n"
	<< "   .create(""name""[,""result""]\n"
	<< "       [public/protected/private][public/protected/private]\n"
	<< "       [,static][,abstract][,definite])\n"
	<< "   .remove(""name"")\n"
	<< "   (""name"").static.[re]set\n"
	<< "   (""name"").abstract.[re]set\n"
	<< "   (""name"").definite.[re]set\n"
	<< "   (""name"").name.set(""name"")\n"
	<< "   (""name"").result.set(""name"")\n"
	<< "   (""name"").local.[re]set\n"
	<< "   (""name"").init.[re]set\n"
	<< "   (""name"").scope.set(public/private/protected\n"
	<< "       [,public/private/protected])\n";
}

EXPORT void msh_help ()
{  
   cout << "Commands:\n"
	<< "   commit, reset, exit, help\n"
	<< "   help.{schema,type,enum,typedef,extern,class,method,comp\n";

   help_schema();
   help_type();
   help_enum();
   help_typedef();
   help_extern();
   help_class();
   help_method();
   help_comp();
}

#ifdef OBST_EVOLUTION_WITH_GENERIC
// The following code may be useful, if schemaevolution is extended to handle
// generic classes. It is not used in the current msh-implementation.

// {schema("").class("").generic.param}.insert("Name"[,<Pos>],["Type])
// {schema("").class("").generic.param}.remove("Name")
// {schema("").class("").generic.param}("Name").type.set ("Name")
// {schema("").class("").generic.param}("Name").name.set ("Name")
LOCAL void parse_generic_param_op (sos_Schema_module sm, sos_Class_type ct)
{  if (parse_keyword (".insert"))
   {  parse_parenthesis();
      sos_String     name = parse_name();
      sos_Type_descr td   = sos_Type_descr::make (sos_Object_type);
      sos_Int	     pos  = INVALID(ct.get_generic_class())
	 			? 1
	 			: ct.get_formal_gen_params().card()+1;

      while (parse_comma (FALSE))
      {  int i = parse_number (FALSE);
	 if (i != NO_NUMBER)
	    pos = i;
         else
	 {  sos_Type_descr t = parse_type (sm, FALSE);
	    if (VALID(t))
	       td = t;
            else
	       err ("<Pos> or \"type-name\" expected");
	 }
      }
      parse_parenthesis();
      parse_end();
      // mta_insert_generic_param (ct, name, pos);
      if (VALID(td))
         ; // mta_modify_generic_param (ct, pos, name, td);
   }
   else
   if (parse_keyword (".remove"))
   {  sos_Int pos = parse_generic_param(ct);
      parse_end();
      // mta_remove_generic_param (ct, pos);
   }
   else
   {  sos_Int pos = parse_generic_param(ct,TRUE,
			     "insert, remove or (\"Gen.par.name\") expected");
      parse_dot();
      sos_Gen_param  gp   = ct.get_formal_gen_params().get_nth(pos);
      sos_String     name = gp.get_name();
      sos_Type_descr td   = gp.get_super_type();
      if (parse_keyword ("type.set"))
      {  parse_parenthesis();
	 td = parse_type (sm);
         parse_parenthesis();
         parse_end();
      }
      else
      if (parse_keyword ("name.set"))
      {  parse_parenthesis();
	 name = parse_name();
         parse_parenthesis();
         parse_end();
      }
      else
	 invalid_keyword ("type.set or name.set expected");
      // mta_modify_generic_param (ct, pos,name,td);
   }
} // *** parse_generic_param_op ***

// {schema("").class("").generic}.param
void parse_generic_op (sos_Schema_module sm, sos_Class_type ct)
{  parse_dot();
   if (parse_keyword ("param"))
      parse_generic_param_op(sm, ct);
   else
      invalid_keyword ("param expected");
} // ** parse_generic_op ***

//    ...class("name").instantiation:

      }
      else
      if (parse_keyword ("instantiation"))
      {  parse_dot (TRUE);
         parse_keyword ("create",TRUE);
         parse_parenthesis();
	 sos_String	   schema_name = parse_name (TRUE); 
	 sos_Schema_module createsm    = sos_Schema_module::lookup(schema_name);
	 if (INVALID (createsm))
	    err("invalid schema name");
         else
	 {
	    sos_Type_descr_List agpl = sos_Type_descr_List::create(
	    							TEMP_CONTAINER);
	    sos_Gen_param_List  fgpl = ct.get_formal_gen_params()
            agg_iterate (fgpl, sos_Gen_param gp)
	       parse_comma (TRUE);
	       sos_Type_descr agp = parse_act_gen_param (sm, ct);
	       agpl.append (agp);
            agg_iterate_end (fgpl, gp)		
            parse_parenthesis();
	    parse_end();

	    ct.lookup_or_create_instantiation (sm, agpl, TRUE);
         }
      }
#endif

EXPORT void msh_wait_key()
{
   char s[100];

   if (!msh_stdio)
   {
      cout << "\nHit ENTER to continue..."; cout.flush();
      // Now read a newline from the old stdin, which should be a terminal!
      fgets (s, 100, old_stdin);
   }
}

EXPORT void msh_prompt()
{  
   if (msh_stdio)
      cout << "msh> "; cout.flush();
}

LOCAL void parse_options (sos_Cstring opt)
{  for (;  opt;  ++ opt)
      if (*opt == 'x')
      {  msh_echo = TRUE;
	 break;
      }
} // *** parse_options ***

LOCAL inline void display_options()
{  cout << "msh [-x] [filename]\n";
}

LOCAL void parse_options (int argc, char* argv[])
{  
   char *filename = NULL;
   
   if (argc == 1)
      msh_stdio = TRUE;
   else
   if (argc == 2)
   {  filename = argv[1];
      if (filename[0] == '-')
      {  msh_stdio = TRUE;
	 parse_options(argv[1]);
      }
   }
   else
   if (argc == 3)
   {  sos_Cstring p = argv[1];
      if (p[0] != '-')
      {  cerr << "invalid options";
	 display_options();
      }
      else
	 parse_options (p);
      filename = argv[2];
   }
   else
      display_options();

   if (filename)  // filename was provided, then associate it with stdin:
   {		  // But: save old stdin to new handle to be able to read
		  // from the keyboard
      int old_stdin_handle = dup (0);
      old_stdin		   = fdopen (old_stdin_handle, "r");
      int handle	   = open (filename, O_RDONLY);
      if (handle==-1)	// file could not be opened
	 msh_error (err_SYS, err_MSH_FILE_NOT_FOUND, filename);
      dup2 (handle, 0);
   }
} // *** parse_options *** 

EXPORT void msh_error (err_class ec, err_msg s, const sos_String& where)
{
   char *w = where.make_Cstring();
   msh_error (ec, s, w);
   delete w;
}

EXPORT void msh_error (err_class ec, err_msg s, char* where /* = 0 */)
{
   smg_String line_msg = smg_String ("line ") + msh_yylineno;

   if (where)
      line_msg += smg_String(": ") + where;

   err_raise (ec, s, line_msg.make_Cstring (SMG_BORROW));
}

EXPORT int msh_main(int argc,char *argv[])
{  T_FILE ("msh.out");
   T_REDEF(NULL);

   parse_options (argc, argv);
   msh_sd = sos_Schema_module::schema_dir();

   mta_open_status (WRITING);
   msh_prompt(); // show initial prompt if reading from stdin
   msh_yyparse();

   return 0;
} // *** main ***
