/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "mta.h"


EXPORT void mta_enter_object_as_sc (sos_Class_type	 ct,
				    sos_Super_class_List scl)
{
   T_PROC ("mta_enter_object_as_sc")
   TT (mta_H, T_ENTER);
 
   if (ct.operator!=(mta_get_object_type()))
   {  sos_Super_class sc = sos_Super_class::create(ct.container());

      sc.set_super_class  (sos_Class_type::make(mta_get_object_type()));
      sc.set_create_params(sos_Expr_List::make(NO_OBJECT));
      sc.set_is_direct	  (TRUE);
      sc.set_offset	  (0);

      scl.append (sc);
   }
 
   TT (mta_H, T_LEAVE);
}


EXPORT sos_Super_class mta_super_class_in_list(const sos_Super_class_List& scl,
					       const sos_Type_descr&	   tp) 
{
   // If the entry looked for is contained in the given superclass list scl,
   // then it is returned. Otherwise NO_OBJECT is the result.

   T_PROC ("mta_super_class_in_list");
   TT (mta_H, T_ENTER);

   sos_Super_class result = sos_Super_class::make(NO_OBJECT);

   if (VALID(scl))
      agg_iterate (scl, sos_Super_class sc)
      {
	 if (sc.get_super_class().operator==(tp))
	 {  result = sc;
	    break;
	 }
      }
      agg_iterate_end (scl, sc);

   TT (mta_H, T_LEAVE);
   return result;
}

LOCAL sos_Expr_List mta_formal_params_to_actuals (sos_Container		cnt, 
						  const sos_Param_List& fpl)
{
   T_PROC ("mta_formal_params_to_actuals");
   TT (mta_H, T_ENTER);

   sos_Expr_List result;

   if (INVALID(fpl))
      result = sos_Expr_List::make (NO_OBJECT);
   else
   {  result = sos_Expr_List::create (cnt);
      agg_iterate (fpl, sos_Param fp)
      {  sos_Identifier id = sos_Identifier::create (cnt, fp.get_name());
	 result.append (id);
      }
      agg_iterate_end (fpl, fp);
   }

   TT (mta_H, T_LEAVE);
   return result;
} 


LOCAL sos_Expr_List mta_expand_params (sos_Container	     cnt,
				       const sos_Expr_List&  original,
				       const sos_Param_List& fpl,
				       const sos_Expr_List&  apl)
// class A (f1, ..., fm) : ... C (o1, ..., on) ...
// class B : ... A (a1, ..., am) ...
// result : subst fi through ai in oi
{
   T_PROC ("expand_params");
   TT (mta_H, T_ENTER);

   sos_Expr_List result;

   if (INVALID(original) OR INVALID(fpl))
      result = original;
   else
   {  result = sos_Expr_List::create (cnt);
      agg_iterate (original, sos_Expr expr)
      {  sos_Expr subst = expr;
	 if (expr.has_type (sos_Identifier_type))
	 {  sos_String id = sos_Identifier::make (expr).get_id();
	    sos_Bool   av = VALID(apl);
	    sos_Cursor c;
	    if (av)
	    {  c  = apl.open_cursor();
	       av = apl.is_valid (c);
	    }

	    agg_iterate (fpl, sos_Param fp)
	    {  sos_String s = fp.get_name();
	       if (VALID(s) AND s.equal (id))
	       {  if (av)
		     subst = apl.get(c);
		  else
		     subst = fp.get_default_expr();
		  break;
	       }
	       if (av)
		  av = apl.to_succ (c);
	    }
	    agg_iterate_end (fpl, fp);
	    if (VALID(apl))
	       apl.close_cursor (c);
	 }
	 result.append (subst);
      }
      agg_iterate_end (original, expr);
   }

   TT (mta_H, T_LEAVE);
   return result;
}


LOCAL sos_Super_class mta_append_super_class (const sos_Super_class_List& scl,
					      const sos_Type_descr&	  ct,
					      const sos_Expr_List&	  cpl,
					      const sos_Bool		  isd)
{  T_PROC ("mta_append_super_class")
   TT (mta_H, T_ENTER);

   sos_Container   cnt = scl.container();
   sos_Super_class sc  = sos_Super_class::create (cnt);

   sc.set_super_class  (ct);
   sc.set_create_params(cpl);
   sc.set_is_direct    (isd);

   scl.append (sc);

   TT (mta_H, T_LEAVE);
   return sc;
}


LOCAL sos_Expr_List mta_get_actual_create_params(const sos_Super_class& which,
						 const sos_Super_class& context)
{
   /* This function determines from a given sos_Super_class object the
      actual create parameter list for the derived class context.
      Therefore the class type is extracted from <which> and searched for
      in the super_closure of context. The actual corresponding create parameter
      list of context's super_closure is returned. */

   T_PROC ("mta_get_actual_create_params");
   TT (mta_H, T_ENTER);

   sos_Expr_List result;

   sos_Class_type  which_class = sos_Class_type::make (which.make_type());
   sos_Super_class context_sc  = MTA_SUPER_CLASS_IN_LIST (
       sos_Class_type::make(context.make_type()).get_super_closure(),
       which_class);

   if (INVALID(context_sc))
   {  smg_String class_name = which_class.get_name();
      err_raise (err_SYS, err_MTA_INTERNAL,
			  class_name.make_Cstring (SMG_BORROW));
   }
   else
      result = context_sc.get_create_params();

   TT (mta_H, T_LEAVE);
   return result;
}


LOCAL void mta_correct_sc_order (sos_Class_type ct)
{
   /* This function establishes an order in the super_classes list of ct.
      The criteria is like in the super_closure list the class hierarchy. */

   T_PROC ("mta_correct_sc_order");
   TT (mta_H, T_ENTER);

   sos_Bool		order_already_correct = TRUE;
   sos_Int		max_index = 0;
   sos_Super_class_List sclo = ct.get_super_closure();
   sos_Super_class_List scla = ct.get_super_classes();

   // The following mapping associates the super_closure indices with the
   // corresponding sos_Super_class objects from the super_classes list.
   sos_Object_sos_Object_Mapping pos_to_sc =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);

   agg_iterate (scla, sos_Super_class sc)
   {
      /* Now find class beloging to sc in the super_closure. Class is found,
	 since superclosure contains at least all classes contained in the
	 super_classes list. */
      sos_Int i = 0; // index of class sc within super_closure

      agg_iterate (sclo, sos_Super_class s2)
      {
	 i++;
	 if (s2.get_super_class().make_type()==sc.get_super_class().make_type())
	    break;
      }
      agg_iterate_end (sclo, s2);
      // order is correct, if indices are in increasing order
      if (i < max_index)
	 order_already_correct = FALSE;
      max_index = i;
      pos_to_sc.insert (make_sos_Int_object(i), sc);
   }
   agg_iterate_end (scla, sc);

   if (!order_already_correct) // order needs to be corrected
   {
      // Now adapt positions of list elements from the gained knowledge:
      scla.clear(); // first remove all elements from the list.
      for (sos_Int i = sclo.card();  i > 0;  i--)
      {
	 sos_Super_class class_to_append = 
	    sos_Super_class::make(pos_to_sc [make_sos_Int_object(i)]);

	 /* NOTE: appending to scla can cause trouble if scla is not located
	    in cnt. This could happen, if the super_classes list was
	    just copied when doing a generic instantiation. But in this case it
	    can be proposed, that this list must have been in the correct
	    order, because the schema containing the super_classes list has
	    been compiled already. So no write access outside cnt
	    will occur. */

	 if (VALID(class_to_append))  // sc with that index was contained
	    scla.append (class_to_append);  // in the super_classes list:
      }
   }
   pos_to_sc.destroy();   // destroy temporary mapping
  
   TT (mta_H, T_LEAVE);
}

EXPORT void mta_set_super_closure (sos_Class_type ct)
   // compute super closure in prefix order using the already
   // computed super closure of the super classes of ct.
   // (i.e. a superclass appears always before any of its subclasses
   //       in the Super_class_List)
{
   T_PROC ("mta_set_super_closure");
   TT (mta_H, T_ENTER);

   /* Three temporary mappings exist:
      - sc2direct maps each object from the new constructed superclosure
	to its corresponding direct superclass of the class, for which the
	superclosure is constructed. This info yields the class,
	from which the create-parameters must be taken for initialization.
      - scl_lookup is intended to speed up the search for an element in the
	new constructed superclosure. Key is an sos_Class_type, info is
	the associated sos_Super_class element contained in the new
	superclosure, that has the sos_Class_type as its super_class entity.
      - acp_lists contains for each element of the new superclosure the
	actual create parameter list, which must be expanded later.
	E.g. if we have class C (c) {...};
			class B (b) : C (b) {...};
			class A : B (x);
        then if we construct A's superclosure, for the class C the actual
	parameter list (b) is entered, since this was the initialization
	yielded by B. Note, that this association may be changed during the
	process: If we have instead class A : B (x), C (y);
	then we must change C's actual parameter list from (b) to (y), as
	well as we have to change the direct association from B to A.        */

   sos_Object_sos_Object_Mapping sc2direct = 
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Object_sos_Object_Mapping scl_lookup =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);
   sos_Object_sos_Object_Mapping acp_lists =
	sos_Object_sos_Object_Mapping::create (TEMP_CONTAINER);

   sos_Container	cnt	= ct.container();
   sos_Param_List       fcp     = ct.get_create_params ();
   sos_Super_class_List scl     = ct.get_super_classes ();
   sos_Super_class_List new_scl = sos_Super_class_List::create (cnt, FALSE);

   agg_iterate (scl, sos_Super_class sc)
   {
      sos_Class_type	   ct1     = sos_Class_type::make(sc.make_type());
      sos_Super_class      prev_sc = sos_Super_class::make(scl_lookup [ct1]);
      sos_Super_class_List scl1    = ct1.get_super_closure();
      if (INVALID(prev_sc))
      {
	 agg_iterate (scl1, sos_Super_class sc2)
	 {
	    sos_Class_type ct2 = sos_Class_type::make(sc2.make_type());
	    sos_Super_class class_in_list = sos_Super_class::make(
							scl_lookup [ct2]);
	    if (INVALID(class_in_list))
	    {				 // ATT2.0 QUEERNESS: explicit operator
	       sos_Bool compare_result = (ct2.operator== (ct1));
	       sos_Super_class inserted_sc =
			MTA_APPEND_SUPER_CLASS (new_scl, ct2,
						sos_Expr_List::make(NO_OBJECT),
						compare_result);
				   // is direct iff it is the supplied sup-cl.
	       scl_lookup.insert (ct2, inserted_sc); // insert in speedup map
					     // save actual create parameters
                  // Error in Sun CC Compiler:
		  // The code doesn't work with create_params passed directly 
		  // without a temporary variable. Thanks to [matsj]
               sos_Expr_List acp = sc2.get_create_params(); 
	       acp_lists.insert (inserted_sc, acp);
	       sc2direct.insert (inserted_sc, sc); // Associate the class to
					 // append with the direct superclass
	    }
	    else// class already in superclosure, but direct-state may change:
	        // if the found class is marked to be direct, it must be marked
		// to be indirect, since it is not ct1 (ct1 is asserted to be
		// not in the superclosure at this point by the check
		// prev_sc == NO_OBJECT).
	       class_in_list.set_is_direct (FALSE);
	 }
	 agg_iterate_end (scl1, sc2);
      }
      else // superclass was previously inserted in the superclosure
      {
	 if (NOT prev_sc.get_is_direct())
	 /* the superclass was not direct and only the direct-flag
	    in the super_classes list needs to be changed, since indirect
	    'overrules' direct.
	    NOTE: aviod access outside container through previous check  */
	    if (sc.get_is_direct())
	       sc.set_is_direct (FALSE);

	 sos_Super_class associated = sos_Super_class::make(sc2direct[prev_sc]);
	 // modify the associations of the superclasses from the constructed
	 // superclosure, which are contained in the superclosure of sc and
	 // are initialized by 'associated'
	 agg_iterate (scl1, sos_Super_class ass_sc)
	 {
	    sos_Class_type ass_ct = sos_Class_type::make (ass_sc.make_type());
	    sos_Super_class ass_sc_from_new_scl = 
		sos_Super_class::make(scl_lookup [ass_ct]);
	    if (sos_Super_class::make(sc2direct [ass_sc_from_new_scl])
		.operator== (associated))
	    {   // ATT2.0 QUEERNESS: explicit operator

	       // Associate it with direct sc.
	       sc2direct.insert (ass_sc_from_new_scl, sc);
	       // Determine the actual create parameters of ass_sc in the
	       // context of the direct superclass sc.
	       sos_Expr_List acp = mta_get_actual_create_params (
					ass_sc_from_new_scl, sc);
	       acp_lists.insert (ass_sc_from_new_scl, acp);
	    }
         } agg_iterate_end (scl1, ass_sc);
      }
   }
   agg_iterate_end (scl, sc);

   // NOTE: the following call to formal_params_to_actuals yields
   // NO_OBJECT in case the formal create parameter list was also NO_OBJECT!
   MTA_APPEND_SUPER_CLASS (new_scl, ct,
			   mta_formal_params_to_actuals (cnt,fcp),
			   FALSE);
			      // the super_closure contains the class itself!

   /* Now that the super_closure is constructed with the order already
      being correct, the create parameters can be expanded using the
      created mapping sc2direct, in which for each class of the constructed
      superclosure the association with the direct superclass is registered. */
   agg_iterate (new_scl, sos_Super_class sc)
   {			 // ATT2.0 QUEERNESS: explicit operator
      if (sc.make_type().operator!= (ct)) // Don't process current class
      {  // Parameters not yet expanded (is the case for all classes but
	 // the class itself (for which the superclosure is constructed),
	 // since for that the call to formal_params_to_actuals filled them in)
	 sos_Super_class assoc_direct = sos_Super_class::make(sc2direct [sc]);
	 sos_Param_List fcp1 = sos_Class_type::make (
				  assoc_direct.make_type()).get_create_params();
	 sos_Expr_List  acp1 = assoc_direct.get_create_params();
	 sos_Expr_List  acp2 = sos_Expr_List::make(acp_lists [sc]);

	 sc.set_create_params (mta_expand_params (cnt, acp2, fcp1, acp1));
      }
   } agg_iterate_end (new_scl, sc);

   ct.set_super_closure (new_scl);
   mta_correct_sc_order (ct); // use superclosure order to sort superclass list

   TT (mta_H, T_LEAVE);
} // set_super-closure


LOCAL sos_Class_type mta_class (const sos_Super_class& sc)
{  // Yields the sos_Class_type belonging to an sos_Super_class object
   T_PROC ("mta_class");
   TT (mta_H, T_ENTER);

   sos_Type_descr td = sc.get_super_class();
   for (;;)
   {  if (td.has_type (sos_Class_type_type))
         break;
      else
      if (td.has_type (sos_Generic_instantiation_type))
         td = sos_Generic_instantiation::make (td).get_instantiation();
      else
	 mta_internal_error ("mta_class");
   }

   TT (mta_H, T_LEAVE);
   return sos_Class_type::make (td);
} // ** mta_class ***

LOCAL sos_Expr_List mta_default_acpl (const sos_Class_type& ct,
				      const sos_Container&  cnt)
{
   // constructs an actual create parameter list conforming the formal
   // create parameter list of class ct. The list is constructed into the
   // container cnt.

   T_PROC ("mta_default_acpl");
   TT (mta_L, T_ENTER);

   sos_Expr_List result = sos_Expr_List::make (NO_OBJECT);

   sos_Param_List fcpl = ct.get_create_params();
   if (VALID (fcpl))
   {
      sos_Bool already_one_non_default = FALSE;
      agg_iterate_reverse (fcpl, sos_Param p)
	 if (INVALID(p.get_default_expr()) AND NOT already_one_non_default)
	 {
	    already_one_non_default = TRUE;
	    result = sos_Expr_List::create (cnt);
	 }
	 if (already_one_non_default) // don't create actual values for the
				      // trailing default parameters
	    result.insert (1, mta_get_default_from_param (p, cnt));
      agg_iterate_reverse_end (fcpl, p)
   }

   TT (mta_L, T_LEAVE);
   return result;
}

// *************************************************************************
void sos_Class_type::insert_superclass (sos_Type_descr super)
// *************************************************************************
// insert a new superclass in the list of superclasses. If the superclass
// is a generic class then pass the actual (will be copied) generic params.
// Don't pass a generic instantiation, but generic class itself
{  sos_Class_type ct = self;
   sos_Container cnt = ct.container();
#ifdef OBST_HAVE_PROT
   sos_ClassModif cmod = sos_ClassModif::make (MTA_GET_TYPE_MOD(cnt, ct));
#endif

   if (    NOT super.has_type (sos_Generic_instantiation_type)
       AND NOT super.has_type (sos_Class_type_type))
      mta_error (err_SYS, err_MTA_CLASS_EXPECTED);

   if (VALID(ct.lookup_superclass (super)))
   {
      // Don't raise an error, do just nothing
      // mta_error (err_SYS, err_MTA_SUPERCLASS_EXISTS);
      return;
   }

   if (VALID(ct.lookup_superclass (ct)))
      mta_error (err_SYS, err_MTA_CIRCLED_DECL);

   if (ct.is_instantiation())
      mta_error (err_SYS, err_MTA_DONT_MODIFY_INSTANTIATION);

   if (ct.is_generic_class())
      mta_error (err_SYS, err_NOT_IMPLEMENTED,
		 "demanded modification of generic class");

   mta_open_for_writing (cnt);
   if (super.has_type (sos_Class_type_type))
   {  sos_Class_type superclass = sos_Class_type::make (super);
      if (superclass.is_generic_class())
	 mta_error (err_SYS, err_MTA_INVALID_SUPERCLASS);

	 // All checks are done, now start modifications
      sos_Super_class sc = sos_Super_class::create (cnt);

      sc.set_create_params (mta_default_acpl (superclass, cnt));
      sc.set_super_class (superclass);
      sc.set_is_direct (TRUE);
#ifdef OBST_HAVE_PROT
      if (VALID(cmod)) // generate protocol information only if needed.
	 cmod.add_super (sc);
#endif

      sos_Super_class_List ct_superclasses = ct.get_super_classes();
      ct_superclasses.append (sc);

         // flat classes that inherit just from sos_Object have this class
         // in the superclass list. Classes in an hierarchy don't, so remove
         // eventually this superclass. This case is indicated by the
         // cardinality of the super_closure, which contains sos_Object
         // and the class itself
      sos_Super_class sos_Object_super = 
         ct.lookup_superclass (sos_Type_descr::make (sos_Object_type));
      if (VALID (sos_Object_super))
         ct_superclasses.remove (ct_superclasses.find (sos_Object_super));

      err_block
         mta_build_class (ct, TRUE, TRUE, TRUE, TRUE);
      err_exception
	 ct_superclasses.remove (ct_superclasses.find (sc));
	 if (ct_superclasses.card() == 0)
	    ct_superclasses.append (sos_Object_super);
#ifdef OBST_HAVE_PROT
	 sos_ModifHistory::trash(sc);
#endif
	 mta_repeat_last_error();
      err_block_end

#ifdef OBST_HAVE_PROT
      if (VALID (sos_Object_super))
         sos_ModifHistory::trash(sos_Object_super);
#endif

      // Now all checks are successful. The following operations
      // assume that there is no error.

      // NOTE: sos_Class_type::get_subclasses for generic classes or
      // most general instantiations returns all instantiations of this
      // generic class.
      sos_Class_type_List subclasses = ct.get_subclasses();
      agg_iterate (subclasses, sos_Class_type sub)
         assert_no_error_block("insert_superclass")
            mta_set_super_closure (sub);
            mta_set_offsets_and_size (sub);
            mta_complete_components (sub);
         assert_no_error_block_end
      agg_iterate_end (subclasses, sub)
   }
   else
      mta_error (err_SYS, err_NOT_IMPLEMENTED,
		 "insertion of non-sos_Class_type superclass");

      // the hierarchy has changed: This would cause
      // invalid results of the cache used in 
      // sos_Class_type::get_subclasses.
      // remove all concerned classes from the cache
   sos_Super_class_List scl = ct.get_super_closure();
   agg_iterate (scl, sos_Super_class sc)
      mta_reset_subclass_cache (mta_class (sc));
   agg_iterate_end (scl, sc)

} // *** sos_Class_type::insert_superclass ***

// **************************************************************************
void sos_Class_type::remove_superclass(sos_Type_descr super)
// **************************************************************************
// remove a class from the superclass list. If an generic instantiation 
// should removed, pass the corresponding generic class
{  sos_Class_type ct  = self;
   sos_Container  cnt = ct.container();
#ifdef OBST_HAVE_PROT
   sos_ClassModif cmod = sos_ClassModif::make (MTA_GET_TYPE_MOD (cnt, ct));
#endif

   if (    NOT super.has_type (sos_Generic_instantiation_type)
       AND NOT super.has_type (sos_Class_type_type))
      mta_error (err_SYS, err_MTA_CLASS_EXPECTED);

   sos_Class_type superclass = sos_Class_type::make (super.make_type());

   if (ct.is_instantiation())
      mta_error (err_SYS, err_MTA_DONT_MODIFY_INSTANTIATION);
 
   if (ct.is_generic_class())
      mta_error (err_SYS, err_NOT_IMPLEMENTED,
                 "demanded modification of generic class");

   mta_open_for_writing (cnt);
   sos_Super_class_List scl = ct.get_super_classes();
   sos_Super_class	sc  = ct.lookup_superclass (superclass);
   if (INVALID(sc))
      mta_error (err_SYS, err_MTA_NOT_FOUND);
 
#ifdef OBST_HAVE_PROT
   if (VALID(cmod)) // generate protocol information only if needed
      cmod.del_super (sc);
#endif
   sos_Class_type_List subclasses = ct.get_subclasses();
   agg_iterate (subclasses, sos_Class_type sub)
      mta_open_for_writing (sub.container());

      sos_Super_class_List sub_scl= sub.get_super_classes();
	 // If ct was generic, remove the superclass from all 
	 // instantiations. If ct is non-generic, remove the 
	 // super class from ct only
      if (   (VALID(ct.get_generic_class()) AND  sub.root() == ct.root())
	  OR sub == ct)
      {     // sub is a class who serves the condition above
	    // look for the sos_Super_class and remove it from 
	    // sub.get_super_classes()
	    // NOTE (must be fixed!): This won't work, if ct is generic
	    // and having subclasses different from instantiations of itself.
	 sos_Super_class sub_sc = sub.lookup_superclass (superclass);
	 if (INVALID (sub_sc))
	    mta_internal_error("remove_superclass");
	 sub_scl.remove (sub_scl.find (sub_sc));

	 sos_Type_descr sc_class = sc.get_super_class();
	 if (sc_class.has_type (sos_Generic_instantiation_type))
	    mta_destroy (sos_Generic_instantiation::make (sc_class));
      }
	 // if the removed super class is the last one,
	 // insert sos_Object as sole super class
      if (sub_scl.card() == 0)
	 mta_enter_object_as_sc (sub, sub_scl);
      
      assert_no_error_block("remove_superclass")
	 mta_set_super_closure (sub);
	 mta_build_methodtable(sub);
	 mta_complete_components (sub);
	 mta_set_offsets_and_size (sub);
      assert_no_error_block_end
   agg_iterate_end (subclasses, sub)

   mta_reset_subclass_cache (superclass);
} // *** sos_Class_type::remove_superclass ***

// *************************************************************************
sos_Super_class sos_Class_type::lookup_superclass (sos_Type_descr super)
// *************************************************************************
// yields the sos_Superclass which contains the type super. This function could
// be written much shorter, but the explicit enumeration of all cases if
// much clearer than intensive usage of make_type() and root().
// super could be a generic class to specify the general class, or a
// sos_Generic_instantiation to specify the exact type. Both cases result
// in the same sos_Super_class
{
   sos_Class_type  ct		= self;
   sos_Super_class result	= sos_Super_class::make (NO_OBJECT);
   sos_Type	   super_type	= super.make_type();
   sos_Type	   super_root	= super.make_type().root();
   sos_Super_class_List scl	= ct.get_super_classes();

   agg_iterate (scl, sos_Super_class tmp_sc)
      sos_Type_descr tmp_td= tmp_sc.get_super_class();
      if (tmp_td.has_type (sos_Class_type_type))
      {  sos_Class_type tmp_ct = sos_Class_type::make (tmp_td);
         if (tmp_ct.is_generic_class())
         {     // This case must not occurr
             mta_error (err_SYS, err_MTA_INTERNAL,
                        "sos_Super_class::lookup_superclass");
         }
         else
         if (tmp_ct.is_instantiation())
         {  // if super is a sos_Generic_instantiation or the 
	    // generic class itself, the function yields the 
	    // corresponding superclass
            if (super_root == tmp_td.make_type().root())
            {  result = tmp_sc;
               break;
            }
         }
	 else
         {  if (tmp_ct == super)
            {  result = tmp_sc;
               break;
            }
         }
      }
      else
      if (tmp_td.has_type (sos_Generic_instantiation_type))
      {  sos_Generic_instantiation gi = sos_Generic_instantiation::make(tmp_td);
         if (gi.get_gen_class() == super)
         {  result = tmp_sc;
            break;
         }
         else
         if (gi.get_instantiation() == super)
         {  result = tmp_sc;
            break;
         }
      }
    agg_iterate_end (scl, tmp_sc)
 
   return result;
} // *** sos_Class_type::lookup_superclass ***
