/* --------------------------------------------------------------------------
 * Copyright 1992-1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * you should have received along with this program.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "OBST Projekt", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */

#include "mta.h"

sos_Bool _mta_use_err_USE /* = FALSE */;

EXPORT sos_Method mta_lookup_method (const sos_Class_type& ct,
				     const sos_String&     name,
				     sos_Int		   num_par /* = -1 */)
{  
   if (NOT ct.get_methods().is_key(name))
      mta_error (err_USE, err_MTA_INVALID_METHOD_NAME);
   else
   {
      sos_Method_List ml = 
	 sos_Method_List::copy(ct.get_methods()[name], TEMP_CONTAINER);

	 // remove methods that are not defined in ct 
      for (sos_Cursor c = ml.open_cursor();  ml.is_valid(c); )
      {  if (ml.get(c).get_defined_in() != ct)
	    ml.remove_at (c);
	 else
	    ml.to_succ(c);
      }

      if (ml.card() == 0)
	 mta_error (err_USE, err_MTA_INVALID_METHOD_NAME);
      else if (ml.card() == 1)
      {  sos_Method m = ml.get_nth(1);
	 ml.destroy();
	 return m;
      }
      else if (num_par == -1)
	 mta_error (err_USE, err_MTA_OVERLOADED_METHOD);
      else
      {  agg_iterate (ml, sos_Method m)
         {  if (m.get_params().card() == num_par)
	    {  ml.close_cursor (agg_current_cursor());
	       ml.destroy();
	       return m;
	    }
	 }
	 agg_iterate_end (ml, m)
	 mta_error (err_USE, err_MTA_OVERLOADED_NOT_FOUND);
      }
      ml.destroy();
   }
   return sos_Method::make (NO_OBJECT); // suppress warning 
} // ** mta_lookup_method ***

LOCAL void mta_insert_methods (const sos_Method_table& mt,
			       const sos_Method_List&  ml)
// If the method is generated from another, check (and correct).
{  T_PROC ("mta_insert_methods");
   TT (mta_H, T_ENTER);
   
   agg_iterate (ml, sos_Method m)
   {  sos_Method m1 = mt.lookup_or_add (m);
      if (m1.operator!=(m))
      {
	 smg_String msg = m1.get_defined_in().get_name();
	 msg += (smg_String) "::" + m1.get_name();
         mta_error (_mta_use_err_USE?err_USE:err_SYS,
		    err_MTA_INVALID_OVERLOADING,
		    msg.make_Cstring (SMG_BORROW));
      }
   }
   agg_iterate_end (ml, m);

   TT (mta_H, T_LEAVE);
}

EXPORT sos_Expr mta_copy (const sos_Expr& gp, sos_Container cnt)
{  T_PROC ("mta_copy");
   TT (mta_H, T_ENTER);
   
   sos_Expr result;

   if (gp.has_type (sos_Identifier_type))
   {  sos_String val = sos_String::copy (sos_Identifier::make(gp).get_id(),
					 cnt);
      result = sos_Identifier::create(cnt, val);
   }
   else
      result = sos_Int_expr::copy (sos_Int_expr::make (gp),cnt);
   
   TT (mta_H, T_LEAVE);
   return result;
}

EXPORT sos_Bool mta_dominates (const sos_Method& m,
			       const sos_Method& m1)
// class of m derived from class of m1
{
   T_PROC ("mta_dominates")
   TT (mta_L, T_ENTER);
 
   // A method does NOT dominate another, if both are defined in
   // instantiations of the same generic class!
 
   sos_Bool result;
   sos_Class_type c  = m.get_defined_in();
   sos_Class_type c1 = m1.get_defined_in();
   sos_Class_type cg = c.get_generic_class();
 
   result = (sos_Bool) (   (INVALID (cg) || cg != c1.get_generic_class())
			&& c.is_derived_from_some(c1.get_root_class()));
 
   TT (mta_L, T_LEAVE);
   return result;
}

LOCAL inline sos_Bool mta_is_predefined_method (const sos_String& name)
{
   smg_String nm     = name;
   sos_Bool   result = (sos_Bool)(   nm.equal("copy")
				  OR nm.equal("create")
				  OR nm.equal("clone")
				  OR nm.equal("destroy")
				  OR nm.equal("assign")
				  OR nm.equal("equal")
				  OR nm.equal("hash_value"));
   return result;
}

LOCAL void mta_destroy_method (sos_Method m)
{  T_PROC ("mta_destroy_method")
   TT (mta_VL, T_ENTER);
 
   sos_Param_List pl = m.get_params();
   if (VALID(pl))
   {  
      agg_iterate (pl, sos_Param p)
#ifdef OBST_PROT_ALL
	 sos_ModifHistory::trash(p);
#else
	 p.destroy();
#endif
      agg_iterate_end (pl, p);
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(pl);
#else
      pl.destroy();
#endif
   }
#ifdef OBST_PROT_ALL
   sos_ModifHistory::trash(m.get_name());
   sos_ModifHistory::trash(m);
#else
   m.get_name().destroy();
   m.destroy();
#endif
 
   TT (mta_VL, T_LEAVE);
}
 
LOCAL sos_Method mta_remove_method (sos_Class_type ct,
				    smg_String     nm, sos_Int pcard)
{  T_PROC ("mta_remove_method")
   TT (mta_L, T_ENTER);
 
   sos_String       mname  = nm.make_String (TEMP_CONTAINER);
   sos_Method_table mtable = ct.get_methods();
   sos_Method_List  ml     = mtable[mname];
   sos_Method       m;
   if (VALID (ml))
   {
      agg_iterate (ml, sos_Method _m)
	 sos_Param_List pl = _m.get_params();
	 if (pcard == (INVALID(pl) ? 0 : pl.card()))
	 {  m = _m;
	    ml.remove_at (agg_current_cursor());
	    break;
	 }
      agg_iterate_end (ml, _m);
 
      mtable.remove (mname);       // The m.get_name() object might be the key
				   // for ml and is hence replaced in any case.
      if (ml.card() == 0)
	 ml.destroy();
      else
	 mtable.insert (ml.get_nth(1).get_name(), ml);
    
      ml = ct.get_local_methods();
      ml.remove (ml.find (m));
   }
 
   TT (mta_L, T_LEAVE);
   return m;
}

 
LOCAL sos_Type_descr mta_SelfTp (sos_Class_type ct)
{  T_PROC ("mta_SelfTp")
   TT (mta_L, T_ENTER);
 
   /* for a non-generic class type mta_SelfTp returns the given ct,
      for generic class types an sos_Generic_instantiation object is
      created representing the root class as an instantiation. */
 
   static sos_Class_type last_ct = sos_Class_type::make (NO_OBJECT);
   static sos_Type_descr last_td;

   sos_Container cnt = ct.container();

   if (ct != last_ct)
   {  last_ct = ct;
      if (ct.is_generic_class())
      {  sos_Gen_param_List  gpl = ct.get_formal_gen_params();
         sos_Type_descr_List tdl = sos_Type_descr_List::create (cnt);
         agg_iterate (gpl, sos_Gen_param gp)
            tdl.append (gp);
         agg_iterate_end (gpl, gp);
 
         sos_Generic_instantiation gi
            = mta_get_instantiation (ct, ct.get_root_class(), tdl);
 
         if (gi.get_act_gen_params() != tdl)
            tdl.destroy();
 
         last_td = gi;
      }
      else
         last_td = ct;
   }
   TT (mta_L, T_LEAVE);
   return last_td;
}


LOCAL void mta_get_agpl_of_superclass_recursive (const sos_Class_type& sub,
				        	 const sos_Class_type& sup,
				        	 const sos_Gen_param_List& r)
{
   T_PROC ("mta_get_agpl_of_superclass_recursive");
   TT (mta_M, T_ENTER);
   /* Determines the actual generic parameters of sub passed through to sup.
      NO_OBJECT is inserted in the list at positions, where something
      different from a formal gen param of sub was used in the instantiation
      of sup. */

   // PRECONDITION: sub and sup are generic classes,
   //		    r contains a list of the same length as the actual
   //		    generic parameter list of sup.

   sos_Bool found = FALSE;
   sos_Class_type sup_root = sup.get_root_class();

   // First check, if the generic class sup is contained in the super_classes
   // list of sub as sos_Generic_instantiation. In this case everything is
   // easy and we don't need recursion!
   // By the way collect all sos_Generic_instantiation objects in the
   // super_classes list.
   sos_Super_class_List scl = sub.get_super_classes();
   sos_Type_descr_List gi_list = sos_Type_descr_List::create (TEMP_CONTAINER);

   agg_iterate (scl, sos_Super_class sub_sc)
   {
      sos_Type_descr sc = sub_sc.get_super_class();
      if (sc.has_type (sos_Generic_instantiation_type))
      {
	 sos_Generic_instantiation gi = sos_Generic_instantiation::make(sc);
	 gi_list.append (gi);
	 sos_Class_type gc = gi.get_gen_class();
	 if (sup == gc)
	 {
	    // Take over all formal gen params into the result list.
	    sos_Type_descr_List agpl = gi.get_act_gen_params();
	    Index pos = 1;
	    agg_iterate (agpl, sos_Type_descr td)
	    {
	       if (td.has_type (sos_Gen_param_type))
	       {
		  r.remove (pos);
		  r.insert (pos, sos_Gen_param::make (td));
	       }
	       pos++;
	    } agg_iterate_end (agpl, td);

	    found = TRUE;
	    break;
	 }
      }
   } agg_iterate_end (scl, sub_sc);

   // If we didn't find sup in the direct superclasses, then scan the
   // direct superclasses' super_closures.
   if (NOT found)
   {
      agg_iterate (gi_list, sos_Type_descr act_gi)
      {
	 sos_Bool chance = FALSE; // Do we have a chance to find an
			   // instantiation of sup in act_gi's super_classes ?
	 sos_Generic_instantiation gi = sos_Generic_instantiation::make(act_gi);
	 // Determine generic class belonging to direct superclass:
	 sos_Class_type act_gen_ct = gi.get_gen_class();
	 sos_Super_class_List act_gen_ct_scl = act_gen_ct.get_super_closure();
	 agg_iterate (act_gen_ct_scl, sos_Super_class closure_class)
	 {
	    sos_Class_type cc_root = sos_Class_type::make (
				closure_class.make_type()).get_root_class();
	    if (cc_root == sup_root)
	    {
	       // an instantiation of sup occurs in gi's super_closure.
	       // So we have a chance to find any further passed through
	       // generic parameters of sub in gi's super_classes.
	       chance = TRUE;
	       break;
	    }
	 } agg_iterate_end (act_gen_ct_scl, closure_class);

	 if (chance)
	 {
	    // Now solve it by recursion:
	    sos_Gen_param_List tmp_gpl = sos_Gen_param_List::create
							    (TEMP_CONTAINER);
	    for (int i = 1; i <= sup.get_formal_gen_params().card(); i++)
	       tmp_gpl.append (sos_Gen_param::make (NO_OBJECT));
	    mta_get_agpl_of_superclass_recursive (act_gen_ct, sup, tmp_gpl);

	    // Now unify resulting genparams with the formal genparams
	    // of sub using gi.get_act_gen_params().
	    Index pos_in_sup_gpl = 1;
	    sos_Gen_param_List act_gen_ct_fgpl =
					act_gen_ct.get_formal_gen_params();
	    sos_Type_descr_List act_gen_ct_agpl =
					gi.get_act_gen_params();
	    agg_iterate (tmp_gpl, sos_Gen_param gp)
	    {
	       if (VALID (gp))
	       {  // genparam of sup was a passed through genparam of act_gen_ct
		  Index pos_in_gen_gpl = act_gen_ct_fgpl.find (gp);
		  sos_Type_descr td = act_gen_ct_agpl.get_nth (pos_in_gen_gpl);
		  if (td.has_type (sos_Gen_param_type))
		  {
		     r.remove (pos_in_sup_gpl);
		     r.insert (pos_in_sup_gpl, sos_Gen_param::make (td));
		  }
	       }
	       pos_in_sup_gpl++;
	    } agg_iterate_end (tmp_gpl, gp);
	    tmp_gpl.destroy();
	 }
      } agg_iterate_end (gi_list, act_gi);
   }
   gi_list.destroy();

   TT (mta_M, T_LEAVE);
   /* POSTCONDITION: Each formal generic parameter of sub, which was passed
		     through up to sup is entered in that place in r, where
		     it occurs in the actual generic param list of sup.
   */
}


EXPORT sos_Gen_param_List mta_get_agpl_of_superclass (const sos_Class_type& sub,
						      const sos_Class_type& sup)
{
   T_PROC ("mta_get_agpl_of_superclass");
   TT (mta_M, T_ENTER);

   sos_Gen_param_List agpl = sos_Gen_param_List::create (TEMP_CONTAINER);
   for (int i=sup.get_formal_gen_params().card(); i>0; i--)
      agpl.append (sos_Gen_param::make (NO_OBJECT));
   mta_get_agpl_of_superclass_recursive (sub, sup, agpl);

   TT (mta_M, T_LEAVE);
   return agpl;
}


EXPORT void mta_build_methodtable (const sos_Class_type& ct)
// computes the methodtable for the passed class. If an error occurs and
// will be catched from an err_block , ct remains unchanged
// The super_closure for the given class must be set correctly,
// since it is needed for redefinition checks.
{  T_PROC ("mta_build_methodtable")
   TT (mta_M, T_ENTER);
   sos_Container cnt = ct.container();

     // backup the old methodtable
   sos_Method_table mt_backup = ct.get_methods();

      // the new methodtable:
   sos_Method_table mt		       = sos_Method_table::create (cnt);
   sos_Bool	    is_abstract_backup = ct.get_is_abstract();

   err_block 
      mta_insert_methods (mt, ct.get_local_methods());
 
      sos_Super_class_List super_classes = ct.get_super_classes();
      sos_Bool             not_abstract  = (sos_Bool) NOT is_abstract_backup;
 
      agg_iterate (super_classes, sos_Super_class stn)
         if (NOT stn.get_is_direct())
            continue;
 
         sos_Class_type   super_class = sos_Class_type::make(stn.make_type());
         sos_Method_table scmt        = super_class.get_methods();
    
         agg_iterate_association (scmt, sos_String name, sos_Method_List ml)
           agg_iterate (ml, sos_Method m)
               if (    m.get_kind() != sos_PRIVATE
		   AND NOT m.get_is_static()
                   AND NOT mta_is_predefined_method(m.get_name()))
               {  sos_Method m1 = mt.lookup_or_add (m);
                  if (m1.operator==(m))    //ATT2.0 QUEERNESS: explicit operator
                  {  if (not_abstract AND m.get_is_abstract())
                     {  not_abstract = FALSE;
                        ct.set_is_abstract (TRUE);
                     }
                  }
                  else
                  {  // m1 - a different method with the same name as m found
                     // m1 doesn't overload m!

                     if (mta_dominates (m1, m))
                     {                     //ATT2.0 QUEERNESS: explicit operator
                        if (ct.operator== (m1.get_defined_in()))
                        {  
			   // nearly all subcases yield an error.
			   // So precalculate the method name for construction
			   // of the message:
			   smg_String msg = m.get_defined_in().get_name();
			   msg += (smg_String) "::" + m.get_name();
			   if (m1.redefines (m) && m.get_kind() <= m1.get_kind())
                           {  // The redefining method is already in the
			      // method table. Don't change it, only do a
			      // definite-check:
			      if (m.get_is_definite())
                                 mta_error (_mta_use_err_USE?err_USE:err_SYS,
					    err_MTA_DEFINITE_REDEFINED,
					    msg.make_Cstring (SMG_BORROW));
                           }
                           else // m1 defined in the given class and no 
                                // redefinition, or access right restriction
                              if ( NOT m1.redefines (m))
			      {   
				 mta_error (_mta_use_err_USE?err_USE:err_SYS,
					    err_MTA_INVALID_REDEFINITION,
					    msg.make_Cstring (SMG_BORROW));
			      }
                              else
			         mta_error (_mta_use_err_USE?err_USE:err_SYS,
					    err_MTA_INVALID_ACCESS_RESTR,
					    msg.make_Cstring (SMG_BORROW));
                        }
                     }
                     else if (mta_dominates (m, m1))
                     {  // no redefinition check neccessary, since
			// it must have been correct during construction
			// of the method table of the class, where
			// m was defined.
			mt.replace_or_add (m);
                        if (not_abstract AND m.get_is_abstract())
                        {  not_abstract = FALSE;
                           ct.set_is_abstract (TRUE);
                        }
                     }
                     else
                        mta_error (_mta_use_err_USE?err_USE:err_SYS,
				   err_MTA_AMBIGUOUS_METHODS,
				   m.get_name());
                  }
               }
            agg_iterate_end (ml, m);
         agg_iterate_association_end (scmt, name, ml);
      agg_iterate_end (super_classes, super_class);

   err_exception
      mta_destroy (mt);        // destroy useless methodtable
      mt = sos_Method_table::make (NO_OBJECT);
      ct.set_methods (mt_backup);
      ct.set_is_abstract (is_abstract_backup);
      mta_repeat_last_error();
   err_block_end 

   ct.set_methods (mt);
   if (VALID(mt_backup))
      mta_destroy (mt_backup);

   TT (mta_M, T_LEAVE);
}
 
EXPORT void mta_build_class (const sos_Class_type& ct,
			     sos_Bool derived,
			     sos_Bool build_methodtable      /* = FALSE */,
			     sos_Bool build_super_closure    /* = FALSE */,
			     sos_Bool build_offsets_and_size /* = FALSE */,
			     sos_Bool complete_components    /* = FALSE */)
// computes some useful components in one class, or (derived = TRUE) a class
// including its subclasses. All errors are catched, recovered and repeated
// An error which is produced from this function need no further treatment,
// All concerned classes remain unchanged
{  
   T_PROC ("mta_build_class");
   TT (mta_H, T_ENTER);

   // NOTE: Changes in the super_closure, the offsets and size and the
   //       components would affect the subclasses. So if derived==FALSE,
   //	    these changes do not take place.
   if (NOT derived)
      mta_build_methodtable (ct);
   else
   {  sos_Class_type_List classes = ct.get_subclasses();

      sos_Class_type  error_class = sos_Class_type::make (NO_OBJECT);
      sos_Object_List mt_backup   = sos_Object_List::create (TEMP_CONTAINER);
      sos_Object_List sc_backup   = sos_Object_List::create (TEMP_CONTAINER);
      err_block
         agg_iterate (classes, sos_Class_type tmp_ct)
            mta_open_for_writing (tmp_ct.container());
	       // mark the last methodtable building without error
            error_class = tmp_ct; 
	    if (build_super_closure)
	    {  sc_backup.append (tmp_ct.get_super_closure());
	       tmp_ct.set_super_closure (sos_Super_class_List::make(NO_OBJECT));
	       mta_set_super_closure (tmp_ct);
            }
            if (build_methodtable)
	    {  mt_backup.append (tmp_ct.get_methods());
               tmp_ct.set_methods (sos_Method_table::make (NO_OBJECT));
               mta_build_methodtable (tmp_ct);
	    }
	    if (build_offsets_and_size)
	       mta_set_offsets_and_size (tmp_ct);
            if (complete_components)
	       mta_complete_components (tmp_ct);

         agg_iterate_end (classes, tmp_st)
      err_exception
            // set the method table of the modified classes inclusive the
            // class which produced the error to the old state
            // Fortunately the old method tables are backup up ...   
         agg_iterate(classes, sos_Class_type tmp_ct)
	    if (build_methodtable)
	    {  sos_Method_table new_mt = tmp_ct.get_methods();
               mta_destroy (new_mt);
               tmp_ct.set_methods(sos_Method_table::make(mt_backup.get_nth(1)));
	       mt_backup.remove(1);
	    }
	    if (build_super_closure)
	    {  sos_Super_class_List new_scl = tmp_ct.get_super_closure();
#ifdef OBST_PROT_ALL
	       sos_ModifHistory::trash(new_scl);
#else
	       new_scl.destroy();
#endif
	       tmp_ct.set_super_closure (sos_Super_class_List::make(
						       sc_backup.get_nth(1)));
	       sc_backup.remove(1);
            }
	    if (build_offsets_and_size)
	       mta_set_offsets_and_size (tmp_ct);
            if (complete_components)
	       mta_complete_components (tmp_ct);
	    if (tmp_ct == error_class)
	       break;
         agg_iterate_end(classes, tmp_ct)
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(mt_backup); 
	 sos_ModifHistory::trash(sc_backup);
#else
         mt_backup.destroy(); 
	 sc_backup.destroy();
#endif
         mta_repeat_last_error(); // exit from procedure
      err_block_end

#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(mt_backup); 
	 sos_ModifHistory::trash(sc_backup);
#else
         mt_backup.destroy(); 
	 sc_backup.destroy();
#endif
   }      

   TT (mta_H, T_LEAVE);
}


LOCAL sos_Bool mta_set_predefined_method
                           (sos_Bool              use_mtable,
                            const sos_Class_type& ct,
                            const smg_String&     n,
                            sos_Bool              is_static,
                            const sos_Param_List& pl,
                            const sos_Type_descr& rt)
{  T_PROC ("mta_set_predefined_method");
   TT (mta_H, T_ENTER);
   
   sos_Method_table mt;
   sos_Method_List  ml;
   sos_Container    cnt = ct.container();
   sos_Method       m = sos_Method::create (cnt);
   sos_Bool         already_defined;
   sos_String       str = n.make_String(cnt);
 
   if (use_mtable)
   {  mt = ct.get_methods();
      ml = mt [str];
      already_defined = VALID (ml);
   }
   else
      already_defined = FALSE;
 
   m.set_name (str);
   m.set_kind ((use_mtable) ? sos_PROTECTED : sos_PUBLIC);
   m.set_is_abstract (FALSE);
   m.set_is_definite (FALSE);
   m.set_is_static (is_static);
   m.set_is_operator (FALSE);
   m.set_is_generated (TRUE);
   m.set_generated_from (sos_Method::make (NO_OBJECT));
   m.set_params (pl);
   m.set_result_type (rt);
   m.set_impls (sos_Method_impl_List::make (NO_OBJECT));
 
   if (already_defined)                    // implies 'use_mtable == TRUE'
   {  if (INVALID (mt.lookup (m)) OR ml.card() > 1)
      {  m.destroy();
         str.destroy();
         mta_error (err_SYS,err_MTA_INVALID_LOCAL_METHOD, m.get_name());
         return TRUE;
      }
 
      str.destroy();
      m.destroy();
      pl.destroy(); // was explicitly created for predefined method, can thus
   }		    // be destroyed.
   else
   {  m.set_defined_in (ct);
 
      ct.get_local_methods().append(m);
      if (use_mtable)
         mt.lookup_or_add (m);
   }
 
   TT (mta_H, T_LEAVE);
   return already_defined;
}
 
LOCAL inline void mta_set_predefined_public_method
                           (const sos_Class_type& ct,
                            const smg_String&     n,
                            sos_Bool              is_static,
                            const sos_Param_List& pl,
                            const sos_Type_descr& rt)
{  (void)mta_set_predefined_method (FALSE,ct,n,is_static,pl,rt); }
 
LOCAL inline sos_Bool mta_set_predefined_local_method (sos_Class_type ct,
                                                 smg_String     n,
                                                 sos_Param_List pl,
                                                 sos_Type_descr rt)
{  return mta_set_predefined_method (TRUE,ct,n,TRUE,pl,rt); }
 
LOCAL void mta_add_param
		   (const sos_Param_List &pl,
                    const smg_String     &n,
                    const sos_Type_descr &tn,
                    const sos_Expr       &de)
// create a sos_Param(n,tn,de) and add it to pl
{  T_PROC ("mta_add_param");
   TT (mta_H, T_ENTER);
   
   sos_Container cnt = pl.container();
   sos_Param p = sos_Param::create (cnt);
 
   p.set_name (n.make_String (cnt));
   p.set_type (tn);
   p.set_default_expr (de);
   p.set_is_ref (FALSE);
 
   pl.append (p);

   TT (mta_H, T_LEAVE);
}
 
LOCAL void mta_set_predefined_methods (const sos_Class_type& ct)
// create and add predefined methods to ct:
//    create, copy, clone, destroy, assign, equal, hash_value
{  T_PROC ("mta_set_predefined_methods");
   TT (mta_H, T_ENTER);
   
   sos_Container cnt = ct.container();
   sos_Param_List pl;
 
   // create
   pl = sos_Param_List::create (cnt);
   mta_add_param (pl,"_ct",sos_Type_descr::make(mta_get_container_type()),
		           sos_Expr::make (NO_OBJECT));
   sos_Param_List tail = ct.get_create_params();
   if VALID(tail)
      pl += tail;
   mta_set_predefined_public_method (ct,"create",TRUE,pl,mta_SelfTp(ct));
 
   // copy
   pl = sos_Param_List::create (cnt);
   mta_add_param (pl,"_y",mta_SelfTp(ct),sos_Expr::make (NO_OBJECT));
   mta_add_param (pl,"_ct",sos_Type_descr::make(mta_get_container_type()),
		           sos_Expr::make (NO_OBJECT));
   mta_set_predefined_public_method (ct,"copy",TRUE,pl,mta_SelfTp(ct));
 
   // clone
   pl = sos_Param_List::create (cnt);
   mta_add_param (pl,"_y",mta_SelfTp(ct),sos_Expr::make (NO_OBJECT));
   mta_add_param (pl,"_ct",sos_Type_descr::make(mta_get_container_type()),
		  	   sos_Expr::make (NO_OBJECT));
   mta_set_predefined_public_method (ct,"clone",TRUE,pl,mta_SelfTp(ct));
 
   // destroy
   mta_set_predefined_public_method (ct,"destroy",FALSE,
                                     sos_Param_List::make (NO_OBJECT),
                                     sos_Type_descr::make(mta_get_void_type()));
 
   // assign
   pl = sos_Param_List::create (cnt);
   mta_add_param (pl,"_y",
		  sos_Type_descr::make(mta_get_object_type()),
		  sos_Expr::make (NO_OBJECT));
   mta_set_predefined_public_method (ct,"assign",FALSE,pl,
				     sos_Type_descr::make(mta_get_void_type()));
 
   // equal
   pl = sos_Param_List::create (cnt);
   mta_add_param (pl,"_y",
		  sos_Type_descr::make(mta_get_object_type()),
		  sos_Expr::make (NO_OBJECT));
   sos_Identifier id = sos_Identifier::create (cnt,
					sos_String::create (cnt, "EQ_STRONG"));
   MTA_ADD_PARAM(pl,"_eq",sos_Type_descr::make(mta_get_eqkind_type()),id);
   mta_set_predefined_public_method (ct,"equal",FALSE,pl,
				     sos_Type_descr::make(mta_get_bool_type()));
 
   // hash_value
   mta_set_predefined_public_method (ct,"hash_value",FALSE,
                                     sos_Param_List::make (NO_OBJECT),
				     sos_Type_descr::make(mta_get_int_type()));

   TT (mta_H, T_LEAVE);
}
  
EXPORT void mta_init_methods (const sos_Class_type& ct)
{  T_PROC ("mta_init_methods");
   TT (mta_H, T_ENTER);
   
   sos_Container cnt = ct.container();
   ct.set_local_methods (sos_Method_List::create (cnt, FALSE));
   if (NOT ct.is_instantiation())
   {  // for instantiations the predefined methods are substituted from
      // the generic class.
      mta_set_predefined_methods (ct);
      ct.set_has_init_comps (FALSE);
   }

   TT (mta_H, T_LEAVE);
}
 
EXPORT void mta_complete_local_methods (sos_Class_type ct)
{  T_PROC ("mta_complete_methods")
   TT (mta_M, T_ENTER);
 
   static sos_String
          nm_t_ass  = sos_String::create(TEMP_CONTAINER,"total_assign"),
          nm_t_eql  = sos_String::create(TEMP_CONTAINER,"total_equal"),
          nm_t_hash = sos_String::create(TEMP_CONTAINER,"total_hash_value"),
          nm_final  = sos_String::create(TEMP_CONTAINER,"local_finalize"),
          nm_init   = sos_String::create(TEMP_CONTAINER,"local_initialize"),
          nm_t_fin  = sos_String::create(TEMP_CONTAINER,"total_finalize"),
          nm_t_init = sos_String::create(TEMP_CONTAINER,"total_initialize");

   sos_Param_List   pl;
   sos_Method_table mt  = ct.get_methods();
   sos_Container    cnt = ct.container();
   sos_Bool         assign_defined, equal_defined, hash_value_defined,
                    finalize_defined   = VALID(mt[nm_final]),
      		    initialize_defined = VALID(mt[nm_init]),
                    t_ass_defined      = VALID(mt[nm_t_ass]),
      		    t_eq_defined       = VALID(mt[nm_t_eql]),
      		    t_hash_defined     = VALID(mt[nm_t_hash]),
                    t_fin_defined      = VALID(mt[nm_t_fin]),
      		    t_init_defined     = VALID(mt[nm_t_init]);
 
/*
   // This code destroys the create method for classes made abstract.
   // It is currently not included, since there are still unsolved problems
   // in connection with schema evolution (should there be a change, if
   // a class toggles is_abstract?)

   if (ct.get_is_abstract())
   {  mta_destroy_method (mta_remove_method (ct, "copy", 2));
 
      sos_Param_List pl = ct.get_create_params();
      sos_Method     m  = mta_remove_method (ct, "create",
                                             INVALID(pl) ? 1
                                                               : 1+ pl.card());
      m.get_params().get_nth (1).destroy();
      m.set_params(sos_Param_List::make (NO_OBJECT));
      mta_destroy_method (m);
   }

   // Don't create local_...-methods for classes without local components.
   // Same problem as above: What, when this changes by schema evolution?
   if (ct.get_local_size() == 0)
   {  static sos_String
             nm_assign = sos_String::create(TEMP_CONTAINER,"local_assign"),
             nm_equal  = sos_String::create(TEMP_CONTAINER,"local_equal"),
             nm_hashv  = sos_String::create(TEMP_CONTAINER,"local_hash_value");
 
      assign_defined     = VALID(mt[nm_assign]);
      equal_defined      = VALID(mt[nm_equal ]);
      hash_value_defined = VALID(mt[nm_hashv ]);
   }
   else
   {  // local_assign
*/
      pl = sos_Param_List::create (cnt);
      mta_add_param (pl,"_x",mta_SelfTp (ct),sos_Expr::make (NO_OBJECT));
      mta_add_param (pl,"_y",sos_Type_descr::make(mta_get_object_type()),
		     sos_Expr::make (NO_OBJECT));
      assign_defined
         = mta_set_predefined_local_method (ct, "local_assign", pl,
			sos_Type_descr::make(mta_get_void_type()));
 
      // local_equal
      pl = sos_Param_List::create (cnt);
      mta_add_param (pl,"_x",mta_SelfTp (ct),sos_Expr::make (NO_OBJECT));
      mta_add_param (pl,"_y",sos_Type_descr::make(mta_get_object_type()),
		     sos_Expr::make (NO_OBJECT));
      mta_add_param (pl,"_eq",sos_Type_descr::make(mta_get_eqkind_type()),
		     sos_Expr::make(NO_OBJECT));
      equal_defined
         = mta_set_predefined_local_method (ct, "local_equal", pl,
			sos_Type_descr::make(mta_get_bool_type()));
 
      // local_hash_value
      pl = sos_Param_List::create (cnt);
      mta_add_param (pl,"_x",mta_SelfTp(ct),sos_Expr::make (NO_OBJECT));
      hash_value_defined
         = mta_set_predefined_local_method (ct, "local_hash_value", pl,
			sos_Type_descr::make(mta_get_int_type()));
// }
   if (assign_defined AND NOT equal_defined)
      mta_error (err_WNG, err_MTA_ASSIGN_WITHOUT_EQUAL);
   if (equal_defined AND NOT hash_value_defined)
      mta_error (err_WNG, err_MTA_EQUAL_WITHOUT_HASH);
   if (assign_defined && t_ass_defined)
      mta_error (err_WNG, err_MTA_L_AND_T_ASSIGN);
   if (equal_defined && t_eq_defined)
      mta_error (err_WNG, err_MTA_L_AND_T_EQUAL);
   if (hash_value_defined && t_hash_defined)
      mta_error (err_WNG, err_MTA_L_AND_T_HASH);
   if (finalize_defined && t_fin_defined)
      mta_error (err_WNG, err_MTA_L_AND_T_FINALIZE);
   if (initialize_defined && t_init_defined)
      mta_error (err_WNG, err_MTA_L_AND_T_INIT);
 
   TT (mta_M, T_LEAVE);
}
 

LOCAL sos_Bool valid_method_name (const sos_Object& o,
			          const sos_String& name)
// TRUE, if sos_Class_type(o) doesn't contain a method named name
{ sos_Class_type ct = sos_Class_type::make (o);
  return (sos_Bool) (NOT ct.get_methods().is_key(name));
}

LOCAL inline void insert (const sos_Object_Set& s, char* n)
{ s.insert(sos_String::create(TEMP_CONTAINER, n));
}

LOCAL sos_Object_Set operator_names()
{  T_PROC ("operator_names");
   TT (mta_M, T_ENTER);
   
   static sos_Bool created = FALSE;
   static sos_Object_Set s;
;
   if (NOT created)
   {  s = sos_Object_Set::create(TEMP_CONTAINER,FALSE,TRUE);

      insert(s,"*");   insert(s,"/");   insert(s,"%");  insert(s,"+");
      insert(s,"-");   insert(s,">>");  insert(s,"<<"); insert(s,">");
      insert(s,"<");   insert(s,"==");  insert(s,"!="); insert(s,"<=");
      insert(s,">=");  insert(s,"&");   insert(s,"|");  insert(s,"&&");
      insert(s,"||");  insert(s,"=");   insert(s,"+="); insert(s,"-=");
      insert(s,"*=");  insert(s,"/=");  insert(s,"^="); insert(s,"|=");
      insert(s,">>="); insert(s,"<<="); insert(s,"()"); insert(s,"[]");
      created = TRUE;
   }

   TT (mta_M, T_LEAVE);
   return s;
}

LOCAL sos_Bool is_operator(const sos_String& name)
{  T_PROC ("is_operator");
   TT (mta_M, T_ENTER);
   
   smg_String nm = name.make_Cstring();
   sos_Bool   is = FALSE;

   if (streqln("operator", nm.make_Cstring(SMG_BORROW), 8))
   {  name.assign_Cstring (nm.make_Cstring(SMG_BORROW) + 8);
      if (NOT operator_names().is_element(name))
	 mta_error (err_SYS, err_MTA_INVALID_NAME);
      else
	 is = TRUE;
   }

   TT (mta_M, T_LEAVE);
   return is;
}

LOCAL inline sos_Bool mta_is_local_method (const sos_String& name)
{
   smg_String nm     = name;
   sos_Bool   result = (sos_Bool)(   nm.equal("local_assign")
				  OR nm.equal("local_equal")
				  OR nm.equal("local_hash_value")
				  OR nm.equal("local_initialize")
				  OR nm.equal("local_finalize"));
   return result;
}


EXPORT void mta_destroy (sos_Param p)
{  T_PROC ("mta_destroy (sos_Param)");
   TT (mta_H, T_ENTER);
   
   if (VALID (p))
   {  sos_String name = p.get_name();
      if (VALID (name))
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(name);
#else
         name.destroy();
#endif
      mta_destroy (p.get_default_expr());
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(p);
#else
      p.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}

EXPORT void mta_destroy (sos_Param_List pl)
{  T_PROC ("mta_destroy (sos_Param_List)");
   TT (mta_H, T_ENTER);
   
   if (VALID (pl))
   {  agg_iterate (pl, sos_Param p)
         mta_destroy (p);
      agg_iterate_end (pl, p)
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(pl);
#else
      pl.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_Method_table mt)
{  T_PROC ("mta_destroy (sos_Method_table)");
   TT (mta_H, T_ENTER);
   
   if (VALID (mt))
   {  
      for (sos_Cursor c = mt.open_cursor();mt.is_valid(c);)
      {  sos_Method_List ml = mt.get_info(c);
         mt.to_succ(c);
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(ml);
#else
         ml.destroy();
#endif
      }
      mt.close_cursor(c);
 
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(mt);
#else
      mt.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_Method_List ml)
{  T_PROC ("mta_destroy (sos_Method_List)");
   TT (mta_H, T_ENTER);
   
   if (VALID (ml))
   {  agg_iterate (ml, sos_Method m)
         mta_destroy (m);
      agg_iterate_end (ml, m)
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(ml);
#else
      ml.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}
 
EXPORT void mta_destroy (sos_Method m)
{  T_PROC ("mta_destroy (sos_Method)");
   TT (mta_H, T_ENTER);
   
   if (VALID (m))
   {  
#ifdef OBST_PROT_ALL
      sos_ModifHistory::trash(m.get_name());
#else
      m.get_name().destroy();
#endif
      sos_Param_List pl = m.get_params();
      if (VALID (pl))
      {
	 agg_iterate (pl, sos_Param p)
	    mta_destroy (p); 
	 agg_iterate_end (pl, p)
#ifdef OBST_PROT_ALL
	 sos_ModifHistory::trash(pl);
#else
	 pl.destroy();
#endif
      }
      sos_Method_impl_List impls = m.get_impls();
      if VALID (impls)
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(impls);
      sos_ModifHistory::trash(m);
#else
         impls.destroy();
      m.destroy();
#endif
   }

   TT (mta_H, T_LEAVE);
}
 
// **************************************************************************
sos_Method sos_Class_type::create_method()
// **************************************************************************
{  T_PROC ("sos_Class_type::create_method");
   TT (mta_H, T_ENTER);
   
   sos_Class_type ct  = self;
   sos_Container  cnt = ct.container();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));
#endif
 
   smg_String smg_name =
      MTA_GENERATE_VALID_NAME ("new_method", ct, valid_method_name);

   mta_open_for_writing (cnt);

   sos_Method new_method = sos_Method::create(cnt);
   sos_String new_name   = smg_name.make_String(cnt);

   new_method.set_name          (new_name);
   new_method.set_kind          (sos_PRIVATE);
   new_method.set_is_static     (FALSE);
   new_method.set_is_abstract   (FALSE);
   new_method.set_is_definite   (FALSE);
   new_method.set_is_operator   (operator_names().is_element(new_name));
   new_method.set_generated_from(sos_Method::make(NO_OBJECT));
   new_method.set_defined_in    (ct);
   new_method.set_params	(sos_Param_List::make (NO_OBJECT));
   new_method.set_result_type	(sos_Type_descr::make(void_type));
   new_method.set_impls		(sos_Method_impl_List::create(cnt));
   new_method.set_comp_descr	(sos_Comp_descr::make (NO_OBJECT));

#ifdef OBST_PROT_ALL
   if (VALID(cmod)) // generate protocol info if needed
      cmod.add_meth(new_method);
#endif

      // if the method is generic, insert the new method in the local
      // method list of the instantiations
   if (ct.is_generic_class())
   {     // Note:ct is element of the subclass List
      sos_Class_type_List subclasses = ct.get_subclasses();
      agg_iterate (subclasses, sos_Class_type sub)
         if (sub.root() == ct.root())
	 {  mta_open_for_writing (sub.container());
	    sub.get_local_methods().append (new_method);
	 }
      agg_iterate_end (subclasses, sub);
   }
   else
      ct.get_local_methods().append (new_method);
 
      // The method is private, so do not pay any 
      // attention to derived classes
   assert_no_error_block("create_method")
      mta_destroy (ct.get_methods());
      ct.set_methods (sos_Method_table::make (NO_OBJECT));
      mta_build_methodtable (ct);
   assert_no_error_block_end
 
   TT (mta_H, T_LEAVE);
   return new_method;
} // *** sos_Method sos_Class_type::create_method ***

// *************************************************************************
void sos_Method::remove ()
// *************************************************************************
{  T_PROC ("sos_Method::remove");
   TT (mta_H, T_ENTER);

   sos_Method 	  m   = self;

   sos_Class_type ct  = m.get_defined_in();
   sos_Container  cnt = m.container();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));
#endif

      // if generated methods will be removed, don't change the method
      // but reset the flag `generate`. Then the method appears in the
      // declaration and wants to be user-implemented
#ifdef OBST_PROT_ALL
   if (VALID(cmod)) // generate protocol information if needed
      cmod.del_meth(m);
#endif
   if (mta_is_local_method (m.get_name()))
   {  if (m.get_is_generated())
         m.set_is_generated (FALSE);
      else
	 mta_error (err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED);
   }
   else
   {
      if (VALID (m.get_comp_descr()))
	 mta_error (err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED);
      
    
      sos_Method_List local_methods = ct.get_local_methods();
      sos_Int	      index	    = local_methods.find (m);
      if (index <=1)
	 mta_error (err_SYS, err_MTA_NOT_FOUND);

      mta_open_for_writing (m.container());
      local_methods.remove(index);

      sos_Bool look_at_derived =(sos_Bool) (    m.get_kind() != sos_PRIVATE
				 	    AND NOT m.get_is_static());
    
      assert_no_error_block("sos_Method::remove")
	 mta_build_class (ct, look_at_derived, TRUE);
      assert_no_error_block_end

      mta_destroy (m);
   }
   TT (mta_H, T_LEAVE);
} // *** sos_Method::remove ***


 
// *************************************************************************
void sos_Method::modify (sos_Type_descr  new_result_type, 
			 sos_String	 new_name, 
			 sos_Method_kind new_kind, 
			 sos_Bool	 new_is_static,
			 sos_Bool	 new_is_abstract, 
                         sos_Bool	 new_is_definite)
// *************************************************************************
{  T_PROC ("sos_Method::modify");
   TT (mta_H, T_ENTER);
   
   sos_Method     m   = self;
   sos_Class_type ct  = m.get_defined_in();
   sos_Container  cnt = m.container();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));
#endif

   if (m.get_is_generated() OR (ct.is_instantiation()))
      mta_error (err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED, m.get_name());

#ifdef OBST_PROT_ALL
   if (VALID(cmod)) // generate protocol information if needed
      cmod.mod_meth (m);
#endif
   if (mta_is_local_method (new_name))
   // That's all:
      m.set_is_generated (TRUE);
   else
   {
	    // backup old method 
      smg_String old_name (m.get_name());
      sos_Method_kind old_kind  = m.get_kind();
      sos_Bool old_is_static    = m.get_is_static();
      sos_Bool old_is_abstract  = m.get_is_abstract();

	 // pass eventual errors to the higher level:
      sos_Bool new_is_operator = is_operator (new_name);
	 
      mta_open_for_writing (cnt);
    
	 // assign all new values:
      m.get_name().assign (new_name);
      m.set_is_static (new_is_static);
      m.set_is_abstract (new_is_abstract);
      m.set_is_definite (new_is_definite);
      m.set_kind (new_kind);
      m.set_result_type (new_result_type);

	 // is it neccessary to take a look to derived classes ? 
      sos_Bool look_to_derived = (sos_Bool)(
				  old_kind != new_kind
				 OR
				 (old_is_static != new_is_static AND
				  old_kind != sos_PRIVATE)
				 OR
				 (old_kind != sos_PRIVATE AND
				  NOT old_name.equal(new_name)));

	 // pass an error to the higher level
      mta_build_class(ct, look_to_derived, TRUE);

	 
      m.set_is_operator(new_is_operator);

	 // if the method is no more abstract, check if there exists 
	 // another abstract method to set the flag is_abstract:
      if (old_is_abstract AND  NOT new_is_abstract)
      {  sos_Bool	 found	       = FALSE;
	 sos_Method_List local_methods = ct.get_local_methods();
	 agg_iterate (local_methods, sos_Method tmp_m)
	    if (tmp_m.get_is_abstract())
	    {  found = TRUE;
	       break;
	    }
	 agg_iterate_end (local_methods,tmp_m)
	 ct.set_is_abstract (found);
      }
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Method::modify ***


// *************************************************************************
void sos_Method::insert_param (sos_Int no, sos_Type_descr td)
// *************************************************************************
{  T_PROC ("sos_Method::insert_param");
   TT (mta_H, T_ENTER);
   
   sos_Method	  m   = self;
   sos_Class_type ct  = m.get_defined_in();
   sos_Container  cnt = m.container();
#ifdef OBST_PROT_ALL
   sos_MethodModif mmod = mta_get_method_mod (cnt, m);
#endif
 
   if (mta_is_local_method(m.get_name()))
      mta_error(err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED);

   mta_open_for_writing (cnt);
 
      // create and initialize new parameter
   sos_Param p = sos_Param::create(cnt);
   p.set_name(sos_String::make(NO_OBJECT));
   p.set_type(td);
   p.set_is_ref(FALSE);
   p.set_default_expr(sos_Expr::make(NO_OBJECT));
 
   sos_Param_List pl  = m.get_params();
   sos_Int	  crd = VALID(pl) ? pl.card() : 0;
   if (no < 1  OR  no > crd+1)
      mta_error(err_SYS, err_MTA_INVALID_POSITION);
   else
   {  
#ifdef OBST_PROT_ALL
      if (VALID (mmod)) // generate protocol info if needed
	 mmod.add_param (p, no);
#endif
      if (INVALID(pl))
      {  pl = sos_Param_List::create (cnt);
         m.set_params (pl);
      }
      pl.insert(no,p);
      sos_Bool inhc = (sos_Bool)
		      (m.get_kind() != sos_PRIVATE AND NOT m.get_is_static());
      err_block 
         mta_build_class (ct,inhc, TRUE);
      err_exception
         sos_Cursor c = pl.open_cursor();
         pl.move_cursor (c,no);
         pl.remove_at(c);
         p.destroy();
         pl.close_cursor (c);
	 if (pl.card() == 0)
	 {  pl.destroy();
	    m.set_params(sos_Param_List::make (NO_OBJECT));
         }
	 mta_repeat_last_error();
      err_block_end
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Method::insert_param ***

// *************************************************************************
void sos_Method::remove_param(sos_Int no)
// *************************************************************************
{  T_PROC ("sos_Method::remove_param");
   TT (mta_H, T_ENTER);
   
   sos_Method	  m   = self;
   sos_Class_type ct  = m.get_defined_in();
   sos_Container  cnt = m.container();
#ifdef OBST_PROT_ALL
   sos_MethodModif mmod = mta_get_method_mod (cnt, m);
#endif
 
   if (mta_is_local_method(m.get_name()))
      mta_error(err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED);
 
   mta_open_for_writing (cnt);
   sos_Param_List pl = m.get_params();
   sos_Int crd = VALID(pl) ? pl.card()
      			   : 0;
   if (no < 1  OR  no > crd)
      mta_error(err_SYS, err_MTA_INVALID_POSITION);
   else
   {
      sos_Param p = pl.get_nth(no);
#ifdef OBST_PROT_ALL
      if (VALID (mmod)) // generate protocol info if needed
	 mmod.del_param (p);
#endif
      pl.remove (no);
      sos_Bool inhcl = (sos_Bool)
		       (m.get_kind() != sos_PRIVATE AND NOT m.get_is_static());
      err_block
         mta_build_class (ct,inhcl, TRUE);
      err_exception
         pl.insert(no,p);
	 mta_repeat_last_error();
      err_block_end

      mta_destroy(p);
      if (pl.card() == 0)
      {  
#ifdef OBST_PROT_ALL
         sos_ModifHistory::trash(pl);
#else
         pl.destroy();
#endif
         m.set_params (sos_Param_List::make (NO_OBJECT));
      }
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Method::remove_param ***

// *************************************************************************
void sos_Method::modify_param(sos_Int	     no, 
			      sos_Type_descr st, 
			      sos_String     name, 
			      sos_Bool	     is_ref,
			      sos_Expr	     e)
// *************************************************************************
{  T_PROC ("sos_Method::modify_param");
   TT (mta_H, T_ENTER);
   
   sos_Method	  m   = self;
   sos_Class_type ct  = m.get_defined_in();
   sos_Container  cnt = m.container();
#ifdef OBST_PROT_ALL
   sos_MethodModif mmod = mta_get_method_mod (cnt, m);
#endif
 
   if (mta_is_local_method(m.get_name()))
      mta_error(err_SYS, err_MTA_METHOD_MUST_NOT_MODIFIED);
 
   sos_Param_List pl  = m.get_params();
   sos_Int	  crd = VALID(pl) ? pl.card() : 0;
   if (no < 1  OR  no > crd)
      mta_error(err_SYS, err_MTA_INVALID_POSITION);
   else
   {  sos_Param_List pl = m.get_params();
      sos_Param	     p  = pl.get_nth(no);
#ifdef OBST_PROT_ALL
      if (VALID (mmod)) // generate protocol info if needed
	 mmod.mod_param (p);
#endif
      sos_Bool inhcl = (sos_Bool)
	 	       (    m.get_kind() != sos_PRIVATE
			AND NOT m.get_is_static()
			AND st.operator!=(sos_Type_descr::make(p.get_type())));
 
      sos_Param bak_p = sos_Param::copy(p,TEMP_CONTAINER);
      bak_p.set_name(sos_String::make(NO_OBJECT));
 
      mta_open_for_writing (cnt);
      p.set_type (sos_Type_descr::make(st));
      p.set_is_ref (is_ref);
      p.set_default_expr (sos_Expr::make(NO_OBJECT));
 
      err_block
         mta_build_class (ct, inhcl, TRUE);
      err_exception
         p.assign(bak_p);
         mta_destroy (bak_p);
	 bak_p = sos_Param::make (NO_OBJECT);
         mta_repeat_last_error();
      err_block_end

      if (VALID (bak_p))
	 mta_destroy (bak_p);
      if (VALID (name))
         p.set_name(sos_String::copy(name,cnt));

      if (VALID (e))
         p.set_default_expr (mta_copy (e,cnt));
   }

   TT (mta_H, T_LEAVE);
} // *** sos_Method::modify_param ***



/* --------------------------------------------------------------------------
 *
 * Methods to modify components
 *
 * --------------------------------------------------------------------------
*/


LOCAL void mta_make_comp_method_names (const sos_String& name,
				       sos_String&       get_name,
				       sos_String&       set_name,
				       sos_Container     cnt = TEMP_CONTAINER)
// creates two Strings get_<name> und set_<name> in cnt
{  T_PROC ("mta_make_comp_method_names");
   TT (mta_M, T_ENTER);
   
   smg_String smg_name(name), fullnm;
 
   fullnm   = smg_String("get_") + smg_name;
   get_name = fullnm.make_String(cnt);
 
   fullnm   = smg_String("set_") + smg_name;
   set_name = fullnm.make_String(cnt);

   TT (mta_M, T_LEAVE);
} // mta_make_comp_method_names
 

LOCAL sos_Bool valid_comp_name (const sos_Object& o,
				const sos_String& name)
// tests the validity of the componentname name: True, if
// set_<name> and get_<name> did not exists in the metdhodtable o
{  T_PROC ("valid_comp_name");
   TT (mta_M, T_ENTER);
   
   sos_String get_name, set_name;
   mta_make_comp_method_names(name, get_name, set_name);

   sos_Method_table tmp_mt = sos_Method_table::make(o);

   sos_Bool result = (sos_Bool) (    NOT tmp_mt.is_key(get_name)
				 AND NOT tmp_mt.is_key(set_name));
   get_name.destroy();
   set_name.destroy();

   TT (mta_M, T_LEAVE);
   return result;
} 

 
EXPORT void mta_complete_components (const sos_Class_type& ct)
// Adds the component-list of each user-specified superclass to the
// components list of the class ct. Components from super_classes will
// be removed before, so complete_components could be called several times
// without creating the whole list.
{
   T_PROC ("mta_complete_components");
   TT (mta_H, T_ENTER);
   
   sos_Comp_descr_List  comp = ct.get_components();
      // remove all components that aren't defined in ct
   for (sos_Cursor c = comp.open_cursor();  comp.is_valid(c); )
   {  if (comp.get(c).get_get_method().get_defined_in().operator!=(ct))
         comp.remove_at (c);
      else
         comp.to_succ(c);
   }
   comp.close_cursor(c);
 
   sos_Super_class_List scl = ct.get_super_classes();
   agg_iterate(scl, sos_Super_class sc)
   {  // sct holds the superclass as an sos_Class_type.
      sos_Class_type	  sct = sos_Class_type::make(sc.make_type());
      sos_Comp_descr_List cdl = sct.get_components();
 
      agg_iterate(cdl, sos_Comp_descr cd)
         comp.find_or_append(cd); // comps. inherited over multiple paths
                                   // are inserted only once.
      agg_iterate_end(cdl, cd);
   }
   agg_iterate_end(scl, sc);

   TT (mta_H, T_LEAVE);
}

LOCAL void mta_set_inherited_offsets (const sos_Class_type& ct)
// Compute the component offsets in ct and all inherited classes
// and the offset of their superclasses.
// precondition: ct.super_classes...local_size have to be set
{ 
   T_PROC ("mta_set_inherited_offsets");
   TT(mta_H, T_ENTER);
   
   sos_Class_type_List inherited_classes = ct.get_subclasses ();
   agg_iterate (inherited_classes,sos_Class_type tmp_ct)
   {  
         mta_open_for_writing (tmp_ct.container());
         mta_set_offsets_and_size (tmp_ct);
   }
   agg_iterate_end (inherited_classes,ct);

   TT (mta_H, T_LEAVE);
}

EXPORT sos_Int mta_comp_size (const sos_Comp_descr& cd)
{
   T_PROC ("mta_comp_size");
   TT (mta_H, T_ENTER);

   sos_Type bt	      = cd.get_get_method().get_result_type().make_type();
   sos_Int  comp_size = 0;

   if (VALID (bt) AND bt.is_scalar())
      comp_size += bt.get_object_size();
   else if (cd.get_is_local())
      comp_size += SOS_OFFSET_SIZE;
   else
      comp_size += SOS_TYPED_ID_SIZE;

   TT (mta_H, T_LEAVE);
   return comp_size;
} 

// ***************************************************************************
sos_Comp_descr sos_Class_type::create_component ()
// ***************************************************************************
// creates a component with generated name (new_comp_xx) in the class ct.
// The new component is private.
{  T_PROC ("sos_Class_type::create_component");
   TT (mta_H, T_ENTER);
   
   sos_Class_type    ct   = self;
   sos_Container     cnt  = ct.container();
   sos_Schema_module sm   = sos_Schema_module::retrieve (cnt);
   sos_Method_table  inhm = ct.get_methods();
 
   mta_open_for_writing (cnt);
   smg_String smg_name =
      MTA_GENERATE_VALID_NAME("new_comp", inhm, valid_comp_name);
   sos_String new_name = smg_name.make_String (cnt);
 
   sos_String get_str, set_str;
   mta_make_comp_method_names(new_name, get_str, set_str, cnt);
 
      // create and initialize get method
   sos_Method get_cm = sos_Method::create(cnt);

   get_cm.set_name	  (get_str);
   get_cm.set_kind	  (sos_PRIVATE);
   get_cm.set_is_static   (FALSE);
   get_cm.set_is_operator (FALSE);
   get_cm.set_impls	  (sos_Method_impl_List::make(NO_OBJECT));
   get_cm.set_defined_in  (ct);
   get_cm.set_params	  (sos_Param_List::make (NO_OBJECT));
   get_cm.set_result_type (sos_Type_descr::make(sos_Object_type));
 
      // create and initialize set method
   sos_Param pa = sos_Param::create (cnt);
   pa.set_name	       (sos_String::make(NO_OBJECT));
   pa.set_type	       (sos_Type_descr::make(sos_Object_type));
   pa.set_default_expr (sos_Expr::make (NO_OBJECT));
   pa.set_is_ref       (FALSE);

   sos_Param_List pl = sos_Param_List::create (cnt, FALSE);
   pl.append (pa);

   sos_Method set_cm = sos_Method::copy(get_cm, cnt);
   set_cm.set_name	  (set_str);
   set_cm.set_params	  (pl);
   set_cm.set_result_type (sos_Type_descr::make(void_type));
 
      // create and initialize sos_Comp_desc
   sos_Comp_descr cd = sos_Comp_descr::create (cnt);

   cd.set_name		 (new_name);
   cd.set_init_expr	 (sos_Expr::make (NO_OBJECT));
   cd.set_is_value	 (FALSE);
   cd.set_is_local	 (FALSE);
   cd.set_offset	 (0xFFFFFFFF);
   cd.set_get_method	 (get_cm);
   cd.set_set_method	 (set_cm);
   get_cm.set_comp_descr (cd);
   set_cm.set_comp_descr (cd);
 
#ifdef OBST_HAVE_PROT
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));
   if (VALID(cmod)) // generate protocol information only when needed
      cmod.add_comp(cd);
#endif

   ct.get_local_methods().append (get_cm);
   ct.get_local_methods().append (set_cm);

      // even private/protected components are contained in the
      // component list of subclasses, but never in the method lists
   sos_Class_type_List subclasses = ct.get_subclasses();
   agg_iterate (subclasses, sos_Class_type sub)
      sos_Comp_descr_List comps = sub.get_components();
      mta_open_for_writing (comps.container());
      comps.append (cd);
   agg_iterate_end (subclasses, sub)

   assert_no_error_block("create_component")
      mta_destroy(ct.get_methods());
      ct.set_methods(sos_Method_table::make (NO_OBJECT));
      mta_build_methodtable (ct);
      mta_set_inherited_offsets (ct);
   assert_no_error_block_end
 
   TT (mta_H, T_LEAVE);
   return cd;
} // *** sos_Class_type::create_comp ***

// *************************************************************************
void sos_Comp_descr::remove()
// *************************************************************************
{  T_PROC ("sos_Comp_descr::remove");
   TT (mta_H, T_ENTER);
   
   sos_Comp_descr cd	     = self;
   sos_Method	  get_method = cd.get_get_method();
   sos_Method	  set_method = cd.get_set_method();
   sos_Class_type ct	     = get_method.get_defined_in();
   sos_Container  cnt	     = cd.container();
#ifdef OBST_HAVE_PROT
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));

   if (VALID(cmod)) // generate protocol info only when needed
      cmod.del_comp(cd);
#endif
   sos_Bool is_static = get_method.get_is_static();
   sos_Bool derived   = (sos_Bool) (    get_method.get_kind() != sos_PRIVATE
                       		    AND NOT get_method.get_is_static());

   sos_Class_type_List subclasses = ct.get_subclasses();
   agg_iterate (subclasses, sos_Class_type sub)
      mta_open_for_writing (sub.container());

          // if the sub is the class ct itsself, remove the component
          // methods from sos_Class_type::local_methods
          // Even derived components are not present in the 
          // local_methods list. So it is sufficient to 
          // remove get/set_method from ct only
      if (sub.operator==(ct))
      {  sos_Method_List local_methods = sub.get_local_methods();
         mta_remove_from_list (local_methods, get_method);
         mta_remove_from_list (local_methods, set_method);
      }

      if (sub.operator==(ct) OR NOT is_static)
         mta_remove_from_list (sub.get_components(), cd);
   agg_iterate_end (subclasses, sub)

   mta_build_class (ct, derived, TRUE);   

   TT (mta_H, T_LEAVE);
} 
  


// ***************************************************************************
void sos_Comp_descr::modify (sos_String	     new_name,
			     sos_Type_descr  new_type,
			     sos_Method_kind new_get_kind, 
			     sos_Method_kind new_set_kind, 
			     sos_Bool	     new_local,
			     sos_Bool	     new_definite,
			     sos_Bool,
			     sos_Expr	     new_init_expr)
// ***************************************************************************
{  T_PROC ("sos_Comp_descr::modify");
   TT (mta_H, T_ENTER);
   
   sos_Comp_descr cd  = self;
   sos_Class_type ct  = cd.get_get_method().get_defined_in();
   sos_Container  cnt = ct.container();
#ifdef OBST_PROT_ALL
   sos_ClassModif cmod = sos_ClassModif::make(MTA_GET_TYPE_MOD(cnt, ct));

   if (VALID(cmod)) // generate protocol information, if needed
      cmod.mod_comp(cd);
#endif

   sos_String new_get_name,new_set_name;
   mta_make_comp_method_names (new_name,new_get_name,new_set_name);

   sos_Int old_comp_size = mta_comp_size (cd);

   sos_Method      get_method	= cd.get_get_method();
   sos_Method      set_method	= cd.get_set_method();
   sos_Method_kind old_get_kind = get_method.get_kind();
   sos_Method_kind old_set_kind = set_method.get_kind();
   sos_Bool        old_local	= cd.get_is_local();
   sos_Bool	   old_definite = get_method.get_is_definite();
   sos_Expr        old_e	= cd.get_init_expr();
   sos_Type_descr  old_type	= get_method.get_result_type();
   sos_String	   old_get_name = sos_String::copy (get_method.get_name(),
						    TEMP_CONTAINER);
   sos_String	   old_set_name = sos_String::copy (set_method.get_name(),
						    TEMP_CONTAINER);
   sos_String	   old_name     = cd.get_name();

   // remark:
   // No check for valid init expression, because it could
   // be defined in *_ext.h. If not, and there's no create parameter and
   // no literal, compiling the generated files will cause an error.

   sos_Bool sizechange = (sos_Bool) (   NOT old_type.identical(new_type)
				     OR old_local != new_local);

   sos_Bool derived = (sos_Bool) (   old_get_kind != new_get_kind
                       		  OR old_set_kind != new_set_kind
		       		  OR old_definite != new_definite
                       		  OR NOT old_get_name.equal(new_get_name)
		       		  OR sizechange);

   mta_open_for_writing (cnt);
   get_method.get_name().assign (new_get_name);
   get_method.set_result_type (new_type);
   get_method.set_kind (new_get_kind);
   get_method.set_is_definite (new_definite);

   set_method.get_name().assign (new_set_name);
   set_method.set_kind (new_set_kind);
   set_method.set_is_definite (new_definite);
   set_method.get_params().get_nth(1).set_type (new_type);

   cd.set_init_expr (VALID(new_init_expr) ? mta_copy(new_init_expr, cnt)
		     			  : sos_Expr::make (NO_OBJECT));
   cd.set_is_local (new_local);
   cd.get_name().assign (new_name);

   err_block
      mta_build_class (ct, derived, TRUE, FALSE, sizechange);
   err_exception				// reset all changes
      get_method.get_name().assign (old_get_name);
      get_method.set_result_type (old_type);
      get_method.set_kind (old_get_kind);
      get_method.set_is_definite (old_definite);

      set_method.get_name().assign (old_set_name);
      set_method.get_params().get_nth(1).set_type(new_type);
      set_method.set_kind (old_set_kind);
      set_method.set_is_definite (old_definite);
      cd.set_is_local (old_local);
      cd.set_init_expr (old_e);
      cd.get_name().assign (old_name);

      mta_repeat_last_error();
   err_block_end

   /*
   sos_Class_type_List subclasses = ct.get_subclasses();
   agg_iterate (subclasses, sos_Class_type sub)
      sos_Comp_descr_List subcomps = sub.get_components();
      agg_iterate (subcomps, sos_Comp_descr sub_cd)
	 if (sub_cd != cd  AND  sub_cd.get_name().equal (cd.get_name()))
#ifdef OBST_HAVE_PROT
	    sos_ModifHistory::trash(sub_cd);
#else
	    sub_cd.destroy();
#endif
      agg_iterate_end (subcomps, sub_cd)
   agg_iterate_end (subclasses, sos_Class_type sub)
*/
#ifdef OBST_HAVE_PROT
   sos_ModifHistory::trash(old_get_name);
   sos_ModifHistory::trash(old_set_name);
#else
   old_get_name.destroy();
   old_set_name.destroy();
#endif

   TT (mta_H, T_LEAVE);
} // *** sos_Comp_descr::modify ***
