/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module knl
// **************************************************************************
// implements methods of classes: sos_Id, sos_Typed_id, sos_Object,
//                                sos_Ordered_object, sos_Scalar_object
//                                sos_String, sos_Type
// **************************************************************************

// ==========================================================================
// method implementations for the types of the kernel schema
//
// Tracing conventions: see trc_knl.h
// ==========================================================================

#define OBST_IMP_STDCONST
#define OBST_IMP_FORMATTED_IO
#define OBST_IMP_STRINGOP
#include "obst_stdinc.h"

#include "_obst_config.h"
#include "obst_progstd.h"
#include "smg.h"
#include "knl_err.h"
#include "trc_knl.h"
#include "cci_use.h"
#include "knl_obst.h"

extern  void _psm_initialize();
externC void _agg_init_obst(int);
externC void _dir_init_obst(int);
externC void _mta_init_obst(int);
externC void _cci_init_obst(int);
#ifdef OBST_HAVE_JOYCE
externC void _sync_init_obst(int);
externC void _trans_init_obst(int);
#endif

// ==========================================================================
// quasi constants, module initialization
// ==========================================================================

EXPORT sos_Bool	    obst_in_create;
EXPORT sos_Object   NO_OBJECT;
LOCAL  sos_Typed_id NOTPID;
LOCAL  sos_Id 	    NOID;

/*
 * Performs the basic initialization steps needed maybe even for
 * _knl_init_obst(). _knl_init_obst(0) must already have been called.
 * (There must not yet be any access to OBST facilities on the object level,
 *  hence the chosen form of initializing NO_OBJECT.)
 */
void _obst_basic_init()
{  static sos_Bool do_init = TRUE;
   if (do_init)
   {  do_init = FALSE;

      _smg_init();
      _psm_initialize();

      NOID      = sos_Id::make (sos_Container::make (0), 0);
      NOTPID    = sos_Typed_id::make (NOID, NOID);
      NO_OBJECT = sos_Object::make(NOTPID, NULL);

      // NULL must be replaced by __sos_Object_obj, if the possibility #2
      // mentioned in _cci_get_impl_obj is employed.
   }
}

/*
 * Initializes the core library, i.e. the boot schemata.
 */
EXPORT void _obst_initcore (OBST_NO_ARGS)
{  static int first_call = TRUE;
   if (first_call)
   {  first_call = FALSE;
      _knl_init_obst(999);
      _agg_init_obst(999);
      _mta_init_obst(999);
      _dir_init_obst(999);
      _cci_init_obst(999);

#ifdef OBST_HAVE_JOYCE
      _sync_init_obst(999);
      _trans_init_obst(999);
#endif
   }
}


// ==========================================================================
// local utility methods
// ==========================================================================

// *************************************************************************
LOCAL inline sos_Comp_result make_comp_result (int i)
// *************************************************************************
{  return (i > 0) ? CMP_GREATER
		  : (i < 0 ) ? CMP_LESS
			     : CMP_EQUAL;
}


// ==========================================================================
// string conversion operations for the external types of the kernel schema
// ==========================================================================

// **************************  sos_Int  ****************************

sos_Cstring make_string_from_sos_Int_object (const sos_Object& i)
{  smg_String s (make_sos_Int (i));
   return s.make_Cstring (SMG_TRANSFER);
}

sos_Object make_sos_Int_object_from_string (const sos_Cstring s)
{  char	      *ptr;
   long       l      = strtol (s, &ptr, 10);
   sos_Object result = (ptr != s AND ptr[0] == '\0')
			   ? make_sos_Int_object (sos_Int (l))
			   : NO_OBJECT;
   return result;
}

// **************************  sos_Float  ****************************

sos_Cstring make_string_from_sos_Float_object (const sos_Object& o)
{  char buf[20];
   sprintf (buf, "%e", (double) make_sos_Float(o));
   char* result = obst_strdup (buf);
   return result;
}

sos_Object make_sos_Float_object_from_string (const sos_Cstring str)
{  char       *ptr;
   double     d = strtod (str, &ptr);
   sos_Object result;

   if (ptr != str && !*ptr)
      result = make_sos_Float_object ((sos_Float)d);
   else
      result = NO_OBJECT;

   return result;
}

// **************************  sos_Char  ****************************

sos_Cstring make_string_from_sos_Char_object (const sos_Object& c)
{  char*  str = new char[2];
   str[0] = make_sos_Char (c);
   str[1] = '\0';

   return str;
}

sos_Object make_sos_Char_object_from_string (const sos_Cstring s)
{  sos_Object result = (strlen (s) == 1) ? make_sos_Char_object (s[0])
					 : NO_OBJECT;
   return result;
}

// **************************  sos_Pointer  ****************************

sos_Cstring make_string_from_sos_Pointer_object (const sos_Object& p)
{  sos_Pointer p1     = make_sos_Pointer (p);
   sos_Cstring result = smg_String((sos_Int)p1, FALSE)
			   .make_Cstring (SMG_TRANSFER);
   return result;
}

sos_Object make_sos_Pointer_object_from_string (const sos_Cstring s)
{  char       *ptr;
   long       l = strtol (s, &ptr, 16);
   sos_Object result = (ptr != s AND ptr[0] == EOS)
			   ? make_sos_Pointer_object (sos_Pointer (l))
			   : NO_OBJECT;
   return result;
}

// **************************  sos_CppFctPtr  ****************************

sos_Cstring make_string_from_sos_CppFctPtr_object (const sos_Object& p)
{  sos_CppFctPtr p1   = make_sos_CppFctPtr (p);
   sos_Cstring result = smg_String((sos_Int)p1, FALSE)
			   .make_Cstring (SMG_TRANSFER);
   return result;
}

sos_Object make_sos_CppFctPtr_object_from_string (const sos_Cstring s)
{  char       *ptr;
   long       l = strtol (s, &ptr, 16);
   sos_Object result = (ptr != s AND ptr[0] == EOS)
			   ? make_sos_CppFctPtr_object (sos_CppFctPtr (l))
			   : NO_OBJECT;
   return result;
}

// **************************  sos_Offset  ****************************

sos_Cstring make_string_from_sos_Offset_object (const sos_Object& o)
{  sos_Offset o1     = make_sos_Offset (o);
   char*      result = smg_String (o1).make_Cstring (SMG_TRANSFER);

   return result;
}

sos_Object make_sos_Offset_object_from_string (const sos_Cstring s)
{  char       *ptr;
   long	      l = strtol (s, &ptr, 10);

   sos_Object result = (ptr != s AND ptr[0] == EOS)
			   ? make_sos_Offset_object (sos_Offset (l))
			   : NO_OBJECT;
   return result;
}

// **************************  sos_Container  ****************************

sos_Cstring make_string_from_sos_Container_object (const sos_Object& ct)
{  sos_Container ct1    = make_sos_Container (ct);
   sos_Cstring	 result = smg_String ((sos_Int)ct1).make_Cstring(SMG_TRANSFER);

   return result;
}

sos_Object make_sos_Container_object_from_string (const sos_Cstring s)
{  char *ptr;
   long l = strtol (s, &ptr, 10);

   sos_Object result
	       = (ptr != s AND ptr[0] == EOS)
		    ? make_sos_Container_object (sos_Container::make ((int) l))
		    : NO_OBJECT;
   return result;
}

// **************************  sos_CString  ****************************

sos_Cstring make_string_from_sos_Cstring_object (const sos_Object& s)
{  return obst_strdup (make_sos_Cstring (s));
}

sos_Object make_sos_Cstring_object_from_string (const sos_Cstring s)
{  return make_sos_Cstring_object (obst_strdup (s));
}

// **************************  sos_Id  ********************************

sos_Cstring make_string_from_sos_Id_object (const sos_Object& i)
{  sos_Id     i1 = make_sos_Id (i);
   smg_String s  = smg_String("<") + (int)i1.container()
				   + "," + (int)i1.offset() + ">";

   return s.make_Cstring (SMG_COPY);
}

sos_Object make_sos_Id_object_from_string (const sos_Cstring s)
{  int        ct, os, l;
   sscanf (s, "<%d,%d>%n", &ct, &os, &l);

   sos_Object result;
   if (l == strlen(s))
      result = make_sos_Id_object
		  (sos_Id::make (sos_Container::make (ct), sos_Offset (os)));
   else
      result = NO_OBJECT;
   return result;
}


// ==========================================================================
// generic conversion operations for scalar types
// ==========================================================================

// *************************************************************************
void _obst_extern_from_object (void*             addr,
			       const sos_Object& o,
			       void              bcopy_to (void*,void*),
			       const sos_Id&     tp)
// *************************************************************************
{  T_PROC ("_obst_extern_from_object")
   TT (knl_M, T_ENTER);

   if (o._type_id() != tp)
      err_raise(err_SYS, err_SOS_TYPE_ERROR, "_obst_extern_from_object",FALSE);

   union {_obst_maxalign_t  dummy;
      	  sos_Id	    id;} u;

   u.id = o._self_id();
   bcopy_to (addr, &u.id);
      
   TT (knl_M, T_LEAVE);
}


// *************************************************************************
sos_Object _obst_object_from_extern (void*   	   addr,
				     void    	   bcopy_from (void*,void*),
				     const sos_Id& tp)
// *************************************************************************
{  T_PROC ("_obst_object_from_extern")
   TT (knl_M, T_ENTER);

   union {_obst_maxalign_t  dummy;
      	  sos_Id	    id;} u;

   u.id = NOID;
   // assures that the mapping scalar value <---> id is a bijection;
   // -> preserves equality and ordering
   //    identical extern values will yield identical objects

   bcopy_from (addr, &u.id);
   sos_Object obj = sos_Object::make(sos_Typed_id::make (u.id, tp));

   TT (knl_M, T_LEAVE);
   return obj;
}

// *************************************************************************
sos_Int _obst_Int_from_enum (const sos_Object& o, const sos_Type& et)
// *************************************************************************
{  T_PROC ("_obst_Int_from_enum")
   TT (knl_M, T_ENTER);

#ifdef BOOT
   err_assert (FALSE, "_obst_Int_from_enum:BOOT");
#endif
   if (NOT o.has_type (et))
      err_raise (err_SYS, err_SOS_TYPE_ERROR, "_obst_Int_from_enum", FALSE);

   sos_Int size = ((sos_Type&)et).get_object_size();
   sos_Id  id   = o._self_id();
   sos_Int result;
   if (size == 1)
      result = *(sos_Char*)&id;
   else
   {  err_assert (size == 2, "_obst_Int_from_enum:invalid size");
      result = *(sos_Short*)&id;
   }

   TT (knl_M, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Scalar_object _obst_enum_from_sos_Int (sos_Int i, const sos_Type& et)
// *************************************************************************
{  T_PROC ("_obst_enum_from_sos_Int")
   TT (knl_M, T_ENTER);

#ifdef BOOT
   err_assert (FALSE, "_obst_enum_from_sos_Int:BOOT");
#endif
   sos_Int size = ((sos_Type&)et).get_object_size();
   sos_Id  id   = NOID;	// see _obst_object_from_extern

   if (size == 1)
      *(sos_Char*)&id = i;
   else
   {  err_assert (size == 2, "_obst_enum_from_sos_Int:invalid size");
      *(sos_Short*)&id = i;
   }
   sos_Scalar_object result =
      sos_Scalar_object::make (sos_Typed_id::make(id, et._self_id()));

   TT (knl_M, T_LEAVE);
   return result;
}


// ==========================================================================
// handle container local references
// ==========================================================================

// *************************************************************************
sos_Typed_id _obst_make_local_typed_id (sos_Offset os, sos_Container ct)
// *************************************************************************
{  if (os == 0) return NOTPID;
   else
   {  sos_Id id = sos_Id::make (ct, os);
      return sos_Typed_id::make (id, id.get_type_id());
   }
}

// *************************************************************************
sos_Offset _obst_local_offset (OBST_PARDECL(sos_Object) o, sos_Container ct_id)
// *************************************************************************
{  sos_Offset os = o._self_id().offset();
   if (os != 0 AND o.container() != ct_id)
      err_raise (err_SYS, err_SOS_INVALID_ASSIGNMENT_TO_LOCAL,
                 "assignment to a component that is declared to be local",
		 FALSE);
   return os;
}

// ==========================================================================
// type constants / queries
// ==========================================================================

// *************************************************************************
sos_Type _obst_type_object (const sos_Id& id, const sos_Id& tp)
// *************************************************************************
{  sos_Typed_id tpid;
   cci_Impl_obj tpimpl;

#define _get_impl(other_tp)  _cci_get_impl_obj(_sos_Type_type, other_tp)
#ifdef BOOT
   static cci_Impl_obj common_impl = _get_impl (_sos_Type_type);

   tpid   = sos_Typed_id::make (id, _sos_Type_type);
   tpimpl = common_impl;

#else
   static cci_Impl_obj Class_impl  = _get_impl (_sos_Class_type_type),
                       Enum_impl   = _get_impl (_sos_Enum_type_type),
                       Extern_impl = _get_impl (_sos_Extern_type_type),
                       Union_impl  = _get_impl (_sos_Union_type_type);
   
   tpid = sos_Typed_id::make (id, tp);
   
   if (tp.offset() == _sos_Class_type_type.offset())
      tpimpl = Class_impl;
   else if (tp.offset() == _sos_Enum_type_type.offset())
      tpimpl = Enum_impl;
   else if (tp.offset() == _sos_Extern_type_type.offset())
      tpimpl = Extern_impl;
   else if (tp.offset() == _sos_Union_type_type.offset())
      tpimpl = Union_impl;
   else
      err_assert (FALSE, "_obst_type_object - unexpected type");
#endif

   return sos_Type::make (tpid, (_sos_Type*)tpimpl);
}

// *************************************************************************
sos_Type _obst_knl_type (register _obst_knl_types what_type)
// *************************************************************************
{  sos_Type result;

#ifndef BOOT
   switch (what_type)
   {  case _THE_VOID_TYPE  : result = void_type;		break;
      case _THE_BOOL_TYPE  : result = sos_Bool_type;		break;
      case _THE_EQKIND_TYPE: result = sos_Eq_kind_type;		break;
      case _THE_INT_TYPE   : result = sos_Int_type;		break;
      case _THE_CNT_TYPE   : result = sos_Container_type;	break;
      case _THE_OBJECT_TYPE: result = sos_Object_type;		break;
      case _THE_SCALAR_TYPE: result = sos_Scalar_object_type;
   }
#else
   static sos_Type 	    *the_types[_THE_SCALAR_TYPE + 1];
   static sos_String        the_string
				 = sos_String::create (TEMP_CONTAINER, "knl");
   static sos_Schema_module the_knl_schema
				 = sos_Schema_module::lookup (the_string);

   static char* the_type_names[] = { "void",          "sos_Bool",
				     "sos_Eq_kind",   "sos_Int",
				     "sos_Container",
				     "sos_Object",    "sos_Scalar_object" };
   if (the_types[what_type])
      result = *the_types[what_type];
   else
   {  err_assert (VALID (the_knl_schema), "_obst_knl_type - no knl schema");

      the_string.assign_Cstring (the_type_names[what_type]);
      the_types[what_type]  = new sos_Type;
      *the_types[what_type] =
      result 		    = the_knl_schema.lookup_type (the_string).
			      make_type();
   }
#endif
   return result;
}


// ==========================================================================
// exported utility methods
// ==========================================================================

// *************************************************************************
sos_Existing_status obst_ID_valid (const sos_Typed_id& obj_tpid)
// *************************************************************************
{  T_PROC ("knl - obst_ID_valid")
   TT (knl_M, T_ENTER);

   // This function relies on some properties of class sos_Container:
   //  o `object_exists' accepts any valid lower bound of the size of an object,
   //    in particular SOS_ID_SIZE (which is the size of sos_Object).
   //  o If `c.object_exists(o,l) == PERHAPS_EXISTING', then `c.read(o,l,...)'
   //    will succeed.
   //    This implies that `id.get_type_id()' will succeed if
   //    `id.container.object_exists(id.offset,SOS_ID_SIZE) == PERHAPS_EXISTING'
   //	 (Note that `object_exists' is very unprecise for the TEMP_CONTAINER.)

   sos_Existing_status result = PERHAPS_EXISTING;
   sos_Id	       tp_id  = obj_tpid.get_tp();

   if (NOTPID EQ obj_tpid)
      result = PERHAPS_EXISTING;
   else if (   obj_tpid.container().object_exists(obj_tpid.offset(),SOS_ID_SIZE)
	       EQ NOT_EXISTING
	    OR tp_id.container().object_exists(tp_id.offset(),SOS_ID_SIZE)
	       EQ NOT_EXISTING)
      result = NOT_EXISTING;
   else
   {  sos_Id tp_of_tp_id = tp_id.get_type_id();
      sos_Id tp_id2	 = obj_tpid.get_id().get_type_id();

      // o The type of the type of the given object must be one of the types
      //   from the OBST meta schema listed below.
      // o The type id stored in the `type_id' component of the given object
      //   (see sos_Object) must be the same as `obj_tpid.get_tp()'.

      if (tp_id2 NEQ tp_id
	  OR (    tp_of_tp_id NEQ _sos_Class_type_type
	      AND tp_of_tp_id NEQ _sos_Enum_type_type
	      AND tp_of_tp_id NEQ _sos_Extern_type_type
	      AND tp_of_tp_id NEQ _sos_Union_type_type))
	 result = NOT_EXISTING;
      else
      {  sos_Typed_id tp_tpid  = sos_Typed_id::make (tp_id, tp_of_tp_id);
	 sos_Type     obj_type = sos_Type::make (tp_tpid);

	 // Check now with the true size of the type object and then with the
	 // true size of the object itself.

	 if (   tp_id.container().object_exists(tp_id.offset(), obj_type.size())
	        EQ NOT_EXISTING
	     OR obj_tpid.container().object_exists(obj_tpid.offset(),
						   obj_type.get_object_size())
		EQ NOT_EXISTING)
	    result = NOT_EXISTING;
      }
   }
   TT (knl_M, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool obst_is_temporary (const sos_Object& o)
// *************************************************************************
{  T_PROC ("knl - obst_is_temporary")
   TT (knl_M, T_ENTER);

   sos_Bool result = (sos_Bool)(
			o.container() EQ TEMP_CONTAINER
			AND VALID (o)
		        AND NOT ((sos_Object&)o).isa (sos_Scalar_object_type));
   TT (knl_M, T_LEAVE);
   return result;
}

// ==========================================================================
// C++ class / external type sos_Id
// ==========================================================================

// *************************************************************************
void sos_Id::set_type_id (const sos_Id& tp) const
// *************************************************************************
{  union { sos_Id dummy;
	   char   c[SOS_ID_SIZE]; } u;

   bcopy_from_sos_Id ((void*)&tp, &u);
   container().write (offset(), SOS_ID_SIZE, &u);
}


// *************************************************************************
sos_Id sos_Id::get_type_id () const
// *************************************************************************
{  sos_Id tp;
   union { sos_Id dummy;
	   char   c[SOS_ID_SIZE]; } u;

   container().read (offset(), SOS_ID_SIZE, &u);
   bcopy_to_sos_Id (&tp, &u);

   return tp;
}

#ifndef INLINE_sos_Id_OP_equal
// *************************************************************************
sos_Bool sos_Id::operator== (const sos_Id& id) const
// *************************************************************************
{  return (sos_Bool) (os == id.os && ct == id.ct);
}
#endif

// ==========================================================================
// C++ class / external type sos_Typed_id
// ==========================================================================

// *************************************************************************
sos_Typed_id sos_Typed_id::make (const sos_Id& id)
// *************************************************************************
{  return sos_Typed_id::make (id, id.get_type_id());
};

// *************************************************************************
sos_Typed_id sos_Typed_id::allocate (const sos_Type& tp, sos_Container ct)
// *************************************************************************
{  T_PROC ("sos_Typed_id::allocate")
   TT (knl_M, T_ENTER);

#ifdef BOOT
   err_assert (FALSE, "sos_Typed_id::allocate : BOOT");
#endif
   sos_Typed_id tpid = allocate (tp._self_id(),
				 ct, ((sos_Type&)tp).get_object_size());
   TT (knl_M, T_LEAVE);
   return tpid;
}

// *************************************************************************
sos_Typed_id sos_Typed_id::allocate (const sos_Id& tp_id,
				     sos_Container ct,   sos_Int object_size)
// *************************************************************************
{  T_PROC ("sos_Typed_id::allocate")
   TT (knl_M, T_ENTER; TI (tp_id.container()); TI (tp_id.offset());
	      TI ((sos_Int) ct); TI (object_size));

   sos_Id id = sos_Id::make (ct, ct.allocate (object_size));
   id.set_type_id (tp_id);
   sos_Typed_id tpid = sos_Typed_id::make (id, tp_id);

   TT (knl_M, T_LEAVE);
   return tpid;
}


// ==========================================================================
// C++ class / external type obst_Root_class
//
// (Method implementations shadowing the respective methods of sos_Object.
//  These are the methods which are applicable to NO_OBJECT.)
// ==========================================================================

// *************************************************************************
sos_Bool obst_Root_class::identical (const obst_Root_class& o) const
// *************************************************************************
{  return _self == o._self;
}

// *************************************************************************
sos_Bool obst_Root_class::operator== (const obst_Root_class& o) const
// *************************************************************************
{  return _self == o._self;
}

// *************************************************************************
sos_Bool obst_Root_class::operator!= (const obst_Root_class& o) const
// *************************************************************************
{  return _self != o._self;
}

// **************************************************************************
sos_Comp_result obst_Root_class::compare_ids (const obst_Root_class& o) const
// **************************************************************************
{  T_PROC ("sos_Object::compare_ids")
   TT (knl_H, T_ENTER);

   sos_Id id1 = _self_id(),
	  id2 = o._self_id();

   register int cmp_val = id1.offset() - id2.offset();

   if (cmp_val EQ 0)
   {  cmp_val = id1.container() - id2.container();

      if (cmp_val EQ 0)
      {  sos_Id tp1 = _type_id(),
		tp2 = o._type_id();

         cmp_val = tp1.offset() - tp2.offset();

	 if (cmp_val EQ 0)
	    cmp_val = tp1.container() - tp2.container();
      }
   }
   sos_Comp_result result = make_comp_result (cmp_val);

   TT (knl_H, T_LEAVE);
   return result;
}


// ==========================================================================
// class sos_Object
// ==========================================================================

// *************************************************************************
sos_Id sos_Object::get_type_id ()
// *************************************************************************
{  err_raise_not_implemented ("sos_Object::get_type_id");
   return self._self_id();
}

// *************************************************************************
void sos_Object::set_type_id (sos_Id _x)
// *************************************************************************
{  err_raise_not_implemented ("sos_Object::set_type_id");
   sos_Id _y = _x;
   _y = self._self_id();
}

// *************************************************************************
sos_Int sos_Object::size ()
// *************************************************************************
{  T_PROC ("sos_Object::size")
   TT (knl_H, T_ENTER);

   sos_Int result = self.container().realsize (this->_size);

   TT (knl_H, T_LEAVE; TI (result));
   return result;
}

// *************************************************************************
sos_Type sos_Object::type ()
// *************************************************************************
{  T_PROC ("sos_Object::type")
   TT (knl_H, T_ENTER);

#ifdef BOOT
   sos_Typed_id tpid = sos_Typed_id::make (self._type_id(), _sos_Type_type);
#else
   sos_Id	id   = self._type_id();
   sos_Typed_id tpid = sos_Typed_id::make (id, id.get_type_id());
#endif
   sos_Type t = sos_Type::make (tpid);

   TT (knl_H, T_LEAVE);
   return t;
}

// *************************************************************************
sos_Bool sos_Object::isa (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Object::isa")
   TT (knl_H, T_ENTER);

   sos_Bool result;
   if (self._type_id() == tp._self_id())
      result = TRUE;
   else
   {
#ifdef BOOT
      err_assert (FALSE, "isa:BOOT");
#endif
      result = self.type().is_derived_from (tp);
   }
   TT (knl_H, T_LEAVE; TB(result));
   return result;
}

// *************************************************************************
sos_Bool sos_Object::is_some (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Object::is_some")
   TT (knl_H, T_ENTER);

#ifdef BOOT
   err_assert (FALSE, "is_some:BOOT");
#endif
   sos_Bool result = (self._type_id() == tp._self_id())
      			? TRUE
      			: self.type().is_derived_from_some (tp);

   TT (knl_H, T_LEAVE; TB(result));
   return result;
}

// *************************************************************************
sos_Bool sos_Object::has_type (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Object::has_type")
   TT (knl_H, T_ENTER);

   sos_Bool b = (self._type_id() == tp._self_id());

   TT (knl_H, T_LEAVE; TB(b));
   return b;
}

// **************************************************************************
sos_Bool sos_Object::is_value ()
// **************************************************************************
{  T_PROC ("sos_Object::is_value")
   TT (knl_H, T_ENTER);

   // value components are not supported in this version!

   err_raise_not_implemented ("sos_Object::is_value");

   TT (knl_H, T_LEAVE);
   return (sos_Bool)(self.offset() == -1);   // FALSE (suppresses warnings)
}

// *************************************************************************
sos_Bool sos_Object::like (sos_Object anobject)
// *************************************************************************
{  return anobject.equal (self, /*eq_kind*/ EQ_WEAK);
}

// *************************************************************************
void sos_Object::local_assign (sos_Object, sos_Object)
// *************************************************************************
{  T_PROC ("sos_Object::local_assign")
   TT (knl_M, T_ENTER);

   // Do nothing.
   // Note that a generated local_assign would copy the component 'type_id'.

   TT (knl_M, T_LEAVE);
}

// *************************************************************************
sos_Bool sos_Object::local_equal (sos_Object,sos_Object,sos_Eq_kind)
// *************************************************************************
{  T_PROC ("sos_Object::local_equal")
   TT (knl_M, T_ENTER);

   // Note that a generated local_equal would compare the component 'type_id'.

   TT (knl_M, T_LEAVE; TB(TRUE));
   return TRUE;
}

// *************************************************************************
sos_Int sos_Object::local_hash_value (sos_Object)
// *************************************************************************
{  T_PROC ("sos_Object::local_hash_value")
   TT (knl_M, T_ENTER);
   TT (knl_M, T_LEAVE; TI (0));

   // Note, that a generated 'hash_value' method would hash on 'type_id' which
   // would in general require that the 'equal' relation is identical to the
   // 'identical' relation.

   return 0;
}


// ==========================================================================
// class sos_Ordered_object
// ==========================================================================

// *************************************************************************
sos_Bool sos_Ordered_object::operator< (sos_Ordered_object o)
// *************************************************************************
{  T_PROC ("sos_Ordered_object::operator<")
   TT (knl_H, T_ENTER);

   sos_Bool result = (sos_Bool)(self.compare (o) == CMP_LESS);

   TT (knl_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool sos_Ordered_object::operator<= (sos_Ordered_object o)
// *************************************************************************
{  T_PROC ("sos_Ordered_object::operator<=")
   TT (knl_H, T_ENTER);

   sos_Bool result = (sos_Bool)(self.compare (o) != CMP_GREATER);

   TT (knl_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool sos_Ordered_object::operator> (sos_Ordered_object o)
// *************************************************************************
{  T_PROC ("sos_Ordered_object::operator>")
   TT (knl_H, T_ENTER);

   sos_Bool result = (sos_Bool)(self.compare (o) == CMP_GREATER);

   TT (knl_H, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Bool sos_Ordered_object::operator>= (sos_Ordered_object o)
// *************************************************************************
{  T_PROC ("sos_Ordered_object::operator>=")
   TT (knl_H, T_ENTER);

   sos_Bool result = (sos_Bool)(self.compare (o) != CMP_LESS);

   TT (knl_H, T_LEAVE);
   return result;
}


// ==========================================================================
// class sos_Scalar_object
// ==========================================================================

// *************************************************************************
void sos_Scalar_object::local_initialize (sos_Scalar_object)
// *************************************************************************
{  T_PROC ("sos_Scalar_object::local_initialize")
   TT (knl_M, T_ENTER);

   err_raise (err_USE, err_SOS_ILLEGAL_EXT_OP,
		       "sos_Scalar_object::local_initialize", FALSE);
   TT (knl_M, T_LEAVE);
}

// *************************************************************************
void sos_Scalar_object::local_finalize (sos_Scalar_object)
// *************************************************************************
{  T_PROC ("sos_Scalar_object::local_finalize")
   TT (knl_M, T_ENTER);

   err_raise (err_USE, err_SOS_ILLEGAL_EXT_OP,
		       "sos_Scalar_object::local_finalize", FALSE);
   TT (knl_M, T_LEAVE);
}

// *************************************************************************
void sos_Scalar_object::total_assign (sos_Scalar_object, sos_Object)
// *************************************************************************
{  T_PROC ("sos_Scalar_object::total_assign");
   TT (knl_M, T_ENTER);

   err_raise (err_USE, err_SOS_ILLEGAL_EXT_OP,
		       "sos_Scalar_object::total_assign", FALSE);
   TT (knl_M, T_LEAVE);
}

// *************************************************************************
sos_Bool sos_Scalar_object::total_equal (sos_Scalar_object x,
				         sos_Object 	   o, sos_Eq_kind)
// *************************************************************************
{  T_PROC ("sos_Scalar_object::total_equal")
   TT (knl_M, T_ENTER);

   sos_Bool result = x.identical (o);

   TT (knl_M, T_LEAVE);
   return result;
}

// *************************************************************************
sos_Int sos_Scalar_object::total_hash_value (sos_Scalar_object scalo)
// *************************************************************************
{  T_PROC ("sos_Scalar_object::total_hash_value")
   TT (knl_M, T_ENTER);

   sos_Int result = scalo.offset();

   TT (knl_M, T_LEAVE);
   return result;
}


// ==========================================================================
// class sos_String
// ==========================================================================

// *************************************************************************
sos_String sos_String::create (sos_Container cnt, sos_Cstring cstr)
// *************************************************************************
{  T_PROC ("sos_String::create/2")
   TT (knl_H, T_ENTER);

   sos_String result = sos_String::create(cnt);
   if (cstr && *cstr)
   {  int	 size;
      sos_Offset addr;

      result.set_length  (size = strlen(cstr));
      result.set_address (addr = cnt.allocate (size));
      cnt.write (addr, size, cstr);
   }
   TT (knl_H, T_LEAVE);
   return result;
}

// *************************************************************************
void sos_String::operator+= (sos_String s)
// *************************************************************************
{  T_PROC ("sos_String::operator+=")
   TT (knl_H, T_ENTER);

   smg_String cat = smg_String (self) + s;
   self.assign_Cstring (cat.make_Cstring (SMG_BORROW));

   TT (knl_H, T_LEAVE);
}

// *************************************************************************
void sos_String::assign_Cstring (sos_Cstring s)
// *************************************************************************
{  T_PROC ("sos_String::assign_Cstring")
   TT (knl_H, T_ENTER);

   sos_Int       size = strlen (s),
		 old_size = self.get_length();
   sos_Container ct = self.container();
 
   if (old_size != size)
   {  self.set_length (size);
      if (old_size > 0)              // remove the old contents
	 ct.deallocate (self.get_address(), old_size);
   }
   if (size > 0)
   {  sos_Offset new_addr;

      if (old_size == size)      // it's not necessary to allocate new space
         new_addr = self.get_address();
      else
      {  new_addr = ct.allocate (size);
         self.set_address (new_addr);
      }  
      ct.write (new_addr, size, s);
   }
   TT (knl_H, T_LEAVE; TI (old_size); TI (size));
}

// *************************************************************************
sos_Cstring sos_String::_make_Cstring ()
// *************************************************************************
{  T_PROC ("sos_String::_make_Cstring")
   TT (knl_H, T_ENTER);

#define OBST_MAKE_CSTRING_BUFS	3

   static sos_Cstring buf[OBST_MAKE_CSTRING_BUFS]      /* = { NULL, ... } */;
   static int	      buf_size[OBST_MAKE_CSTRING_BUFS] /* = { 0, ..., 0 } */;
   static int	      curr_buf			       /* = 0 */;

   int size = self.get_length();

   if (-- curr_buf < 0)
      curr_buf = OBST_MAKE_CSTRING_BUFS-1;

   if (size >= buf_size[curr_buf])
   {  if (buf_size[curr_buf])
	 delete buf[curr_buf];
      buf[curr_buf] = new char[buf_size[curr_buf] = size+1];
   }
   self.container().read (self.get_address(), size, buf[curr_buf]);
   buf[curr_buf][size] = '\0';

   TT (knl_H, T_LEAVE; TI (size));
   return buf[curr_buf];
}

// *************************************************************************
sos_Cstring sos_String::make_Cstring ()
// *************************************************************************
{  T_PROC ("sos_String::make_Cstring")
   TT (knl_H, T_ENTER);

   int         size = self.get_length();
   sos_Cstring str  = new char[size+1];

   self.container().read (self.get_address(), size, str);
   str[size] = '\0';

   TT (knl_H, T_LEAVE; TI (size));
   return str;
}

// **************************************************************************
sos_Int sos_String::size()
// **************************************************************************
{  T_PROC ("sos_String::size")
   TT (knl_H, T_ENTER);

   sos_Int sz = self.container().realsize (self.get_length())
		+ self.sos_Object::size();

   TT (knl_H, T_LEAVE);
   return sz;
}

// **************************************************************************
sos_Comp_result sos_String::compare (sos_Ordered_object o)
// **************************************************************************
{  T_PROC ("sos_String::compare")
   TT (knl_H, T_ENTER);

   sos_Comp_result result;
   if (o.isa (sos_String_type))
   {  smg_String s1 = self;
      smg_String s2 = sos_String::make (o);

      result = make_comp_result (s1.compare (s2));
   }
   else
   {  err_raise (err_USE, err_SOS_TYPE_ERROR, "sos_String::compare", FALSE);
      result = CMP_LESS;
   }
   TT (knl_H, T_LEAVE);
   return result;
}

// *************************************************************************
void sos_String::local_initialize (sos_String str)
// *************************************************************************
{  T_PROC ("sos_String::local_initialize")
   TT (knl_M, T_ENTER);

   str.set_length (0);
 
   TT (knl_M, T_LEAVE);
}

// *************************************************************************
void sos_String::local_finalize (sos_String str)
// *************************************************************************
{  T_PROC ("sos_String::local_finalize")
   TT (knl_M, T_ENTER);

   sos_Int size = str.get_length();

   if (size > 0)
      str.container().deallocate (str.get_address(), size);

   TT (knl_M, T_LEAVE);
}

// *************************************************************************
void sos_String::local_assign (sos_String x, sos_Object o)
// *************************************************************************
{  T_PROC ("sos_String::local_assign")
   TT (knl_M, T_ENTER);

   smg_String ostring = sos_String::make (o);
   x.assign_Cstring (ostring.make_Cstring (SMG_BORROW));
 
   TT (knl_M, T_LEAVE);
}

// *************************************************************************
sos_Bool sos_String::local_equal (sos_String x, sos_Object o, sos_Eq_kind)
// *************************************************************************
{  T_PROC ("sos_String::local_equal")
   TT (knl_M, T_ENTER);

   sos_String y   = sos_String::make (o);
   sos_Int    l_y = y.get_length();
   
   sos_Bool result = (sos_Bool)(x.get_length() == l_y AND
				x.container().equal(x.get_address(), l_y,
						    y.container(),
						    y.get_address()));
   TT (knl_M, T_LEAVE; TB(result));
   return result;
}

// *************************************************************************
sos_Int sos_String::local_hash_value (sos_String x)
// *************************************************************************
{  T_PROC ("sos_String::local_hash_value")
   TT (knl_M, T_ENTER);

   sos_Int result = x.container().hash_value (x.get_address(), x.get_length());

   TT (knl_M, T_LEAVE; TI (result));
   return result;
}


// ==========================================================================
// class sos_Type
// ==========================================================================

// *************************************************************************
sos_Bool sos_Type::is_derived_from (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Type::is_derived_from")
   TT (knl_H, T_ENTER);

   sos_Bool result;

   if (   tp._self_id() == self._self_id()
#ifdef BOOT
       OR tp._self_id() == _obst_knl_type (_THE_OBJECT_TYPE)._self_id()
#else
       OR tp._self_id() == _sos_Object_type
#endif
      )
      result = TRUE;

   // the above is equivalent to 'self.equal(tp) OR tp.equal (sos_Object_type)'
   // (the method is redefined if the above would not hold or be not correct)

   else if (tp._type_id() == _sos_Union_type_type)
      result = sos_Union_type::make(tp).is_base_of (self);

   else
      result = FALSE;

   TT (knl_H, T_LEAVE; TB (result));
   return result;
}

// *************************************************************************
sos_Bool sos_Type::is_derived_from_some (sos_Type tp)
// *************************************************************************
{  T_PROC ("sos_Type::is_derived_from_some")
   TT (knl_H, T_ENTER);

   sos_Bool result = self.is_derived_from (tp);

   TT (knl_H, T_LEAVE; TB (result));
   return result;
}

// *************************************************************************
sos_Bool sos_Type::is_scalar ()
// *************************************************************************
{  T_PROC ("sos_Type::is_scalar")
   TT (knl_H, T_ENTER);
   TT (knl_H, T_LEAVE; TB(FALSE));

   return FALSE;			// default implementation
}

// *************************************************************************
sos_Type sos_Type::root ()
// *************************************************************************
{  T_PROC ("sos_Type::root")
   TT (knl_H, T_ENTER);

   sos_Type t = self;

   TT (knl_H, T_LEAVE);
   return t;
}
