/* --------------------------------------------------------------------------
 * Copyright 1994 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the GNU
 * General Public License as published by the Free Software Foundation;
 * either version 2, or (at your option) any later version.
 *
 * This license agreement differs from the standard agreement for OBST
 * software since this file includes part of a source file covered by
 * GPL version 1 (see below).
 * The addition of this file to the OBST distribution is seen as a mere
 * aggregation of work that does not influence the licensing terms of the
 * OBST distribution itself.
 *
 * If you want additional information on OBST, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
// **************************************************************************
// implementation of incremental loading based on GNU's dld
// **************************************************************************

#define OBST_COMPILE_DEBUG_STUFF 0
#if OBST_COMPILE_DEBUG_STUFF && defined(NO_TT)
#  undef NO_TT
#endif

// --------------------------------------------------------------------------
// Tracing conventions: see trc_cci.h
// --------------------------------------------------------------------------

#include "_obst_config.h"

#define OBST_IMP_STDCONST
#define OBST_IMP_STRINGOP
#define OBST_IMP_FILE
#define OBST_IMP_MALLOC
#if OBST_COMPILE_DEBUG_STUFF
#  define OBST_IMP_A_OUT
#endif
#include "obst_stdinc.h"

#include "obst_progstd.h"
#include "obst.h"
#include "smg.h"
#include "cci_err.h"
#include "trc_cci.h"
#include "cci_obst.h"
#include "cci_incrload.h"

#if OBST_INCRLD == LD_COLLECT  ||  OBST_INCRLD == LD_GNU
	// We are not using "__GLOBAL_$" since some GCC installations seem not
	// to use '$' for symbol names.
#  define COMMONprefix	"__GLOBAL_"
#  define KEYoffset	1
#  define CTORkey	'I'
#  define DTORkey	'D'
#endif

#if OBST_INCRLD == LD_PATCH  ||  OBST_INCRLD == LD_MUNCH
#  define COMMONprefix	"___st"
#  define KEYoffset	0
#  define CTORkey	'i'
#  define DTORkey	'd'
#endif


extern "C" {
/* ---------------------------------------------------------------------------
 * The following definitions are taken from the dld 3.2.3 distribution.
 * They are covered by the following copyright and license:
 */

/* Copyright (C) 1990 by W. Wilson Ho.

   The author can be reached electronically by how@ivy.ucdavis.edu or
   through physical mail at:

   W. Wilson Ho
   Division of Computer Science
   University of California at Davis
   Davis, CA 95616
 */

/* This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 1, or (at your option) any
   later version. */

/* ------------------------------------------
 * This portion is taken from "dld.h" (i.e. the standard external interface).
 * They are included here to make this file self-contained (and because
 * this file does already include some dld source code).
 */

extern int dld_undefined_sym_count;     /* # of undefined global symbols */

extern int dld_init (const char*);      /* initialize the dld routines */
extern dld_link (const char*);	        /* dynamically link and load an object
					   file */
extern int
dld_create_reference (const char*);	/* explicitly create a reference to the
                                           given symbol. */
extern int
dld_function_executable_p (const char*);/* return true if the named C function
					   is executable */
extern char **
dld_list_undefined_sym ();	        /* return array of undefined symbols */


/* ------------------------------------------
 * This portion is taken from "defs.h" (i.e. the internal interface of dld).
 * In the production version, we only need an access path to all symbols loaded
 * by dld. This is underlined by adding just a few definitions and by
 * additionally making the glosym type partially opaque.
 * In the debugging version, we access further internal data structures.
 */
#if !OBST_COMPILE_DEBUG_STUFF
   typedef
   struct glosym
   {
      /* Pointer to next symbol in this symbol's hash bucket.  */
      struct glosym *link;
      /* Name of this symbol.  */
      char *name;
      /* Value of this symbol as a global symbol.  */
      long value;
   }
   symbol;

   /* Number of buckets in symbol hash table */
#  define TABSIZE 1009

   /* The symbol hash table: a vector of TABSIZE pointers to struct glosym. */
   extern symbol *_dld_symtab[TABSIZE];

#else

/* Each input file, and each library member ("subfile") being loaded,
   has a `file_entry' structure for it.

   For files specified by command args, these are contained in the vector
   which `file_table' points to.

   For library members, they are dynamically allocated,
   and chained through the `chain' field.
   The chain is found in the `subfiles' field of the `file_entry'.
   The `file_entry' objects for the members have `superfile' fields pointing
   to the one for the library.  */

struct file_entry {
    /* Name of this file.  */
    char *filename;
    /* Name to use for the symbol giving address of text start */
    /* Usually the same as filename, but for a file spec'd with -l
       this is the -l switch itself rather than the filename.  */
    char *local_sym_name;

    /* For library member, points to next entry for next member.
       For object or library *file*, points to previously loaded entry */
    struct file_entry *chain;

    /* number of undefined symbols referenced by this module */
    int undefined_symbol_count;

    /* chain of file_entry that defines symbols this file references */
    struct file_chain *refs;

    /* chain of file_entry that references symbols defined in this file */
    struct file_chain *refs_by;

    /* reference count -- number of entries referenceing myself */
    int ref_count;

    /* Describe the layout of the contents of the file */

    /* The file's a.out header.  */
    struct exec header;

    /* Describe data from the file loaded into core */

    /* Symbol table of the file.  */
    struct nlist *symbols;
    /* Size in bytes of string table.  */
    int string_size;
    /* Pointer to the string table.
       The string table is not kept in core all the time,
       but when it is in core, its address is here.  */
    char *strings;

    /* Relocation information of the file. */

    /* Start of this file's text relocation information. */
    struct dld_reloc_info *text_reloc;
    /* Start of this file's data relocation information. */
    struct dld_reloc_info *data_reloc;

    /* Relation of this file's segments to the output buffer */

    /* Start of this file's text seg in the output file core image.  */
    int text_start_address;
    /* Start of this file's data seg in the output file core image.  */
    int data_start_address;
    /* Start of this file's bss seg in the output file core image.  */
    int bss_start_address;

    /* For library members only */

    /* For a library, points to chain of entries for the library members.  */
    struct file_entry *subfiles;
    /* For a library member, offset of the member within the archive.
       Zero for files that are not library members.  */
    int starting_offset;
    /* Size of contents of this file, if library member.  */
    int total_size;
    /* For library member, points to the library's own entry.  */
    struct file_entry *superfile;

    /* 1 if file is a library. */
    char library_flag;

    /* 1 if file's header has been read into this structure.  */
    char header_read_flag;

    /* 1 if this module has all external references resolved */
    char all_symbols_resolved_flag;

    /* 1 if functions in this module can be safely executed. */
    char executable_flag;

    /* 1 if this module has already been (soft) unlinked. */
    char already_unlink;
    /* 1 means search a set of directories for this file.  */
    /* char search_dirs_flag; */
};


/* format of file_entry chain */
struct file_chain {
    struct file_chain *next;
    struct file_entry *entry;
};

/* Symbol table */

/* Global symbol data is recorded in these structures,
   one for each global symbol.
   They are found via hashing in 'symtab', which points to a vector of buckets.
   Each bucket is a chain of these structures through the link field.  */

typedef
  struct glosym
    {
      /* Pointer to next symbol in this symbol's hash bucket.  */
      struct glosym *link;
      /* Name of this symbol.  */
      char *name;
      /* Value of this symbol as a global symbol.  */
      long value;
      /* Points to the file_entry that defines this symbol */
      struct file_entry *defined_by;
      /* chain of file_entry that contains reference to this symbol */
      struct file_chain *referenced_by;
      /* Nonzero means a definition of this global symbol is known to exist.
         Library members should not be loaded on its account.  */
      char defined;
      /* Nonzero means a reference to this global symbol has been seen
         in a file that is surely being loaded. */
      char referenced;
    }
  symbol;

/* Number of buckets in symbol hash table */
#define TABSIZE 1009

/* The symbol hash table: a vector of TABSIZE pointers to struct glosym. */
extern symbol *_dld_symtab[TABSIZE];

/* pointer to the lastest (newest) file entry */
extern struct file_entry *_dld_latest_entry;


#endif /* OBST_COMPILE_DEBUG_STUFF */

/* End of excerpt from dld 3.2.3
 * ---------------------------------------------------------------------------
 */
} /* extern "C" */


// --------------------------------------------------------------------------
// module initialization
// --------------------------------------------------------------------------

extern "C" cci_LoadFct incrld_DLD;    // must be referenced to load this module

LOCAL void init_DLD()
{  T_PROC ("DLD_incrld : init_DLD")
   TT (cci_M, T_ENTER);

   cci_load_fct = incrld_DLD;

   TT (cci_M, T_LEAVE);
}
LOCAL int _dummy = (init_DLD(),0);


#if OBST_COMPILE_DEBUG_STUFF
// --------------------------------------------------------------------------
// debugging stuff for dld structures
// --------------------------------------------------------------------------

#define _BOOL(b)   ((b) ? "j" : "n")
#define _FNAME(f)  ((f && f->filename) ? f->filename : "<NULL>")

LOCAL void _dld_print_file (int all, struct file_entry *fe = _dld_latest_entry)
{  if (!fe)
      return;

   cout << "\nfile            : "   << _FNAME(fe)
        << "\nnext file       : "   << _FNAME(fe->chain)
        << "\nlocal sym name  : "   << fe->local_sym_name
	<< "\nundef syms (out): "   << fe->undefined_symbol_count
	<< "\nhas undef syms  : "   << _BOOL(! fe->all_symbols_resolved_flag)
	<< "\nis executable   : "   << _BOOL(fe->executable_flag)
	<< "\nreferences (in) : "   << fe->ref_count
	<< "\nsymbol table    : 0x" << (unsigned long)fe->symbols
	<< "\nstring table    : 0x" << (unsigned long)fe->strings
	<< "\nstring table sz.: "   << fe->string_size
	<< "\ntext reloc. tbl.: 0x" << (unsigned long)fe->text_reloc
	<< "\ndata reloc. tbl.: 0x" << (unsigned long)fe->data_reloc
	<< "\ntext rel. addr. : 0x" << (unsigned long)fe->text_start_address
	<< "\ndata rel. addr. : 0x" << (unsigned long)fe->data_start_address
	<< "\nbss  rel. addr. : 0x" << (unsigned long)fe->bss_start_address
	<< "\nheader read     : "   << _BOOL(fe->header_read_flag)
	<< "\nis library      : "   << _BOOL(fe->library_flag)
	<< "\nis lib member   : "   << _BOOL(fe->starting_offset)
	;

   if (fe->starting_offset)
      cout << "\nstarting offset : 0x" << (unsigned long)fe->starting_offset
	   << "\ntotal size      : "   << fe->total_size
	   << "\nlibrary         : "   << _FNAME(fe->superfile)
	   ;

   if (fe->header_read_flag)
      cout << "\n(hdr) dynamic   : "   << _BOOL(fe->header.a_dynamic)
	   << "\n(hdr) tool vers.: "   << fe->header.a_toolversion
	   << "\n(hdr) mach type : "   << (int)fe->header.a_machtype
	   << "\n(hdr) magic no. : "   << fe->header.a_magic
	   << "\n(hdr) text size : "   << (long)fe->header.a_text
	   << "\n(hdr) data size : "   << (long)fe->header.a_data
	   << "\n(hdr) bss  size : "   << (long)fe->header.a_bss
	   << "\n(hdr) sym size  : "   << (long)fe->header.a_syms
	   << "\n(hdr) txt reloc.: "   << (long)fe->header.a_trsize
 	   << "\n(hdr) data reloc: "   << (long)fe->header.a_drsize
	   << "\n(hdr) entry pnt.: 0x" << fe->header.a_entry
	   ;

   cout << "\n";

   if (all)
   {  if (fe->library_flag)
	 _dld_print_file (all, fe->subfiles);

      _dld_print_file (all, fe->chain);
   }
}

LOCAL void _dld_print_sym (int all, struct glosym* sym = (struct glosym*)-1)
{  if (! sym)
      return;

   if (sym == (struct glosym*)-1)
   {  if (all)
      {  for (int i = 0;  i < TABSIZE;  ++ i)
	    _dld_print_sym (all, _dld_symtab[i]);
      }
      return;
   }

   cout << "\nsymbol    : "   << sym->name
        << "\nvalue     : 0x" << (unsigned long)sym->value
	<< "\ndefined by: "   << _FNAME(sym->defined_by)
	<< "\nreferenced: "   << _BOOL(sym->referenced)
	<< "\ndefined   : "   << _BOOL(sym->defined)
	<< "\n";

   if (all)
      _dld_print_sym (all, sym->link);
}

LOCAL void _dld_find_by_prefix (char* prefix)
{  int preflen = strlen(prefix);

   struct glosym** outer;
   for (outer = _dld_symtab + TABSIZE-1;  outer >= _dld_symtab;  -- outer)
   {  struct glosym* inner;

      for (inner = *outer;  inner;  inner = inner->link)
	 if (streqln (prefix, inner->name, preflen))
	    _dld_print_sym (FALSE, inner);
   }
}

#endif /* OBST_COMPILE_DEBUG_STUFF */


// --------------------------------------------------------------------------
// auxiliary functions: ?tor handling
// --------------------------------------------------------------------------

// **************************************************************************
LOCAL void read_xtors (cci_XtorList& ctors,
		       cci_XtorList& dtors,
		       sos_Bool      do_enter)
// **************************************************************************
{  T_PROC ("DLD_incrld - read_xtors")
   TT (cci_VL, T_ENTER);

   // DLD does not offer an efficient access path to all newly read
   // symbols with a common prefix.
   // Hence, we have to record (and later ignore) all previously seen xtors;
   // and we have to scan DLD's global symbol table each time.

   static sos_Object_sos_Object_Mapping_Hash
      known_xtors = sos_Object_sos_Object_Mapping_Hash::create (TEMP_CONTAINER);

   int preflen = strlen (COMMONprefix);

   struct glosym** outer;
   for (outer = _dld_symtab + TABSIZE-1;  outer >= _dld_symtab;  -- outer)
   {  struct glosym* inner;

      for (inner = *outer;  inner;  inner = inner->link)
	 if (streqln (COMMONprefix, inner->name, preflen))
	 {  void*      addr = (void*)inner->value;
	    sos_Object key  = make_sos_Pointer_object (addr);

#if OBST_COMPILE_DEBUG_STUFF
	    sos_Bool is_key = known_xtors.is_key (key);

	    _dld_print_sym (FALSE, inner);
	    cout << "known     : " << _BOOL(is_key)
	         << "\nentering  : " << _BOOL(!is_key && do_enter)
	         << "\n\n";
#endif
	    if (known_xtors.is_key (key))
	       continue;
	    known_xtors.insert (key, NO_OBJECT);

	    // The xtor function entry of a single object module is a function
	    // which calls all other xtor's of that module.
	    // Hence, entering the symbols value suffices.

	    if (do_enter)
	       if (inner->name[preflen+KEYoffset] == CTORkey)
		  ctors.add ((cci_Xtor_p)addr);
	       else
	       {  err_assert (inner->name[preflen+KEYoffset] == DTORkey,
			      "DLD_incrld - read_xtors: unknown symbol");

		  dtors.add ((cci_Xtor_p)addr);
	       }
	 }
   }
   TT (cci_VL, T_LEAVE);
}


// --------------------------------------------------------------------------
// main functions for incremental loading
// --------------------------------------------------------------------------

// **************************************************************************
EXPORT void incrld_DLD (const sos_Object_List &schemas,
		        const sos_String_List &objects,
		        const sos_String_List &libs)
// **************************************************************************
{  T_PROC ("incrld_DLD")
   TT (cci_M, T_ENTER);

   static const char*     here = "incrld_DLD";
   static cci_XtorList    ctors;
   static sos_String_List stdlibs;

   static sos_Bool first_load = TRUE;
   if (first_load)
   {  first_load = FALSE;

      if (cci_incr_ld.echo)
	 cerr << "...initializing with " << cci_incr_ld.executable << "\n";

      if (dld_init (cci_incr_ld.executable))
	 err_raise (err_SYS,
		    err_CCI_READ_HEADER_FAILED, cci_incr_ld.executable);

      ctors.init();
      read_xtors (ctors, cci_incr_ld.dtors, FALSE);

      stdlibs = sos_String_List::make (NO_OBJECT);

      smg_String libstr;
      if (cci_incr_ld.ld_param1)
      {  libstr = (char*)cci_incr_ld.ld_param1;
	 if (cci_incr_ld.ld_params)
	    libstr += smg_String(" ") + cci_incr_ld.ld_params;
      }
      else if (LDLIBSXX)
	 libstr = LDLIBSXX;

      if (libstr.length())
      {  stdlibs = sos_String_List::create (TEMP_CONTAINER);

	 char*       str = libstr.make_Cstring (SMG_BORROW);
	 const char* sep = " \t";
	 char*	     _fname;

	 while (_fname = strtok (str, sep))
	 {  smg_String fname;

	    if (streqln (_fname, "-l", 2))
	       fname = smg_String("lib") + (_fname+2) + ".a";
	    else
	       fname = _fname;

	    int fd = cci_incr_ld.find_file (fname);
	    close (fd);

	    stdlibs.append (fname.make_String (TEMP_CONTAINER));

	    str = NULL;
	 }
      }
   }

   if (VALID (schemas))
   {  smg_String initfct;
      agg_iterate (schemas, sos_Object _sm)
	 initfct = smg_String("_")
	 	   + sos_Named::make(_sm).get_name() + "_init_obst";

         if (dld_create_reference (initfct.make_Cstring (SMG_BORROW)))
	    err_raise (err_SYS, "can't create reference to symbol",
		       		initfct.make_Cstring (SMG_BORROW));
      agg_iterate_end (schemas, _sm)
   }

   for (int i = 0;  i < 3;  ++ i)
   {  sos_String_List sl;

      if (i == 0)
	 sl = objects;
      else if (i == 1)
	 sl = libs;
      else
	 sl = stdlibs;

      if (VALID (sl))
         agg_iterate (sl, sos_String s)
         {  smg_String path = s;

	    if (i == 1)
	    {  char* p = path.make_Cstring (SMG_BORROW);
	       if (streqln (p, "-l", 2))
	          path = smg_String("lib") + (p+2) + ".a";
	    }
	    int fd = cci_incr_ld.find_file (path);
	    
	    if (cci_incr_ld.echo)
	       cerr << "...loading " << path << "\n";
	    
	    if (dld_link (path.make_Cstring (SMG_BORROW)))
	       err_raise (err_SYS, err_CCI_INC_LOAD_FAILED,
			  	   path.make_Cstring (SMG_BORROW));
	    close (fd);
	 }	 
	 agg_iterate_end (sl, s);
   }

   char** symlist = dld_list_undefined_sym();
   if (dld_undefined_sym_count)
   {  for (int i = 0; i < dld_undefined_sym_count; i++)
	 cerr << "\tundefined symbol: " << symlist[i] << "\n";
      free (symlist);

      err_raise (err_SYS, err_CCI_INC_LOAD_FAILED, here);
   }

#if OBST_COMPILE_DEBUG_STUFF
   _dld_print_file(TRUE);
   //_dld_print_sym(TRUE);
#endif

   read_xtors (ctors, cci_incr_ld.dtors, TRUE);

   ctors.process (FALSE);
   ctors.clear();

   TT (cci_M, T_LEAVE);
}
