/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
/* ========================================================================= */
/* MODULE IMPLEMENTATION                                                     */
/* ========================================================================= */
/*                                                                           */
/* MODULE : Extended Locking Synchronization                                 */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.10.1993              */
/*                                                                           */
/* CHANGES: see .changes                                                     */
/*                                                                           */
/* PROBLEMS: this module redefines the commit procedure of locking           */
/*           transactions to cope with the existence of other                */
/*           synchronization protocols.                                      */
/*                                                                           */
/* VERSION: 1.1                                                              */
/* ========================================================================= */

#include "obst_progstd.h"
#include "sync_obst.h"
#include "ext_sync_obst.h"
#include "sync_decl.h"
#include "sync_err.h"
#include "trc_tsy.h"
#include "sync_trc.h"

/* ========================================================================= */
/* CLASS IMPLEMENTATION                                                      */
/* ========================================================================= */
/*                                                                           */
/* CLASS-NAME: ExtLockingTA                                                  */
/*                                                                           */
/* ORIGINAL                                                                  */
/* AUTHOR: Michael Ranft                       DATE: 01.04.1992              */
/*                                                                           */
/* ========================================================================= */


/* ================================================================ */
/* Method: ExtLockingTA::commit                                        */
/*                                                                  */
/* Purpose: Closes all opended containers, if ta not implicit.      */
/*          Calls commit method on every accessed data item,        */
/*          which releases locks and awakes waiting transactions.   */
/*                                                                  */
/* Error: SYNC_OK                                                   */
/* ================================================================ */

void ExtLockingTA::commit()
{
   T_PROC("ExtLockingTA::commit");
   TT(tsy_H, T_ENTER);
   TTA(self);

   SyncCont        syct;        // root to all sync data
   sos_TaTab       optatab;
   sos_PidSet      pid_set;
   sos_HostProc_Id pid;
   sos_Timestamp   acts;
   Transaction     ta;
   OptimisticTA    opta;


   // is ta aborted internally ?
   if (self.get_status() == ABORTED)
     { self.set_error(SYNC_OK);
       TT(tsy_H, TXT("Intern aborted"); T_LEAVE);
       return; }

   if (!self.get_writeset().is_empty())
    { // distribute sets among optimistic transactions
      TT(tsy_L, TXT("distribute writeset"));
      syct    = SyncCont::get_root();
 
      if (!syct.get_opta_tab().is_empty())
        { // there are optimistic ta
          acts    = syct.get_ts() - SOS_OLD_TA_TS;
          pid_set = sos_PidSet::create(TEMP_CONTAINER);

          optatab = syct.get_opta_tab(); 
          agg_iterate_association (optatab, pid, ta)
           { TT(tsy_VL, TI(pid); TI(ta.get_type()));
 
	     opta = OptimisticTA::make(ta);
             if ((opta.get_ts() < acts) &&
                 (!syct.process_check(pid)))
              { // process of old optimistic ta is aborted
                // (only for long running (old) optimistic ta a process_check
                //  is performed to keep the effort low. An OptTa is old if
                //  SOS_OLD_TA_TS have been started since the OptTa was 
                //  started)
                pid_set.insert(pid);
              }
             else
              { // distribute
                opta.get_commitset() += self.get_writeset();
                TSOSET(opta.get_commitset());
              }
 
           } agg_iterate_association_end (optatab, pid, ta);
 
 
          // remove aborted process
          if (!pid_set.is_empty())
           { agg_iterate ( pid_set, pid )
              {
                syct.process_remove(pid);
              }
             agg_iterate_end (pid_set, pid);
             pid_set.clear();
           }
        }
    }

   self.LockingTA::commit();

   // end commit
   TT(tsy_H, T_LEAVE);
   return;
}

