/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the license
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* OBST LIBRARY MODULE */
// **************************************************************************
// Module err                                         Bernhard Schiefer (bs)
//                                                           Juergen Uhl (ju
//                                                      Dietmar Theobald (dt)
// **************************************************************************
// OBST error handler               
// **************************************************************************
//
// tracing conventions: see trc_err.h

#define OBST_IMP_STDCONST
#define OBST_IMP_STREAM
//      cerr
#define OBST_IMP_FORMATTED_IO
//      fprintf
// #define OBST_IMP_FILE
#define OBST_IMP_MALLOC
//      REALLOC
#define OBST_IMP_PROCEXEC
#define OBST_IMP_STRINGOP
//      strlen strcpy

#include "obst_stdinc.h"


#include "trc_err.h"
#include "obst_progstd.h"
#include "obst_err.h"
#include "obst.h"

/*
 * Result of implicit initialization at process startup in comments.
 */
void (*err_output_handler)() /* = NULL */;

LOCAL int err_sys_sum /* = 0 */;
LOCAL int err_use_sum /* = 0 */;
LOCAL int err_wng_sum /* = 0 */;

LOCAL err_class last_class  /* = err_SYS */;
LOCAL err_msg   last_raised /* = NULL */;
LOCAL err_msg   last_origin /* = NULL */;

// --------------------------------------------------------------------------

class err_buffer
{  char *buf;
   int  size;

 public:
   //err_buffer() { buf = NULL; size = 0; }

   char* buffer_string (const char*);
};

/* Note: the initializations performed by the default constructor are
 *	 done implicitely for these variables - even if the constructor is
 *	 not invoked (as happens if this file is part of a SunOS 4.1* shared
 *	 object.)
 */
LOCAL err_buffer err_raised_buffer;
LOCAL err_buffer err_origin_buffer;

char* err_buffer::buffer_string (const char *str)
{  T_PROC ("err_buffer::buffer_string")
   TT (err_L, T_ENTER);

   char* result = NULL;
   if (str)
   {  int strsize = strlen (str) + 1;

      if (strsize > size)
      {  if (buf)
	    delete buf;
	 buf  = new char[strsize];
	 size = strsize;
      }
      result = strcpy (buf, str);
   }
   TT (err_L, T_LEAVE);
   return result;
}

// --------------------------------------------------------------------------

EXPORT void err_raise (err_class ec, err_msg msg, err_msg where /* = NULL */,
		       int copy /* = 1 */)
{  T_PROC ("err_raise")
   TT (err_H, T_ENTER; TI (ec));

   if (copy)
   {  last_raised = err_raised_buffer.buffer_string (msg);
      last_origin = err_origin_buffer.buffer_string (where);
   }
   else
   {  last_raised = msg;
      last_origin = where;
   }
   last_class = ec;

   if (ec != err_SYS  OR  NOT err_top_env)
   {  if (err_output_handler)
	 (*err_output_handler)();

      else if (obst_initialized)
      {  cerr << ((ec == err_WNG) ? " Warning " : " ERROR ");
	 if (where)
	    cerr << ((ec == err_WNG) ? "for " : "in ")
		 << where << " ";

	 cerr << ": " << msg << "\n";
	 cerr.flush();
      }
      else
      {  fprintf(stderr, ((ec == err_WNG) ? " Warning " : " ERROR "));
	 if (where)
	    fprintf(stderr, "%s %s ", ((ec == err_WNG) ? "for" : "in"),
				      where);

	 fprintf(stderr, ": %s\n", msg);
	 fflush(stderr);
      }
   }
   switch (ec)
   {  case err_SYS : {  err_sys_sum++;
			if (err_top_env)
			{  sigjmp_buf* top_env = err_top_env;
			   err_pop_env();

			   siglongjmp (*top_env, 1);
			}
			else
			   abort();
		     }
      case err_USE : err_use_sum++; break;
      case err_WNG : err_wng_sum++; break;
   }  // switch

   TT (err_H, T_LEAVE);
}

EXPORT err_class err_last_class ()
{  return last_class;
}

EXPORT err_msg err_last_raised ()
{  return last_raised;
}

EXPORT err_msg err_last_origin ()
{  return last_origin;
}

EXPORT int err_occurred (err_class ec)
{  T_PROC ("err_occurred")
   TT (err_H, T_ENTER; TI(ec));

   int result;

   switch (ec)
   {  case err_SYS : result = err_sys_sum; break;
      case err_USE : result = err_use_sum; break;
      case err_WNG : result = err_wng_sum; break;
   }
   TT (err_H, T_LEAVE; TI(result));
   return result;
}

EXPORT void err_reset ()
{  T_PROC ("err_reset")
   TT (err_H, T_ENTER);
   
   err_sys_sum = 0;   err_use_sum = 0;   err_wng_sum = 0;

   TT (err_H, T_LEAVE);
}

// --------------------------------------------------------------------------

#define ENVSTACK_INCR	2
#define ENVSTACK_LIMIT  1024
	   // ..LIMIT must be a multiple of ..INCR

EXPORT sigjmp_buf* err_top_env;
LOCAL  sigjmp_buf* err_envstack;
LOCAL  int	   err_envstack_size /* = 0 */;
LOCAL  int	   err_envstack_next /* = 0 */;

EXPORT void err_push_env()
{  T_PROC ("err_push_env")
   TT (err_L, T_ENTER);

   if (err_envstack_next >= err_envstack_size)
   {  if (err_envstack_size)
         err_envstack = (sigjmp_buf*)
			REALLOC (err_envstack,
				 sizeof(sigjmp_buf)
				    * (err_envstack_size += ENVSTACK_INCR));
      else
      {  err_envstack 	    = (sigjmp_buf*)new sigjmp_buf[ENVSTACK_INCR];
	 err_envstack_size += ENVSTACK_INCR;
      }
#ifndef ERR_NO_ASSERT
      if (err_envstack_size == ENVSTACK_LIMIT)		// warn just once!
	 err_raise (err_WNG, "err_push_env", "possible memory leak");
#endif
   }
   err_top_env = err_envstack + (err_envstack_next ++);
   
   TT (err_L, T_LEAVE);
}

EXPORT void err_pop_env()
{  T_PROC ("err_pop_env")
   TT (err_L, T_ENTER);

   if (-- err_envstack_next > 0)
      err_top_env = err_envstack + err_envstack_next - 1;
   else
   {  err_assert (err_envstack_next == 0,
		  "err_pop_env - exception stack underflow");

      err_top_env = NULL;
   }
   TT (err_L, T_LEAVE);
}
