/* --------------------------------------------------------------------------
 * Copyright 1992-1993 by Forschungszentrum Informatik (FZI)
 *
 * You can use and distribute this software under the terms of the licence
 * version 1 you should have received along with this software.
 * If not or if you want additional information, write to
 * Forschungszentrum Informatik, "STONE", Haid-und-Neu-Strasse 10-14,
 * D-76131 Karlsruhe, Germany.
 * --------------------------------------------------------------------------
 */
/* **************************************************************************
 * check_machdeps
 *
 * This program is intended to verify that the dependencies of OBST on C++
 * types are fulfilled by the mapping of OBST-->C++ types on a given machine.
 *
 * The first two tests (type sizes, alignment) must be passed by any mapping,
 * the last test should be passed if the memory layout of types on a Sun
 * workstation is to be used for the persistent representation on disk.
 * Such a standard persistent representation will be necessary for any
 * heterogeneous environment, where the OBST database will be accessed from
 * different machines.
 * In case all tests are passed, there should be no need to perform a full
 * bootstrap, i.e. the short (sun) bootstrap form can be used when installing
 * OBST.
 * **************************************************************************
 */

#include <stdio.h>
#include <memory.h>

// --------------------------------------------------------------------------

// Define some basic types/operations as they would be in the OBST installation
// on the given machine.
// It should suffice to adapt the typedefs sos_Char .. sos_Offset as well as
// the bcopy-Operations.

typedef double		_obst_maxalign_t;
typedef char		sos_Char;
typedef short		sos_Short;
typedef int		sos_Int;
typedef float		sos_Float;
typedef void*		sos_Pointer;
typedef char*		sos_Cstring;
typedef unsigned	sos_Offset;

class sos_Container { sos_Int i; };
class sos_Id	    { sos_Offset os; sos_Container ct; };
class sos_Typed_id  { sos_Id id1, id2; };
class sos_Object    { sos_Typed_id tp; void* p; };

typedef sos_Object 	(*cci_Fun)(sos_Object&, sos_Object&);

#define SOS_CHAR_SIZE		1
#define SOS_SHORT_SIZE		2
#define SOS_INT_SIZE		4
#define SOS_FLOAT_SIZE		4
#define SOS_POINTER_SIZE	4
#define SOS_CSTRING_SIZE	4
#define SOS_OFFSET_SIZE		4
#define SOS_CONTAINER_SIZE	SOS_INT_SIZE
#define SOS_ID_SIZE		8
#define SOS_TYPED_ID_SIZE	16
#define CCI_FUN_SIZE		4

inline void bcopy_from_sos_Int (register void *i, register void *c)
{  *(sos_Int *)c = *(sos_Int *)i;
}
inline void bcopy_from_sos_Float (register void *f, register void *c)
{  *(sos_Float *)c = *(sos_Float *)f;
}
inline void bcopy_from_sos_Offset (register void *o, register void *c)
{  *(sos_Offset *)c = *(sos_Offset *)o;
}
inline void bcopy_from_sos_Pointer (register void *p, register void *c)
{  *(sos_Pointer *)c = *(sos_Pointer *)p;
}

/*
 * sos_Short is not an external type. But the equivalent to the following
 * implementation is used in the kernel routines which convert from/to an
 * external type.
 * These are: _obst_Int_from_enum, _obst_enum_from_sos_Int plus the component
 *	      access methods generated in genCC.
 */
void bcopy_from_sos_Short (register void *s, register void *c)
{  *(sos_Short *)c = *(sos_Short *)s;
}

// --------------------------------------------------------------------------

int error_occured;
int status=0;

void wrong_IntProperty (char *property, char* tpname, int wanted, int real)
{  printf ("property %s (%s) == %d, required: %d\n",
	   property, tpname, real, wanted);
   error_occured = 1;
   status++;
}

void wrong_Property (char *property, char* tpname)
{  printf ("property %s (%s): error\n", property, tpname);
   error_occured = 1;
   status++;
}

void start_test()
{  error_occured = 0;
}

void report_test (char* descr)
{  printf ("\ntest - %s: %s\n\n", descr, (error_occured ? "failed" : "passed"));
}

// --------------------------------------------------------------------------

int main (int argc, char* argv[])
{
   start_test();	// *** type sizes ***

#  define assert_typesize(tpname, wanted, real) \
      if ((wanted)!=(real)) wrong_IntProperty ("sizeof", tpname, wanted, real)

   assert_typesize ("sos_Char",      SOS_CHAR_SIZE,	 sizeof(sos_Char));
   assert_typesize ("sos_Short",     SOS_SHORT_SIZE,	 sizeof(sos_Short));
   assert_typesize ("sos_Int",       SOS_INT_SIZE, 	 sizeof(sos_Int));
   assert_typesize ("sos_Float",     SOS_FLOAT_SIZE, 	 sizeof(sos_Float));
   assert_typesize ("sos_Pointer",   SOS_POINTER_SIZE,   sizeof(sos_Pointer));
   assert_typesize ("sos_Cstring",   SOS_CSTRING_SIZE,   sizeof(sos_Cstring));
   assert_typesize ("sos_Offset",    SOS_OFFSET_SIZE,    sizeof(sos_Offset));
   assert_typesize ("sos_Container", SOS_CONTAINER_SIZE, sizeof(sos_Container));
   assert_typesize ("sos_Id", 	     SOS_ID_SIZE,        sizeof(sos_Id));
   assert_typesize ("sos_Typed_id",  SOS_TYPED_ID_SIZE,  sizeof(sos_Typed_id));
   assert_typesize ("cci_Fun",	     CCI_FUN_SIZE,	 sizeof(cci_Fun));

   report_test ("type sizes");


   start_test();	// *** alignment ***

   /*
    * check alignment (see bundle_main.C for reasons).
    */
   typedef double maxalign_t;

   int tp_al, dummyforal;

#define compute_alignment(tp, var) \
   {  struct { maxalign_t _x; char reffield; tp tpfield;} s;	\
      var = (int)&s.tpfield - (int)&s.reffield; }

#define op_identical(a,b)  ((a)==(b))
#define op_multiple(a,b)   ((b)%(a)==0)
#define assert_alignment(tp, tpname, wanted, cmp_op) \
   compute_alignment (tp, dummyforal);	\
      if (!cmp_op(dummyforal,wanted))	\
	 wrong_IntProperty ("alignment", tpname, wanted, dummyforal)

   int obst_maxal, int_al, offset_al, id_al;
   
   compute_alignment (_obst_maxalign_t, obst_maxal);
   compute_alignment (sos_Int,		int_al);
   compute_alignment (sos_Offset,	offset_al);
   compute_alignment (sos_Id,		id_al);

   assert_alignment (char,	    "char",	     1, 	 op_identical);
   assert_alignment (sos_Char,	    "sos_Char",      1, 	 op_identical);
   assert_alignment (sos_Short,	    "sos_Short",     id_al, 	 op_multiple);
   assert_alignment (sos_Int,	    "sos_Int",       obst_maxal, op_multiple);
   assert_alignment (sos_Float,	    "sos_Float",     obst_maxal, op_multiple);
   assert_alignment (sos_Pointer,   "sos_Pointer",   obst_maxal, op_multiple);
   assert_alignment (sos_Cstring,   "sos_Cstring",   obst_maxal, op_multiple);
   assert_alignment (sos_Offset,    "sos_Offset",    obst_maxal, op_multiple);
   assert_alignment (sos_Container, "sos_Container", obst_maxal, op_multiple);
   assert_alignment (sos_Id,	    "sos_Id",	     obst_maxal, op_multiple);
   assert_alignment (sos_Typed_id,  "sos_Typed_id",  int_al, 	 op_identical);
   assert_alignment (cci_Fun, 	    "cci_Fun",	     obst_maxal, op_multiple);

   report_test ("alignment");


   start_test();	// byte ordering 

   char tstbuf[4];
   int  i;

#  define get_tstval(tp)\
   union U { char buf[20]; tp val; };U _u;\
   for (i = 0;  i < 20;  ++ i)\
      _u.buf[i] = i+1;

get_tstval(sos_Float)

#  define assert_order(tp, tpname, bcopy_from, tstval)\
   {  tp tstvar = (tp)tstval;                    \
      memset (tstbuf, 0, 4);                     \
      bcopy_from ((void*)&tstvar, (void*)tstbuf);\
      for (i = 0;  i < sizeof(tp);  ++ i)        \
      {  if (tstbuf[i] != i+1)                   \
            wrong_Property ("byte order", tpname); }}
   
   // Note, that sos_Short is only needed for enumeration types with more than
   // 256 literals. That means, that a failure of the following assertion
   // can be ignored in practice, IFF the role of sos_Short has not been
   // extended in the meantime. I.e. sos_Short must only be used in the two
   // kernel functions mentioned above plus the genCC function which generates
   // component access methods.
   assert_order (sos_Short,  "sos_Short",  bcopy_from_sos_Short,  0x0102);

   assert_order (sos_Int,    "sos_Int",    bcopy_from_sos_Int,    0x01020304);
   assert_order (sos_Offset, "sos_Offset", bcopy_from_sos_Offset, 0x01020304);
   assert_order (sos_Pointer,"sos_Pointer",bcopy_from_sos_Pointer,0x01020304);

   printf ("(The following test might fail simply due to differences in");
   printf (" the string-->float conversion routines.)\n");

   assert_order (sos_Float,  "sos_Float",  bcopy_from_sos_Float,
		 2.38793926e-38);

   report_test ("SUN byte ordering");

   return (status);
}
