/* resched.c  -  resched */

#include <conf.h>
#include <kernel.h>
#include <proc.h>
#include <q.h>
#include <magic.h>
#include <quantum.h>

/*------------------------------------------------------------------------
 * resched  --  reschedule processor to highest priority ready process
 *
 * Notes:	Upon entry, currpid gives current process id.
 *		Proctab[currpid].pstate gives correct NEXT state for
 *			current process if other than PRCURR.
 *------------------------------------------------------------------------
 */
# define BADSP(pp) \
    kprintf("bad SP (%x) [%x,%x] for pid %d\n", \
	    pp->pregs[SSP], pp->pbase, pp->plimit, currpid );

int	resched()
{
	register struct	pentry	*optr;	/* pointer to old process entry */
	register struct	pentry	*nptr;	/* pointer to new process entry */
	register int i;

	/* no switch needed if current process priority higher than next*/

	if ( ( (optr= &proctab[currpid])->pstate == PRCURR) ) {
	    if( (lastkey(rdytail)<optr->pprio) )
		return(OK);
	}

	/* force context switch */
	if (optr->pstate == PRCURR) {
		optr->pstate = PRREADY;
		insert(currpid,rdyhead,optr->pprio);
	}

	/* remove highest priority process at end of ready list */

	nptr = &proctab[ (currpid = getlast(rdytail)) ];
	nptr->pstate = PRCURR;		/* mark it currently running	*/
#ifdef RTCLOCK
	preempt = QUANTUM;		/* reset preemption counter	*/
#endif RTCLOCK

#ifndef NOSTACKCHECK
	if( *( (int *)nptr->pbase ) != MAGIC ) {
	    kprintf("bad MAGIC for pid %d\n", currpid );	    
	    panic("stack corrupted");
	}
	if( ((unsigned)nptr->pregs[SSP]) < ((unsigned)nptr->plimit) ) {
	    BADSP(nptr);
	    panic("stack overflow");
	}
	else if( ((unsigned)nptr->pregs[SSP]) > ((unsigned)nptr->pbase) ) {
	    BADSP(nptr);
	    panic("stack underflow");
	}
#endif NOSTACKCHECK

#ifdef TRACE
	if( optr != nptr ) {
	    kprintf("%s -> %s\n", optr->pname, nptr->pname );
/*	    prdump(); /**/
	}
#endif
	ctxsw(optr->pregs,nptr->pregs);

	/* The OLD process returns here when resumed. */
#ifdef TRACE
	if( currpid != NULLPROC ) {
	    kprintf("back in %s fp= %x sp=%x\n", optr->pname,
		    optr->pregs[FP], optr->pregs[SSP] );
	}
#endif TRACE
#ifdef STACKDUMP
	{
	    long *p, i;
	    p = (long *) optr->pregs[SSP];
	    for( i = 0; i < 5 ; i++ ) {
		kprintf("%6x/ %8x  ", p, *p );
		p++;
		kprintf("%6x/ %8x\n", p, *p );
		p++;
	    }
	}
#endif STACKDUMP
	return(OK);
}
