/* NSC 32000 ROM debugger.
 * Bruce Culbertson  Bob Krause
 *
 * Expression parser.
 */

#include "debugger.h"

/* Parse an expression from the passed string.  For consistency, all
 * command handlers should use this for parsing numeric arguments.
 * Advances *p past the current expression.
 *
 * Return NO_NUM, BAD_NUM or GOT_NUM.  See help for = command for
 * expression syntax.
 */
int
getIntScan (p, c)
char **p;
long *c;
{
  long accum, val;
  char op;
  int ret;

  ret = getIntScan1 (p, &accum);
  if (ret != GOT_NUM) return ret;
  for (;;) {
    scan (p);
    op = **p;
    if (op != '-' && op != '+' && op != '|') break;
    ++(*p);
    if (getIntScan1 (p, &val) != GOT_NUM) return BAD_NUM;
    switch (op) {
      case '+':	accum += val; break;
      case '-':	accum -= val; break;
      case '|':	accum |= val; break;
    }
  }
  *c = accum;
  return GOT_NUM;
}

/* Part of expression parser, parses a term.
 */
int
getIntScan1 (p, c)
char **p;
long *c;
{
  long accum, val;
  char op;
  int ret;

  ret = getIntScan2 (p, &accum);
  if (ret != GOT_NUM) return ret;
  for (;;) {
    scan (p);
    op = **p;
    if (op != '*' && op != '/' &&
        op != '%' && op != '<' &&
        op != '>' && op != '&')
      break;
    if (op == '<' || op == '>') {
      if (op != *(*p + 1)) return BAD_NUM;
      ++(*p);
    }
    ++(*p);
    if (getIntScan2 (p, &val) != GOT_NUM) return BAD_NUM;
    switch (op) {
      case '*':	accum *= val; break;
      case '/':	accum /= val; break;
      case '%':	accum %= val; break;
      case '<':	accum <<= val; break;
      case '>':	accum >>= val; break;
      case '&':	accum &= val; break;
    }
  }
  *c = accum;
  return GOT_NUM;
}

/* Part of expression parser, parses a factor.
 */
int
getIntScan2 (p, c)
char **p;
long *c;
{
  long j, k;
  char c0, c1;

  scan (p);
  c0 = tolower(**p);
  c1 = tolower(*(*p+1));
  switch (c0) {
    case '\n':
    case '\0':
      return NO_NUM;
    case '-':					/* -exp */
      ++*p;
      j = getIntScan2 (p, &k);
      *c = -k;
      return j;
    case '~':					/* ~exp */
      ++*p;
      j = getIntScan2 (p, &k);
      *c = ~k;
      return j;
    case '(':					/* (exp) */
      ++*p;
      if (GOT_NUM != getIntScan (p, c))
        return BAD_NUM;
      scan (p);
      if (**p != ')')
        return BAD_NUM;
      ++*p;
      return GOT_NUM;
    case '\'':					/* ASCII code */
      ++*p;
      *c = *(*p)++;
      return GOT_NUM;
    case 'b':
      if (c1 == ':') {				/* b:adr, fetch char */
        *p += 2;
        j = getIntScan2 (p, &k);
        *c = CHAR_STAR(k+BASE);
        return j;
      }
      if (c1 == '\'') {				/* b'num, base = 2 */
        *p += 2;
	return getNum (p, c, 2L);
      }
      break;
    case 'w':					/* w:adr, fetch short */
      if (c1 == ':') {
        *p += 2;
        j = getIntScan2 (p, &k);
        *c = SHORT_STAR(k+BASE);
        return j;
      }
      break;
    case 'd':					/* d:adr, fetch long */
      if (c1 == ':') {
        *p += 2;
        j = getIntScan2 (p, &k);
        *c = LONG_STAR(k+BASE);
        return j;
      }
      if (c1 == '\'') {				/* d'num, base 10 */
        *p += 2;
	return getNum (p, c, 10L);
      }
      break;
    case 'h':					/* base 16 number */
      if (c1 == '\'') {
        *p += 2;
	return getNum (p, c, 16L);
      }
      break;
    case 'r':					/* r'<reg> */
      /* force "f0" to be interpretted as a reg instead of a number */
      if (c1 == '\'') {
        *p += 2;
	return getReg (p, c);
      }
      break;
    case 'v':					/* v(vaddr[,ptb]) */
      if (c1 == '(' || c1 == ' ' || c1 == '\t') {
	++*p;
	return getVaddr (p, c);
      }
      break;
  }
  if (getNum (p, c, defaultBase) == GOT_NUM) return GOT_NUM;
  return getReg (p, c);
}

/* Part of expression parser.  **p should point to a register name.
 * If so, *c is assigned the current value of the register.
 */
int
getReg (p, c)
char **p;
long *c;
{
  char regbuf [80];
  struct regTable *r, *find_reg();

  scanreg (p, regbuf);				/* get register */
  if (NULL != (r = find_reg (regbuf))) {
    switch (r->type & T_LEN) {
      case T_CHAR:	*c = CHAR_STAR (r->ptr); break;
      case T_SHORT:	*c = SHORT_STAR (r->ptr); break;
      case T_LONG:	*c = LONG_STAR (r->ptr); break;
    }
    return GOT_NUM;
  }
  return BAD_NUM;
}

/* Part of expression parser.  **p should point to a number.
 * If so, *c is assigned the number.
 */
int
getNum (p, c, base)
char **p;
long *c, base;
{
  long j, k;

  if (-1 != (j = tohex (**p)) || j >= base) {	/* get number */
    ++*p;
    while (-1 != (k = tohex (**p)) && k < base) {
      ++*p;
      j = j * base + k;
    }
    *c = j;
    return GOT_NUM;
  } else return BAD_NUM;
}

/* Like getIntScan, but p is char * instead of char **.  See getIntScan.
 */
int
getInt (p, val)
long *val;
char *p;
{
  return getIntScan (&p, val);
}
  
/* Convert a char, interpretted as a hex digit, to an int.
 */
int
tohex (ch)
char ch;
{
  ch = toupper (ch);  /* bug fix WBC */
  if (ch >= '0' && ch <= '9') return ch - '0';
  if (ch >= 'A' && ch <= 'F') return ch - 'A' + 10;
  return -1;
}
