/* 
   Copyright (C) 1993, 1994 Free Software Foundation

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2, or (at
   your option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA. */

#include "ports.h"
#include <assert.h>
#include <cthreads.h>

typedef int (*demuxer)(mach_msg_header_t *, mach_msg_header_t *);
static spin_lock_t reqthreadlock = SPIN_LOCK_INITIALIZER;
static int nreqthreads = 0;
static int hadrequest = 0;

#define TIMEOUT (1000 * 60) /* One minute */

#define NDEATH_TIMEOUTS 10

/* Notice that TIMEOUT * NDEATH_TIMEOUTS should be the same as
   DEATH_TIMEOUT in manage-ops-one.c. */

/* Demultiplex an incoming message INP placing the reply in OUTP.  */
static int
our_demuxer (mach_msg_header_t *inp, mach_msg_header_t *outp)
{
  int spawn = 0;
  int status;

  spin_lock (&reqthreadlock);
  assert (nreqthreads);
  if (!--nreqthreads)
    spawn = 1;
  hadrequest = 1;
  spin_unlock (&reqthreadlock);

  if (spawn)
    cthread_detach (cthread_fork ((cthread_fn_t) 
				  ports_manage_port_operations_multithread, 
				  (any_t) 0));
  status = ports_demuxer (inp, outp);

  spin_lock (&reqthreadlock);
  nreqthreads++;
  spin_unlock (&reqthreadlock);

  return status;
}

/* Wait for messages on all the ports created with ports_allocate_port;
   for each message call ports_demuxer.  */
void
ports_manage_port_operations_multithread ()
{
  error_t err;
  int ntimeouts = 0;
  
  if (ports_wire_threads)
    thread_wire (ports_wire_threads, mach_thread_self (), 1);
  if (ports_wire_cthreads)
    cthread_wire ();
  spin_lock (&reqthreadlock);
  nreqthreads++;
  spin_unlock (&reqthreadlock);
  while (1)
    {
      err = mach_msg_server_timeout (our_demuxer, 0, _libports_request_portset,
				     MACH_RCV_TIMEOUT, TIMEOUT);
      if (err == MACH_RCV_TIMED_OUT)
	{
	  spin_lock (&reqthreadlock);
	  if (nreqthreads > 1)
	    {
	      nreqthreads--;
	      spin_unlock (&reqthreadlock);
	      cthread_exit (0);
	    }
	  else
	    {
	      /* We are the only thread.  Clear hadrequest and increment
		 ntimeouts.  When we've had ten timeouts, then that's
		 the ten minute death timeout (the same as manage-ops-one.c) 
		 and we tell the user we're idle. */
	      if (hadrequest)
		{
		  hadrequest = 0;
		  ntimeouts = 0;
		}
	      ntimeouts++;
	      if (ntimeouts == NDEATH_TIMEOUTS)
		{
		  spin_unlock (&reqthreadlock);
		  spin_lock (&_ports_count_lock);
		  ports_notice_idle (_ports_nhardports, _ports_nsoftports);
		  spin_unlock (&_ports_count_lock);
		  ntimeouts = 0;
		}
	      else
		spin_unlock (&reqthreadlock);
	    }
	}
    }
}


