
#ifndef lint
static char    *copyright = "Copyright (C) 1994, Steve Cumming";
#endif

/*
 * Copyright (C) 1994 by Steve Cumming (stevec@geog.ubc.ca)
 *
 * Permission to use, copy, modify, and distribute this software 
 * for any purpose and without fee is hereby granted, provided
 * that the above copyright notices appear in all copies and that both the
 * copyright notice and this permission notice appear in supporting
 * documentation.  This software is provided "as is" without express or
 * implied warranty.
 */

#include <stdio.h>
#include <math.h>
#include <string.h>
#include "sys.h"
#include "lib.h"
#include "search.h"
#include "gp.h"
#include "array.h"
#include "mlib.h"


/*
  implement array data type

  NULL array of any type is 0 length array, with NULL pointer to data
*/

int 
    size_type(enum DATA_TYPES type, int n){
	int s;
	switch (type) {
	case INT:
	    s = n * sizeof(int);
	    break;
	case FLOAT:
	    s = n * sizeof(float);
	    break;
	case DOUBLE:
	    s = n * sizeof(double);
	    break;
	case CMPLX:
	    s= n * sizeof(struct cmplx);
	    break;
	case POINTER:
	    s = n * sizeof(void *);
	    break;
	default:
	    s = 0;
	    error("illegal type of array");
	    break;
	}
	return s;
    }



struct array * copy_coerce_array(struct array * a, enum DATA_TYPES type){

    struct value v;
    int i;
    struct array * n = new_array(type, a->n);
    for (i = 0; i < a->n; i++){	
	get_array_value(a,i,&v);
	put_array_value(n,i,coerce_value(&v,type));
    }
    return n;
}
    
struct array *
    new_array(enum DATA_TYPES type, int n){

	int s = size_type(type,n);
	struct array * a = (struct array *)xalloc(sizeof(struct array),"new_array()");
	a->n = n;
	a->type = type;
	if (s){
	    a->a = (void *)xalloc(s,"new_array() space");
	    bzero((char *)a->a,s);
	} else {
	    a->a = NULL;
	}
	a->free = NULL;
	return a;
    }

void 
    set_array_free(struct array * a, void (*fcn)(void *))
{
    a->free = fcn;
}


/*
  the semantics of cat etc work better if
  nul dups are allowed;
*/

struct array *
    dup_array(struct array * a){
	int n;
	struct array * new;

	if (a == NULL)
	    return NULL;

	n = size_type(a->type,a->n);
	new = new_array(a->type, a->n);
	bcopy(a->a,new->a,n);	
	return new;
    }

	   
void 
    zero_array(struct array * a){
	int sz = size_type(a->type,a->n);
	bzero((char *)a->a,sz);
	return;
    }

void
    free_array(struct array * a){
	int i;
	if (a){
	    if (a->n > 0)
		if (a->free) {	/* some complex thing */
		    for (i = 0; i < a->n; i++)
			(*a->free)(((void **)a->a)[i]);
		}
		else 
		    free(a->a);
	    free((void *)a);
	}
	return;
    }

struct array *
    cat_array(struct array * a, struct array * b){
	struct array * res;
	if (b == NULL){
	    if (a)
		return dup_array(a);
	    else
		return a;
	}
	if (a == NULL)
	    return dup_array(b);
	if (a->type != b->type)
	    fatal("cat_array(): mixed types not supported\n");
	res = dup_array(a);
	res = resize_array(res,a->n + b->n);
	bcopy((char *)(b->a), (char *)(res->a) + size_type(a->type,a->n), size_type(b->type,b->n));
	return res;
    }
	      

struct array *
    append_array(struct array * a, struct array * b){
	int sz;
	if (b == NULL){
	    return a;
	}
	if (a == NULL)
	    return dup_array(b);
	if (a->type != b->type)
	    fatal("cat_array(): mixed types not supported\n");
	sz = size_type(a->type,a->n);
	a = resize_array(a ,a->n + b->n);
	
	bcopy((char *)(b->a), (char *)(a->a) + sz, size_type(b->type,b->n));
	return a;
    }
	      


struct array * shift_array(struct array * a, int n){
    struct array * tmp;
    tmp = new_array(a->type,a->n + n);
    bcopy((char *)(a->a), (char *)(tmp->a) + size_type(a->type,n),size_type(a->type,a->n));
    free_array(a);
    return tmp;
}
    
    

struct array *
    resize_array(struct array * a, int n){
	void * tmp;
	int sz;
	
	Assert((n >= 0),"resize_array to negative!");

	if (a->n == 0){	/* not initialised yet */
	    sz = size_type(a->type,n);
	    a->a = (void *)xalloc(sz,"resize_array() new");
	    bzero((char *)a->a,sz);
	    a->n = n;
	}
	else if (n < a->n){
	    sz = size_type(a->type,n);
	    tmp = (void *)xalloc(sz,"resize_array() shrink");
	    bcopy((char *)a->a,(char *)tmp,sz);
	    free(a->a);
	    a->a = tmp;
	    a->n = n;
	} else if (n > a->n){
	    sz = size_type(a->type,n);
	    tmp = (void *)xalloc(sz,"resize_array() grow");
	    bzero((char *)tmp,sz);
	    sz = size_type(a->type,a->n);
	    bcopy((char *)a->a,(char *)tmp,sz);
	    free(a->a);
	    a->a = tmp;
	    a->n = n;
	}
	return a;
    }


struct array *	
    promote_array(struct array * a, enum DATA_TYPES type){
	int sz;
	void * tmp;
	double * d;
	float * f;
	int i;

	if (a->type == type)
	    return a;

	if (a->n == 0){
	    a->type = type;
	    return a;
	}

        sz = size_type(type,a->n);
	tmp = (void *) xalloc(sz,"promote_array()");
	switch (type){
	case DOUBLE:	
	    d = (double *)tmp;
	    switch(a->type){
	    case INT:
		for (i = 0; i < a->n; i++)
		    d[i] = (double)((int *)a->a)[i];
		break;
	    case FLOAT:
		for (i = 0; i < a->n; i++)
		    d[i] = (double)((float *)a->a)[i];
		break;
	    default:
		error("impossible type in promote_array(a)");
		break;
	    }
	    break;
	case FLOAT:
	    f = (float *) tmp;
	    switch(a->type){
	    case INT:
		for (i = 0; i < a->n; i++)
		    f[i] = (float)((int *)a->a)[i];
		break;
	    default:
		error("impossible type in promote_array(a)");
		break;
	    }
	default:
	    error("impossible type in promote_array(a)");
	    break;
	}
	free(a->a);
	a->type = type;
	a->a = tmp;
	return a;
}    


void 
    init_array(struct array * a, struct value * val)
{
    int i;
    int ival, *ia;
    double dval, *da;
    float fval, *fa;
    if (a->type != val->type)
	error("init_array(): non conforming arguments");
    i = 0;
    switch (a->type){
    case INT:
	ia = (int *)a->a;
	ival = val->v.int_val;
	while (i < a->n)
	    ia[i++] = ival;
	break;
    case FLOAT:
	fa = (float *)a->a;
	fval = val->v.f_val;
	while (i < a->n)
	    fa[i++] = fval;
	break;
    case DOUBLE:
	da = (double *)a->a;
	dval = val->v.d_val;
	while (i < a->n)
	    da[i++] = dval;
	break;
    default:
	error("init_array(): invalid or unimplemented type");
	break;
    }
    return;
}



void put_double_array(struct array * a, int n, double v){
    if (n+1 > a->n)
	a = resize_array(a,n+1);
    a = promote_array(a,DOUBLE);
    ((double *)a->a)[n] = v;
    return;
}

double get_double_array(struct array * a, int n){
    struct value v;
    (void) get_array_value(a,n,&v);
    coerce_value(&v,DOUBLE);
    return v.v.d_val;
}

void put_ptr_array(struct array * a, int n, void * v){
    if (n+1 > a->n)
	a = resize_array(a,n+1);
    a = promote_array(a,POINTER);
    ((void **)a->a)[n] = v;
    return;
}

void put_array_value(struct array * a, int n, struct value * v){
    
    if (n + 1 > a->n)
	a = resize_array(a,n+1);
    coerce_value(v,a->type);

    switch(a->type){
    case INT:
	((int *)(a->a))[n] = v->v.int_val;
	break;
    case FLOAT:
	((float *)(a->a))[n] = v->v.f_val;	
	break;
    case DOUBLE:
	((double *)(a->a))[n] = v->v.d_val;
	break;
    default:
	fatal("put_array_value() type insupported");
	break;
    }
    return;
}

	

void get_array_value(struct array * a, int n, struct value * v){

    if (n + 1 > a->n)
	fatal("get_array_value(): index out of bounds\n");

    switch(a->type){
    case INT:
	v->v.int_val = ((int *)(a->a))[n];
	break;
    case FLOAT:
	 v->v.f_val = ((float *)(a->a))[n];
	break;
    case DOUBLE:
	 v->v.d_val = ((double *)(a->a))[n];
	break;
    default:
	fatal("get_array_value() type insupported");
	break;
    }
    v->type = a->type;
    return;
}

	

/*
  add two arrays together with type promotion and size adjustment,
  
  modify argument one directly

  hence	  add_array(a,b) == a+= b;

*/

struct array *
    add_array(struct array * a, struct array * b){

	int i;
	a = promote_array(a,highest_type(a->type,b->type));
	a = resize_array(a,max(a->n,b->n));
	
	switch (a->type){
	    int *ia, *ib;
	    double *da, *db;

	case INT:
	    ia = (int *) a->a;
	    switch (b->type){
	    case INT:
		ib = (int *)b->a;
		i = 0;
		while (i < b->n){
		    ia[i]+=ib[i];
		    i++;
		}
		break;
	    default:
		error("add_array(,b): invalid type");
		break;
	    }
	    break;
	case DOUBLE:
	    da = (double *)a->a;
	    switch (b->type){
	    case INT:
		ib = (int *)b->a;
		i = 0;
		while (i < b->n){
		    da[i]+= (double)ib[i];
		    i++;
		}
		break;
	    case DOUBLE:
		db = (double *)b->a;
		i = 0;
		while (i < b->n){
		    da[i] += db[i];
		    i++;
		}
		break;
	    default:
		error("add_array(,b): invalid type");
		break;
	    }
	    break;
	default:
	    error("add_array(a,): invalid type");
	    break;
	}
	return a;
    }


/*
  component-wise division with type promotion 
  and divide by zero protection
  
  modify argument one directly

  hence	  divide_array(a,b) == a/= b;

*/

struct array *
    divide_array(struct array * a, struct array * b){

	int i;
	int xi;
	int xd;
	int fperror = 0;

	a = resize_array(a,max(a->n,b->n));
	if (a->n != b->n)
	    error("can't divide non confoming arrays");

	a = promote_array(a,highest_type(a->type,b->type));
	
	switch (a->type){
	    int *ia, *ib;
	    double *da, *db;

	case INT:
	    ia = (int *) a->a;
	    switch (b->type){
	    case INT:
		ib = (int *)b->a;
		i = 0;
		while (i < b->n){
		    xi = ib[i];
		    if (xi != 0)
			ia[i] /= ib[i];
		    else{
			fperror++;
			ia[i] = 0;
		    }
		    i++;
		}
		break;
	    default:
		error("divide_array(,b): invalid type");
		break;
	    }
	    break;
	case DOUBLE:
	    da = (double *)a->a;
	    switch (b->type){
	    case INT:
		ib = (int *)b->a;
		i = 0;
		while (i < b->n){
		    xi = ib[i];
		    if (xi != 0)
			da[i] /= (double)xi;
		    else {
			fperror++;
			da[i] = 0;
		    }
		    i++;
		}
		break;
	    case DOUBLE:
		db = (double *)b->a;
		i = 0;
		while (i < b->n){
		    xd = db[i];
		    if (xd != 0.0) 
			da[i] /= db[i];
		    else {
			fperror++;
			da[i] = 0;
		    }
		    i++;
		}
		break;
	    default:
		error("divide_array(,b): invalid type");
		break;
	    }
	    break;
	default:
	    error("divide_array(a,): invalid type");
	    break;
	}
	if (fperror)
	    fprintf(stderr,"divide_array: %d divide by zero(s) prevented\n",fperror);
	return a;
    }



double 
    sum_array(struct array * a)
{

    int i;
    int * ia;
    double *da;
    float *fa;
    double sumd = 0.0;
    int    sumi = 0;

    i = 0;
    switch (a->type){
    case INT:
	ia = (int *)a->a;
	while (i++ < a->n)
	    sumi += *ia++;
	sumd = (double) sumi;
	break;
    case FLOAT:
	fa = (float *)a->a;
	while (i++ < a->n)
	    sumd += (double)*fa++;
	break;
    case DOUBLE:
	da = (double *)a->a;
	while (i++ < a->n)
	    sumd += *da++;
	break;
    default:
	error("sum_array() invalid array type");
	break;
    }
    return sumd;
}	

/*
   it is possible that really large or really small
   floats or doubles will not appear correctly....
*/

static char * ret_str = NULL;
static char str_term = 0x0;

char *
array_to_string(struct array * a){
    char tmp[32];
    int i,n;
    int * ia;
    double * da;
    float * fa;
    double x;
    int len;
    int maxa = 0;


/*
    hc("array_to_string() entry");
*/

    if (ret_str)
	free((void *)ret_str);

    n = a->n;
    switch (a->type){
    case INT:
	ia = (int *)a->a;
	for (i = 0; i < n ; i++)
	    maxa = max(maxa,abs(ia[i]));
	maxa = max(1, maxa);			/* guard against zero */
	x = log10((double)maxa);
	len = (int)x;
	len = max(len,4);
	len =  len + 1 + 2;
	len *= n;
	len++;
	ret_str = (char *) xalloc(len,"array_to_string");
	*ret_str = 0x0;
	for (i = 0; i < n; i++){
	    sprintf(tmp,"  %4d%c",ia[i],str_term);
	    strcat(ret_str,tmp);
	}	
	break;
    case FLOAT:
	fa = (float *)a->a;
	len = 10 + 1 + 4;
	len *= n;
	len++;
	ret_str = (char *) xalloc(len,"array_to_string");
	*ret_str = 0x0;
	for (i = 0; i < n; i++){
	    x = fa[i];
	    if (fabs(x) < 10000.0)
		sprintf(tmp,"  %8.4f%c",x,str_term);
	    else
		sprintf(tmp,"  %10.4g%c",x,str_term);
	    strcat(ret_str,tmp);
	}	
	break;
    case DOUBLE:
	da = (double *)a->a;
	len = 10 + 1 + 4;
	len *= n;
	len++;
	ret_str = (char *) xalloc(len,"array_to_string");
	*ret_str = 0x0;
	for (i = 0; i < n; i++){
	    x = da[i];
	    if (fabs(x) < 10000.0)
		sprintf(tmp,"  %8.4f%c",x,str_term);
	    else
		sprintf(tmp,"  %10.4g%c",x,str_term);
	    strcat(ret_str,tmp);
	}	
	break;
    default:
	error("array_to_string() unprintable array type");
	break;
    }
/*
    hc("array_to_string() exit");
*/
    return ret_str;
}

/*
  construct array from a stream of tokens
*/

struct array * read_array(enum DATA_TYPES t, FILE * f, int tok){

    char * str;
    struct array * a;
    int n = 0;
    struct value v;
    int sz = 10;
    
    a = new_array(t,sz);

    while (str = nth_token(f,tok)){

	string_to_value(str,t,&v);
	put_array_value(a, n++, &v);

	if (n == sz){
	    sz <<= 2;
	    resize_array(a,sz);
	}
    }
    resize_array(a, n);
    return a;
}

void read_array_pair(struct array ** r1, enum DATA_TYPES t1, struct array ** r2, enum DATA_TYPES t2, FILE * f){

    char * str;
    struct array * a1, * a2;
    int n = 0;
    struct value v;
    int sz = 10;
    
    a1 = new_array(t1,sz);
    a2 = new_array(t2,sz);

    while (str = nxt_token(f)){

	string_to_value(str,t1,&v);
	put_array_value(a1, n, &v);
	str = nxt_token(f);
	if (!str)
	    fatal("read_array_pair(): bad input file\n");
	string_to_value(str,t2,&v);
	put_array_value(a2, n++, &v);

	if (n == sz){
	    sz <<= 2;
	    resize_array(a1,sz);
	    resize_array(a2,sz);
	}
    }
    resize_array(a1, n);
    resize_array(a2, n);
    *r1 = a1;
    *r2 = a2;
    return;
}

void
write_array(struct array * a, FILE * f){
    int i,n;
    int * ia;
    double * da;
    float * fa;
    struct cmplx * ca;
    fprintf(f,"# some histogram\n");
    n = a->n;
    switch (a->type){
    case INT:
	ia = (int *)a->a;
	for (i = 0; i < n; i++)
	    fprintf(f,"%d\t%d\n",i,ia[i]);
	break;
    case FLOAT:
	fa = (float *)a->a;
	for (i = 0; i < n; i++)
	    fprintf(f,"%d\t%f\n",i,(double)fa[i]);
	break;
    case DOUBLE:
	da = (double *)a->a;
	for (i = 0; i < n; i++)
	    fprintf(f,"%d\t%f\n",i,da[i]);
	break;
    case CMPLX:
	ca = (struct cmplx *)a->a;
	for (i = 0; i < n; i++)
	    fprintf(f,"%d\t{%f.%f}\n",i,ca[i].real,ca[i].imag);
	break;
    default:
	error("array_to_string() unprintable array type");
	break;
    }

    fprintf(f,"#\n");
    fflush(f);
    return;
}

void
write_array_element(struct array * a, int i, FILE * f){
    int n;
    int * ia;
    double * da;
    float * fa;
    struct cmplx * ca;
    n = a->n;
    if (i >= n)
	fatal("write_array_element(): out of bounds");
    
    switch (a->type){
    case INT:
	ia = (int *)a->a;
	fprintf(f,"%d ",ia[i]);
	break;
    case FLOAT:
	fa = (float *)a->a;
	fprintf(f,"%f ",(double)fa[i]);
	break;
    case DOUBLE:
	da = (double *)a->a;
	fprintf(f,"%f ",da[i]);
	break;
    case CMPLX:
	ca = (struct cmplx *)a->a;
	fprintf(f,"{%f,%f} ",ca[i].real,ca[i].imag);
	break;
    default:
	error("write_array_element(): unprintable array type");
	break;
    }
}

#ifdef MAP

/*

  writes out a list of arrays in row major order,
  assumes all of same length

*/


void
write_array_list(struct lcb * index, struct lcb * as, FILE * f){

    struct array * a;	
    struct value * v;
    int i,n;
    reset_lcb(index);
    
    fprintf(f,"# ");
    while(v = (struct value *)nxt_entry(index))
	fprintf(f,"%s ",value_to_string(v));
    fprintf(f,"\n");

    reset_lcb(as);
    n = (a = (struct array *)nxt_entry(as))->n;

    for (i = 0; i < n; i++){
	reset_lcb(as);
	while(a = (struct array *)nxt_entry(as))
	    write_array_element(a,i,f);
	fprintf(f,"\n");
    }
	
    return;
}	

#endif

void write_array_pair(struct array * a, struct array * b, FILE * f){

    int i;

    if (a->n != b->n)
	fatal("write_array_pair()");

    for (i = 0; i < a->n; i++){
	write_array_element(a,i,f);
	write_array_element(b,i,f);
	fprintf(f,"\n");
    }
    fflush(f);
    return;
}

/*
  make new array

  new->a[i] = operand->a[i] op val

  used to do this in place, but think of
  
        x = a*y^2 + b*y + c 

  where x and y are arrays...

  Note and fix the inconcistent return of struct values 

*/


struct array *
    op_array(struct array * aop, enum GP_OPS op, struct value * v){

	struct array * a;
	struct value av;
	int i;

	enum DATA_TYPES t = highest_type(aop->type, v->type);
	a = new_array(t,aop->n);
	for (i = 0; i < a->n; i++){
	    get_array_value(aop, i, &av);
	    put_array_value(a, i, gp_binary(&av,v,op));
	}
	return a;
    }

int cmp_double(double * d1, double * d2){
    double x = (*d1) - (*d2);
    return (x < 0.0 ? -1 : (x == 0.0 ? 0 : 1));
    }

int cmp_float(float * d1, float * d2){
    float x = (*d1) - (*d2);
    return (x < 0.0 ? -1 : (x == 0.0 ? 0 : 1));
    }

int cmp_int(int * d1, int * d2){
    int x = (*d1) - (*d2);
    return (x < 0 ? -1 : (x == 0.0 ? 0 : 1));
    }

void sort_array_in_place(struct array * a){
    
    switch (a->type){
    case DOUBLE: 
	qsort(a->a, (size_t)a->n, sizeof(double), cmp_double);
	break;
    case FLOAT:
	qsort(a->a, (size_t)a->n, sizeof(float), cmp_float);
	break;
    case INT:
	qsort(a->a, (size_t)a->n, sizeof(int), cmp_int);
	break;
    case STR:
	qsort(a->a, (size_t)a->n, sizeof(int), strcmp);
	break;
    default:
	fatal("sort_array_in_place() Bad type");
	break;
    }
        
    return;
}


/*
  Turn a into a cummulative probability fcn
*/
 
void write_dist(struct array * a, FILE * f){

    int i;
    double x;
    struct array * new = dup_array(a);
    new = promote_array(new,DOUBLE);
    sort_array_in_place(new);


    for (i = 0; i < new->n; i++){
	x = (double)(i + 1)/(double)new->n;
	fprintf(f,"%f\t%f\n",x, ((double *)new->a)[i]);
    }

    free_array(new);
    return;

}





void array_bounds(struct array * a, double * mn, double * mx){

    register double x,lmx,lmn;
    int i;
    lmn = HUGE;
    lmx = lmn * -1.0;

    for (i = 0; i < a->n; i++){
	x = get_double_array(a,i);
	if (x > lmx) lmx = x;
	if (x < lmn) lmn = x;
	
    }
    *mn = lmn;
    *mx = lmx;
    return;
}
    





void subsample_arrays(double p, struct array * a1, struct array * a2, struct array **r1, struct array ** r2)
{
    
    extern double ran2(int *);

    struct value v;
    struct array * x, * y, * indices, *ivals;
    static int iseed = -1;
    int i,j,n;
    int * ia, *ja;
    
    if (a1->n != a2->n)
	fatal("sub_sample_arrays() array lengths differ\n");

    n = a1->n * p;

    x = new_array(a1->type, n);
    y = new_array(a1->type, n);
    indices = new_array(INT, n);
    ivals = new_array(INT, a1->n); 	/* init'd to 0 */
    ia = (int *)indices->a;
    ja = (int *)ivals->a;
    /* get the indices 			*/

    i = 0;
    while (i < n){
	j = ran2(&iseed) * a1->n;
	if (ja[j]++ == 0)
	    ia[i++] = j;
    }

    /* and transfer the chosen ones 	*/


    for (i = 0; i < n; i++){
	j = ia[i];
	get_array_value(a1,j,&v);
	put_array_value(x,i,&v);
	get_array_value(a2,j,&v);
	put_array_value(y,i,&v);
    }

    free_array(indices);
    free_array(ivals);
    *r1 = x;
    *r2 = y;
    return;
}
    

void subsample_array(double p, struct array * a, struct array ** r)
{
    
    extern double ran2(int *);

    struct value v;
    struct array * x, * indices, *ivals;
    static int iseed = -1;
    int i,j,n;
    int * ia, * ja;
    
    n = a->n * (p > 1.0 ? 1.0 : p);
    x = new_array(a->type, n);
    indices = new_array(INT, n);
    ivals = new_array(INT, a->n); 	/* init'd to 0 */
    ia = (int *)indices->a;
    ja = (int *)ivals->a;

    /* get the indices 			*/

    i = 0;
    while (i < n){
	j = ran2(&iseed) * a->n;
	if (ja[j]++ == 0)
	    ia[i++] = j;
    }

    /* and transfer the chosen ones 	*/


    for (i = 0; i < n; i++){
	j = ia[i];
	get_array_value(a,j,&v);
	put_array_value(x,i,&v);
    }

    free_array(indices);
    free_array(ivals);
    *r = x;
    return;
}
    

