/* Miscellaneous index routines for use with GNATS. 
   Copyright (C) 1993 Free Software Foundation, Inc.
   Contributed by Brendan Kehoe (brendan@cygnus.com) and
   by Tim Wicinski (wicinski@barn.com).

This file is part of GNU GNATS.

GNU GNATS is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

GNU GNATS is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with GNU GNATS; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.  */

#include <signal.h>

#include "config.h"
#include "gnats.h"
#include "globals.h"
#include "pathmax.h"

void
create_index_entry (buf)
     char *buf;
{
  char *resp, *r;

  resp = field_value (RESPONSIBLE);
  if (! resp)
    return;

  resp = (char *) strdup (resp);

  r = (char *) strchr (resp, ' ');
  if (r != NULL)
    *r = '\0';
  	
  sprintf (buf, "%s/%s:%s:%s:%s:%s:%s:%s\n", field_value (CATEGORY),
	   field_value (NUMBER), field_value (SUBMITTER),
	   resp, field_value (STATE), field_value (CONFIDENTIAL), 
	   field_value (SEVERITY), field_value (PRIORITY));
}


/* Return the next index entry from FP.  */
Index *
next_index_entry (fp)
     FILE *fp;
{
  char *start, *end, *b;
  char *buf = (char *) xmalloc (STR_MAX);
  Index *p = (Index *) xmalloc (sizeof (Index));

  b = buf;
  do
    if (fgets (buf, STR_MAX, fp) == NULL)
      goto no_entry;
  while (buf[0] == '#');

  start = b;
  end = (char *) strchr (start, '/');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->category = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->number = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->submitter = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->responsible = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->state = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->confidential = start;

  start = end + 1;
  end = (char *) strchr (start, ':');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->severity = start;

  start = end + 1;
  end = (char *) strchr (start, '\n');
  if (end == NULL)
    goto no_entry;
  *end = '\0';
  p->priority = start;

  p->next = NULL;
  return p;

no_entry:
  free (p);
  free (buf);
  return (Index *) NULL;
}

/* Find problem report NUMBER in the index file, returning its category.  */
char *
find_pr_category (fp, number)
     FILE *fp;
     char *number;
{
  char *start, *end, *b;
  char buf[STR_MAX];
  char *category;

  b = &buf[0];

  while (1)
    {
      do
	if (fgets (&buf[0], STR_MAX, fp) == NULL)
	  return NULL;
      while (buf[0] == '#');

      start = b;
      end = (char *) strchr (start, '/');
      if (end == NULL)
	return NULL;
      *end = '\0';
      category = start;

      start = end + 1;
      end = (char *) strchr (start, ':');
      if (end == NULL)
	return NULL;
      *end = '\0';

      if (strcmp (start, number) == 0)
	return category;
    }
}

/* get_index - reads in the entire index. useful for searching or
   modifying index records.  */
Index*
get_index ()
{
  FILE *fp = open_index ();
  Index *i;
  Index *index_chain = NULL;
  Index *index_chain_end = NULL;

  if (fp == (FILE *) NULL)
    return (Index *) NULL;

  while ((i = next_index_entry (fp)))
    {
      if (index_chain_end == NULL)
        index_chain = index_chain_end = i;
      else
        {
          index_chain_end->next = i;
          index_chain_end = i;
        }

      i->next = NULL;
    }

  close_index (fp);
  return (index_chain);
}

FILE *
open_index ()
{
  FILE *fp;
  char *path = (char *) xmalloc (PATH_MAX);

  sprintf (path, "%s/gnats-adm/%s", gnats_root, INDEX);
  fp = fopen (path, "r");
  if (fp == NULL)
    {
      fprintf (stderr, "%s: can't open the index file %s\n",
	       program_name, path);
      return (FILE *) NULL;
    }

  free (path);
  return fp;
}

void
close_index (fp)
     FILE *fp;
{
  fclose (fp);
}
