;;; -*- Mode: Emacs-Lisp;  -*-
;;; File: thing.el
;;; Authors: Martin Boyer, IREQ <mboyer@ireq-robot.hydro.qc.ca>
;;;              adapted from Heinz Schmidt's thing.el for sky-mouse
;;;          Heinz Schmidt, ICSI (hws@ICSI.Berkeley.EDU)
;;;              adapted from Dan L. Pierson's epoch-thing.el
;;;          Dan L. Pierson <pierson@encore.com>, 2/5/90
;;;              adapted from Joshua Guttman's Thing.el
;;;          Joshua Guttman, MITRE (guttman@mitre.org)
;;;              adapted from sun-fns.el by Joshua Guttman, MITRE.  
;;;
;;; Copyright (C) International Computer Science Institute, 1991
;;;
;;; COPYRIGHT NOTICE: This code is provided "AS IS" WITHOUT ANY WARRANTY.
;;; It is subject to the terms of the GNU EMACS GENERAL PUBLIC LICENSE
;;; described in a file COPYING in the GNU EMACS distribution or to be obtained
;;; from Free Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
;;;* FUNCTION: Things are language objects contiguous pieces of text
;;;*           whose boundaries can be defined by syntax or context.
;;;*
;;;* RELATED PACKAGES: various packages built on this.
;;;*
;;;* HISTORY: 
;;;* Last edited: May 24 00:45 1991 (hws)
;;;*  Nov 28 17:40 1991 (mb): Cleaned up, and added thing-bigger-alist.
;;;*  May 24 00:33 1991 (hws): overworked and added syntax.
;;;* Created: 2/5/90 Dan L. Pierson
;;;*~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~

(provide 'thing)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;  Customization and Entry Point  ;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defvar thing-boundary-alist
  '((?w thing-word)
    (?_ thing-symbol)
    (?\( thing-sexp-start)
    (?\$ thing-sexp-start)
    (?' thing-sexp-start)
    (?\" thing-sexp-start)
    (?\) thing-sexp-end)
    (?  thing-whitespace)
    (?< thing-comment)
    (?. thing-next-sexp))
  "*List of pairs of the form (SYNTAX-CHAR FUNCTION) used by THING-BOUNDARIES."
  )

(defvar *last-thing*
  "The last thing found by thing-boundaries.  Used for chaining commands.")

(defvar thing-bigger-alist
  '((word-symbol thing-symbol)
    (symbol thing-sexp)
    (word-sexp thing-sexp)
    (sexp thing-up-sexp)
    (sexp-up thing-up-sexp)
    (line thing-paragraph)
    (paragraph thing-page)
    (char thing-word)
    (word-sentence thing-sentence)
    (sentence thing-paragraph))
  "List of pairs to go from one thing to a bigger thing.
See mouse-select-bigger-thing and mouse-delete-bigger-thing.")

(defvar thing-word-next 'word-sentence
  "*The next bigger thing after a word.  A symbol.
Supported values are: word-symbol, word-sexp, and word-sentence.
Default value is word-sentence.
Automatically becomes local when set in any fashion.")
(make-variable-buffer-local 'thing-word-next)

(defun thing-boundaries (here)
  "Return start and end of text object at HERE using syntax table and
thing-boundary-alist.  Thing-boundary-alist is a list of pairs of the
form (SYNTAX-CHAR FUNCTION) where FUNCTION takes a single position
argument and returns a cons of places (start end) representing
boundaries of the thing at that position.

Typically:
 Left or right Paren syntax indicates an s-expression.	
 The end of a line marks the line including a trailing newline. 
 Word syntax indicates current word. 
 Symbol syntax indicates symbol.
 If it doesn't recognize one of these it selects just the character HERE."
  (interactive "d")
  (setq *last-thing* nil)
  (if (save-excursion (goto-char here) (eolp))
      (thing-get-line here)
    (let* ((syntax (char-syntax (char-after here)))
	   (pair (assq syntax thing-boundary-alist)))      
      (if pair
	  (funcall (car (cdr pair)) here)
	(setq *last-thing* 'char)
	(cons here (1+ here))))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;  Code Delimiters  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun thing-symbol (here)
  "Return start and end of symbol at HERE."
  (setq *last-thing* 'symbol)
  (let ((end (scan-sexps here 1)))
    (cons (min here (scan-sexps end -1)) end)))

(defun thing-sexp-start (here)
  "Return start and end of sexp starting HERE."
  (setq *last-thing* 'sexp-start)
  (cons here (scan-sexps here 1)))

(defun thing-sexp-end (here)
  "Return start and end of sexp ending HERE."
  (setq *last-thing* 'sexp-end)
  (cons (scan-sexps (1+ here) -1) (1+ here)))

(defun thing-sexp (here)
  "Return start and end of the sexp at HERE."
  (setq *last-thing* 'sexp)
  (save-excursion 
    (goto-char here)
    (cons (progn (backward-up-list 1) (point))
	  (progn (forward-list 1) (point)))))

(defun thing-up-sexp (here)
  "Return start and end of the sexp enclosing the selected area."
  (setq *last-thing* 'sexp-up)
  ;; Keep going up and backward in sexps.  This means that thing-up-sexp
  ;; can only be called after thing-sexp or after itself.
  (setq here (or (zone-start drag-zone)
		 here))
  (save-excursion 
    (goto-char here)
    (cons (progn (backward-up-list 1) (point))
	  (progn (forward-list 1) (point)))))

;;; Allow punctuation marks not followed by white-space to include
;;; the subsequent sexp. Useful in foo.bar(x).baz and such.
(defun thing-next-sexp (here)
  "Return from HERE to the end of the sexp at HERE,
if the character at HERE is part of a sexp."
  (setq *last-thing* 'sexp-next)
  (if (= (char-syntax (char-after (1+ here))) ? )
      (cons here (1+ here))
    (cons here 
	  (save-excursion (forward-sexp) (point)))))

;;; Allow click to comment-char to extend to end of line
(defun thing-comment (here)
  "Return rest of line from HERE to newline."
  (setq *last-thing* 'comment)
  (save-excursion (goto-char here)
		  (end-of-line)
		  (cons here (point))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;  Text Delimiters  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun thing-word (here)
  "Return start and end of word at HERE."
  (setq *last-thing* thing-word-next)
  (save-excursion
    (goto-char here)
    (forward-word 1)
    (let ((end (point)))
      (forward-word -1)
      (cons (point) end))))

(defun thing-sentence (here)
  "Return start and end of the sentence at HERE."
  (setq *last-thing* 'sentence)
  (save-excursion 
    (goto-char here)
    (cons (progn (backward-sentence) (point))
	  (progn (forward-sentence) (point)))))

(defun thing-whitespace (here)
  "Return start to end of all but one char of whitespace HERE, unless 
there's only one char of whitespace.  Then return start to end of it."
  (setq *last-thing* 'whitespace)
  (save-excursion
    (let ((start (progn (skip-chars-backward " \t") (1+ (point))))
	  (end (progn (skip-chars-forward " \t") (point))))
      (if (= start end)
	  (cons (1- start) end)
	(cons start end)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;  Physical Delimiters  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun thing-get-line (here)
  "Return whole of line HERE is in, with newline unless at eob."
  (setq *last-thing* 'line)
  (save-excursion
    (goto-char here)
    (let* ((start (progn (beginning-of-line 1) (point)))
	   (end (progn (forward-line 1) (point))))
      (cons start end))))

(defun thing-paragraph (here)
  "Return start and end of the paragraph at HERE."
  (setq *last-thing* 'paragraph)
  (save-excursion 
    (goto-char here)
    (cons (progn (backward-paragraph) (point))
	  (progn (forward-paragraph) (point)))))

(defun thing-page (here)
  "Return start and end of the page at HERE."
  (setq *last-thing* 'page)
  (save-excursion 
    (goto-char here)
    (cons (progn (backward-page) (point))
	  (progn (forward-page) (point)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;  Support functions  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun kill-thing-at-point (here)
  "Kill text object using syntax table.
See thing-boundaries for definition of text objects"
  (interactive "d")
  (let ((bounds (thing-boundaries here)))
    (kill-region (car bounds) (cdr bounds))))

(defun copy-thing-at-point (here)
  "Copy text object using syntax table.
See thing-boundaries for definition of text objects"
  (interactive "d")
  (let ((bounds (thing-boundaries here)))
    (copy-region-as-kill (car bounds) (cdr bounds))))
