; Save and restore Epoch screen/window configuration.

;; Copyright (C) 1991 Salvador Pinto Abreu <spa@fct.unl.pt>.
;; Copyright (C) 1991 Free Software Foundation, Inc.

;; This file is part of GNU Emacs.

;; GNU Emacs is distributed in the hope that it will be useful,
;; but WITHOUT ANY WARRANTY.  No author or distributor
;; accepts responsibility to anyone for the consequences of using it
;; or for whether it serves any particular purpose or works at all,
;; unless he says so in writing.  Refer to the GNU Emacs General Public
;; License for full details.

;; Everyone is granted permission to copy, modify and redistribute
;; GNU Emacs, but only under the conditions described in the
;; GNU Emacs General Public License.   A copy of this license is
;; supposed to have been given to you along with GNU Emacs so you
;; can know your rights and responsibilities.  It should be in a
;; file named COPYING.  Among other things, the copyright notice
;; and this notice must be preserved on all copies.

(provide 'epoch-config)

(defun epoch-info ()
  (mapcar 'epoch-screen-info (epoch::screen-list)))

(defun dump-epoch-info (buffer)
  (let ((info (epoch-info)))
    (set-buffer buffer)
    (goto-char (point-max))
    (while info
      (cond ((epoch-screen-ok (car info)) (print (car info))))
      (setq info (cdr info)))))

(defun epoch-screen-ok (info)
  "Decide whether a screen should be saved in the configuration or not.
For the moment, this saves everything except screens whose name starts with
a '*'."
  (not (= ?* (aref (nth 1 info) 0))))

(defun save-epoch-configuration (the-file)
  "Save the current window configuration to THE-FILE."
  (interactive "FSave configuration to file: ")
  (let* ((the-buffer (get-buffer-create "*screen-config*"))
	 (standard-output the-buffer))
    (set-buffer the-buffer) (erase-buffer)
    (insert ";;; File automatically generated by epoch-save-configuration.\n")
    (insert (format ";;; Dumped by %s@%s (%s) at %s.\n"
		    (user-real-login-name) (system-name) (user-full-name)
		    (current-time-string)))
    (dump-epoch-info the-buffer)
    (set-buffer the-buffer) (write-file the-file)
    (kill-buffer the-buffer)))

(defun epoch-screen-info (screen)
  (epoch::select-screen screen)

  (let* ((parent-window    (car (cdr (epoch::query-tree screen))))
	 (vdesk-window     (car (cdr (epoch::query-tree parent-window))))
	 (root-window      (car (epoch::query-tree screen)))

	 (vdesk-info       (cond ((eq vdesk-window root-window) nil)
				 (t (screen-information vdesk-window))))

	 (x-offset (cond (vdesk-info (- (nth 0 vdesk-info))) (t 0)))
	 (y-offset (cond (vdesk-info (- (nth 1 vdesk-info))) (t 0)))

	 (screen-info      (screen-information))
	 (screen-x         (+ (nth 0 screen-info) x-offset))
	 (screen-y         (+ (nth 1 screen-info) y-offset))
	 (screen-width     (epoch::screen-width screen))
	 (screen-height    (epoch::screen-height screen))
	 (screen-title     (get-property "WM_NAME"      screen))
	 (screen-icon-name (get-property "WM_ICON_NAME" screen))
	 (border-width     (nth 4 screen-info))
	 (map-state        (nth 5 screen-info))
	 (windows          (mapcar 'epoch-window-info
				   (epoch-window-list screen))))
    (list 'epoch-select-one-screen
	  screen-title screen-icon-name
	  screen-x screen-y
	  screen-width screen-height
	  (cons 'list windows))))


(defun epoch-window-list (screen &optional first this)
  "Return the list of windows displayed on SCREEN."

  (cond (first (cond
		((and (not (eq first this))
		      (eq screen (epoch::screen-of-window this)))
		 (cons this
		       (epoch-window-list screen first (next-window this))))))
	(t (let ((first (epoch::first-window screen)))
	     (cons first
		   (epoch-window-list screen first (next-window first)))))))


(defun epoch-window-info (window)
  "Return a list (BUFFER-NAME FILE-NAME MAJOR-MODE) for WINDOW."
  (save-window-excursion (select-window window)
			 (` (list (, (buffer-name)) (, (buffer-file-name))
				  (quote (, major-mode))))))


(defun epoch-select-one-screen (title icon-name x y width height windows)
  "This function is called from files saved by (dump-epoch-info)."

  (let* ((n-windows (length windows))
	 (screen-attributes
	  (append
	   (list (cons 'title title)
		 (cons 'icon-name icon-name)
		 (cons 'geometry (compute-epoch-geometry x y width height)))
	   epoch::screen-properties))
	 (screen (create-screen
		  (cond ((> n-windows 1) (get-buffer-create-screen "*junk*"))
			(t (let* ((the-window (car windows))
				  (the-buffer (nth 0 the-window))
				  (the-file   (nth 1 the-window))
				  (the-mode   (nth 2 the-window)))
			     (prog1
				 (set-buffer (find-file-noselect the-file))
			       (funcall the-mode)))))
		  screen-attributes)))

    (epoch::select-screen screen)

;;; Split screen in as many windows as needed.

    (cond ((> n-windows 1)
	   (let ((w-height (/ height n-windows))
		 (n n-windows))
	     (progn
	       (save-window-excursion
		 (while (> (setq n (1- n)) 0)
		   (split-window (selected-window) w-height)
		   (other-window 1)))

;;; Now select the appropriate buffer for each window.

	       (while windows
		 (let* ((the-window (car windows))
			(the-buffer (nth 0 the-window))
			(the-file   (nth 1 the-window))
			(the-mode   (nth 2 the-window)))
		   (switch-to-buffer (find-file the-file))
		   (funcall the-mode)
		   (other-window 1))
		 (setq windows (cdr windows)))

	       (kill-buffer "*junk*")))))))

(defun compute-epoch-geometry (x y w h)
  (format "%dx%d+%d+%d" w h x y))

(global-set-key "\C-xL" 'load-file)
(global-set-key "\C-xS" 'save-epoch-configuration)
