;----Le laboratoire de robotique de l'Institut de recherche d'Hydro-Quebec-----
; 
; Nom     : (require 'imouse)
; Fonction: Utilisation de la souris avec Epoch.
; Fichiers: imouse.el
; Notes   : 
; 
; Cree    : 29 octobre 91 ------- Martin Boyer <mboyer@ireq-robot.hydro.qc.ca>
; Modifie : 10 fevrier 92 -----3- Martin Boyer <mboyer@ireq-robot.hydro.qc.ca>
;           Copyright (c) 1991 Hydro-Quebec
; 
; Historique: 
; 
; 10 fevrier 92 -----3- Martin Boyer <mboyer@ireq-robot.hydro.qc.ca>
; 	Converti a Epoch 4.0 Beta 0 (button => zone).
; 
; 21 janvier 92 -----2- Martin Boyer <mboyer@ireq-robot.hydro.qc.ca>
; 	Commented the calls to epoch::store-cut-buffer in mouse::copy-button
; 	to solve the X selection bug (thanks to Gardner Cohen
; 	<beldar@rimulac.microunity.com>).  Wrapped the binding calls with
; 	imouse-do-bindings to allow users to customize them even when imouse
; 	is preloaded.  Split auto-X-selection into auto-set-X-selection and
; 	auto-get-X-selection.  Prevented mouse-copy-thing and mouse-move-thing
; 	from chaining kills.
;------------------------------------------------------------------------------

(require 'cl)
(require 'motion)
(require 'thing)
(autoload 'move-modeline-up "drag")
(autoload 'move-window-up "drag")

(provide 'imouse)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;  Configuration  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst imouse-version "1.01 of February 10, 1991"
  "The version and release date of the IREQ mouse library.")

(defvar mouse-select mouse-left
  "*The mouse button used to select.  Usually mouse-left.")

(defvar mouse-modify mouse-middle
  "*The mouse button used to modify.
Usually mouse-middle for Open Look, and mouse-right for twm.")

(defvar mouse-paste mouse-right
  "*The mouse button used to paste.
Usually mouse-right for Open Look, and mouse-middle for twm.")

(defvar mouse-delete-modifier mouse-shift
  "*The modifier used to delete with the mouse.")

(defvar mouse-thing-modifier mouse-control
  "*The modifier used to operate 
on objects with the mouse.")

(defvar mouse-window-modifier mouse-meta
  "*The modifier used to operate on emacs windows with the mouse.")

(defvar mouse-show-position t
  "*If t, display the mouse character position for each mouse click.")

(defvar auto-set-X-selection t
  "*If t, setting the emacs selection also sets the X selection.
If nil, the X selection has to be set explicitly.")

(defvar auto-get-X-selection t
  "*If t, the PASTE button inserts the X selection or cut-buffer.
If nil, the X selection has to be retrieved explicitly,
and the PASTE button will either insert the epoch selection, if set,
or yank the kill buffer.")

(defvar auto-X-selection-to-killbuffer t
  "*If t, setting the X selection also copies it to the kill buffer.")

(defvar drag-scroll-glyph 52
  "*The cursor glyph used to indicate drag scrolling.")

(defvar drag-modeline-glyph 116
  "*The cursor glyph used to indicate drag scrolling.")

(defconst imouse::tutorial-file
  (expand-file-name "IMOUSE-TUTORIAL" exec-directory)
  "The location of the IREQ MOUSE tutorial.")


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;  Useful functions  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun what-cursor-coordinates ()
  "Print line and column number of the cursor."
  (interactive)
  (message "(%d, %d)"
	   (current-column) (1+ (count-lines 1 (point)))))

(defun imouse::region-selected ()
  (and drag-zone
       (zone-start drag-zone)
       (/= (zone-start drag-zone) (zone-end drag-zone))))

(defun set-selection ()
  "Set the X selection if auto-set-X-selection is on."
  (interactive)
  (when auto-set-X-selection
    (set-X-selection)
    (when (interactive-p) (message "X selection set."))))

(defun get-selection ()
  "Insert the epoch selection, or the X selection if auto-get-X-selection is on."
  (interactive)
  (cond (auto-get-X-selection (get-X-selection) (message "X inserted"))
	((imouse::region-selected) (insert (zone-text drag-zone))
	 (message "Selection inserted"))
	(t (yank) (message "Yanked!"))))

(defun set-X-selection ()
  "Copy the selected (underlined) region to the X selection."
  (interactive)
  (if drag-zone
      (mouse::copy-zone drag-zone)
    (error "No region selected.")))

(defun get-X-selection ()
  "Insert the current X selection or cut-buffer at point."
  (interactive)
  ;; Stolen from mouse::paste-cut-buffer
  ;; Get text from selection PRIMARY, insert text once it's there.
  (let ((text (epoch::convert-selection mouse::selection-atom
					"STRING" "PROPERTY")))
    (if (stringp text)
	(insert text)			; selection is owned and has a value
      (insert (get-cut-buffer))))	; grab cut-buffer instead
  )

(defun imouse::set-drag-position (begin end &optional buffer)
  "Set the beginning and end of the drag-zone to BEGIN and END,
optional BUFFER defaults to the current buffer."
  (unless buffer (setq buffer (current-buffer)))
  (if drag-zone
      (setq drag-zone (epoch::move-zone drag-zone
					    begin end buffer))
    (setq drag-zone (add-zone begin end motion::style nil buffer)))
  )

;;;; eval-in-window
;;; From Jeff Peck's emacstool.
(defmacro eval-in-window (window &rest forms)
  "Switch to WINDOW, evaluate FORMS, return to original window."
  (` (let ((OriginallySelectedWindow (selected-window)))
       (unwind-protect
           (progn
             (select-window (, window))
             (,@ forms))
         (select-window OriginallySelectedWindow)))))
(put 'eval-in-window 'lisp-indent-hook 1)

;;;; mouse::copy-zone
;;; Similar to original, but optionally sets the kill buffer based on the
;;; value of auto-X-selection-to-killbuffer.  The calls to 
;;; epoch::store-cut-buffer are commented out until store-cut-buffer is fixed.

(defun mouse::copy-zone (zone &optional kill)
  "Copy the text in the BUTTON to the appropriate selection atom and to
the X cut-buffer. If auto-X-selection-to-killbuffer is true, the selection
is also copied to the kill buffer. If the optional KILL is true, the text
is killed to the killbuffer."
  (if (zonep zone)
      (let ((beg (epoch::zone-start zone))
	    (end (epoch::zone-end zone)))
	(if (null beg) (setq beg 1))
	(if (null end) (setq end 1))
	;(epoch::store-cut-buffer (buffer-substring beg end))
	;; Assert ownership of PRIMARY selection
	(epoch::acquire-selection mouse::selection-atom)
	;; Store data so we people can paste from Epoch to other clients.
	(setq epoch::selection-alist
	      (alist-delete mouse::selection-atom epoch::selection-alist))
	(setq epoch::selection-alist
	      (cons
	       (cons mouse::selection-atom (buffer-substring beg end))
	       (alist-delete mouse::selection-atom epoch::selection-alist)))
	(when (and auto-X-selection-to-killbuffer (/= beg end))
	  (if kill
	      (delete-region beg end)
	    (copy-region-as-kill beg end)))
	)
    ;(epoch::store-cut-buffer "")
    (setq epoch::selection-alist
	  (cons
	   (cons mouse::selection-atom "")
	   (alist-delete mouse::selection-atom epoch::selection-alist)))
    )
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;  Corrections to Standard Epoch  ;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Handler to dispatch mouse events.
;;; Changed to record the previous and last command.

(defun mouse::handler (type value scr)
  ;; first, calculate the index
  (let*
    (
      (number (nth 3 value))
      (edge (nth 0 value))
      (modstate (nth 4 value))
      index
      (epoch::event-handler-abort nil)	;prevent lossage
      (arg (epoch::coords-to-point (nth 1 value) (nth 2 value) scr))
      (buffer (and arg (nth 1 arg)))
      ;; Find which button table.  We want to stay in the same set of tables
      ;; (window, mode, minibuf) as any down press.
      (number-offset (if mouse::down-buffer
			 mouse::down-number-offset
		       (if (and (eq (nth 2 arg) (minibuffer-window))
				(= (minibuffer-depth) 0))
			   mouse-minibuf
			 (if (null (car arg))
			     mouse-mode
			   1)))) 
    )
    ;; Minibuf presses get no args.
    (if (= number-offset mouse-minibuf)
	(setq arg nil))

    ;; Count clicks as a convenience for some functions.
    (if (car value)
      ;; down-click - do timing check
      (let ((elapsed (- (aref epoch::event 3) mouse::time-stamp)))
	(if (< elapsed mouse::interval) ; multi-click
	  (setq mouse::clicks (1+ mouse::clicks))
	  (setq mouse::clicks 1)
	)
      )
      ;; up-click - update time stamp
      (setq mouse::time-stamp (aref epoch::event 3))
    )

    (setq mouse::event-data value)
    (setq mouse::x (nth 1 value))
    (setq mouse::y (nth 2 value))

    (setq number (+ number (1- number-offset)))
    ;; find the handler list and try to dispatch
    (let*
      (
        (index
	  (+
	    (if edge mouse-down mouse-up)
	    (if (/= 0 (logand modstate shift-mod-mask)) mouse-shift 0)
	    (if (/= 0 (logand modstate control-mod-mask)) mouse-control 0)
	    (if (/= 0 (logand modstate meta-mod-mask)) mouse-meta 0)
	    (* mouse::button-size (1- number))
	  )
	)
  	(map
  	  (if (and mouse::down-buffer (not edge))
              ;; force release into press buffer, for simulated grab
	      (symbol-buffer-value 'mouse::local-map mouse::down-buffer)
  	    ;; ELSE if there's an arg, use the arg buffer
	    (and arg (symbol-buffer-value 'mouse::local-map buffer))
	  )
	)

	(handler
	  (or
	    (and (vectorp map) (aref map index))
	    (aref mouse::global-map index)
	  )
	)
      )
      ;; Record down circumstances for next event.
      (setq mouse::down-buffer (and edge buffer))
      (if edge
	  (setq mouse::down-number-offset number-offset))

      ;; Do it.
      (when (and handler (functionp handler))
	(setq this-command handler)
        (funcall handler arg)
	(setq last-command this-command)
	(undo-boundary)
      )
    )
    (setq mouse::last-spot arg)
  )
)


;;; Changed to remove a call to undo-boundary; the mouse::handler does
;;; that automatically.

(defun mouse::paste-cut-buffer (arg)
  "Retrieve text from appropriate selection, or X cut-buffer if none"
  (let ( (buff (nth 1 arg)) )
    (when (and buff (bufferp buff))
      (save-excursion
        (set-buffer buff)
        (goto-char (car arg))
	; get text from selection PRIMARY
	; insert text once it's there.
	(let
	  (
	    (text
	      (epoch::convert-selection mouse::selection-atom
		"STRING" "PROPERTY"))
	  )
	  (if (stringp text)
	    (insert text)	; selection is owned and has a value
	    (insert (get-cut-buffer))	; grab cut-buffer instead
	  )
	)
))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;  Mouse dragging support  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defconst *imouse-selection-starters*
  '(mouse-start-selection)
  "List of commands that force extend-mouse-drag to start a new selection,
instead of modifying the current selection.")

(defconst *imouse-selecters*
  (append *imouse-selection-starters*
	  '(mouse-select-word mouse-select-line select-thing extend-mouse-drag
			      mouse-set-point-or-select set-selection))
  "The list of commands that set or modify the selection.")

(defvar imouse::selection-type nil
  "Current type of selection.  Must be set by all *imouse-selecters*.")

;;;; end-mouse-drag
;;; Similar to the original, but conditionally sets the X selection

(defun end-mouse-drag (mdata)
  "Stop update in motion handler, set the selection and set point and mark
around it.  Usually bound to an UP button transition or used to simulate one."
  (setq mouse::last-point -1)		;always do this cleanup
  (setq mouse::downp nil)		;stop update in motion handler
  (set-selection)
  (let ((s (and drag-zone (zone-start drag-zone)))
	(e (and drag-zone (zone-end drag-zone))))
    (if (null s) (setq s 1))
    (if (null e) (setq e 1))
    (if (zonep drag-zone)
	(if (<= (point) s)
	    (progn
	      (push-mark e t)
	      (goto-char s))
	  ;; ELSE point is past drag zone start
	  (progn
	    (push-mark s t)
	    (goto-char e)))
      (error "No zone in end-mouse-drag!?!")))
  )


;;;; extend-mouse-drag
;;; Similar to the original, but checks for the meaning of the
;;; previous mouse button (the selection state and type).

(defun extend-mouse-drag (mdata)
  (let ((m1 (and drag-zone (zone-start drag-zone)))
	(m2 (and drag-zone (zone-end drag-zone)))
	(spot (car mdata))		;point of the mouse click.
	(buf (cadr mdata)))		;buffer of the mouse click.
    (if (null m1) (setq m1 0))
    (if (null m2) (setq m2 0))
    (setq mouse::downp t)

    ;(db-print "emd: (%s, %s) spot: %d, m1: %d, m2: %d\n"
    ;      last-command imouse::selection-type spot m1 m2)

    (cond

     ;; No zone or no mouse marker, or buffer switch,
     ;; create a zone from the point to the mouse.
     ((or (null drag-zone) (null mouse-down-marker)
	  (not (eq buf (current-buffer))))
      (set-buffer buf)
      (imouse::set-drag-position (point) spot buf)
      (set-mouse-marker))

     ;; Previous mouse click was starting a selection,
     ;; create a new selection from there to the mouse.
     ((memq last-command *imouse-selection-starters*)
      (imouse::set-drag-position mouse-down-marker spot buf)
      (set-mouse-marker))

     ;; Anything else,
     ;; adjust the selection from the mouse to the farthest.

     ;; This should be reworked to re-use the code and to add more
     ;; selection-types.

     ;; Extend beginning
     ((<= spot m1)
      (case imouse::selection-type
	('word
	 (goto-char spot)
	 (forward-word -1)
	 (setq spot (point)))
	('line
	 (goto-char spot)
	 (beginning-of-line 1)
	 (setq spot (point))))
      (imouse::set-drag-position spot m2 buf)
      (set-mouse-marker m2))

     ;; Extend end
     ((>= spot m2)
      (case imouse::selection-type
	('word
	 (goto-char spot)
	 (forward-word 1)
	 (setq spot (point)))
	('line
	 (goto-char spot)
	 (beginning-of-line 2)
	 (setq spot (point))))
      (imouse::set-drag-position m1 spot buf)
      (set-mouse-marker m1))

     ;; Near beginning, shorten it
     ((< (abs (- spot m1)) (/ (- m2 m1) 2))
      (case imouse::selection-type
	('word
	 (goto-char spot)
	 (forward-word -1)
	 (setq spot (point)))
	('line
	 (goto-char spot)
	 (beginning-of-line 1)
	 (setq spot (point))))
      (imouse::set-drag-position spot (max m2 mouse-down-marker) buf)
      (set-mouse-marker m2))

     ;; Last case, shorten end.
     (t
      (case imouse::selection-type
	('word
	 (goto-char spot)
	 (forward-word 1)
	 (setq spot (point)))
	('line
	 (goto-char spot)
	 (beginning-of-line 2)
	 (setq spot (point))))
      (imouse::set-drag-position m1 spot buf)
      (set-mouse-marker m1))
     )
    (epoch::redisplay-screen)
    (setq mouse::last-point (point))
    (message "Mark set"))
  )



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;  Object selection  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-select-word (mdata)
  "Select the word around the mouse."
  (let (b)
    (mouse::set-point mdata)
    (setq mouse::downp t)
    (forward-word -1)
    (setq b (point))
    (forward-word 1)
    (setq imouse::selection-type 'word)
    (imouse::set-drag-position b (point))
    (end-mouse-drag mdata)))

(defun mouse-select-line (mdata)
  "Select the line where the mouse is."
  (let (b)
    (mouse::set-point mdata)
    (setq mouse::downp t)
    (beginning-of-line 1)
    (setq b (point))
    (end-of-line)
    (setq imouse::selection-type 'line)
    (imouse::set-drag-position b (point))
    (end-mouse-drag mdata)))

(defun mouse-start-selection (mdata)
  "Set point at mouse and prepare for drag selection. If mouse-show-position
is non-nil, displays the current line and column number."
  (setq imouse::selection-type 'char)
  (setq this-command 'mouse-start-selection)
  (abort-isearch)
  (set-mouse-marker)
  (setq mouse::last-point (point))
  (if mouse-show-position (what-cursor-coordinates)))

(defun mouse-set-point-or-select (mdata)
  "Set point at mouse.  With double-click, selects the current word,
with triple-click, selects the current line.  If mouse-show-position
is non-nil, displays the current line, column number and buffer."
  (mouse::set-point mdata)
  (setq mouse::downp t)
  (if (and (eq (car mouse::last-spot) (car mdata))
	   (eq (cadr mouse::last-spot) (cadr mdata))
	   (> mouse::clicks 1))
      (case mouse::clicks
	(2 (mouse-select-word mdata))
	(3 (mouse-select-line mdata)))
    (mouse-start-selection mdata)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;  Deleting and Killing  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-kill-region (mdata)
  "Kill (to kill buffer) between point and the mouse."
  (let ((b (point)))
    (mouse::set-point mdata)
    (kill-region b (point)))
  (epoch::delete-zone drag-zone)
  ;; I don't think it makes sense to append kills;
  ;; mouse operations are too complex already.
  (setq this-command 'not-kill))

(defun kill-selection (&optional mdata)
  "Kill (to kill buffer) the selected text."
  (interactive)
  (if (and drag-zone (zone-start drag-zone) (zone-end drag-zone))
      (when (or (pos-visible-in-window-p (zone-start drag-zone)
					 (selected-window))
		(pos-visible-in-window-p (zone-end drag-zone)
					 (selected-window))
		(beep)
		(confirm "Selection not visible, really kill?"))
	(kill-region (zone-start drag-zone) (zone-end drag-zone))
	(epoch::delete-zone drag-zone)
	;; I don't think it makes sense to append kills;
	;; mouse operations are too complex already.
	(setq this-command 'not-kill))
    (error "No region selected.")))

(defun mouse-delete-char (mdata)
  "Move point at the mouse and delete the character after point."
  (mouse::set-point mdata)
  (delete-char 1))

(defun delete-char-or-selection (&optional arg)
  "Delete the next character, or the selected region if the previous command is
a selection."
  (interactive "p")
  (if (memq last-command *imouse-selecters*)
      (kill-selection)
    (delete-char arg)))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;  Yanking, Copying, and Moving  ;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-yank (mdata)
  "Move point at the mouse and yank the kill buffer."
  (mouse::set-point mdata)
  (yank)
  (setq this-command 'yank))

(defun mouse-set-point-and-paste (mdata)
  "Move point at the mouse and retrieve text from the appropriate selection."
  (mouse::set-point mdata)
  (get-selection))

(defun mouse-move-selection (mdata)
  "Move (cut and paste) at the mouse the current selection."
  ;; Note that mouse::set-point operates on numbers, not markers. Thus,
  ;; we need to set the point now because we're deleting a region later.
  (mouse::set-point mdata)
  (when (not (imouse::region-selected))
    (error "No region selected."))
  (kill-region (zone-start drag-zone) (zone-end drag-zone))
  (epoch::delete-zone drag-zone)
  (yank)
  (setq this-command 'yank))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;  Interface to "thing.el"  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-copy-thing (mdata)
  "Copy at point the text object pointed to by the mouse."
  (save-excursion
    (set-buffer (cadr mdata))
    (setq last-command 'not-kill)	;Do not chain mouse kills
    (copy-thing-at-point (car mdata)))
    (yank)
    (setq this-command 'yank))

(defun mouse-move-thing (mdata)
  "Move (cut and paste) at point the text object pointed to by the mouse."
  (save-excursion
    (set-buffer (cadr mdata))
    (setq last-command 'not-kill)	;Do not chain mouse kills
    (kill-thing-at-point (car mdata)))
  (yank)
  (setq this-command 'yank))

(defun mouse-select-thing (mdata)
  "Select the object where the mouse is."
  (let (bounds)
    (mouse::set-point mdata)
    (setq mouse::downp t)
    (setq bounds (thing-boundaries (car mdata)))
    (setq imouse::selection-type *last-thing*)
    (when (memq *last-thing* '(word-symbol word-sexp word-sentence))
      (setq imouse::selection-type 'word))
    (imouse::set-drag-position (car bounds) (cdr bounds))
    (end-mouse-drag mdata))
  (setq this-command 'select-thing))

(defun mouse-select-bigger-thing (mdata)
  "Select a bigger object where the mouse is."
  (let (bounds bigger-fun)
      (mouse::set-point mdata)
      (setq bounds
	    (if (and (eq last-command 'select-thing)
		     (setq bigger-fun (assoc *last-thing* thing-bigger-alist)))
		(funcall (car (cdr bigger-fun)) (car mdata))
	      (thing-boundaries (car mdata))))
      (setq imouse::selection-type *last-thing*)
      (when (memq *last-thing* '(word-symbol word-sexp word-sentence))
	(setq imouse::selection-type 'word))
      (imouse::set-drag-position (car bounds) (cdr bounds))
      (end-mouse-drag mdata))
  (setq this-command 'select-thing))

(defvar *bigger-thing-kill-point* nil
  "Where the last bigger thing was deleted from.")

(defun mouse-kill-thing (mdata)
  "Kill (to kill buffer) the text object pointed to by the mouse."
  (save-excursion
    (mouse::set-point mdata)
    (let ((bounds (thing-boundaries (car mdata))))
      (kill-region (car bounds) (cdr bounds)))
    (setq *bigger-thing-kill-point* (point))
    (epoch::delete-zone drag-zone))
  ;; SIDE EFFECT: don't append kills
  (setq this-command 'kill-thing))

(defun mouse-kill-bigger-thing (mdata)
  "Kill a bigger object where the mouse is.  See also mouse-kill-thing and
mouse-select-bigger-thing."
  (let (bounds bigger-fun)
    (save-excursion 
      (mouse::set-point mdata)
      (setq bounds
	    (if (and (eq last-command 'kill-thing)
		     (setq bigger-fun (assoc *last-thing* thing-bigger-alist)))
		(funcall (car (cdr bigger-fun)) (car mdata))
	      (thing-boundaries (car mdata))))
      (if bigger-fun
	  (progn
	    (kill-append (buffer-substring (car bounds)
					   *bigger-thing-kill-point*) t)
	    (kill-append (buffer-substring *bigger-thing-kill-point*
					   (cdr bounds)) nil)
	    (delete-region (car bounds) (cdr bounds)))
	(kill-region (car bounds) (cdr bounds)))
      (setq *bigger-thing-kill-point* (point))
      (epoch::delete-zone drag-zone)))
  ;; SIDE EFFECT: don't append kills
  (setq this-command 'kill-thing))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;  Miscellaneous Commands  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-exec-kbd-macro (mdata)
  "Executes the current keyboard macro at the position set by the mouse."
  (mouse::set-point mdata)
  (call-last-kbd-macro))

(defun mouse-indent (mdata)
  (mouse::set-point mdata)
  (indent-according-to-mode))

(defun mouse-fill (mdata)
  (error "mouse-fill is not implemented yet."))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;  Screen and Window Handling  ;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Switching focus is unsupported.  See the full sky-mouse package
;;; from Heinz Schmidt <hws@icsi.berkeley.edu> for a complete
;;; implementation.

;(require 'epoch-focus)
;
;(defun mouse-toggle-focus-screen (mdata)
;  "Toggles whether the screen pointed to with the mouse holds the focus.
;The first toggle click on a screen sets the focus. The next releases it.
;Clicking to another screen moves the focus."
;  (let ((scr (nth 3 mdata)))
;    ;; Do not support it on the minibuffer for now;
;    ;; it works, but commands like top-level select another screen
;    ;; without going through select-screen...
;    (if (eq scr (minibuf-screen))
;	(progn (beep) (message "Come on. Minibuffer focuses automatically.")) 
;      (if (eq scr current-focus-screen)
;	  (release-kbd-focus)
;	(set-kbd-focus scr))
;      ;; Must set point to understand where we are.
;      (mouse::set-point mdata)))
;  )

(defun mouse-toggle-focus-screen (mdata)
  (error "mouse-toggle-focus-screen is not implemented yet."))

;;; Original by Heinz Schmidt <hws@icsi.berkeley.edu>
(defun mouse-warp-to-point (&optional mdata)
  "Warps mouse to the current point."
  (interactive)
  (let ((coords (epoch::query-cursor)))
    (epoch::warp-mouse (car coords) (cdr coords))))

(defun mouse-delete-other-windows (mdata)
  "Keep only the window where the mouse is."
  (mouse::set-point mdata)
  (delete-other-windows))

(defun mouse-delete-window (mdata)
  "Keep only the window where the mouse is."
  (let ((screen (epoch::current-screen)))
    (epoch::select-screen (nth 3 mdata))
    (delete-window (nth 2 mdata))
    (epoch::select-screen screen)))

(defun mouse-split-window (mdata)
  "Split in two the window where the mouse is."
  (mouse::set-point mdata)
  (split-window))

(defun mouse-switch-to-buffer-other-window (mdata)
  "Split in two the window where the mouse is and select another buffer."
  (let (buf)
    (mouse::set-point mdata)
    ;; BUG: a RET character is echoed in the minibuffer.
    (call-interactively 'switch-to-buffer-other-window)))
  


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;  Scrolling  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-line-to-top (mdata)
  "Put the line where the mouse is at the top of the window."
  (eval-in-window (nth 2 mdata)
    (mouse::set-point mdata)
    (recenter 0)
    (message "Line to top")))

(defun mouse-line-to-center (mdata)
  "Put the line where the mouse is at the center of the window."
  (eval-in-window (nth 2 mdata)
    (mouse::set-point mdata)
    (recenter '(1))
    (message "Line to center")))

(defun mouse-line-to-bottom (mdata)
  "Put the line where the mouse is at the center of the window."
  (eval-in-window (nth 2 mdata)
    (mouse::set-point mdata)
    (recenter -1)
    (message "Line to bottom")))

(defvar imouse::drag-scroll-start nil
  "Where mouse-drag-scroll started.")
(defvar imouse::drag-scroll-win nil
  "The window where mouse-drag-scroll started.")
(defvar imouse::drag-scroll-crs nil
  "The cursor used before mouse-drag-scroll started.")

(defun mouse-drag-scroll (mdata)
  "Scroll the whole text around."
  (setq mouse::downp nil) 
  (setq imouse::drag-scroll-start (epoch::query-pointer))
  (setq imouse::drag-scroll-win (nth 2 mdata))
  (setq imouse::drag-scroll-crs (epoch::cursor-glyph nil))
  (epoch::cursor-glyph drag-scroll-glyph)
  (message "Hold button down to drag text, release it when done."))

;;;; mouse-drag-scroll-terminate
;;; We have to drag by pixels because the text width can vary from
;;; character to character .  Further, epoch::query-mouse returns 0 as
;;; the x value on empty lines.  Finally, there is no function to
;;; scroll by pixels.
(defun mouse-drag-scroll-terminate (mdata)
  "The up click handler that goes with mouse-drag-scroll.
This actually scrolls the window in one fell swoop."
  (unwind-protect
      (let* ((drag-scroll-end (epoch::query-pointer))
	     (delta-x	    (- (nth 0 imouse::drag-scroll-start)
			       (nth 0 drag-scroll-end)))
	     (delta-y	    (- (nth 1 imouse::drag-scroll-start)
			       (nth 1 drag-scroll-end)))
	     (pixwidth	    (window-pixwidth imouse::drag-scroll-win))
	     (pixheight	    (window-pixheight imouse::drag-scroll-win))
	     (width	    (window-width imouse::drag-scroll-win))
	     (height	    (window-height imouse::drag-scroll-win)))
	(unless (and (< (abs delta-x) 2)
		     (< (abs delta-y) 2))
	  (eval-in-window imouse::drag-scroll-win
	    (scroll-left (/ (* delta-x width) pixwidth))
	    (scroll-up (/ (* delta-y height) pixheight)))))
    ;; Restore normal environment.
    (epoch::cursor-glyph imouse::drag-scroll-crs)
    (message "top line: %d, left offset: %d"
             (1+ (count-lines (point-min) (window-start))) (window-hscroll)))
  )

(defun mouse-move-modeline (mdata)
  "Shrink or stretch a window by dragging its modeline."
  (setq mouse::downp nil) 
  (setq imouse::drag-scroll-win (nth 2 mdata))
  (setq imouse::drag-scroll-crs (epoch::cursor-glyph nil))
  (let ((modeline (nth 3 (window-pixedges imouse::drag-scroll-win))))
    (epoch::warp-pointer (/ (window-pixwidth imouse::drag-scroll-win) 2)
			 (- modeline 3)))
  (setq imouse::drag-scroll-start (epoch::query-pointer))
  (epoch::cursor-glyph drag-modeline-glyph)
  (message "Hold button down to shrink or stretch window, release it when done."))

(defun mouse-move-modeline-terminate (mdata)
  "The up click handler that goes with mouse-move-modeline.
This actually resizes the window in one fell swoop"
  (unwind-protect
      (let* ((drag-scroll-end   (epoch::query-pointer))
	     (delta-y	    (- (nth 1 imouse::drag-scroll-start)
			       (nth 1 drag-scroll-end)))
	     (pixheight	    (window-pixheight imouse::drag-scroll-win))
	     (height	    (window-height imouse::drag-scroll-win)))
	(unless (< (abs delta-y) 2)
	  (eval-in-window imouse::drag-scroll-win
	    ;; BUG: move-modeline-up gets confused
	    ;; in epoch when called on the bottom window.
	    (move-modeline-up (/ (* delta-y height) pixheight)))))
    ;; Restore normal environment.
    (epoch::cursor-glyph imouse::drag-scroll-crs)
    (message "width: %d, height: %d"
	     (window-width imouse::drag-scroll-win)
	     (window-height imouse::drag-scroll-win)))
  )

(defun mouse-move-window (mdata)
  (error "mouse-move-window is not implemented yet."))
(defun mouse-move-window-terminate (mdata)
  (error "mouse-move-window-terminate is not implemented yet."))

(defun mouse-menu (mdata)
  (error "mouse-menu is not implemented yet."))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;  Help  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun mouse-summary (&optional mdata)
  "Display a summary of the mouse bindings."
  (interactive)
  (with-output-to-temp-buffer "*Help*"
    ;; This should put the button columns in order,
    ;; according to the current options.
    (princ 
"                    SELECT                MODIFY                 PASTE
		                                                             
Normal      	set point or select   set mark or bounds   paste sel. at mouse
DELETE      	delete character      kill region          move sel. at mouse
THING       	select thing          select bigger thing  copy thing at point
DELETE+THING    kill thing            kill bigger thing    move thing at point

WINDOW      	toggle focus screen   drag scroll          menu
WINDOW+DELETE	indent                fill                 exec kbd macro
MODELINE        this window only      move modeline        split window
MODELINE+DELETE delete window         move buffer          split to buffer
            	                                                             

		   LEFT                  MIDDLE               RIGHT

WINDOW+THING  	line to top           line to center       line to bottom

C+S+Meta        this summary          mouse play           mouse tutorial")
    (print-help-return-message))
  )


(defun mouse-play (&optional mdata)
  (error "mouse-play is not implemented yet."))

;;; Stolen from help-with-tutorial
(defun mouse-tutorial (&optional mdata)
  "Select the IREQ MOUSE learn-by-doing tutorial."
  (interactive)
  (let ((file (expand-file-name "~/IMOUSE-TUTORIAL")))
    (delete-other-windows)
    (if (get-file-buffer file)
	(switch-to-buffer (get-file-buffer file))
      (switch-to-buffer (create-file-buffer file))
      (setq buffer-file-name file)
      (setq default-directory (expand-file-name "~/"))
      (setq auto-save-file-name nil)
      (insert-file-contents imouse::tutorial-file)
      (goto-char (point-min))
      (set-buffer-modified-p nil))))



;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;  Standard mouse bindings  ;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun imouse-do-bindings ()
  "Bind the mouse keys according to user preferences.
First bind mouse-select, mouse-modify, and mouse-paste to either
mouse-left, mouse-middle, or mouse-right, then bind
mouse-delete-modifier, mouse-thing-modifier, and mouse-window-modifier
to either mouse-shift, mouse-control, or mouse-meta.
Calling this function will do the actual binding."
  (let ((SELECT       mouse-select)
	(MODIFY	      mouse-modify)
	(PASTE	      mouse-paste)
	(LEFT	      mouse-left)
	(MIDDLE	      mouse-middle)
	(RIGHT	      mouse-right)
	(MODE-SELECT  (+ mouse-select mouse-mode -1))
	(MODE-MODIFY  (+ mouse-modify mouse-mode -1))
	(MODE-PASTE   (+ mouse-paste mouse-mode -1))
	(DOWN	      mouse-down)
	(UP	      mouse-up)
	(DELETE	      mouse-delete-modifier)
	(THING	      mouse-thing-modifier)
	(WINDOW	      mouse-window-modifier)
	(ALL	      mouse-meta-control-shift))
    (global-set-mouse SELECT	  DOWN		    'mouse-set-point-or-select)
    (global-set-mouse SELECT	  UP		    nil)
    (global-set-mouse SELECT	  DELETE	    'mouse-delete-char)
    (global-set-mouse SELECT	  THING		    'mouse-select-thing)
    (global-set-mouse SELECT	  (+ DELETE THING)  'mouse-kill-thing)

    (global-set-mouse MODIFY	  DOWN		    'extend-mouse-drag)
    (global-set-mouse MODIFY	  UP		    'end-mouse-drag)
    (global-set-mouse MODIFY	  DELETE	    'mouse-kill-region)
    (global-set-mouse MODIFY	  THING		    'mouse-select-bigger-thing)
    (global-set-mouse MODIFY	  (+ DELETE THING)  'mouse-kill-bigger-thing)

    (global-set-mouse PASTE	  DOWN		    'mouse-set-point-and-paste)
    (global-set-mouse PASTE	  UP		    nil)
    (global-set-mouse PASTE	  DELETE	    'mouse-move-selection)
    (global-set-mouse PASTE	  THING		    'mouse-copy-thing)
    (global-set-mouse PASTE	  (+ DELETE THING)  'mouse-move-thing)

    (global-set-mouse SELECT	  WINDOW	    'mouse-toggle-focus-screen)
    (global-set-mouse SELECT	  (+ WINDOW DELETE) 'mouse-indent)

    (global-set-mouse MODIFY	  WINDOW	    'mouse-drag-scroll)
    (global-set-mouse MODIFY	  (+ WINDOW UP)	    'mouse-drag-scroll-terminate)
    (global-set-mouse MODIFY	  (+ WINDOW DELETE) 'mouse-fill)

    (global-set-mouse PASTE	  WINDOW	    'mouse-menu)
    (global-set-mouse PASTE	  (+ WINDOW DELETE) 'mouse-exec-kbd-macro)

    (global-set-mouse LEFT	  (+ WINDOW THING)  'mouse-line-to-top)
    (global-set-mouse MIDDLE	  (+ WINDOW THING)  'mouse-line-to-center)
    (global-set-mouse RIGHT	  (+ WINDOW THING)  'mouse-line-to-bottom)

    (global-set-mouse LEFT	  ALL		    'mouse-summary)
    (global-set-mouse MIDDLE	  ALL		    'mouse-play)
    (global-set-mouse RIGHT	  ALL		    'mouse-tutorial)

    (global-set-mouse MODE-SELECT DOWN		    'mouse-delete-other-windows)
    (global-set-mouse MODE-SELECT DELETE	    'mouse-delete-window)

    (global-set-mouse MODE-MODIFY DOWN		    'mouse-move-modeline)
    (global-set-mouse MODE-MODIFY UP		    'mouse-move-modeline-terminate)
    (global-set-mouse MODE-MODIFY DELETE	    'mouse-move-window)
    (global-set-mouse MODE-MODIFY (+ DELETE UP)	    'mouse-move-window-terminate)

    (global-set-mouse MODE-PASTE  DOWN		    'mouse-split-window)
    (global-set-mouse MODE-PASTE  DELETE	    'mouse-switch-to-buffer-other-window)
    ))


;;; It's more efficient to call this from the default.el or the .emacs file,
;;; but it's here for safety, to provide *some* bindings.
(imouse-do-bindings)
