# Solfege - free ear training software
# Copyright (C) 2007 Tom Cato Amundsen
# License is GPL, see file COPYING

import unittest
from src.tree import TreeNode, Cursor

class TestTreeNode(unittest.TestCase):
    def test_create(self):
        tree = TreeNode('tree')
        cursor = Cursor(tree)
        self.assert_(cursor.get() == tree)
        self.assert_(cursor.go_next() == False)
        self.assert_(cursor.get() == tree)
        n0 = TreeNode('n0')
        tree.add(n0)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n0)
        self.assert_(cursor.go_next() == False)
        n00 = TreeNode('n00')
        n0.add(n00)
        self.assert_(cursor.get() == n0)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n00)
        self.assert_(cursor.go_next() == False)
        n000 = TreeNode('n000')
        n00.add(n000)
        self.assert_(cursor.get() == n00)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n000)
        self.assert_(cursor.go_next() == False)
        n001 = TreeNode('n001')
        n00.add(n001)
        self.assert_(cursor.get() == n000)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n001)
        self.assert_(cursor.go_next() == False)
        n1 = TreeNode('n1')
        tree.add(n1)
        self.assert_(cursor.get() == n001)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n1)
        self.assert_(cursor.go_next() == False)
        n2 = TreeNode('n2')
        tree.add(n2)
        self.assert_(cursor.go_next() == True)
        self.assert_(cursor.get() == n2)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == n1)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == n001)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == n000)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == n00)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == n0)
        self.assert_(cursor.go_prev() == True)
        self.assert_(cursor.get() == tree)
        cursor.last()
        self.assert_(cursor.get() == n2)
        cursor.goto([0, 0, 1])
        self.assert_(cursor.get() == n001)
        self.assert_(cursor.go_parent() == True)
        self.assert_(cursor.get() == n00)
        self.assert_(cursor.go_parent() == True)
        self.assert_(cursor.get() == n0)
        self.assert_(cursor.go_parent() == True)
        self.assert_(cursor.get() == tree)
        cursor.goto([0, 0, 1])
        self.assert_(cursor.get() == n001)
        cursor.delete()
        self.assert_(cursor.get() == n000)
        cursor.delete()
        self.assert_(cursor.get() == n00)
        cursor.goto([2])
        self.assert_(cursor.get() == n2)
        cursor.delete()
        self.assert_(cursor.get() == n1)
        cursor.delete()
        self.assert_(cursor.get() == n00)
        cursor.delete()
        self.assert_(cursor.get() == n0)
        cursor.delete()
        self.assert_(cursor.get() == tree)
    def setUp(self):
        self.tree = TreeNode('tree')
        self.n0 = TreeNode('n0')
        self.tree.add(self.n0)
        self.n00 = TreeNode('n00')
        self.n0.add(self.n00)
        self.n000 = TreeNode('n000')
        self.n00.add(self.n000)
        self.c = Cursor(self.tree)
    def test_new_point_to_node(self):
        c = Cursor.new_point_to_node(self.n0)
        self.assertEqual(c.get().m_name, 'n0')
    def test_go_prev(self):
        c = Cursor.new_point_to_node(self.n0)
        self.assert_(c.go_prev(), "go_prev should return True on success")
        self.assert_(not c.go_prev(), "go_prev when at first should return False")
    def test_go_next(self):
        c = Cursor.new_point_to_node(self.n00)
        self.assert_(c.go_next(), "go_next should return True on success")
        self.assert_(not c.go_next(), "go_next when at last node should return False")
    def test_XX_prev_sibling(self):
        """
        Test go_prev_sibling and get_prev_sibling.
        """
        x = TreeNode('x')
        self.n0.add(x)
        c = Cursor(self.tree)
        c.goto_node(x)
        self.assertEqual(c.get_prev_sibling(), self.n00)
        self.assert_(c.go_prev_sibling())
        self.assertEqual(c.get(), self.n00)
        self.assert_(not c.go_prev_sibling())
        self.assertEqual(c.get(), self.n00)
        self.assertEqual(c.get_next_sibling(), x)
        self.assert_(c.go_next_sibling())
        self.assertEqual(c.get(), x)
    def test_goto_node(self):
        self.assert_(self.c.goto_node(self.n0))
    def test_cursor_constructor(self):
        c = Cursor(self.tree)
        self.assertEqual(c.get(), self.tree)
        self.assertEqual(c.m_tree, self.tree)
        c = Cursor(self.n00)
        self.assertEqual(c.get(), self.tree)
        self.assert_(c.goto_node(self.n000))
        self.assertEqual(c.get(), self.n000, "goto_node failed")
    def test_is_child_of(self):
        c = Cursor(self.tree)
        self.assertEqual(c.get().m_name, 'tree')
        self.assert_(c.go_next())
        self.assertEqual(c.get().m_name, 'n0')
        self.assert_(c.go_next())
        self.assertEqual(c.get().m_name, 'n00')
        self.assert_(c.go_next())
        self.assertEqual(c.get().m_name, 'n000')
        self.assert_(c.is_child_of('n00'))
        self.assert_(c.is_child_of('n0'))
        self.assert_(c.is_child_of('tree'))
        self.assert_(c.is_child_of('tree', 'n0', 'n00'))
        c.go_prev()
        self.assert_(c.is_child_of('n0'))
        self.assert_(not c.is_child_of('n00'))
    def test_insert(self):
        tree = TreeNode('tree')
        n0 = TreeNode('n0')
        tree.add(n0)
        n1 = TreeNode('n1')
        tree.add(n1)
        n2 = TreeNode('n2')
        tree.add(n2)
        cursor = Cursor(tree)
        cursor.goto([1])
        self.assert_(cursor.get() == n1)
        n0_1 = TreeNode('n0.1')
        cursor.insert(n0_1)
        self.assert_(cursor.get() == n0_1)
        nfirst = TreeNode('nfirst')
        cursor.goto([0])
        cursor.insert(nfirst)
        self.assert_(cursor.get() == nfirst)
        self.assert_(cursor.m_pos == [0])
        nv = TreeNode('nv')
        cursor.goto([1])
        cursor.get().add(nv)
        cursor.go_next()
        x = TreeNode('x')
        cursor.insert(x)
    def test_iterate_tree(self):
        v = list(self.n0.iterate_tree())
        self.assertEqual(v[0], self.n0)
        self.assertEqual(v[1], self.n00)
        self.assertEqual(v[2], self.n000)
        self.assertEqual(len(v), 3)
    def test_iterate_children(self):
        v = list(self.n0.iterate_children())
        self.assertEqual(v[0], self.n00)
        self.assertEqual(v[1], self.n000)
        self.assertEqual(len(v), 2)

suite = unittest.makeSuite(TestTreeNode)
