/* getaline.c - routines used to read a line in vary length 
 * Copyright (C) 2005-2006 Jia Wang (skyroam@gmail.com)
 *
 *
 *
 * This file is part of proxyknife.
 * Proxyknife is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the 
 * Free Software Foundation; either version 2 of the License, or (at your 
 * option) any later version.
 *
 * Proxyknife is distributed in the hope that it will be useful, but WITHOUT ANY 
 * WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License 
 * for more details.
 *
 *
 * You should have received a copy of the GNU General Public License 
 * along with Proxyknife; if not, write to the Free Software 
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA
 */

#include <proxyknife.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
/* Read a entire line and store the address of the buffer  containing
   the text into *line. The buffer is newline-terminated.

   If *line is NULL, getaline() routine will allocate a new buffer in 
   engough size(try  *size firstly)  to contain the text. 
   If *line is too small to contain the entire line, the 
   routine will realloc buffer until it is big enough. 
   
   So *line should be free by user.
   
   SIZE: size of init-buffer if *line == NULL.
   
   Return value:
   -1: failed.
   number of characters including \n: sucess.
   */
/* need a speed version including last line */
/* fgets may no use for lines in every length */
int
getaline (unsigned char **line, int *size, FILE * f)
{
  unsigned char *p, *q, ret;
  int offset, n, D = 256;

  if (*size <= 0)
    return -1;

  if (*line == NULL)
    {
      p = malloc (*size);
      if (p == NULL)
	{
	  *line = NULL;		/*no memory allocated,no need clean mem */
	  /* return -1; */
	  xexit (EXIT_FAILURE);
	}
    }
  else
    {
      p = *line;
    }

  offset = 0;
  while (1)
    {
      if (offset + 2 > *size)
	{
	  *size += D;
	  q = xrealloc ((void *) p, *size);
	  /*
	     if (q == NULL)
	     { */
	  /* this will not work if realloc destroy p for bug */
	  *line = p;
	  /* return -1; */
	  /*xexit (EXIT_FAILURE);
	     }
	     else */
	  p = q;
	}
      n = fread (p + offset, 1, 1, f);
      if (n != 1)
	{
	  *line = p;
	  return -1;
	}
      if (p[offset] == '\n')
	{
	  p[offset + 1] = '\0';	/* total offset+2 len */
	  *line = p;
	  return (offset + 1);	/* including '\n'; */
	}
      offset++;
    }
}

/* The same as above,but used between pthread mutex.
   Inner mutex was used in pxmalloc and pxrealloc. 
   Use fgets for performance,so it should be protected by the  
   mutex outside.

   The struct thread_mem is only used for pxmalloc and pxrealloc. 
 */


int
getaline_r (char **line, int *size, FILE * f, struct thread_mem *thread_mem)
{
  char *p, *q, *s;
  int offset, n, D = 256, linelen;
  size_t len;
  linelen = 80;

  if (*size <= 0)
    return -1;

  if (*line == NULL)
    {
      p = pxmalloc (*size, thread_mem);
      if (p == NULL)		/* for mutex, we shouldn't exit here? */
	return PROXYKNIFE_THREAD_EXIT;
    }
  else
    {
      p = *line;
    }

  offset = 0;
  while (1)
    {
      if (offset + linelen > *size)
	{
	  *size = offset + linelen + D;
	  q = pxrealloc (p, *size, thread_mem);
	  if (q == NULL)
	    return PROXYKNIFE_THREAD_EXIT;
	  p = q;
	}

      /* ftell */
      s = fgets (p + offset, linelen, f);
      /* ftell ,sub= read */


      if (s == NULL)
	{
	  *line = p;		/* for free */
	  return -1;		/*  end of file or errors */
	}
      else
	{
	  len = strlen (p + offset);	/* count before NULL, ignore others */
	  if (*(p + offset + len - 1) == '\n')
	    {
	      *line = p;
	      return (offset + len);	/* including '\n'; */
	    }
	  else
	    {
	      offset += len;
	    }
	}
    }
}
