/*
 * PWRAP
 *
 * This program performs simple universe wrap-around. That is, a ship that
 * flies beyond the defined boundaries of the universe is moved to the
 * other "edge" of the universe, so that it always remains in-bounds. Two
 * universe shapes are currently supported, round and square. For square
 * universes, if a ship flies beyond one edge of the square it is moved
 * to the other edge. For round universes, the ship is moved to the other
 * side of the diameter that the ship is on. In both cases, the distance
 * that a ship is placed from its new edge is equal to the distance that
 * the ship overflew the original edge. The farther over the edge the
 * ship went, the closer to the center (on the other side) the ship will
 * be.
 *
 * This program is meant to be executed in the AUXHOST 2 phase of host
 * processing.
 *
 * This file is copyrighted by Andrew Sterian in 1995. You may copy and
 * distribute this file freely as long as you retain this notice and
 * explicitly document any changes you make to this file (along with your name
 * and date) before distributing modified versions.
 *
 */

#define PWRAP_VERSION_MAJOR 1
#define PWRAP_VERSION_MINOR 0

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <math.h>
#include <memory.h>
#include <assert.h>
#include "phostpdk.h"
#include "pgetopt.h"

#define  AND   &&
#define  OR    ||
#define  EQ    ==
#define  NEQ   !=
#define  GT    >
#define  GE    >=
#define  LT 	<
#define  LE    =<

#ifndef M_PI
#define M_PI        3.14159265358979323846
#endif

#ifndef min
#define min(a,b) ((a) < (b) ? (a) : (b))
#endif
#ifndef max
#define max(a,b) ((a) > (b) ? (a) : (b))
#endif
#define sgn(x) ((x) < 0 ? -1 : ((x) > 0 ? 1 : 0))

typedef enum {
    Square,
    Circular
} MapShape_Def;

typedef struct {
    MapShape_Def mShape;
    Uns16        mRadius;       /* Or square side length */
    Uns16        mCenterX;      /* The center of the square or circle */
    Uns16        mCenterY;
} MConfig_Struct;

static MConfig_Struct gConfig = {
    Square,
    2000,
    2000,
    2000
};

static void
usage(char *argv[])
{
    fprintf(stderr,
"Usage: %s [options] [GameDirectory [RootDirectory]]\n"
"\n"
"options:\n"
"        -c        -- circular universe (default is square)\n"
"        -r        -- size of universe (radius for circle, length of one\n"
"                     side of the square for a square universe)\n"
"        -x        -- X co-ordinate of universe center (default 2000)\n"
"        -y        -- Y co-ordinate of universe center (default 2000)\n"
"        -h        -- prints this help summary and exits\n"
"        -v        -- prints program version and exits\n"
        ,argv[0]);
    exit(1);
}

static void
processOptions(int argc, char *argv[])
{
    int c;

    while ((c=pgetopt(argc, argv, "hHcr:x:y:v")) NEQ EOF) {
        switch (c) {
         case 'h': case 'H': default:
            usage(argv);

         case 'c':
            gConfig.mShape = Circular;
            break;

         case 'r':
            gConfig.mRadius = atoi(poptarg);
            break;

         case 'x':
            gConfig.mCenterX = atoi(poptarg);
            break;

         case 'y':
            gConfig.mCenterY = atoi(poptarg);
            break;

         case 'v':
            printf("Portable WRAP version %u.%u\n", PWRAP_VERSION_MAJOR, PWRAP_VERSION_MINOR);
            exit(0);
        }
    }

    if (poptind < argc) {
        gGameDirectory = argv[poptind++];
    }
    if (poptind < argc) {
        gRootDirectory = argv[poptind++];
    }
    if (poptind < argc) {
        fprintf(stderr,"Too many parameters. First extraneous parameter is '%s'\n"
                       "Type '%s -h' for a usage summary.\n",
                       argv[poptind], argv[0]);
        exit(1);
    }
}

static void doSquareWrap(Uns16 pShip)
{
    static Int16 lBoundX1, lBoundX2, lBoundY1, lBoundY2;
    static Boolean lFirstTime = True;
    Int16 lX, lY;
    Boolean lWrapped;

    if (lFirstTime) {
        lBoundX1 = gConfig.mCenterX - gConfig.mRadius/2;
        lBoundX2 = lBoundX1 + gConfig.mRadius;
        lBoundY1 = gConfig.mCenterY - gConfig.mRadius/2;
        lBoundY2 = lBoundY1 + gConfig.mRadius;
    }

    lX = ShipLocationX(pShip); lY = ShipLocationY(pShip);

    lWrapped = False;
    if (lX < lBoundX1) {
        lX += gConfig.mRadius; lWrapped = True;
    } else if (lX > lBoundX2) {
        lX -= gConfig.mRadius; lWrapped = True;
    }
    if (lY < lBoundY1) {
        lY += gConfig.mRadius; lWrapped = True;
    } else if (lY > lBoundY2) {
        lY -= gConfig.mRadius; lWrapped = True;
    }

    if (lWrapped) {
        PutShipLocationX(pShip, lX); PutShipLocationY(pShip, lY);
        PutShipRelWaypointX(pShip, 0); PutShipRelWaypointY(pShip, 0);
    }
}

static void doRoundWrap(Uns16 pShip)
{
    Int16 lX, lY, lDX, lDY;
    double lRadius, lAngle;

    lX = ShipLocationX(pShip); lY = ShipLocationY(pShip);
    lDX = lX - gConfig.mCenterX;
    lDY = lY - gConfig.mCenterY;

    if ( (lRadius=sqrt((double)((Int32)lDX*lDX + (Int32)lDY*lDY))) > gConfig.mRadius) {
        lRadius = 2*gConfig.mRadius - lRadius;
        lAngle = atan2((double) -lDY, (double) -lDX);

        lX = lRadius*cos(lAngle) + gConfig.mCenterX + 0.5;
        lY = lRadius*sin(lAngle) + gConfig.mCenterY + 0.5;

        PutShipLocationX(pShip, lX);
        PutShipLocationY(pShip, lY);
        PutShipRelWaypointX(pShip, 0);
        PutShipRelWaypointY(pShip, 0);
    }
}

int
main(int argc, char *argv[])
{
    Int16 lDummy;
    Uns16 lShip;

    processOptions(argc, argv);

    /* Initialize the PDK */
    InitPHOSTLib();

    gLogFile = OpenOutputFile("pwrap.log", GAME_DIR_ONLY | TEXT_MODE);

    Info("PWRAP Version %u.%u", PWRAP_VERSION_MAJOR, PWRAP_VERSION_MINOR);

    /* The only thing we need to read in is the list of ships and the
       list of planet positions. */
    if (! Read_Ships_File(&lDummy)) exit(1);
    if (! Read_Planets_File(&lDummy)) exit(1);
    if (! Read_Xyplan_File()) exit(1);

    Info("Starting to wrap ships...");

    for (lShip=1; lShip <= SHIP_NR; lShip++) {
        if (! IsShipExist(lShip)) continue;

        switch (gConfig.mShape) {
         case Square:
            doSquareWrap(lShip);
            break;

         case Circular:
            doRoundWrap(lShip);
            break;

         default: assert(0);
        }
    }

    /* Write back the list of ships */
    if (! Write_Ships_File(0)) exit(1);

    Info("Done.");

    FreePHOSTLib();

    return 0;
}
