/*************************************************************************
Legends Soldier Calculator
Copyright (c) 1994 by Mark R. Nuiver.  All rights reserved.

Module Name      => lsc.c
Version          => 4.0.0
Change Date      => 94/03/09
Programmer       => M.R. Nuiver
Operating System => AmigaDOS 3.00                  MS-DOS 6.0
Compiler         => SAS/C C 5.10                   Power C 2.0.0
Modules Used     => lscdata lsccalc
  (Amiga Only)   => c.o
Header Files     => lsc.h
Libraries Used   => lcm.lib, lc.lib amiga2.0.lib   PCLIB.MIX, PCAUTO.MIX
Compiler Options => None
Data Files       => ARMOR.LSC. MOUNTS.LSC, RACES.LSC, SHIELDS.LSC,
                    STATUS.LSC, TYPES.LSC, WEAPONS.LSC
Documentation    => LSC.DOC (user documentation)

Description

  This purpose of this program is to calculate the basic strength of
legions in the play-by-mail fantasy game Legends.  Because of the
incredible complexity and interdependence of numerous factors, this is
a time consuming and error prone manual process.

Credits

  Legends is a play-by-mail fantasy wargame system designed by Jim Landes
and moderated commercially by Midnight Games.  Legends features aspects
of both power wargaming and role-playing, with cities, armies, civilians,
player characters, monsters, gods, and magic all interacting in a
complex, detailed, and often mysterious manner.
  Legends is a trademark of Midnight Games.
  Thanks to Dennis Vandenberg a.k.a. Zardon for help with understanding
morale and for the kibitzing while he was working on his own project for
Warriors & Wizards.
  Thanks to J. B. Jones for bug reports on version 3.0.0.

Revision Notes

Version Date      Programmer      Reason
------- --------  --------------  ---------------------------------------
4.0.0   94/02/25  Mark Nuiver     * Tweaks and fixes for MS-DOS compile:
                                    - forgot to declare Total[] a long (oops)
4.0.0   94/01/17  Mark Nuiver     * Added a note when Tactics overall
                                    modifier exceeds 300%.
                                  * Created print_magic_overall(),
                                    print_character_overall(),
                                    print_terrain_overall(),
                                    print_forts_overall(), and
                                    print_subtotals_per_soldier().
                                  * Changed print_results to show soldier
                                    subtotals prior to magic bonuses (which
                                    corresponds to the soldier data as it
                                    prints on a battle report).
                                  * Various changes to print functions
                                    using all the lsccalc functions to
                                    compute the values to print.
4.0.0   94/01/08  Mark Nuiver     * Added expanded character data.
4.0.0   93/12/27  Mark Nuiver     * Changed forts and terrain to use revised
                                    symbols in lsc.h.
                                  * Added call to validate_legion() in
                                    main().
                                  * Added call to print_reserve() in
                                    print_results().
                                  * Made printing of base forts bonuses
                                    dependent on reserve slot options.
                                  * Added insert_legion_notes().
                                  * Added call to print_losses() in
                                    print_results().
4.0.0   93/12/22  Mark Nuiver     * Changed entire print_results() logic.
                                    All calculations are now performed in
                                    one call to compute_legion().  This
                                    ensures that LSC and LBS use identical
                                    logic in computing the legion totals.
                                  * Removed most extern function references
                                    to lsccalc functions, added extern to
                                    compute_base_df().
                                  * Added print_terrain().
                                  * Added Terrain Overall Modifier to
                                    print_slot_overall().
                                  * Moved print_terrain() to lscdata.c to
                                    share with lbs.c.
4.0.0   93/12/21  Mark Nuiver     * Changed version.
                                  * Changed Character[] and Forts[] to use
                                    the new OVER symbol in place of the
                                    misleading re-use of CAFB.
                                  * Removed print_special_type().
                                  * Removed print_special_status().
------- --------  --------------  ---------------------------------------
3.0.1   93/12/19  Mark Nuiver     * Changed overlay functions to make the
                                    size of the target string for the
                                    sprintf() functions large enough to
                                    handle the largest possible long integer.
                                  * Fixed the length of a couple of other
                                    sprintf() targets, changed "Unknown
                                    ITEM" message so that ID# would fit.
                                  * Changed the columns and title line for
                                    larger output widths.
                                  * Added code to check that the output
                                    numbers do not exceed the column width,
                                    and if they do print ssterisks to show
                                    that there is an "overflow".
                                  * Changed all int to short so that Amiga
                                    and DOS version will perform same.
------- --------  --------------  ---------------------------------------
3.0.0   93/12/14  Mark Nuiver     * Some minor tweaking that Power C wanted
                                    for the MS-DOS version.
3.0.0   93/12/12  Mark Nuiver     * Added code to handle DAM and INV.
3.0.0   93/12/11  Mark Nuiver     * Various changes moving file functions
                                    and computations into separate modules,
                                    for re-use in LBS program.
------- --------  --------------  ---------------------------------------
2.1.0   This version was not released.
2.1.0   93/12/07  Mark Nuiver     * Changed compute_soldier_totals() to make
                                    the maximum possible AFB and MAFB 400%.
                                  * Changed the columns and title line for
                                    6 digit AF, MAF, and CAF.
2.1.0   93/11/24  Mark Nuiver     * Fortifications were added to this
                                    version.
                                  * Added Fortificationfields[] array to
                                    define fortifications input fields.
                                  * Added Fortifications[] array to hold
                                    fortifications input data.
                                  * Changed load_legion() to input forts.
                                  * Added compute_forts() function to
                                    calculate and print the base (i.e. w/o
                                    overall modifier) fortifications adds.
                                  * Changed compute_slt_overall() to include
                                    all 3 current overall modifier sources:
                                    character slot magic, force overall
                                    leader modifier, and fortifications.
                                    compute_overall() was eliminated.
                                    There was a slight logic problem with
                                    the previous version that was corrected
                                    by doing this.  Previously, the DF
                                    bonus was calculated separately for
                                    both of the overall modifiers, which
                                    was incorrect.  It should have been
                                    done for all the modifiers combined.
                                    This would rarely have altered the
                                    results because of rounding, but it
                                    would have become obvious with the
                                    inclusion of the large fortifications
                                    DF bonus.
------- --------  --------------  ---------------------------------------
2.0.0   93/03/09  Mark Nuiver     * Changed the Typefields[], Racefields[],
                                    and Statusfields[] arrays to include
                                    the MAFB and SAFB fields.  These data
                                    files have a new format as of this
                                    version.
                                  * Changed compute_item() "special
                                    considerations" for RACE, TYPE, STATUS.
                                    Removed code that used the AFB as the
                                    MAFB.
                                  * Changed special_type():
                                      (1) Removed all Surprise MAFB note
                                          insertion.
                                      (2) Deleted G, I, and J special codes
                                          sections.  These are superfluous
                                          with the surprise bonus taken
                                          out.  Special note: code C
                                          replaces G now.  Previously, G
                                          gave a +50% bonus for ALL missile
                                          weapons.  C only gives it for
                                          "bows".  This is consistent with
                                          the Archers training, however.
                                      (3) Separated out the AFB and MAFB.
                                          There is still some question
                                          whether the bonuses given for
                                          missile weapons also apply to
                                          the melee AFB for those weapons.
                                          At this point, the bonus for
                                          missile weapons is added to both
                                          melee and missile.
                                  * Created new function insert_slot_notes()
                                    to add notes based on the final slot
                                    results.  At the moment all it does is
                                    add an appropriate note for Surprise
                                    Missile Round MAFB since that isn't
                                    printed.
                                  * Fixed a bug in the input of the data
                                    files: use of sscanf() on a blank or
                                    bad entry wouldn't overwrite the
                                    variable, which contained either garbage
                                    (auto variable) or the value from the
                                    previous entry.  Changed all sscanf()
                                    calls to atoi() calls; atoi() returns
                                    zero on "no data".
                                  * Added one validation to data file input:
                                    ID# field cannot be less than 1.
                                  * Removed ERROR_TOO_MUCH_DATA case from
                                    cleanup().  Not an error that can occur
                                    since several versions back.
                                  * Miscellaneous style changes and made
                                    the code a little neater.
        93/03/10  Mark Nuiver     * Expanded the weapon codes and put new
                                    notes into insert_slot_notes() warning
                                    of shield used with 2handed weapon,
                                    weapon used with mount when it can't be,
                                    and no mount when mount required.  Also
                                    fixed the new Surprise Round notes so
                                    that they only get inserted if soldiers
                                    also have a missile weapon!  Getting
                                    sloppy.
                                  * Changed the way weapon classes are
                                    stored and associated logic; class is
                                    stored as bit mask and ItemClass char
                                    array in slot_struct has been removed.
                                  * Changed determination of whether slot
                                    has missile weapon or not in
                                    compute_totals() to look at weapon
                                    class.
                                  * Changed compute_totals() so that soldier
                                    totals don't print when only a character
                                    is in the slot.  Also all character
                                    calculations and printing are dependent
                                    on the character ID field being non-zero
                                    in the legion description file + one or
                                    more appropriate fields being non-zero
                                    (i.e. CF, AF, MAF, CAF, SAV for combat
                                    stuff, MAV and AFB for magic, OVER for
                                    overall bonus).
        93/03/15  Mark Nuiver     * Changed the logic for character output
                                    yet again; was generating spurious
                                    output when Character ID# in slot with
                                    no other numbers.  Made ID# optional,
                                    one of the other fields mandatory for
                                    any character output.  Moved all the
                                    logic for what calculations/output to
                                    do into compute_results().  Split
                                    compute_overall() and compute_totals()
                                    into several smaller functions.  I hate
                                    it when I do this!
------- --------  --------------  ---------------------------------------
1.0.4   93/03/03  Mark Nuiver     * Corrected an error in compute_results()
                                    logic.  Overall commander bonus was not
                                    being applied if there was no character
                                    assigned to the slot.
------- --------  --------------  ---------------------------------------
1.0.3   93/02/24  Mark Nuiver     * Added the overall modifier effect to DF.
                                  * Character magic CAFB didn't make sense;
                                    made it an overall bonus to slot instead
                                    and using that field for overall leader
                                    overall bonus too.  Both print with
                                    preceeding 'O' character.
                                  * Changed output so that Training level
                                    line won't print for slots with no
                                    soldiers but a character, and added the
                                    slot number to the character output.
------- --------  --------------  ---------------------------------------
1.0.2   93/02/20  Mark Nuiver     * Fixed problems that showed up only in
                                     the POWER C compiler:
                                     (1) removed function prototype for
                                         main().
                                     (2) changed most of the integer
                                         variables from long to int.  PC
                                         won't switch() a long.
                                     (3) added the "l" modifier in the
                                         printf() and sprintf() functions
                                         that print long integers.
                                     (4) escaped %'s in printf() format
                                         strings.
------- --------  --------------  ---------------------------------------
1.0.1   93/02/02  Mark Nuiver     * Added new special training type code
                                     'I' for +200% MAFB only (Assassins).
                                  * Changed output so that slots with no
                                     soldiers or character will not print.
*************************************************************************/
#include "lsc.h"

/* external functions */
extern void cleanup(enum lsc_errors, char *, short);
extern void load_legion(char *, struct legion_struct *);
extern void load_tables(struct legion_struct *);
extern void init_tables(void);
extern enum lsc_symbols decode_terrain(short);
extern void compute_trl(struct slot_struct *, long *);
extern void compute_forts(struct legion_struct *, short, long *);
extern void compute_magic(struct slot_struct *, long *);
extern void compute_character_overall(struct legion_struct *, short, short, long *);
extern void compute_forts_overall(struct legion_struct *, short, long *);
extern void compute_terrain_overall(struct legion_struct *, short, long *);
extern void compute_totals_per_soldier(struct legion_struct *, short, long *);
extern void compute_legion(struct legion_struct *);
extern void print_terrain(short *);
extern void print_reserve(short *);
extern void print_losses(short *);
extern void validate_legion(struct legion_struct *);
extern short use_forts(struct legion_struct *, short);
extern long compute_minmax_afb(long);
extern long compute_minmax_mafb(long);

/* external variables */
extern char *Sfieldnames[LASTSFIELD]; /* names of soldier fields */
extern char *Fieldnames[LASTFIELD]; /* names of item fields */

/* local variables */
static char title[] =
"SLT: Amt :Description         :  CF   : DF :   AF   :  MAF   :  CAF   :MAV :  SAV  :MAD:SAD:DAM:INV:";
static char separ[] =
"---:-----:--------------------:-------:----:--------:--------:--------:----:-------:---:---:---:---:";
static char columns[] =
"   :     :                    :       :    :        :        :        :    :       :   :   :   :   :";

static struct legion_struct Legion; /* all the legion data */
static long Total[LASTFIELD]; /* results of lsccalc functions */
static char Notes[MAXNOTES][NOTESIZE + 1];    /* notes about the output */
static short notecount;                       /* number of note lines */
static char overflow[] = "*************";

/* local function prototypes */
static void help(void);
static void print_results(char *);
static void print_item(short, short, char *);
static void print_trl(short, char *);
static void print_forts(short, char *); /* v2.1.0 */
static void print_character(short, char *);
static void print_magic(short, char *);
static void print_magic_overall(short, char *);
static void print_character_overall(short, char *);
static void print_forts_overall(short, char *);
static void print_terrain_overall(short, char *);
static void print_subtotals_per_soldier(short, char *);
static void print_totals_per_soldier(short, char *);
static void print_soldier_totals(short, char *);
static void print_character_totals(short, char *);
static void print_slot_totals(short, char *);
static void insert_slot_notes(short);
static void insert_legion_notes(void);
static void print_grand_totals(char *);
static void insert_note(char *);
static void overlayslot(short, char *);
static void overlaycode(char, short, char *);
static void overlayamt(long, char *);
static void overlaydesc(char *, char *);
static void overlaytext(char *, char *);
static void overlaycf(long, char *);
static void overlaydf(long, char *);
static void overlayaf(long, char *);
static void overlayafb(long, char *);
static void overlaymaf(long, char *);
static void overlaymafb(long, char *);
static void overlaycaf(long, char *);
static void overlaycafb(long, char *);
static void overlaymav(long, char *);
static void overlaysav(long, char *);
static void overlaymad(long, char *);
static void overlaysad(long, char *);
static void overlaydam(long, char *);
static void overlayinv(long, char *);

/*************************************************************************
help:
*************************************************************************/
static void help()
  {

    printf("\n\nCommand syntax: 'lsc [<filename> | ? ]'");
    printf("\n\n  '<filename>' is the name of your legion description");
    printf("\nfile.");
    printf("\n  No arguments or '?' displays this help.");
    printf("\n  Use I/O re-direction to write the output into a file, e.g.");
    printf("\n'lsc f1234 >lsc.out'.");
    printf("\n");
    cleanup(ERROR_NO_ERROR, NULL, NULL);

  } /* end of help() */

/*************************************************************************
print_results:
*************************************************************************/
static void print_results(Legionfile)
char *Legionfile;
  {

    short i;
    short slot;
    long t;
    char outline[OUTLINESIZE + 1];

    (void) time(&t);
    printf("\n\nCalculations for legion %s on %s", Legionfile, ctime(&t));
    print_terrain(&(Legion.Terrain[0]));
    print_reserve(&(Legion.Reserve[0]));
    print_losses(&(Legion.Losses[0]));

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* do all soldier slots */

        if(Legion.Slot[slot].SoldierItems[AMT] > 0 || /* soldiers in slot */
           Legion.Slot[slot].Character[CF] != 0 || /* or character in slot */
           Legion.Slot[slot].Character[AF] != 0 || /* NOTE: [AFB] and [OVER] */
           Legion.Slot[slot].Character[MAF] != 0 || /* ignored; they should */
           Legion.Slot[slot].Character[CAF] != 0 || /* not print if no soldiers*/
           Legion.Slot[slot].Character[MAV] != 0 ||
           Legion.Slot[slot].Character[SAV] != 0)
          { /* something in slot */

            printf("\n\n%s\n%s", title, separ);

            if(Legion.Slot[slot].SoldierItems[AMT] > 0)
              { /* soldiers in slot */

                print_trl(slot, outline);

                for(i = 0; i < LASTTABLE; i++)
                  { /* print items in each table */

                    print_item(slot, i, outline);

                  } /* print items in each table */

                if(Legion.Forts[F_DF] > 0 ||
                   Legion.Forts[F_AFB] > 0)
                  { /* v2.1.0 base Fortifications bonuses */

                    if(use_forts(&Legion, slot))
                      { /* slot receives base fort bonuses */

                        print_forts(slot, outline);

                      } /* slot receives base fort bonuses */

                  } /* v2.1.0 base Fortifications bonuses */

                if(Legion.Forts[F_OVER] > 0)
                  { /* forts overall modifier */

                    print_forts_overall(slot, outline);

                  } /* forts overall modifier */

                if(Legion.Slot[MAXCSLOTS].Character[OVER] != 0)
                  { /* force leader overall modifier */

                    print_character_overall(slot, outline);

                  } /* force leader overall modifier */

                if(Legion.Terrain[T_BASIC] != 0 ||
                   Legion.Terrain[T_WALLS] != 0 ||
                   Legion.Terrain[T_SPECIAL] != 0)
                  { /* terrain DF overall modifier */

                    print_terrain_overall(slot, outline);

                  } /* terrain DF overall modifier */

              } /* soldiers in slot */

            if(Legion.Slot[slot].SoldierItems[AMT] > 0 &&
               (Legion.Slot[slot].Character[AFB] != 0 ||
                Legion.Slot[slot].Character[OVER] != 0))
              { /* slot has soldiers and character magic stuff */

                printf("\n%s", separ);
                print_subtotals_per_soldier(slot, outline);
                printf("\n%s", separ);

                if(Legion.Slot[slot].Character[AFB] != 0)
                  { /* character w/ magic AFB assigned to this slot */

                    print_magic(slot, outline);

                  } /* character w/ magic AFB assigned to this slot */

                if(Legion.Slot[slot].Character[OVER] != 0)
                  { /* character w/ magic overall modifier in slot */

                    print_magic_overall(slot, outline);

                  } /* character w/ magic overall modifier in slot */

              } /* slot has soldiers and character magic stuff */

            if(Legion.Slot[slot].Character[CF] != 0 ||
               Legion.Slot[slot].Character[AF] != 0 ||
               Legion.Slot[slot].Character[MAF] != 0 ||
               Legion.Slot[slot].Character[CAF] != 0 ||
               Legion.Slot[slot].Character[MAV] != 0 ||
               Legion.Slot[slot].Character[SAV] != 0)
              { /* character w/ combat attributes assigned to this slot */

                print_character(slot, outline);

              } /* character w/ combat attributes assigned to this slot */

            printf("\n%s", separ);

            if(Legion.Slot[slot].SoldierItems[AMT] > 0)
              { /* soldiers in slot */

                print_totals_per_soldier(slot, outline);
                print_soldier_totals(slot, outline);

              } /* soldiers in slot */

            if(Legion.Slot[slot].Character[CF] != 0 ||
               Legion.Slot[slot].Character[AF] != 0 ||
               Legion.Slot[slot].Character[MAF] != 0 ||
               Legion.Slot[slot].Character[CAF] != 0 ||
               Legion.Slot[slot].Character[MAV] != 0 ||
               Legion.Slot[slot].Character[SAV] != 0)
              { /* slot leader combat attributes */

                print_character_totals(slot, outline);

                if(Legion.Slot[slot].SoldierItems[AMT] > 0)
                  { /* soldiers with character */

                    print_slot_totals(slot, outline);

                  } /* soldiers with character */

              } /* slot leader combat attributes */

            insert_slot_notes(slot);

          } /* something in slot */

      } /* do all soldier slots */

    print_grand_totals(outline);

    insert_legion_notes();

    if(notecount > 0)
      { /* we have special notes */

        printf("\n\nNOTES:");

        for(i = 0;i < notecount;i++)
          { /* print the notes */

            printf("\n(%d) %s", i + 1, Notes[i]);

          } /* print the notes */

      } /* we have special notes */

    /* print the printout key */
    printf("\n\n\
KEY:\n\
SLT=Slot number or F-Forts, C-character, M-character magic, O-overall bonus\n\
Amt=Amount  CF=combat factor  DF=defense factor  AF=attack factor\n");
    printf("\
MAF=missile attack factor     CAF=charge attack factor\n\
MAV=magical attack value      SAV=special attack value\n\
MAD=magical attack defense    SAD=special attack defense\n\
DAM=damage rating             INV=invulnerability rating\n");


  } /* end of print_results() */

/*************************************************************************
print_item:
    prints all the numbers for the soldier items, e.g. race, training,
  weapon, armor, etc.
*************************************************************************/
static void print_item(slot, table_id, outline)
short slot;
short table_id;
char *outline;
  {

    char note[NOTESIZE + 1];
    char udesc[DESCSIZE + LONGWIDTH];

    if(Legion.Slot[slot].Item[table_id][ID] !=
       Legion.Slot[slot].SoldierItems[table_id])
      { /* unknown item */

        strcpy(outline, columns);
        sprintf(udesc, "? %s ID#%d", Sfieldnames[table_id],
                Legion.Slot[slot].SoldierItems[table_id]);
        overlaydesc(udesc, outline);
        printf("\n%s", outline);
        sprintf(note, "%s in slot %d ignored.", udesc, slot);
        insert_note(note);

      } /* unknown item */

    else
      { /* item in table or no item */

        if(Legion.Slot[slot].Item[table_id][ID] != 0)
          { /* slot has this item */

            /* print the numbers for this item */
            strcpy(outline, columns);
            overlaydesc(Legion.Slot[slot].ItemDesc[table_id], outline);
            overlaycf(Legion.Slot[slot].Item[table_id][CF], outline);
            overlaydf(Legion.Slot[slot].Item[table_id][DF], outline);
            overlayafb(Legion.Slot[slot].Item[table_id][AFB], outline);
            overlaymafb(Legion.Slot[slot].Item[table_id][MAFB], outline);
            overlaycafb(Legion.Slot[slot].Item[table_id][CAFB], outline);
            overlaymav(Legion.Slot[slot].Item[table_id][MAV], outline);
            overlaysav(Legion.Slot[slot].Item[table_id][SAV], outline);
            overlaymad(Legion.Slot[slot].Item[table_id][MAD], outline);
            overlaysad(Legion.Slot[slot].Item[table_id][SAD], outline);
            overlaydam(Legion.Slot[slot].Item[table_id][DAM], outline);
            overlayinv(Legion.Slot[slot].Item[table_id][INV], outline);
            printf("\n%s", outline);

          } /* slot has this item */

      } /* item in table or no item */

  } /* end of print_item() */

/*************************************************************************
print_trl:
  prints the training level bonuses for given slot.
*************************************************************************/
static void print_trl(slot, outline)
short slot;
char *outline;
  {

    char udesc[DESCSIZE + LONGWIDTH];

    Total[CF] = 0;
    Total[DF] = 0;

    compute_trl(&(Legion.Slot[slot]), Total);

    /* print the training bonuses */
    strcpy(outline, columns);
    overlayslot(slot, outline);
    overlayamt(Legion.Slot[slot].SoldierItems[AMT], outline);
    sprintf(udesc, "Training %d", Legion.Slot[slot].SoldierItems[TRL]);
    overlaydesc(udesc, outline);
    overlaycf(Total[CF], outline);
    overlaydf(Total[DF], outline);
    overlayafb(0, outline);
    overlaymafb(0, outline);
    overlaycafb(0, outline);
    overlaymav(0, outline);
    overlaysav(0, outline);
    overlaymad(0, outline);
    overlaysad(0, outline);
    overlaydam(0, outline);
    overlayinv(0, outline);
    printf("\n%s", outline);

  } /* end of print_trl() */

/*************************************************************************
print_forts:
  prints the base fortifications bonuses for given slot.
*************************************************************************/
static void print_forts(slot, outline)
short slot;
char *outline;
  {

    Total[DF] = 0;
    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_forts(&Legion, slot, Total);

    /* print the fortifications bonuses */
    strcpy(outline, columns);
    overlaycode('F', slot, outline);
    overlaydesc("Fortifications", outline);
    overlaydf(Total[DF], outline);
    overlayafb(Total[AFB], outline);
    overlaymafb(Total[MAFB], outline);
    printf("\n%s", outline);

  } /* end of print_forts() */

/*************************************************************************
print_character:
  prints combat ability of character in this slot.
*************************************************************************/
static void print_character(slot, outline)
short slot;
char *outline;
  {

    /* print the character combat abilities */
    strcpy(outline, columns);
    overlaycode('C', slot, outline);
    overlayamt(Legion.Slot[slot].Character[ID], outline);
    overlaydesc(Legion.Slot[slot].CharacterDesc, outline);
    overlaycf(Legion.Slot[slot].Character[CF], outline);
    overlaydf(Legion.Slot[slot].Character[DF], outline);
    overlayaf(Legion.Slot[slot].Character[AF], outline);
    overlaymaf(Legion.Slot[slot].Character[MAF], outline);
    overlaycaf(Legion.Slot[slot].Character[CAF], outline);
    overlaymav(Legion.Slot[slot].Character[MAV], outline);
    overlaysav(Legion.Slot[slot].Character[SAV], outline);
    overlaymad(Legion.Slot[slot].Character[MAD], outline);
    overlaysad(Legion.Slot[slot].Character[SAD], outline);
    overlaydam(Legion.Slot[slot].Character[DAM], outline);
    overlayinv(Legion.Slot[slot].Character[INV], outline);
    /* @@ include slot leader morale bonus: +Leadership (tactics) */
    printf("\n%s", outline);

  } /* end of print_character() */

/*************************************************************************
print_magic:
  prints character magic AFB for this slot.
*************************************************************************/
static void print_magic(slot, outline)
short slot;
char *outline;
  {

    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_magic(&(Legion.Slot[slot]), Total);

    /* print character magic effects */
    strcpy(outline, columns);
    overlaycode('M', slot, outline);
    overlayamt(Legion.Slot[slot].Character[ID], outline);
    overlaydesc(Legion.Slot[slot].CharacterDesc, outline);
    overlayafb(Total[AFB], outline);
    overlaymafb(Total[MAFB], outline);
    printf("\n%s", outline);

  } /* end of print_magic() */

/*************************************************************************
print_magic_overall:
  prints overall bonus to slot from:
    character in slot casting a spell
*************************************************************************/
static void print_magic_overall(slot, outline)
short slot;
char *outline;
  {

    Total[DF] = 0;
    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_character_overall(&Legion, slot, slot, Total);

    /* print slot leader magic overall bonuses */
    strcpy(outline, columns);
    overlaycode('M', slot, outline);
    overlayamt(Legion.Slot[slot].Character[ID], outline);
    overlaydesc(Legion.Slot[slot].CharacterDesc, outline);
    overlaydf(Total[DF], outline);
    overlayafb(Total[AFB], outline);
    overlaymafb(Total[MAFB], outline);
    printf("\n%s", outline);

  } /* end of print_magic_overall() */

/*************************************************************************
print_character_overall:
  prints overall bonus to slot from:
    overall leader tactical bonus
*************************************************************************/
static void print_character_overall(slot, outline)
short slot;
char *outline;
  {

    Total[DF] = 0;
    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_character_overall(&Legion, slot, MAXCSLOTS, Total);

    /* print overall leader/other bonus */
    strcpy(outline, columns);
    overlaycode('O', slot, outline);
    overlayamt(Legion.Slot[MAXCSLOTS].Character[ID], outline);
    overlaydesc(Legion.Slot[MAXCSLOTS].CharacterDesc, outline);
    overlaydf(Total[DF], outline);
    overlayafb(Total[AFB], outline);
    overlaymafb(Total[MAFB], outline);
    printf("\n%s", outline);

  } /* end of print_character_overall() */

/*************************************************************************
print_forts_overall:
  prints overall bonus to slot from:
    fortifications tactical bonus
*************************************************************************/
static void print_forts_overall(slot, outline)
short slot;
char *outline;
  {

    Total[DF] = 0;
    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_forts_overall(&Legion, slot, Total);

    /* print overall fortifications bonus */
    strcpy(outline, columns);
    overlaycode('O', slot, outline);
    overlaydesc("Fortifications", outline);
    overlaydf(Total[DF], outline);
    overlayafb(Total[AFB], outline);
    overlaymafb(Total[MAFB], outline);
    printf("\n%s", outline);

  } /* end of print_forts_overall() */

/*************************************************************************
print_terrain_overall:
  prints DF bonus to slot from:
    terrain overall modifiers
*************************************************************************/
static void print_terrain_overall(slot, outline)
short slot;
char *outline;
  {

    Total[DF] = 0;
    Total[AFB] = 0;
    Total[MAFB] = 0;

    compute_terrain_overall(&Legion, slot, Total);

    strcpy(outline, columns);
    overlaycode('O', slot, outline);
    overlaydesc("Terrain", outline);
    overlaydf(Total[DF], outline);
    /* NOTE: the AFB/MAFB would show up separately in each item, so */
    /* don't show it again */
    printf("\n%s", outline);

  } /* end of print_terrain_overall() */

/*************************************************************************
print_subtotals_per_soldier:
  prints the subtotals per soldier BEFORE magic AFB and Overall
*************************************************************************/
static void print_subtotals_per_soldier(slot, outline)
short slot;
char *outline;
  {

    short i;

    /* clear out  the total array */
    for(i = 0; i < LASTFIELD; i++)
      { /* all elements */

        Total[i] = 0;

      } /* all elements */

    /* NOTE: this only works because there has already been a call */
    /* to compute_legion(), which calls compute_items() in lsccalc. */
    /* recompute the raw soldier numbers */
    compute_totals_per_soldier(&(Legion), slot, Total);

    /* print the slot totals per soldier */
    strcpy(outline, columns);
    overlaytext("Totals per soldier:", outline);
    overlaycf(Total[CF], outline);
    overlaydf(Total[DF], outline);
    overlayaf(Total[AF], outline);
    overlaymaf(Total[MAF], outline);
    overlaycaf(Total[CAF], outline);
    overlaymav(0, outline);
    overlaysav(Total[SAV], outline);
    overlaymad(Total[MAD], outline);
    overlaysad(Total[SAD], outline);
    overlaydam(Total[DAM], outline);
    overlayinv(Total[INV], outline);
    printf("\n%s", outline);

  } /* end of print_subtotals_per_soldier() */

/*************************************************************************
print_totals_per_soldier:
  prints the totals per soldier.
*************************************************************************/
static void print_totals_per_soldier(slot, outline)
short slot;
char *outline;
  {

    long amt = Legion.Slot[slot].SoldierItems[AMT];

    /* print the slot totals per soldier */
    strcpy(outline, columns);
    overlaytext("Totals per soldier:", outline);
    overlaycf(Legion.Slot[slot].Total[CF] / amt, outline);
    overlaydf(Legion.Slot[slot].Total[DF], outline);
    overlayaf(Legion.Slot[slot].Total[AF] / amt, outline);
    overlaymaf(Legion.Slot[slot].Total[MAF] / amt, outline);
    overlaycaf(Legion.Slot[slot].Total[CAF] / amt, outline);
    overlaymav(Legion.Slot[slot].Total[MAV] / amt, outline);
    overlaysav(Legion.Slot[slot].Total[SAV] / amt, outline);
    overlaymad(Legion.Slot[slot].Total[MAD], outline);
    overlaysad(Legion.Slot[slot].Total[SAD], outline);
    overlaydam(Legion.Slot[slot].Total[DAM], outline);
    overlayinv(Legion.Slot[slot].Total[INV], outline);
    printf("\n%s", outline);

  } /* end of print_totals_per_soldier() */

/*************************************************************************
print_soldier_totals:
  prints the total for all soldiers.
*************************************************************************/
static void print_soldier_totals(slot, outline)
short slot;
char *outline;
  {

    /* print slot totals for all soldiers */
    strcpy(outline, columns);
    overlaytext("Totals for all soldiers:", outline);
    overlaycf(Legion.Slot[slot].Total[CF], outline);
    overlaydf(Legion.Slot[slot].Total[DF], outline);
    overlayaf(Legion.Slot[slot].Total[AF], outline);
    overlaymaf(Legion.Slot[slot].Total[MAF], outline);
    overlaycaf(Legion.Slot[slot].Total[CAF], outline);
    overlaymav(Legion.Slot[slot].Total[MAV], outline);
    overlaysav(Legion.Slot[slot].Total[SAV], outline);
    overlaymad(Legion.Slot[slot].Total[MAD], outline);
    overlaysad(Legion.Slot[slot].Total[SAD], outline);
    overlaydam(Legion.Slot[slot].Total[DAM], outline);
    overlayinv(Legion.Slot[slot].Total[INV], outline);
    printf("\n%s", outline);

  } /* end of print_soldier_totals() */

/*************************************************************************
print_character_totals:
  print the slot leader's combat numbers.
*************************************************************************/
static void print_character_totals(slot, outline)
short slot;
char *outline;
  {

    /* print slot leader totals */
    strcpy(outline, columns);
    overlaytext("Totals for character:", outline);
    overlaycf(Legion.Slot[slot].Character[CF], outline);
    overlaydf(Legion.Slot[slot].Character[DF], outline);
    overlayaf(Legion.Slot[slot].Character[AF], outline);
    overlaymaf(Legion.Slot[slot].Character[MAF], outline);
    overlaycaf(Legion.Slot[slot].Character[CAF], outline);
    overlaymav(Legion.Slot[slot].Character[MAV], outline);
    overlaysav(Legion.Slot[slot].Character[SAV], outline);
    overlaymad(Legion.Slot[slot].Character[MAD], outline);
    overlaysad(Legion.Slot[slot].Character[SAD], outline);
    overlaydam(Legion.Slot[slot].Character[DAM], outline);
    overlayinv(Legion.Slot[slot].Character[INV], outline);
    printf("\n%s", outline);

  } /* end of print_character_totals() */

/*************************************************************************
print_slot_totals:
  print the grand totals for slot
*************************************************************************/
static void print_slot_totals(slot, outline)
short slot;
char *outline;
  {

    /* print grand totals for slot */
    strcpy(outline, columns);
    overlaytext("Totals for slot:", outline);
    overlaycf(Legion.Slot[slot].Total[CF] +
              Legion.Slot[slot].Character[CF],
              outline);
    overlayaf(Legion.Slot[slot].Total[AF] +
              Legion.Slot[slot].Character[AF],
              outline);
    overlaymaf(Legion.Slot[slot].Total[MAF] +
               Legion.Slot[slot].Character[MAF],
               outline);
    overlaycaf(Legion.Slot[slot].Total[CAF] +
               Legion.Slot[slot].Character[CAF],
               outline);
    overlaymav(Legion.Slot[slot].Total[MAV] +
               Legion.Slot[slot].Character[MAV],
               outline);
    overlaysav(Legion.Slot[slot].Total[SAV] +
               Legion.Slot[slot].Character[SAV],
               outline);
    printf("\n%s", outline);

  } /* end of print_slot_totals() */

/*************************************************************************
insert_legion_notes()
  puts miscellaneous notes about the legion into the notes array.
*************************************************************************/
static void insert_legion_notes()
  {

    char note[NOTESIZE + 1];

    if(Legion.Slot[MAXCSLOTS].Character[OVER] + Legion.Forts[F_OVER] > 300)
      { /* Tactics Overall Modifier exceeds 300% */

        sprintf(
          note,
          "The total leadership overall modifier should not exceed 300%."
               );
        insert_note(note);

      } /* Tactics Overall Modifier exceeds 300% */

  } /* end of insert_legion_notes() */

/*************************************************************************
insert_slot_notes()
    puts miscellaneous notes about the slot into the notes array.
*************************************************************************/
static void insert_slot_notes(slot)
short slot;
  {

    char note[NOTESIZE + 1];

    if(Legion.Slot[slot].Total[SAFB] != 0)
      { /* surprise missile round MAFB */

        if(Legion.Slot[slot].Item[WEAPON][CLASS] & M_MISSILE)
          { /* and soldiers have missile weapon */

            sprintf(
              note,
              "Soldiers of slot %d receive %+d%% MAFB in Surprise Missile Round.",
              slot,
              Legion.Slot[slot].Total[SAFB]
              );
            insert_note(note);

          } /* and soldiers have missile weapon */

      } /* surprise missile round MAFB */

    if(Legion.Slot[slot].Item[WEAPON][ID] != 0)
      { /* soldiers have a weapon */

        if(Legion.Slot[slot].Item[SHIELD][ID] != 0)
          { /* soldiers have a shield */

            if(Legion.Slot[slot].Item[WEAPON][CLASS] & M_2HANDED)
              { /* 2-handed weapon */

                sprintf(
                  note,
                  "Slot %d weapon ID#%d cannot be used with a shield.",
                  slot,
                  Legion.Slot[slot].Item[WEAPON][ID]
                  );
                insert_note(note);

              } /* 2-handed weapon */

          } /* soldiers have a shield */

        if(Legion.Slot[slot].Item[MOUNT][ID] != 0)
          { /* soldiers have a warmount */

            if(!(Legion.Slot[slot].Item[WEAPON][CLASS] &
                (M_MOUNT | M_MOUNT_ONLY)))
              { /* weapon not allowed with warmount */

                sprintf(
                  note,
                  "Slot %d weapon ID#%d cannot be used with a warmount.",
                  slot,
                  Legion.Slot[slot].Item[WEAPON][ID]
                  );
                insert_note(note);

              } /* weapon not allowed with warmount */

          } /* soldiers have a warmount */

        else
          { /* soldiers have no warmount */

            if(Legion.Slot[slot].Item[WEAPON][CLASS] & M_MOUNT_ONLY)
              { /* weapon requires warmount */

                sprintf(
                  note,
                  "Slot %d weapon ID#%d must be used with a warmount.",
                  slot,
                  Legion.Slot[slot].Item[WEAPON][ID]
                  );
                insert_note(note);

              } /* weapon requires warmount */

          } /* soldiers have no warmount */

      } /* soldiers have a weapon */

  } /* end of insert_slot_notes() */

/*************************************************************************
print_grand_totals:
    prints the grand totals of all slots.
*************************************************************************/
static void print_grand_totals(outline)
char *outline;
  {

    long cf = 0, af = 0, maf = 0, caf = 0, mav = 0, sav = 0;
    short slot;

    for(slot = 1; slot <= MAXSSLOTS; slot++)
      { /* all slots */

        /* sum character data */
        cf += Legion.Slot[slot].Character[CF];
        af += Legion.Slot[slot].Character[AF];
        maf += Legion.Slot[slot].Character[MAF];
        caf += Legion.Slot[slot].Character[CAF];
        mav += Legion.Slot[slot].Character[MAV];
        sav += Legion.Slot[slot].Character[SAV];

      } /* all slots */

    /* print the grand totals */
    strcpy(outline, columns);
    overlaytext("Grand totals for force:", outline);
    overlaycf(Legion.Slot[0].Total[CF] + cf, outline);
    overlayaf(Legion.Slot[0].Total[AF] + af, outline);
    overlaymaf(Legion.Slot[0].Total[MAF] + maf, outline);
    overlaycaf(Legion.Slot[0].Total[CAF] + caf, outline);
    overlaymav(Legion.Slot[0].Total[MAV] + mav, outline);
    overlaysav(Legion.Slot[0].Total[SAV] + sav, outline);
    /* @@ include overall force morale: 100+overall leader tactics */
    printf("\n\n%s", outline);

  } /* end of print_grand_totals() */

/*************************************************************************
insert_note:
*************************************************************************/
static void insert_note(string)
char *string;
  {

    if(notecount < (MAXNOTES - 1))
      { /* room for more notes */

        strcpy(Notes[notecount], string);
        notecount = notecount + 1;

      } /* room for more notes */

    else
      { /* no room for more notes */

        if(notecount == (MAXNOTES - 1))
          { /* write final note */

            strcpy(Notes[notecount],
            "Warning: no more space for notes!  Some may be missing.");
            notecount = notecount + 1;

          } /* write final note */

      } /* no room for more notes */

  } /* end of insert_note() */

/*************************************************************************
overlayslot:
*************************************************************************/
static void overlayslot(slot, string)
short slot;
char *string;
  {

    char s[LONGWIDTH];

    sprintf(s, "%*d", SLTWIDTH, slot);
    strncpy(&string[SLTPOS], s, SLTWIDTH);

  } /* end of overlayslot() */

/*************************************************************************
overlaycode:
*************************************************************************/
static void overlaycode(code, slot, string)
char code;
short slot;
char *string;
  {

    char s[LONGWIDTH + 1];

    sprintf(s, "%c%*d", code, SLTWIDTH - 1, slot);
    strncpy(&string[SLTPOS], s, SLTWIDTH);

  } /* end of overlaycode() */

/*************************************************************************
overlayamt:
*************************************************************************/
static void overlayamt(amt, string)
long amt;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", AMTWIDTH, amt) > AMTWIDTH)
      { /* overflow */

        strncpy(&string[AMTPOS], overflow, AMTWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[AMTPOS], s, AMTWIDTH);

      } /* number fits */

  } /* end of overlayamt() */

/*************************************************************************
overlaydesc:
*************************************************************************/
static void overlaydesc(desc, string)
char *desc;
char *string;
  {

    char s[DESCWIDTH + 1];

    sprintf(s, "%-*.*s", DESCWIDTH, DESCWIDTH, desc);
    strncpy(&string[DESCPOS], s, DESCWIDTH);

  } /* end of overlaydesc() */

/*************************************************************************
overlaytext:
*************************************************************************/
static void overlaytext(text, string)
char *text;
char *string;
  {

    char s[TEXTWIDTH + 1];

    sprintf(s, "%-*.*s", TEXTWIDTH, TEXTWIDTH, text);
    strncpy(&string[SLTPOS], s, TEXTWIDTH);

  } /* end of overlaydesc() */

/*************************************************************************
overlaycf:
*************************************************************************/
static void overlaycf(cf, string)
long cf;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", CFWIDTH, cf) > CFWIDTH)
      { /* overflow */

        strncpy(&string[CFPOS], overflow, CFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[CFPOS], s, CFWIDTH);

      } /* number fits */

  } /* end of overlaycf() */

/*************************************************************************
overlaydf:
*************************************************************************/
static void overlaydf(df, string)
long df;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", DFWIDTH, df) > DFWIDTH)
      { /* overflow */

        strncpy(&string[DFPOS], overflow, DFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[DFPOS], s, DFWIDTH);

      } /* number fits */

  } /* end of overlaydf() */

/*************************************************************************
overlayaf:
*************************************************************************/
static void overlayaf(af, string)
long af;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", AFWIDTH, af) > AFWIDTH)
      { /* overflow */

        strncpy(&string[AFPOS], overflow, AFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[AFPOS], s, AFWIDTH);

      } /* number fits */

  } /* end of overlayaf() */

/*************************************************************************
overlayafb:
*************************************************************************/
static void overlayafb(afb, string)
long afb;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld%%", AFWIDTH - 1, afb) > AFWIDTH)
      { /* overflow */

        strncpy(&string[AFPOS], overflow, AFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[AFPOS], s, AFWIDTH);

      } /* number fits */

  } /* end of overlayafb() */

/*************************************************************************
overlaymaf:
*************************************************************************/
static void overlaymaf(maf, string)
long maf;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", MAFWIDTH, maf) > MAFWIDTH)
      { /* overflow */

        strncpy(&string[MAFPOS], overflow, MAFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[MAFPOS], s, MAFWIDTH);

      } /* number fits */

  } /* end of overlaymaf() */

/*************************************************************************
overlaymafb:
*************************************************************************/
static void overlaymafb(mafb, string)
long mafb;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld%%", MAFWIDTH - 1, mafb) > MAFWIDTH)
      { /* overflow */

        strncpy(&string[MAFPOS], overflow, MAFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[MAFPOS], s, MAFWIDTH);

      } /* number fits */

  } /* end of overlaymafb() */

/*************************************************************************
overlaycaf:
*************************************************************************/
static void overlaycaf(caf, string)
long caf;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", CAFWIDTH, caf) > CAFWIDTH)
      { /* overflow */

        strncpy(&string[CAFPOS], overflow, CAFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[CAFPOS], s, CAFWIDTH);

      } /* number fits */

  } /* end of overlaycaf() */

/*************************************************************************
overlaycafb:
*************************************************************************/
static void overlaycafb(cafb, string)
long cafb;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld%%", CAFWIDTH - 1, cafb) > CAFWIDTH)
      { /* overflow */

        strncpy(&string[CAFPOS], overflow, CAFWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[CAFPOS], s, CAFWIDTH);

      } /* number fits */

  } /* end of overlaycafb() */

/*************************************************************************
overlaymav:
*************************************************************************/
static void overlaymav(mav, string)
long mav;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", MAVWIDTH, mav) > MAVWIDTH)
      { /* overflow */

        strncpy(&string[MAVPOS], overflow, MAVWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[MAVPOS], s, MAVWIDTH);

      } /* number fits */

  } /* end of overlaymav() */

/*************************************************************************
overlaysav:
*************************************************************************/
static void overlaysav(sav, string)
long sav;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", SAVWIDTH, sav) > SAVWIDTH)
      { /* overflow */

        strncpy(&string[SAVPOS], overflow, SAVWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[SAVPOS], s, SAVWIDTH);

      } /* number fits */

  } /* end of overlaysav() */

/*************************************************************************
overlaymad:
*************************************************************************/
static void overlaymad(mad, string)
long mad;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", MADWIDTH, mad) > MADWIDTH)
      { /* overflow */

        strncpy(&string[MADPOS], overflow, MADWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[MADPOS], s, MADWIDTH);

      } /* number fits */

  } /* end of overlaymad() */

/*************************************************************************
overlaysad:
*************************************************************************/
static void overlaysad(sad, string)
long sad;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", SADWIDTH, sad) > SADWIDTH)
      { /* overflow */

        strncpy(&string[SADPOS], overflow, SADWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[SADPOS], s, SADWIDTH);

      } /* number fits */

  } /* end of overlaysad() */

/*************************************************************************
overlaydam:
*************************************************************************/
static void overlaydam(dam, string)
long dam;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", DAMWIDTH, dam) > DAMWIDTH)
      { /* overflow */

        strncpy(&string[DAMPOS], overflow, DAMWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[DAMPOS], s, DAMWIDTH);

      } /* number fits */

  } /* end of overlaydam() */

/*************************************************************************
overlayinv:
*************************************************************************/
static void overlayinv(inv, string)
long inv;
char *string;
  {

    char s[LONGWIDTH];

    if(sprintf(s, "%*ld", INVWIDTH, inv) > INVWIDTH)
      { /* overflow */

        strncpy(&string[INVPOS], overflow, INVWIDTH);

      } /* overflow */
    else
      { /* number fits */

        strncpy(&string[INVPOS], s, INVWIDTH);

      } /* number fits */

  } /* end of overlayinv() */

/*************************************************************************
main program
*************************************************************************/
void main(argc, argv)
int argc;
char *argv[];
  {

    char *Program = "Legends Soldier Calculator";
    char *Version = "4.0.0";
    char *Date = "09-Mar-94";
    char *Author = "Mark R. Nuiver";

    printf("\n%s version %s of %s.", Program, Version, Date);
    printf("\nCopyright (C) 1994 by %s.  All rights reserved.", Author);
    if(argc == 1) help();
    if(strcmp(argv[1], "?") == 0) help();
    /* initialize table data */
    init_tables();
    /* load the legion data */
    load_legion(argv[1], &Legion);
    /* validate the legion inputs */
    validate_legion(&Legion);
    /* load the predefined tables */
    load_tables(&Legion);
    /* compute the legion */
    compute_legion(&Legion);
    /* print results */
    print_results(argv[1]);
    cleanup(ERROR_NO_ERROR, NULL, NULL);
  } /* end of main() */
