/**************************************************************
 * CREATE.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Game creation process.                     *
 *------------------------------------------------------------*
 * Author  Damian G Walker                                    *
 * Date    18-Feb-97                                          *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <string.h>
#include <ctype.h>
#include "astrowar.h"
#include "detail.h"
#include "empire.h"
#include "planet.h"
#include "fleets.h"
#include "scrlog.h"
#include "tmpmap.h"
#include "config.h"
#include "gamlst.h"


/* global data ************************************************/


static char letter[2][24][3] = {
    {
        "aa", "ae", "ai", "au", "ay", "ar",
        "ea", "ee", "ei", "eo", "eu", "er",
        "ia", "ie", "io", "iu", "ir", "ue",
        "oa", "oe", "oi", "oo", "ou", "ur"
    },{
        "br", "cl", "dr", "fl", "gr", "j",
        "kr", "ll", "mn", "ng", "ph", "qu",
        "rr", "sh", "th", "vl", "wr", "xc",
        "zz", "ch", "ph", "sh", "th", "zh"
    }
};


/* level 3 routines *******************************************/


/* create_name() - create a planet name */
void create_name(char *name)
{
    int namelen,   /* length of name */
        vowel,     /* vowel (!=0) or consonant */
        letterno,  /* letter number */
        letterlen, /* letter length */
        pos;       /* position in string */

    /* initialise */
    memset(name, 0, 16);
    namelen = ( random() % (( random() % 13 ) + 1) ) + 3;
    vowel = random() % 2;
    pos = 0;

    /* generate word */
    while(pos < namelen)
    {
        letterno = random() % 24;
        letterlen =
            random() % strlen( letter[vowel][letterno] );
        name[pos++] = letter[vowel][letterno][0];
        if( letterlen && (pos < namelen) )
            name[pos++] = letter[vowel][letterno][1];
        vowel = 1 - vowel;
    }
    name[0] = toupper( name[0] );
}

/* create_position() - create a planet position */
pos create_position(detail *d)
{
    pos  pp;     /* planet position */
    long randno; /* random number */

    randno = 2 * (random() & 32767) + (random() % 2);
    pp.x = randno % ( detail_getxmax(d) - detail_getxmin(d) )
        + detail_getxmin(d);
    randno = 2 * (random() & 32767) + (random() % 2);
    pp.y = randno % ( detail_getymax(d) - detail_getymin(d) )
        + detail_getymin(d);

    return pp;
}

/* create_production() - create planetary production */
int create_production(void)
{
    return random() % 16;
}

/* create_ships() - create ships in orbit */
long create_ships(int prod)
{
    return ( random() % 15 ) + ( random() % (prod + 1) ) + 1;
}


/* level 2 routines *******************************************/


/* create_planet() - create a single planet */
void create_planet(detail *d)
{
    planet *p = planet_new(); /* planet record */
    char    name[16];         /* planet name */
    pos     pp;               /* planet position */
    int     prod;             /* planet production */
    long    ships;            /* ships in orbit */

    /* generate details */
    do
        create_name(name);
    while( planet_find(p, name) == R_OK );
    do
        pp = create_position(d);
    while( tmpmap_find(pp) == R_OK );
    tmpmap_write(pp);
    prod = create_production();
    ships = create_ships(prod);

    /* write details */
    planet_setname(p, name);
    planet_setpos(p, pp);
    planet_setprod(p, prod);
    planet_setships(p, ships);
    planet_write(p);

    planet_old(p);
}


/* level 1 routines *******************************************/


/* create_interpret() - interpret command line */
result create_interpret(char *gamename, char **argp)
{
    int argc; /* argument counter */

    argc = 0; strcpy(gamename, "");
    while(argc < 10 && argp[argc] != NULL)
    {
        if(!strcmp( argp[argc], "-game" ) && argc < 9)
        {
            ++argc;
            if(strlen( argp[argc] ) > 15)
            {
                strncpy( gamename, argp[argc], 15 );
                gamename[15] = '\0';
            }
            else strcpy( gamename, argp[argc] );
        }
        else
        {
            scrlog_error();
            return R_BADCMD;
        }
        ++argc;
    }
    return R_OK;
}

/* create_open() - open creation process */
result create_open(char *gamepath, detail **d, char *path, char
    *gamename)
{
    result  r;             /* result returned to calling process */
    int     line;          /* file line number for log */
    char    logfile[128];  /* name of log file */

    /* attempt to open config */
    config_open(path);
    switch( r = config_read(path, &line) )
    {
        case R_FILE:   scrlog_cannotreadconfig(path);
                       return R_FILE;
        case R_BADCMD: scrlog_badcommandinconfig(path, line);
                       return R_BADCMD;
        case R_OK:     config_getlogfile(logfile);
                       if( ( r = scrlog_openlog(logfile, 0xffff) )
                           != R_OK)
                       {
                           scrlog_cannotopenlog(logfile);
                           return R_FILE;
                       }
                       scrlog_create();
                       scrlog_configread(path);
                       break;
        default:       scrlog_badreturncode(r, "config_read");
                       return r;
    }

    /* attempt to open game list */
    switch( r = gamlst_open(path, &line) )
    {
        case R_FILE:   scrlog_cannotopengamelist(path);
                       goto closeconfig;
        case R_EOF:    scrlog_badfileingamelist(path, line);
                       goto closeconfig;
        case R_BADCMD: scrlog_badlineingamelist(path, line);
                       goto closeconfig;
        case R_OK:     scrlog_gamelistread(path);
                       break;
        default:       scrlog_badreturncode(r, "gamlst_open");
                       goto closeconfig;
    }

    /* now open game detail file */
    if(( r = gamlst_find(gamename) ) != R_OK)
    {
        scrlog_gamenotfound(gamename);
        goto closegamelist;
    }
    gamlst_getpath(gamepath);
    switch( r = detail_open(d, gamepath) )
    {
        case R_FILE:  scrlog_cannotopendetail(gamepath);
                       goto closegamelist;
        case R_MEMORY: scrlog_outofmemory("loading detail file");
                       goto closegamelist;
        case R_BADCMD: scrlog_badcommandindetail(gamepath);
                       goto closegamelist;
        case R_OK:     scrlog_detailread(gamepath);
                       break;
        default:       scrlog_badreturncode(r, "detail_open");
                       goto closegamelist;
    }

    /* return OK code */
    return R_OK;

    /* clean up after fatal error */
    closegamelist:
        gamlst_close();
    closeconfig:
        config_close();
    return r;
}

/* create_empire() - create the empire file */
result create_empire(char *path)
{
    result r; /* returned to calling process */

    switch( r = empire_create(path) )
    {
        case R_EXISTS: scrlog_fileexists(path, "empire.data");
                       break;
        case R_FILE:   scrlog_cannotcreatefile(path, "empire.data");
                       break;
        default:       scrlog_filecreated(path, "empire.data");
    }

    return r;
}


/* create_fleets() - create the fleets file */
result create_fleets(char *path)
{
    result r; /* returned to calling process */

    switch( r = fleets_create(path) )
    {
        case R_EXISTS: scrlog_fileexists(path, "fleets.data");
                       break;
        case R_FILE:   scrlog_cannotcreatefile(path, "fleets.data");
                       break;
        default:       scrlog_filecreated(path, "fleets.data");
    }

    return r;
}


/* create_planets() - create the planets file */
result create_planets(detail *d, char *path)
{
    result r;       /* returned to calling process */
    long   planets, /* planets to create */
           curr;    /* current planet number */

    /* initialise planet creation */
    srandom( time(0) );
    if(( r = tmpmap_open(path) ) != R_OK)
    {
        scrlog_cannotcreatefile(path, "tmpmap");
        return r;
    }
    switch( r = planet_create(path) )
    {
        case R_EXISTS: scrlog_fileexists(path, "planet.data");
                       return r;
        case R_FILE:   scrlog_cannotcreatefile(path, "planet.data");
                       return r;
        case R_OK:     if(( r = planet_open(path) ) == R_OK ) break;
                       scrlog_cannotopenfile(path, "planet.data");
        default:       return r;
    }
    scrlog_filecreated(path, "planet.data");

    /* create planets */
    planets = detail_getplanets(d);
    for(curr = 1; curr <= planets; curr++)
    {
        scrlog_creatingplanet(curr, planets);
        create_planet(d);
    }
    scrlog_planetsdone(planets);

    /* close everything */
    planet_close();
    tmpmap_close();
    return R_OK;
}


/* level 0 routines *******************************************/


/* create_main() - main game creation function */
result create_main(char *path, char **argp)
{
    detail *d;            /* game detail record */
    result  r;            /* returned to calling process */
    char    gamename[16], /* name of game */
            gamepath[72]; /* path to game */

    if(( r = create_interpret(gamename, argp) ) == R_OK )
    {
        if(( r = create_open(gamepath, &d, path, gamename) ) != R_OK)
            return r;
        if( !create_empire(gamepath) )
            if( !create_fleets(gamepath) )
                create_planets(d, gamepath);
        scrlog_finished();
        detail_close(d);
    }

    return r;
}
