/**************************************************************
 * FORECS.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * ASH 1.00 - Forecast file library.                          *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     24-Jul-97                                         *
 **************************************************************/


#define _FORECS_C_


/* included headers *******************************************/


#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "ash.h"
#include "forecs.h"


/* some macros ************************************************/


/* record sizes */
#define DATASIZE 37  /* size of packed data record */
#define IND1SIZE 20  /* size of packed fleet number index */

/* flags */
#define ZF_DELETED 0x01 /* deleted flag */


/* structures *************************************************/


/* hidden forecs structure */
typedef struct {

    /* record fields */
    char  deleted,   /* 'deleted' flag */
          name[16],  /* planet name */
          owner[16]; /* forecs's owner */
    long  ships;     /* ships in orbit */

    /* index maintenance fields */
    char  infile;    /* 0=new record, !0=somewhere in file */
    long  pos;       /* record position when last read */
    char  oname[16]; /* original forecs name */

} forecs;

/* indexes */
typedef struct {   /* forecs number index */
    long pos;          /* position of record in data file */
    char name[16];     /* forecs name */
} ind1;


/* global variables *******************************************/


FILE *forecs_data, /* forecs data file */
     *forecs_ind1; /* forecs number index file */


/* level 3 routines *******************************************/


/* forecs_readind1() - read a number index record */
result forecs_readind1(ind1 *i)
{
    long pos; /* first byte of position in file */

    if(( pos = fgetc(forecs_ind1) ) == EOF)
        return R_EOF;
    i->pos = pos;
    FREADLONGR(i->pos, forecs_ind1);
    fread(i->name, 16, 1, forecs_ind1);
    return R_OK;
}


/* level 2 routines *******************************************/


/* forecs_findind1() - find a trans index record */
long forecs_findind1(ind1 *i, char *name)
{
    long pos,    /* original position in index file */
         start,  /* lower bound of binary search area */
         middle, /* midpoint of binary search area */
         end;    /* upper bound of binary search area */

    /* store current file pointer in case of no find */
    pos = ftell(forecs_ind1); fseek(forecs_ind1, 0, SEEK_END);
    if( ftell(forecs_ind1) == 8 ) return 0;

    /* initialise start/end/middle of binary search */
    end = ( ftell(forecs_ind1) - 8 ) / IND1SIZE - 1;
    start = 0; middle = start + (end - start) / 2;

    /* proceed with binary search */
    fseek(forecs_ind1, 8 + IND1SIZE * middle, SEEK_SET);
    forecs_readind1(i);
    while( stricmp(i->name, name) && start <= end )
    {
        if( stricmp(i->name, name) < 0 )
            start = middle + 1;
        else
            end = middle - 1;
        middle = start + (end - start) / 2;
        fseek(forecs_ind1, 8 + IND1SIZE * middle, SEEK_SET);
        forecs_readind1(i);
    }

    /* return value */
    if( !stricmp(i->name, name) )
        return 8 + middle * IND1SIZE;
    fseek(forecs_ind1, pos, SEEK_SET);
    return 0;
}

/* forecs_writeind1() - write a number index record */
result forecs_writeind1(ind1 i)
{
    if( fputc(i.pos & 0xff, forecs_ind1) == EOF ) return R_FILE;
    FWRITELONGR(i.pos,  forecs_ind1);
    fwrite(i.name, 16, 1, forecs_ind1);
    return R_OK;
}


/* level 1 routines *******************************************/


/* forecs_readdata() - read a data record */
result forecs_readdata(forecs *z)
{
    int  deleted; /* deleted flag */
    long pos;     /* position in file */

    /* check for EOF */
    pos = ftell(forecs_data);
    if(( deleted = fgetc(forecs_data) ) == EOF)
        return R_EOF;

    /* read rest of record */
    z->deleted = deleted;
    fread(z->name,  16, 1, forecs_data);
    fread(z->owner, 16, 1, forecs_data);
    FREADLONG(z->ships,    forecs_data);

    /* set index maintenance fields and return */
    z->infile = 1;
    z->pos = pos;
    strcpy(z->oname, z->name);
    return R_OK;
}

/* forecs_writedata() - write a data record */
result forecs_writedata(forecs *z)
{
    long pos; /* position in file */

    /* check that record can be written */
    pos = ftell(forecs_data);
    if( fputc(z->deleted, forecs_data) == EOF )
        return R_FILE;

    /* write rest of record */
    fwrite(z->name,  16, 1, forecs_data);
    fwrite(z->owner, 16, 1, forecs_data);
    FWRITELONG(z->ships,    forecs_data);

    /* set index maintenance fields and return */
    z->infile = 1;
    z->pos = pos;
    strcpy(z->oname, z->name);
    return R_OK;
}

/* forecs_sortind1() - single bi-directional pass of sort */
int forecs_sortind1(void)
{
    long   pos;      /* stored position in index */
    char   name[16]; /* stored forecs name */
    ind1   i1,       /* first index record to compare */
           i2;       /* second index record to compare */
    result r;        /* returned to calling process */

    /* obtain information about current position */
    pos = ftell(forecs_ind1);
    if(pos > 8)
    {
        fseek(forecs_ind1, -IND1SIZE, SEEK_CUR);
        forecs_readind1(&i1);
        strcpy(name, i1.name);
    }
    else strcpy(name, "");

    /* ensure file is big enough to need sorting */
    fseek(forecs_ind1, 0, SEEK_END);
    if( ftell(forecs_ind1) <= 8 + IND1SIZE )
    {
        fseek(forecs_ind1, pos, SEEK_SET);
        return R_OK;
    }
    r = R_OK;

    /* forward pass of sort */
    fseek(forecs_ind1, 8, SEEK_SET); forecs_readind1(&i1);
    while( forecs_readind1(&i2) == R_OK )
    {
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(forecs_ind1, -2 * IND1SIZE, SEEK_CUR);
            forecs_writeind1(i2); forecs_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(forecs_ind1, -IND1SIZE, SEEK_CUR);
        forecs_readind1(&i1);
    }

    /* backward pass of sort */
    fseek(forecs_ind1, -IND1SIZE, SEEK_END);
    while( ftell(forecs_ind1) > 8 )
    {
        fseek(forecs_ind1, -IND1SIZE, SEEK_CUR);
        forecs_readind1(&i1); forecs_readind1(&i2);
        if( stricmp(i1.name, i2.name) > 0 )
        {
            fseek(forecs_ind1, -2 * IND1SIZE, SEEK_CUR);
            forecs_writeind1(i2); forecs_writeind1(i1);
            r = R_SWAPS;
        }
        fseek(forecs_ind1, -2 * IND1SIZE, SEEK_CUR);
    }

    /* clean up */
    if(pos > 8)
        forecs_findind1(&i1, name);
    else
        fseek(forecs_ind1, pos, SEEK_SET);
    return r;
}


/* level 0 routines *******************************************/


/* forecs_open() - open forecs files */
result forecs_open(char *path)
{
    char dataname[128], /* name of data file */
         ind1name[128], /* name of forecs number index file */
         dataheader[8], /* data file header */
         ind1header[8]; /* index 1 header */

    /* initialise data names */
    sprintf(dataname, "%sforecs.data", path);
    sprintf(ind1name, "%sforecs.index", path);

    /* attempt to open files */
    if(( forecs_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    if(( forecs_ind1 = fopen(ind1name, "r+b") ) == NULL)
    {
        fclose(forecs_data);
        return R_FILE;
    }

    /* verify files */
    fread(dataheader, 8, 1, forecs_data);
    fread(ind1header, 8, 1, forecs_ind1);
    if(( strncmp(dataheader, "ASH100F", 8) ) ||
       ( strncmp(ind1header, "ASH100f", 8) ))
    {
        fclose(forecs_data);
        fclose(forecs_ind1);
        return R_HEADER;
    }

    /* files checked out OK */
    return R_OK;
}

/* forecs_create() - create forecs files */
result forecs_create(char *path)
{
    char  dataname[128], /* name of data file */
          ind1name[128]; /* name of forecs number index file */
    FILE *test;          /* used to test existence of files */

    /* initialise data names */
    sprintf(dataname, "%sforecs.data", path);
    sprintf(ind1name, "%sforecs.index", path);

    /* check for files' existence */
    if(( test = fopen(dataname, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }
    if(( test = fopen(ind1name, "rb") ) != NULL)
    {
        fclose(test);
        return R_EXISTS;
    }

    /* attempt to create files */
    if(( forecs_data = fopen(dataname, "wb") ) == NULL)
        return R_FILE;
    if(( forecs_ind1 = fopen(ind1name, "wb") ) == NULL)
    {
        fclose(forecs_data); remove(dataname);
        return R_FILE;
    }

    /* write headers */
    fwrite("ASH100F", 8, 1, forecs_data);
    fwrite("ASH100f", 8, 1, forecs_ind1);

    /* close files and return */
    fclose(forecs_data);
    fclose(forecs_ind1);
    return R_OK;
}

/* forecs_close() - close forecs files */
result forecs_close(void)
{
    fclose(forecs_data);
    fclose(forecs_ind1);
    return R_OK;
}

/* forecs_new() - allocate memory for new forecs record */
forecs *forecs_new(void)
{
    return calloc( 1, sizeof(forecs) );
}

/* forecs_old() - free memory for new forecs record */
void forecs_old(forecs *z)
{
    free(z);
}

/* forecs_clear() - clear details from existing forecs record */
void forecs_clear(forecs *z)
{
    memset( z, 0, sizeof(forecs) );
}

/* forecs_setname() - set the forecs name */
char *forecs_setname(forecs *z, char *name)
{
    return strcpy(z->name, name);
}

/* forecs_getname() - return the forecs name */
char *forecs_getname(char *name, forecs *z)
{
    return strcpy(name, z->name);
}

/* forecs_setowner() - set the owner */
char *forecs_setowner(forecs *z, char *owner)
{
    return strcpy(z->owner, owner);
}

/* forecs_getowner() - return the owner */
char *forecs_getowner(char *owner, forecs *z)
{
    return strcpy(owner, z->owner);
}

/* forecs_setships() - set the ships in orbit */
long forecs_setships(forecs *z, long ships)
{
    return z->ships = ships;
}

/* forecs_getships() - return the ships in orbit */
long forecs_getships(forecs *z)
{
    return z->ships;
}

/* forecs_write() - write a record */
result forecs_write(forecs *z)
{
    ind1   i;  /* trans index record */
    long   p1; /* position of index record 1 */

    /* write new record */
    if(!z->infile)
    {
        if( forecs_findind1(&i, z->name) ) return R_DUPE;
        fseek(forecs_data, 0, SEEK_END);
        fseek(forecs_ind1, 0, SEEK_END);
        i.pos = ftell(forecs_data);
        strcpy(i.name, z->name);
        if( forecs_writedata(z) != R_OK )  return R_FILE;
        if( forecs_writeind1(i) != R_OK )  return R_CORRUPT;
        forecs_sortind1();
        return R_OK;
    }

    /* rewrite existing record */
    p1 = forecs_findind1(&i, z->oname);
    if(!p1)                                return R_CORRUPT;
    fseek(forecs_data, i.pos, SEEK_SET);
    if( forecs_writedata(z) != R_OK )      return R_FILE;
    if( stricmp(i.name, z->name) )
    {
        strcpy(i.name, z->name);
        fseek(forecs_ind1, p1, SEEK_SET);
        if( forecs_writeind1(i) != R_OK )  return R_CORRUPT;
        forecs_sortind1();
    }
    return R_OK;
}

/* forecs_first() - read the first record */
result forecs_first(forecs *z, zindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case Z_NONE:
            fseek(forecs_data, 8, SEEK_SET);
            return forecs_readdata(z);
        case Z_NAME:
            fseek(forecs_ind1, 8, SEEK_SET);
            if(( r = forecs_readind1(&i) ) != R_OK) return r;
            fseek(forecs_data, i.pos, SEEK_SET);
            return forecs_readdata(z);
        default:
            return R_INDEX;
    }
}

/* forecs_next() - read the next record */
result forecs_next(forecs *z, zindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case Z_NONE:
            return forecs_readdata(z);
        case Z_NAME:
            if(( r = forecs_readind1(&i) ) != R_OK) return r;
            fseek(forecs_data, i.pos, SEEK_SET);
            return forecs_readdata(z);
        default:
            return R_INDEX;
    }
}

/* forecs_prev() - read the previous record */
result forecs_prev(forecs *z, zindex inum)
{
    ind1   i; /* number index record */
    result r;  /* result of index read */

    switch(inum)
    {
        case Z_NONE:
            if(( ftell(forecs_data) - 2 * DATASIZE ) < 8)
                return R_EOF;
            fseek(forecs_data, -2 * DATASIZE, SEEK_CUR);
            return forecs_readdata(z);
        case Z_NAME:
            if(( ftell(forecs_ind1) - 2 * IND1SIZE ) < 8)
                return R_EOF;
            fseek(forecs_ind1, -2 * IND1SIZE, SEEK_CUR);
            if(( r = forecs_readind1(&i) ) != R_OK) return r;
            fseek(forecs_data, i.pos, SEEK_SET);
            return forecs_readdata(z);
        default:
            return R_INDEX;
    }
}

/* forecs_last() - read the last record */
result forecs_last(forecs *z, zindex inum)
{
    ind1   i;   /* number index record */
    result r;   /* result of index read */
    long   pos; /* current data/index file position */

    switch(inum)
    {
        case Z_NONE:
            pos = ftell(forecs_data);
            if( fseek(forecs_data, -DATASIZE, SEEK_END) )
                return R_EOF;
            if( ftell(forecs_data) < 8 )
            {
                fseek(forecs_data, pos, SEEK_SET);
                return R_EOF;
            }
            return forecs_readdata(z);
        case Z_NAME:
            pos = ftell(forecs_ind1);
            if( fseek(forecs_ind1, -IND1SIZE, SEEK_END) )
                return R_EOF;
            if( ftell(forecs_ind1) < 8 )
            {
                fseek(forecs_ind1, pos, SEEK_SET);
                return R_EOF;
            }
            if(( r = forecs_readind1(&i) ) != R_OK) return r;
            fseek(forecs_data, i.pos, SEEK_SET);
            return forecs_readdata(z);
        default:
            return R_INDEX;
    }
}

/* forecs_find() - find a record by name or name */
result forecs_find(forecs *z, char *name)
{
    ind1   i;      /* forecs name index record */

    if( forecs_findind1(&i, name) == 0 ) return R_EOF;
    fseek(forecs_data, i.pos, SEEK_SET);
    return forecs_readdata(z);

    return R_OK;
}

/* forecs_delete() - mark a record as deleted */
result forecs_delete(forecs *z)
{
    if(z->pos == 0) return R_EOF;
    z->deleted ^= ZF_DELETED;
    fseek(forecs_data, z->pos, SEEK_SET);
    return forecs_writedata(z);
}

/* forecs_deleted() - return the deleted status of a record */
int forecs_deleted(forecs *z)
{
    if(z->pos == 0) return R_EOF;
    return z->deleted & ZF_DELETED;
}

/* forecs_pack() - pack a close data file */
result forecs_pack(char *path)
{
    forecs *z;             /* forecs record */
    char    dataname[128], /* name of data file */
            ind1name[128], /* name of forecs number index file */
            tempname[128], /* name of temporary file */
            dataheader[8]; /* data file header */
    FILE   *forecs_temp;   /* temporary file */
    ind1    i;             /* number index record */

    /* initialise data names */
    sprintf(dataname, "%sforecs.data", path);
    sprintf(tempname, "%sforecs.temp", path);

    /* attempt to open data & temp file */
    if(( forecs_data = fopen(dataname, "r+b") ) == NULL)
        return R_FILE;
    fread(dataheader, 8, 1, forecs_data);
    if(( strncmp(dataheader, "ASH100F", 8) ) ||
       ( forecs_temp = fopen(tempname, "w+b") ) == NULL)
    {
        fclose(forecs_data);
        return R_HEADER;
    }

    /* allocate memory */
    if( (z = malloc( sizeof(forecs) )) == NULL )
    {
        fclose(forecs_data);
        fclose(forecs_temp);
        remove(tempname);
        return R_MEMORY;
    }

    /* copy non-deleted records to temporary file and back */
    while( forecs_readdata(z) == R_OK )
        if( !(z->deleted & ZF_DELETED) )
            fwrite(z, sizeof(forecs), 1, forecs_temp);
    fclose(forecs_data);
    remove(dataname);
    if(( forecs_data = fopen(dataname, "w+b") ) == NULL)
    {
        free(z);
        fclose(forecs_temp);
        remove(tempname);
        return R_FILE;
    }
    fwrite("ASH100F", 8, 1, forecs_data);
    fseek(forecs_temp, 0, SEEK_SET);
    while( fread(z, sizeof(forecs), 1, forecs_temp) == 1 )
        forecs_writedata(z);
    fclose(forecs_temp);
    remove(tempname);

    /* recreate fleet number index */
    sprintf(ind1name, "%sforecs.index", path);
    remove(ind1name);
    if(( forecs_ind1 = fopen(ind1name, "w+b") ) == NULL)
    {
        free(z);
        fclose(forecs_data);
        return R_FILE;
    }
    fwrite("ASH100f", 8, 1, forecs_ind1);
    fseek(forecs_data, 8, SEEK_SET); i.pos = ftell(forecs_data);
    while( forecs_readdata(z) == R_OK )
    {
        strcpy(i.name, z->name);
        forecs_writeind1(i);
        i.pos = ftell(forecs_data);
    }
    while( forecs_sortind1() == R_SWAPS );
    fclose(forecs_ind1);

    /* clean up */
    free(z);
    fclose(forecs_data);
    return R_OK;
}
