/**************************************************************
 * DETAIL.C                  Copyright (C) Damian Walker 1997 *
 *------------------------------------------------------------*
 * AstroWar 1.00 - Detail file library.                       *
 *------------------------------------------------------------*
 * Author   Damian G Walker                                   *
 * Date     14-Feb-97                                         *
 **************************************************************/


/* included headers *******************************************/


#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include "astrowar.h"


/* structures *************************************************/


typedef struct {
    char name[16]; /* name/tag of game */
    long planets;  /* number of planets in galaxy */
    int  xmin,     /* minimum X co-ordinate on map */
         xmax,     /* maximum X co-ordinate on map */
         ymin,     /* minimum Y co-ordinate on map */
         ymax,     /* maximum Y co-ordinate on map */
         maxdist,  /* maximum distance ships can be sent */
         speed,    /* speed at which ships travel */
         moves,    /* process fleet movement y/n */
         prod,     /* process planet production y/n */
         vict,     /* check for victory y/n */
         orders;   /* process orders y/n */
} detail;


/* constants **************************************************/


#define radius 20


/* level 1 routines *******************************************/


/* detail_procstr() - process instruction w/string argument */
char *detail_procstr(char *outstr)
{
    char *wkstr; /* work string (pointer) */

    if(( wkstr = strtok(NULL, "\n") ) != NULL)
        return strcpy(outstr, wkstr);
    else
        return strcpy(outstr, "");
}

/* detail_proclong() - process instruction w/long argument */
long detail_proclong(void)
{
    char *num; /* pointer to text version of number */

    if(( num = strtok(NULL, "\n") ) != NULL)
        return atol(num);
    else
        return 0;
}

/* detail_procint() - process instruction w/int argument */
int detail_procint(void)
{
    char *num; /* pointer to text version of number */

    if(( num = strtok(NULL, "\n") ) != NULL)
        return atoi(num);
    else
        return 0;
}


/* level 0 routines *******************************************/


/* detail_open() - open and read detail file */
result detail_open(detail **outd, char *path)
{
    char    filename[144], /* name of input file */
            input[81],     /* line input buffer */
            buf[81],       /* line work buffer */
           *keyword;       /* pointer to keyword */
    FILE   *dtlfile;       /* detail file handle */
    detail *d;             /* detail to work on */

    /* attempt to open file */
    sprintf(filename, "%sdetail.data", path);
    if(( dtlfile = fopen(filename, "r") ) == NULL)
        return R_FILE;

    /* initialise 'd' */
    if( (d = malloc( sizeof(detail) )) == NULL )
        return R_MEMORY;
    strcpy(d->name, "");
    d->planets = ((radius / 2) + 1) * ((radius / 2) + 1);
    d->xmin    = -radius;
    d->xmax    = radius;
    d->ymin    = -radius;
    d->ymax    = radius;
    d->maxdist = radius * 2;
    d->speed   = 4;
    d->prod    = 1;
    d->moves   = 1;
    d->vict    = 1;
    d->orders  = 1;

    /* read lines from file */
    while( fgets(input, 81, dtlfile) != NULL )
    {
        /* copy to work buffer & read past CR */
        strcpy(buf, input);
        while( strchr(input, '\n') == NULL )
            if( fgets(input, 81, dtlfile) == NULL )
                strcpy(input, "\n");
        input[80] = '\n';

        /* interpret keyword */
        if( !stricmp(buf, "\n") )
            ; /* skip blank lines... */
        else if ( buf[0] == ';' )
            ; /* ...and comments */
        else if( !stricmp(buf, "nomovement\n") )
            d->moves = 0;
        else if( !stricmp(buf, "noproduction\n") )
            d->prod = 0;
        else if( !stricmp(buf, "novictory\n") )
            d->vict = 0;
        else if( !stricmp(buf, "noorders\n") )
            d->orders = 0;
        else
        {
            /* two-part keyword processing */
            keyword = strtok(buf, " ");
            if(keyword == NULL)
            {
                fclose(dtlfile);
                free(d);
                return R_BADCMD;
            }
            else if( !stricmp(keyword, "name") )
                detail_procstr(d->name);
            else if( !stricmp(keyword, "planets") )
                d->planets = detail_proclong();
            else if( !stricmp(keyword, "xmin") )
                d->xmin = detail_procint();
            else if( !stricmp(keyword, "xmax") )
                d->xmax = detail_procint();
            else if( !stricmp(keyword, "ymin") )
                d->ymin = detail_procint();
            else if( !stricmp(keyword, "ymax") )
                d->ymax = detail_procint();
            else if( !stricmp(keyword, "maxdist") )
                d->maxdist = detail_procint();
            else if( !stricmp(keyword, "speed") )
                d->speed = detail_procint();
            else
            {
                fclose(dtlfile);
                free(d);
                return R_BADCMD;
            }
        }
    }
    fclose(dtlfile);

    /* check for illegal values */
    if( !stricmp(d->name, "") ||
        (d->planets == 0)    ||
        (d->xmin >= d->xmax) ||
        (d->ymin >= d->ymax) ||
        (d->maxdist == 0)    ||
        (d->speed == 0) )
    {
        free(d);
        return R_BADCMD;
    }

    /* return all OK */
    *outd = d;
    return R_OK;
}

/* detail_getname() - return the game name */
char *detail_getname(char *name, detail *d)
{
    return strcpy(name, d->name);
}

/* detail_getplanets() - return the number of planets */
long detail_getplanets(detail *d)
{
    return d->planets;
}

/* detail_getxmin() - return minimum X co-ordinate */
int detail_getxmin(detail *d)
{
    return d->xmin;
}

/* detail_getxmax() - return maximum X co-ordinate */
int detail_getxmax(detail *d)
{
    return d->xmax;
}

/* detail_getymin() - return minimum Y co-ordinate */
int detail_getymin(detail *d)
{
    return d->ymin;
}

/* detail_getymax() - return maximum Y co-ordinate */
int detail_getymax(detail *d)
{
    return d->ymax;
}

/* detail_getmaxdist() - return maximum distance */
int detail_getmaxdist(detail *d)
{
    return d->maxdist;
}

/* detail_getspeed() - return ship speed */
int detail_getspeed(detail *d)
{
    return d->speed;
}

/* detail_getmoves() - return movement processing */
int detail_getmoves(detail *d)
{
    return d->moves;
}

/* detail_getprod() - return production processing */
int detail_getprod(detail *d)
{
    return d->prod;
}

/* detail_getvict() - return victory checking */
int detail_getvict(detail *d)
{
    return d->vict;
}

/* detail_getorders() - return orders processing */
int detail_getorders(detail *d)
{
    return d->orders;
}

/* detail_close() - close the detail file */
void detail_close(detail *d)
{
    free(d);
}
