/* -*-C-*-
*******************************************************************************
*
* File:         tsl.c
* RCS:          $Header: tsl.c,v 1.2 89/09/14 20:33:52 chip Rel $
* Description:  Text Storage Library
* Author:       Chip Chapin, Hewlett Packard Company
* Created:      Thu Aug 24 15:37:16 1989
* Modified:     Thu Sep 14 12:35:24 1989 (Chip Chapin) chip@hpcllcc
* Language:     C
* Package:      Bible Retrieval System
* Status:       Experimental (Do Not Distribute)
*
* $Log:	tsl.c,v $
 * Revision 1.2  89/09/14  20:33:52  20:33:52  chip (Chip Chapin)
 * Release 1-2.  Supports -f and -l options for formatting the output.
 * Updates primarily brl.c, bible.c, and bible.1.
 * 
 * Revision 1.1  89/09/05  17:49:19  17:49:19  chip (Chip Chapin)
 * Initial revision
 * 
*
*******************************************************************************
*/

/*----------------------------------------------------------------------
|   NAME:
|       tsl.c
|       
|   PURPOSE:
|       This file implements the library of routines that are
|       dependent on the storage structure of the text database.
|
|   FUNCTIONS:
|       tsl_gettext
|       	Return text for a particular range of lines.
|	tsl_printtext
|		Write text to stdout instead of returning a buffer.
|       
|       tsl_init
|       	Initialize.
|       	
|       tsl_close
|       	Wrap up.
|
|   HISTORY:
|       890824 cc Extracted storage-dependent functions from brl.c
|
\*----------------------------------------------------------------------*/

#include <stdio.h>
#include <varargs.h>
#include "tsl.h"


#define TRUE	(1)

static char rcs_ident[]="@(#)$Header: tsl.c,v 1.2 89/09/14 20:33:52 chip Rel $";

char *malloc();
char *strtok();

FILE *tfp;	/* Text data file pointer */
long tsl_wsize;	/* Window size (bytes) */
int tsl_wnum;	/* Number of windows */
long *tsl_wtable=NULL;/* Table of window starting offsets in data file */

/* buffer structures.
   We maintain a doubly linked list of uncompressed text buffers, sorted in
   LRU order.
   */
struct buffer_rec {
    struct buffer_rec	*prev, *next;	/* doubly linked list */
    int		win;		/* number of associated window */
    char	*bufferp;	/* the buffer */
};

struct buffer_rec tsl_firstbuffer;	/* ptr to first buffer. */
struct buffer_rec tsl_lastbuffer;	/* take a guess... */
struct buffer_rec **tsl_wbuffers=NULL;	/* table for associating a window with
					   a particular buffer.  Indexed by
					   window number, the table yields a
					   pointer to a buffer_rec. */

char *tsl_cmpbuffer=NULL;	/* Global buffer for compressed text */
int tsl_numbuffs;		/* Count how many buffers active */
int tsl_maxbuffs;		/* Maximum number of buffers we're allowed */
int tsl_maxbuffusage=0x100000;	/* Max buffer mem usage (bytes)	 */



tsl_error( fatal, va_alist )
/*----------------------------------------------------------------------
|   NAME:
|       tsl_error
|
|   ALGORITHM:
|       Report an error specific to the TSL library.
|       
|       fatal	 TRUE if the error should cause an exit.
|       va_alist Variable argument list for printing the error
|       	 report.
|
|   HISTORY:
|       890904 cc Created.
|
\*----------------------------------------------------------------------*/

int fatal;
va_dcl
{
    va_list ap;
    char *format;

    va_start(ap);

    format = va_arg(ap, char *);
    vfprintf(stderr, format, ap);
    putc('\n', stderr);

    va_end(ap);
    if (fatal) exit(-1);
} /* tsl_error */




int tsl_gettext( vn, vc, vb, vbsize )
/*----------------------------------------------------------------------
|   NAME:
|       tsl_gettext
|
|   ALGORITHM:
|       Stuff buffer "vb" with text of line number "vn" and the
|       "vc-1" following lines, but no more than "vbsize" (buffer
|       size) characters.
|       
|       Returns the size (characters) of the text, *not* counting
|       the terminating null.
|
|   HISTORY:
|       890114 cc Initial implementation using simple plain text file.
|       890829 cc Updated to think about buffer size limits.
|
\*----------------------------------------------------------------------*/

int vn, vc;
char *vb;
int vbsize;
{
    long vstart, vsize;
    
    vstart = line_locator[vn];
    vsize = line_locator[ vn+vc ] - vstart;
    if (vsize >= vbsize)
	vsize = vbsize-1;	/* Leave room for trailing null */
    return tsl_textread( vstart, vsize, vb );
} /* tsl_gettext */
	


int tsl_printtext( vn, vc )
/*----------------------------------------------------------------------
|   NAME:
|       tsl_printtext
|
|   ALGORITHM:
|       Write text of line number "vn" and the "vc-1" following
|       lines to stdout.
|       
|       Returns the number of characters written.
|
|   HISTORY:
|       890902 cc Creation from tsl_gettext.
|
\*----------------------------------------------------------------------*/

int vn, vc;
{
    long vstart, vsize;
    
    vstart = line_locator[vn];
    vsize = line_locator[ vn+vc ] - vstart;

    return tsl_textread( vstart, vsize, NULL );
} /* tsl_printtext */
	


tsl_textread( start, vsize, vb )
/*----------------------------------------------------------------------
|   NAME:
|       tsl_textread
|
|   ALGORITHM:
|       Get text starting at absolute byte location "start", and
|       continuing for "vsize" bytes.  If "vb" is NULL, then write
|       the text to stdout, otherwise put it into the buffer
|       pointed to by "vb" and append a null (\0).
|       
|       Returns the size (characters) of the text, *not* counting
|       the terminating null.
|
|   HISTORY:
|       890824 cc Rewritten to handle windowed compressed files.
|       890829 cc Added all the buffer handling -- used to throw
|       	them away each time.
|       890902 cc Added stdout option.
|       890904 cc Iterate on multiple windows, instead of recursing.
|
\*----------------------------------------------------------------------*/

long start, vsize;
char *vb;
{
#ifdef PLAINTEXT
    /* here's the version that works with a plain text file */
    if (fseek( tfp, (long)start, 0 ) == EOF) {
	tsl_error( TRUE, "Cannot seek" );
    } 
    else {
	if (fread( vb, 1, vsize, tfp ) != vsize) {
	    tsl_error( TRUE, "Short read" );
	}
	vb[vsize] = '\0';
    }
#else
    /* here's the version that works with a windowed compressed file */
    /* We use the starting byte in the original file ("start"), and
       the window size ("tsl_wsize") to determine which window we need.
       Using "tsl_wtable", we know where the compressed window starts in the
       file, and how big it is.  Read the compressed window and uncompress it.
       Now we can locate the start of the text and return it.
       */
    long window;			/* current window number */
    long bstart;			/* starting byte relative to beginning
					   of uncompressed window */
    long wstart;			/* starting position of compressed
					   window within the data file */
    long cmpwsize;			/* size of compressed window within the
					   data file */
    long bytes_remaining;		/* Number of bytes yet to be done */
    long size;				/* bytes needed from current window */
    char *uncbuf;			/* buffer for uncompressed data */
    struct buffer_rec *brp;		/* current buffer rec */
    char *cp, *ep;			/* Handy pointers */
    
    bytes_remaining = vsize;
    while (bytes_remaining > 0) {
	window = start / tsl_wsize;	/* which window? */
	if (window >= tsl_wnum)		/* Yikes!  Off the end! */
	    tsl_error( TRUE, "Window %d out of range 0-%d", window, tsl_wnum );
	bstart = start % tsl_wsize;	/* where in [uncompressed] window? */
	if (bstart+bytes_remaining > tsl_wsize)
	    /* Request crosses boundary into next window */
	    size = tsl_wsize-bstart;
	else
	    /* Request can be completed in current window */
	    size = bytes_remaining;
	start += size;
	
	/* Notes on buffer handling ...
	   Three main cases are recognized:
	   1) The buffer for this window is already present.
	   2) It's not present and we can allocate a new buffer.
	   3) It's not present and we reuse an existing buffer.
	   */
	
	if (tsl_wbuffers[window] != NULL) {
	    /* Buffer is already present */
	    brp = tsl_wbuffers[window];
	    uncbuf = brp->bufferp;
	    
	    /* Unlink the buffer from the list.
	       We completely unlink it so the code for putting it back in
	       can be the same regardless of whether or not this is a new buffer.
	       */
	    brp->prev->next = brp->next;
	    brp->next->prev = brp->prev;
	    
	} else {
	    wstart   = tsl_wtable[window];	/* window start in file */
	    cmpwsize = tsl_wtable[window+1]
		- wstart;			/*Size of compressed window*/
	    if (fseek( tfp, (long)wstart, 0 ) == EOF) {
		tsl_error( TRUE, "Bad seek" );
	    } 
	    if (cmpwsize > tsl_wsize) {
		/* This should never happen */
		tsl_error( TRUE, "Compressed window bigger than window size!");
	    }
	    if (fread( tsl_cmpbuffer, 1, cmpwsize, tfp ) != cmpwsize) {
		tsl_error( TRUE, "Short read" );
	    }
	    
	    /* Need a new buffer */
	    if ( tsl_numbuffs >= tsl_maxbuffs ) {
		/* We're at the limit -- need to recycle one
		   Grab the buffer at the end of the LRU list.
		   */
		brp = tsl_lastbuffer.prev;	/* there it is */
		brp->prev->next = brp->next;	/* unlink it */
		brp->next->prev = brp->prev;
		uncbuf = brp->bufferp;
		tsl_wbuffers[brp->win] = NULL;	/* former owner loses */
	    } else {
		/* allocate a new buffer */
		tsl_wbuffers[window] = brp =
		    (struct buffer_rec *)malloc( sizeof(struct buffer_rec) );
		if (brp == NULL)
		    tsl_error( TRUE, "Bad malloc" );
		brp->bufferp = uncbuf = malloc( tsl_wsize );
		if (uncbuf == NULL)
		    tsl_error( TRUE, "Bad malloc" );
		tsl_numbuffs++;
	    } /* new buffer */
	    tsl_wbuffers[window] = brp;
	    brp->win = window;
	    
	    if (cmp_decompress( tsl_cmpbuffer, uncbuf, cmpwsize ) != tsl_wsize) {
		/* Last window is probably small.  Just ignore its size */
		if (window != (tsl_wnum-1)) {
		    free(uncbuf);
		    tsl_error( TRUE, "Bad decompression, result is wrong size" );
		}
	    }
	} /* else we read and decompressed the window */
	
	/* Insert this buffer at head of list */
	brp->next = tsl_firstbuffer.next;
	brp->next->prev = brp;
	tsl_firstbuffer.next = brp;
	brp->prev = &tsl_firstbuffer;
	
	/* If we've gotten this far, we have a nice decompressed buffer to use */
	cp = &uncbuf[bstart];		/* starting address */
	if (vb == NULL) {
	    ep = cp+size;
	    while (cp != ep) putchar( *cp++ );
	} else {
	    memcpy( vb, cp, size );
	    vb += size;
	}
	bytes_remaining -= size;
    } /* while */
    
    if (vb != NULL) *vb = '\0';
#endif
    
    return vsize - bytes_remaining;
} /* tsl_textread */



tsl_init(dfname, path, memlimit)
/*----------------------------------------------------------------------
|   NAME:
|       tsl_init
|
|   ALGORITHM:
|       Initialize the TSL library.
|       
|       dfname		Name of data file.
|       path   		Search path to use for file.
|       memlimit	Limit (in Kbytes) on buffer space to use.
|
|   HISTORY:
|       890825 cc Rewrite for compressed windowed files.
|       890830 cc Added memlimit.
|       890904 cc Implemented search path.
|
\*----------------------------------------------------------------------*/

char *dfname, *path;
int  memlimit;
{
    struct tsl_fileheader fh;
    int i;
    int tablesize;
    char *dfpath, *d;
    int  file_not_open, got_a_path;

    if (memlimit > 0)
	tsl_maxbuffusage = memlimit<<10;	/* times 1024 */
    
    /* Open input file */
    file_not_open = TRUE;
    got_a_path = (d=strtok(path, " :")) != NULL;
    while (file_not_open && got_a_path) {
	i = strlen(d);
	dfpath = malloc( strlen(dfname) +i+2 );
	strcpy( dfpath, d );
	if (d[i-1] != '/')
	    strcat( dfpath, "/");
	strcat( dfpath, dfname );
	file_not_open = ((tfp=fopen( dfpath, "r")) == NULL);
	got_a_path = ((d=strtok(NULL, " :")) != NULL);
	free( dfpath );
    }
    if (file_not_open) {
	tsl_error( TRUE, "Cannot open data file %s", dfname );
    }
	
    /* What do we have here?  Let's check out the file header... */
    if (!fread( &fh, sizeof(fh), 1, tfp )) {
	tsl_error( TRUE, "Cannot read data file %s", dfname );
    }
    if ((fh.magic[0] != TSL_MAGIC1) || (fh.magic[1] != TSL_MAGIC2))
	tsl_error( TRUE, "Cannot use data file: Bad magic number" );
    if ((fh.version[0] != TSL_FVERSION1) || (fh.version[1] != TSL_FVERSION2))
    	tsl_error( TRUE, "Cannot use data file: Wrong version" );
    if ((fh.bytesexflag != TSL_BYTEFLAG))
	tsl_error( TRUE, "Cannot use data file: Wrong byte order" );

    tsl_wsize = fh.wsize;
    tsl_wnum  = fh.wnum;

    /* Grab the window table */
    tablesize = sizeof(int)*(tsl_wnum+1);	/* +1 for ending entry */
    if ((tsl_wtable = (long *)malloc( tablesize )) == NULL)
	tsl_error( TRUE, "Bad malloc" );
    if (!fread( tsl_wtable, tablesize, 1, tfp )) {
	tsl_error( TRUE, "Error reading data file %s", dfname );
    }

    /* Create buffer table (parallel array to window table) */
    if ((tsl_wbuffers =
	 (struct buffer_rec **)malloc(sizeof(*tsl_wbuffers)*tsl_wnum )) == NULL)
	tsl_error( TRUE, "Bad malloc" );
    for (i=0; i< tsl_wnum; i++) tsl_wbuffers[i] = NULL;
    
    tsl_numbuffs = 0;		/* active buffers of uncompressed text */
    tsl_maxbuffs = tsl_maxbuffusage / tsl_wsize;
    if (tsl_maxbuffs < 1) tsl_maxbuffs = 1;
    tsl_firstbuffer.next = &tsl_lastbuffer;
    tsl_firstbuffer.prev = NULL;
    tsl_firstbuffer.win  = NULL;
    tsl_firstbuffer.bufferp = NULL;
    tsl_lastbuffer.prev = &tsl_firstbuffer;
    tsl_lastbuffer.next = NULL;
    tsl_lastbuffer.win  = NULL;
    tsl_lastbuffer.bufferp = NULL;

    /* Global buffer for compressed text.  Much bigger than needed. :-) */
    if ((tsl_cmpbuffer = malloc( tsl_wsize )) == NULL)
	tsl_error( TRUE, "Bad malloc" );
    
    cmp_init();		/* Initialize decompression */
} /* tsl_init */


tsl_close()
/*----------------------------------------------------------------------
|   NAME:
|       tsl_close
|
|   ALGORITHM:
|       Tidy up before leaving the TSL library.
|
|   HISTORY:
|
\*----------------------------------------------------------------------*/
{
    int i;
    struct buffer_rec *bufp, *nbufp;
    
    fclose( tfp);

    /* Free all kinds of buffers and tables */
    bufp=tsl_firstbuffer.next;
    while (bufp != &tsl_lastbuffer) {
	nbufp = bufp->next;
	if (bufp->bufferp != NULL)
	    free(bufp->bufferp);	       	/* free the buffer */
	free(bufp);				/* free the buffer rec */
	bufp = nbufp;				/* on to next buffer rec */
    }
    if (tsl_wtable != NULL) free(tsl_wtable);
    if (tsl_cmpbuffer != NULL) free(tsl_cmpbuffer);
} /* tsl_close */
