/* -*-C-*-
*******************************************************************************
*
* File:         brl.c
* RCS:          $Header: brl.c,v 1.6 89/09/14 20:33:45 chip Rel $
* Description:  Bible Retrieval Library
* Author:       Chip Chapin, Hewlett Packard Company
* Created:      Jan 14 1989, Chip Chapin
* Modified:     Thu Sep 14 19:30:39 1989 (Chip Chapin) chip@hpcllcc
* Language:     C
* Package:      Bible Retrieval System
* Status:       Experimental (Do Not Distribute)
*
* $Log:	brl.c,v $
 * Revision 1.6  89/09/14  20:33:45  20:33:45  chip (Chip Chapin)
 * Release 1-2.  Supports -f and -l options for formatting the output.
 * Updates primarily brl.c, bible.c, and bible.1.
 * 
 * Revision 1.5  89/09/13  21:49:13  21:49:13  chip (Chip Chapin)
 * Implement -f and -l options for pretty-printing and linewidth limitation.
 * 
 * Revision 1.4  89/09/08  13:22:47  13:22:47  chip (Chip Chapin)
 * Better error checking on verse syntax; automatic test suite.
 * 
 * Revision 1.3  89/09/08  09:01:22  09:01:22  chip (Chip Chapin)
 * Bug fix and simplification: send whole input lines or arguments to BRL,
 * and let BRL worry about multiple references.  We don't care.
 * 
 * Revision 1.2  89/09/05  20:22:59  20:22:59  chip (Chip Chapin)
 * Workaround 6.2 C compiler enum gripes.
 * 
 * Revision 1.1  89/09/05  17:49:15  17:49:15  chip (Chip Chapin)
 * Initial revision
 * 
* 
*******************************************************************************
*/

/*----------------------------------------------------------------------
|   NAME:
|       brl.c
|
|   PURPOSE:
|       Provide a set of routines for retrieving Bible text from
|       the Text Storage Library (tsl).  All of these routines
|       should take a high-level view of the text itself; they may
|       know something about the structure of the Bible, but
|       should *not* depend in any way upon the manner in which
|       the text is stored.  Call on the TSL for that.
|
|   FUNCTIONS:
|       Grammar Functions: Used within the BRL to parse verse
|       specifications.
|       	verse_spec
|       	verse_id
|       	verse_continuation
|       	
|       Access Functions: Called from application programs to
|       retrieve Bible text.
|		brl_getverse
|		brl_printverse
|       
|       Utility Functions: Called from application programs.
|       	brl_init
|       	brl_close
|
|   HISTORY:
|       890114 cc Initial creation.
|	890824 cc Improved partitioning between BRL and TSL.  Created
|		brl_getverse to hide verse_spec and verse numbers from
|		user programs.  Fix bug in verse_id parsing "jn".
|
\*----------------------------------------------------------------------*/


#include <stdio.h>
#include <ctype.h>
#include "brl.h"

#define FALSE	(0)

static char rcs_ident[]="@(#)$Header: brl.c,v 1.6 89/09/14 20:33:45 chip Rel $";

char *booknamestr[] = { "Genesis",
			   "Exodus",
			   "Leviticus",
			   "Numbers",
			   "Deuteronomy",
			   "Joshua",
			   "Judges",
			   "Ruth",
			   "1 Samuel",
			   "2 Samuel",
			   "1 Kings",
			   "2 Kings",
			   "1 Chronicles",
			   "2 Chronicles",
			   "Ezra",
			   "Nehemiah",
			   "Esther",
			   "Job",
			   "Psalms",
			   "Proverbs",
			   "Ecclesiastes",
			   "Song of Solomon",
			   "Isaiah",
			   "Jeremiah",
			   "Lamentations",
			   "Ezekiel",
			   "Daniel",
			   "Hosea",
			   "Joel",
			   "Amos",
			   "Obadiah",
			   "Jonah",
			   "Micah",
			   "Nahum",
			   "Habakkuk",
			   "Zephaniah",
			   "Haggai",
			   "Zechariah",
			   "Malachi",
			   "Matthew",
			   "Mark",
			   "Luke",
			   "John",
			   "Acts",
			   "Romans",
			   "1 Corinthians",
			   "2 Corinthians",
			   "Galatians",
			   "Ephesians",
			   "Philippians",
			   "Colossians",
			   "1 Thessalonians",
			   "2 Thessalonians",
			   "1 Timothy",
			   "2 Timothy",
			   "Titus",
			   "Philemon",
			   "Hebrews",
			   "James",
			   "1 Peter",
			   "2 Peter",
			   "1 John",
			   "2 John",
			   "3 John",
			   "Jude",
			   "Revelation",
			   "BAD"
};


/* Default values for book, chapter verse.
   Preserves state from one call to the next.
   */
int brl_book, brl_chapter, brl_verse;


/*----------------------------------------------------------------------
|
|       Verse Specification Grammar
|       
|       The following routines implement the grammar by which
|       verses may be specified.
|       See each routine, starting with verse_spec, for the grammar.
|
\*----------------------------------------------------------------------*/

#define getnum(s,n)	n=0; get_nonblank(s); \
    while (isdigit(*s)) n = 10 * n + (*s++ - '0');



int get_book( s, book )
/*----------------------------------------------------------------------
|   NAME:
|       get_book
|
|   ALGORITHM:
|       Parse a bible book name or abbreviation.  Defaults to the
|       book number passed in "book" if no book is specified.
|       
|       Returns the number of the book, 0..65, -1 if an
|       unrecognizable book name is given, or -2 if no book name
|       is given (needs a bit of cleanup!).
|
|   HISTORY:
|       890902 cc Extracted from verse_id.
|       890904 cc Revised to allow non-fatal errors.
|
\*----------------------------------------------------------------------*/

char **s;
int book;
{
    int c;
    char *s1;

    book = -2;		/* assume no book is given */
    get_nonblank(*s);
    s1 = *s;
    if (*s1 && (isalpha(*s1) || isalpha(*(s1+1)))) {
	switch (c = *s1++) {
	  case '1':	/* 1Sa, 1Ki, 1Ch, 1Co, 1Th, 1Ti, 1Pe, 1Jn */
	  case '2':	/* 2Sa, 2Ki, 2Ch, 2Co, 2Th, 2Ti, 2Pe, 2Jn */
	  case '3':	/* 3Jn */
	    get_nonblank(s1);
	    switch (*s1++) {
	      case 'c': 
		if (*s1=='h') book= (int)CHRON1;
		else if (*s1=='o') book= (int)COR1;
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
		break;
	      case 'j':
		book= (int)JOHN1;
		break;
	      case 'k':
		book= (int)KINGS1;
		break;
	      case 'p':
		book= (int)PET1;
		break;
	      case 's':
		book= (int)SAM1;
		break;
	      case 't':
		if (*s1=='h') book= (int)THESS1;
		else if (*s1=='i') book= (int)TIM1;
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
		break;
	      default:
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    if (c != '1') book++;
	    if (c == '3') 
		if (book == (int)JOHN2) book++;
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    break;
	  case 'a': /* amos, acts */
	    c = *s1++;
	    if (c == 'm') book = (int)AMOS;
	    else if (c == 'c') book = (int)ACTS;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 'c': /* col */
	    book = (int)COLOS;
	    break;
	  case 'd': /* deut, dan */
	    c = *s1++;
	    if (c == 'e') book = (int)DEUT;
	    else if (c == 'a') book = (int)DANIEL;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 'e': /* exodus, ezra, esther, eccl, ezekiel, eph */
	    c = *s1++;
	    if (c == 'x') book = (int)EXODUS;	/* ex */
	    else if (c == 's') book = (int)ESTHER;	/* es */
	    else if (c == 'c') book = (int)ECCL;	/* ec */
	    else if (c == 'p') book = (int)EPH;	/* ep */
	    else {
		if (c == 'z') c = *s1++;
		if (c == 'r') book = (int)EZRA;	/* ezr, er */
		else if (c == 'e' || c == 'k')
		    book = (int)EZEKIEL; /* eze,ee,ezk,ek */
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    break;
	  case 'g': /* gen, gal */
	    c = *s1++;
	    if (c == 'e' || c == 'n') book = (int)GENESIS;
	    else if (c == 'a' || c == 'l') book = (int)GAL;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 'h': /* hebrews, hosea, habakkuk, haggai */
	    c = *s1++;
	    if (c == 'o') book = (int)HOSEA;		/* ho */
	    else if (c == 'e' || c == 'b')
		book = (int)HEBREWS;	/* he, hb (more likely than Hab.) */
	    else {
		if (c == 'a') c = *s1++;
		if (c == 'b') book = (int)HABAK;	/* hab */
		else if (c == 'g') book = (int)HAGGAI; /* hag, hg */
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    break;
	  case 'i': /* isaiah */
	    book = (int)ISAIAH;
	    break;
	  case 'j': /* josh, judges, job, jer, joel, jonah, john, jam, jude */
	    c = *s1++;
	    if (c == 'a') book = (int)JAMES;		/* ja */
	    else if (c == 'e' || c == 'r') book = (int)JEREM;	/* je, jr */
	    else if (c == 'b') book = (int)JOB;	/* jb */
	    else if (c == 'd') {
		if ((c = *s1++) == 'g') book = (int)JUDGES;	/* jdg */
		else if (c == 'e') book = (int)JUDE;	/* jde */
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    else if (c == 'g') book = (int)JUDGES;	/* jg */
	    else if (c == 'l') book = (int)JOEL;	/* jl */
	    else if (c == 'n') book = (int)JOHN;	/* jn */
	    else if (c == 'u') {
		if ((c = *s1++) == 'd') c = *s1++;
		if (c == 'g') book = (int)JUDGES;	/* judg, jug (ha-ha) */
		else if (c == 'e') book = (int)JUDE;	/* jude, jue (ha-ha) */
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    else if (c == 'o') {
		if ((c = *s1++) == 's') book = (int)JOSHUA;	/* jos */
		else if (c == 'b') book = (int)JOB;		/* job */
		else if (c == 'e') book = (int)JOEL;		/* joe */
		else if (c == 'n') book = (int)JONAH;	/* jon */
		else if (c == 'h') book = (int)JOHN;		/* joh */
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    break;
	  case 'l': /* lev, lam, luke */
	    c = *s1++;
	    if (c == 'e' || c == 'v') book = (int)LEVIT;
	    else if (c == 'a' || c == 'm') book = (int)LAMENT;
	    else if (c == 'u' || c == 'k') book = (int)LUKE;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 'm': /* micah, malachi, matt, mark */
	    c = *s1++;
	    if (c == 'i' || c == 'c') book = (int)MICAH;
	    else {
		if (c == 'a') c = *s1++;
		if (c == 'l') book = (int)MALACHI;
		else if (c == 't') book = (int)MATT;
		else if (c == 'r' || c == 'k') book = (int)MARK;
	    }
	    break;
	  case 'n': /* num, neh, nahum */
	    c = *s1++;
	    if (c == 'u' || c == 'm') book = (int)NUM;
	    else if (c == 'e') book = (int)NEHEM;
	    else if (c == 'a') book = (int)NAHUM;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 'o': /* obadiah */
	    book = (int)OBADIAH;
	    break;
	  case 'p': /* psalms, proverbs, philippians, philemon */
	    c = *s1++;
	    if (c == 's') book = (int)PSALMS;
	    else if (c == 'r') book = (int)PROV;
	    else {
		if (c == 'h') c = *s1++;
		if (c == 'i') c = *s1++;
		if (c == 'l') c = *s1++;
		if (c == 'i' || c == 'p') book = (int)PHILIP;
		else if (c == 'e' || c == 'm') book = (int)PHILEM;
		else {
		    tsl_error( FALSE, BADBOOK, *s );
		    return -1;
		}
	    }
	    break;
	  case 'r': /* ruth, romans, revelation */
	    c = *s1++;
	    if (c == 'u' || c == 't') book = (int)RUTH;
	    else if (c == 'o' || c == 'm') book = (int)ROMANS;
	    else if (c == 'e' || c == 'v') book = (int)REV;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  case 's': /* song */
	    book = (int)SONG;
	    break;
	  case 't': /* titus */
	    book = (int)TITUS;
	    break;
	  case 'z': /* zeph, zech */
	    if ((c = *s1++) == 'e') c = *s1++;
	    if (c == 'p') book = (int)ZEPH;
	    else if (c == 'c') book = (int)ZECH;
	    else {
		tsl_error( FALSE, BADBOOK, *s );
		return -1;
	    }
	    break;
	  default:
	    tsl_error( FALSE, BADBOOK, *s );
	    return -1;
	} /* switch on first letter of book name */
	
	/* Skip any remaining letters in book name */
	while (isalpha(*s1)) s1++;
	get_nonblank(s1);
    }
    *s = s1;
    return book;
} /* get_book */



int verse_continuation( s, book, chapter, absverse )
/*----------------------------------------------------------------------
|   NAME:
|       verse_continuation
|
|   ALGORITHM:
|       <verse continuation> ::= <null>
|                            ::= - <verse id>
|       		     
|       "s" is pointing to the first char of the <verse
|       continuation>.  If it is a null continuation we leave it
|       where it is, otherwise on exit we update it to point to
|       the first non-blank after the continuation.
|       	
|       book     -- Number of the book in which the verse spec began.
|       chapter  -- Number of the chapter ...
|       absverse -- ABSOLUTE VERSE NUMBER where verse spec began
|       	(this is more useful than the relative num).
|       	
|       Returns the total number of verses to be fetched, unless
|       there was a problem.  Returns zero if there was a problem. 
|       	
|
|   HISTORY:
|       890829 cc Initial implementation (formerly always returned 1). 
|       890908 cc Error return.
|
\*----------------------------------------------------------------------*/

char **s;
int book, chapter;
long absverse;
{
    int	n;
    int verse;
    char *s1;

    s1 = *s;	/* For easier handling.  Be sure to update it */
    n = 1;	/* Assume NO following verses */

    if (*s1++ == '-') {
	/* yup, there's a continuation */
	if (verse_id( &s1, &book, &chapter, &verse ) == 0)
	    return 0;
	*s = s1;

	/* Determine absolute verse number of ending verse */
	n = verse_num( book, chapter, verse );

	/* how many verses is that? */
	n = n - absverse +1;
    }
    
    return n;
} /* verse_continuation */



int verse_id( s, bookp, chapterp, versep )
/*----------------------------------------------------------------------
|   NAME:
|       verse_id
|
|   ALGORITHM:
|       <verse id>  ::= <book> <chapter> : <verse>
|       	    ::= <book> <chapter>	# implies vs. 1
|           	    ::= <book> 			# implies 1:1
|       	    ::= <chapter> : <verse>	# implies current book
|       	    ::= <verse>			# implies current book/chapter
|       
|       "s" is pointing to the first character of the <verse id>.
|       On exit, we update it to be pointing to the next non-blank
|       after what we've parsed.
|       
|       The values of *bookp, and *chapterp are used as
|       defaults in case of a partial verse spec.  For example,
|       "15" would mean verse 15 in the default book/chapter.
|           
|       We return an absolute verse number (Gen 1:1 == 1) for the
|       referenced verse.  Also update *bookp, *chapterp*, and
|       *versep with the book, chapter, and verse numbers.
|       
|       If a problem occurs, we return 0.
|
|   HISTORY:
|       890902 cc Revised to handle incomplete verse specs.
|       890908 cc Fix to blank line handling.  Revised error return.
|
\*----------------------------------------------------------------------*/

char **s;
int *bookp, *chapterp, *versep;
{
    short book, chapter, verse;
    int   num;
    char  *s1;

    s1 = *s;		/* For easier handling.  Be sure to update it */
    if ((book = get_book( &s1, *bookp )) == -1) return 0;
    getnum(s1, num);
    if (num < 1) {
	/* <verse spec> ::= <book> */
	if (book < 0) {
	    /* Special case: if this is a blank line, then print
	       the very next verse.  But if there's still junk on
	       the line, then what we have here is an error.
	       */
	    if (*s1) {
		/* Bad news, gang */
		tsl_error( FALSE, "Extra garbage on line: '%s'", s1 );
		return 0;
	    }

	    /* Get the last verse printed, increment it,
	       and translate that back into book/chapter/verse,
	       while making sure it stays in the proper range.
	       */
	    num = verse_num( *bookp, *chapterp, *versep );
	    num++;
	    num = extract_num( num, bookp, chapterp, versep );
	    *s = s1;
	    return( num );
	} else {
	    chapter = 1;
	    verse   = 1;
	}
    } else {
	get_nonblank(s1);
	if (*s1 != ':') {
	    if (book < 0) {
		/* <verse spec> ::= <verse> */
		book    = *bookp;
		chapter = *chapterp;
		verse   = num;
	    } else {
		/* <verse spec> ::= <book> <chapter> */
		chapter = num;
		verse   = 1;
	    }
	} else {
	    /* <verse spec> ::= <book> <chapter> : <verse> */
	    /* <verse spec> ::= <chapter> : <verse> */
	    s1++;	/* skip past the ':' */
	    if (book < 0)
		book = *bookp; 		/* default book */
	    chapter = num;
	    getnum(s1, verse);
	    if (verse < 1) {
		tsl_error( FALSE, NO_VERSE, *s );
	    } else {
		get_nonblank(s1);
	    }
	}
    }

    *s = s1;		/* Update the pointer we were passed */
    *bookp    = book;
    *chapterp = chapter;
    *versep   = verse;
    /* return absolute verse number */
    return( verse_num( book, chapter, verse ) );
} /* verse_id */



int extract_num( absverse, bp, cp, vp )
/*----------------------------------------------------------------------
|   NAME:
|       extract_num
|
|   ALGORITHM:
|       Extract the book, chapter, [relative] verse corresponding
|       to the absolute verse number passed in "absverse", if
|       "absverse" is valid.  If it is not valid, then coerce it
|       into something reasonable.  Update book, chapter, and
|       verse through their pointers "bp", "cp", and "vp",
|       respectively.  Return the [possibly changed] value of
|       absverse. 
|
|   HISTORY:
|       890904 cc Created (in a hurry)
|
\*----------------------------------------------------------------------*/

long absverse;
int *bp, *cp, *vp;
{
    int bk, chp;

    if (absverse < 1)
	absverse = 1;	/* cheap insurance */
    
    for (bk= (int)GENESIS; bk <= (int)REV; bk++)
	if (absverse <= start_verse[start_chapter[bk+1]+1])
	    /* we've got the right book */
	    for (chp=start_chapter[bk]+1; chp <= start_chapter[bk+1]; chp++)
		if (absverse <= start_verse[chp+1]) {
		    /* we've got the right chapter */
		    *bp = bk;
		    *cp = chp - start_chapter[bk];
		    *vp = absverse - start_verse[chp];
		    return absverse;
		}
    /* if we got here, then things are messed up.
     Assume that the verse is off the back.
     */
    *bp = (int)REV;
    *cp = start_chapter[ (int)REV+1 ] - start_chapter[(int)REV];
    *vp = start_verse[(*cp)+1] - start_verse[*cp];
    return verse_num( *bp, *cp, *vp );
} /* extract_num */



int verse_num(b, c, v)
/*----------------------------------------------------------------------
|   NAME:
|       verse_num
|
|   ALGORITHM:
|       Return the absolute verse number, given the book, chapter
|       and [relative] verse number.
|       
|       Look up the starting absolute verse number of the
|       specified book/chapter, using the "start_verse" and
|       "start_chapter" tables, then add the [relative] verse
|       number. 
|       
|       Error checking: Ensure chapter is not too big for the
|       book, and that verse is not too big for the chapter.
|
|   HISTORY:
|       890830 cc Created.  The original lookup didn't do any
|       	error checking.
|
\*----------------------------------------------------------------------*/

int b, c, v;
{
    int abschapter, absverse;

    /* force book into proper range */
    if (b < (int)GENESIS) b = (int)GENESIS;
    else if (b > (int)REV) b = (int)REV;
    
    if ((abschapter = start_chapter[b] + c) > start_chapter[b+1])
	abschapter = start_chapter[b+1];
    if ((absverse = start_verse[abschapter] + v) > start_verse[abschapter+1])
	absverse = start_verse[abschapter+1];
    return absverse;
} /* verse_num */



int verse_spec( s, n )
/*----------------------------------------------------------------------
|   NAME:
|       verse_spec
|
|   ALGORITHM:
|          
|           <verse spec> ::= <verse id> <verse continuation>
|       
|       Translate a <verse specifier> string s into a starting
|       absolute verse number, returned as the function result,
|       and a count n of verses to read.  Gen. 1:1 == verse 1.
|       
|       Returns 0 if something went wrong.
|
|   HISTORY:
|
\*----------------------------------------------------------------------*/

char	**s;
int	*n;
{
    long v;
    char *s1;

    /* Munge off leading spaces,
       convert string to lower case,
       convert commas into blanks, and
       dispose of possible yucky '\n'.
     */
    get_nonblank(*s);
    s1 = *s;
    while (*s1) {
	*s1 = tolower(*s1);
	if (*s1 == ',')
	    *s1 = ' ';
	else if (*s1 == '\n')
	    *s1 = '\0';
	s1++;
    }

    if ((v = verse_id( &(*s), &brl_book, &brl_chapter, &brl_verse )) == 0)
	return 0;
    if (**s) {
	if ((*n = verse_continuation( &(*s), brl_book, brl_chapter, v )) == 0)
	    return 0;
    } else {
	*n = 1;
    }
    return v;
} /* verse_spec */



int brl_getverse( vs, vb, vbsize, pretty, lwidth )
/*----------------------------------------------------------------------
|   NAME:
|       brl_getverse
|
|   ALGORITHM:
|       Stuff buffer "vb" with text of verses specified by string
|       "vs", a <verse spec>.
|       
|       "pretty" and "lwidth" are formatting options passed to
|       tsl_gettext. 
|
|   HISTORY:
|       890824 cc Created to hide internal functions
|       	tsl_gettext and verse_spec from user programs.
|       890912 cc Added pretty and lwidth.
|
\*----------------------------------------------------------------------*/

char *vs;
char *vb;
int  vbsize;
int  pretty, lwidth;
{
    int vn, vc, bytecount;

    bytecount = 0;
    while (*vs && (vn = verse_spec( &vs, &vc )))
	   bytecount += tsl_gettext( vn, vc, vb, vbsize );

    return bytecount;
} /* brl_getverse */



brl_printverse( vs, pretty, lwidth )
/*----------------------------------------------------------------------
|   NAME:
|       brl_printverse
|
|   ALGORITHM:
|       Print text to stdout of verses specified by string "vs".
|       
|       pretty -- If true, then we want special output formatting.
|       lwidth -- If non-zero, then insert new-lines if necessary
|       	between words to prevent lines from being longer
|       	than lwidth.
|
|   HISTORY:
|       890902 cc Created as alternative to brl_getverse.
|       890912 cc Added pretty and lwidth.
|
\*----------------------------------------------------------------------*/

char *vs;
int  pretty;
int  lwidth;
{
    int vn, vc, bytecount;

    bytecount = 0;
    if (pretty || lwidth) {
	int len, indent, i;
#define VBSIZE 512
	char vb1[VBSIZE], vb2[VBSIZE];
	char *srcp, *dstp, *endp;
	int  oldbook, oldchapter, oldverse;

	oldbook = brl_book; oldchapter = brl_chapter; oldverse = brl_verse;
	while (*vs && (vn = verse_spec( &vs, &vc )))
	    for (; vc; vc--) {
		/* get text for a single verse */
		len=tsl_gettext( vn++, 1, vb1, VBSIZE );
		srcp = vb1;
		dstp = vb2;
		indent = 0;
		if (pretty) {
		    extract_num( vn-1, &brl_book, &brl_chapter, &brl_verse );
		    /* print book/chapter heading?? */
		    if (brl_book != oldbook || brl_chapter != oldchapter) {
			/* verse == 1 || bytecount == 0) { */
			if (brl_book != oldbook) {
			    /* chapter == 1 || bytecount == 0) { */
			    /* print book heading */
			    dstp += sprintf( dstp, "\n%s\n",
					    booknamestr[brl_book] );
			}
			/* print chapter heading */
			dstp += sprintf( dstp, "\n Chapter %d\n\n",
					brl_chapter );
		    }
		    oldbook = brl_book;
		    oldchapter = brl_chapter;
		    oldverse = brl_verse;
		    
		    /* advance to the verse number in source */
		    while (*srcp != ':') srcp++;
		    srcp++;

		    /* insert indentation in dest */
		    indent = 2;		/* two for verse */
		    for (i=indent; i--; ) *dstp++ = ' ';
		}

		if (lwidth) {
		    /* Line width limitation. */
		    while (strlen(srcp) > (lwidth-indent)) {
			/* split this line.  But where? */
			for( endp=srcp+lwidth-indent; *endp!=' '; endp--) ;

			/* style is to indent only the first line of the
			   verse, before the verse number.  Subsequent lines
			   are *not* indented.
			   */
			indent=0;
			
			len = endp-srcp;
			strncpy(dstp, srcp, len);
			dstp += len;
			*dstp++ = '\n';
			srcp = ++endp;		/* past the ' ' */
		    }
		    /* last line, get the rest */
		    strcpy(dstp, srcp);
		} else {
		    /* No line length limit.
		       Just copy the rest of the verse line.
		       */
		    strcpy(dstp, srcp);
		}
		/* now print out the verse */
		printf( vb2 );
		bytecount += strlen(vb2);
	    }
    } else {
	while (*vs && (vn = verse_spec( &vs, &vc )))
	    bytecount += tsl_printtext( vn, vc );
	
	return bytecount;
    }
} /* brl_printverse */


brl_init( dfname, dfpath, memlimit )
/*----------------------------------------------------------------------
|   NAME:
|       brl_init
|
|   ALGORITHM:
|       Initialize the library.
|       
|       memlimit	Limit (in Kbytes) on buffer space used by TSL.
|       dfname		Name of the data file.
|       dfpath		List of paths to use in searching for the
|       		data file.
|
|   HISTORY:
|       890830 cc Added memlimit.
|       890905 cc Added filename and paths.
|
\*----------------------------------------------------------------------*/

int memlimit;
char *dfname, *dfpath;
{
    if (dfname == NULL)
	dfname = "bible.data";
    if (dfpath == NULL)
	dfpath = "./ /usr/local/lib/";
    tsl_init( dfname, dfpath, memlimit );

    /* Set (low) illegal values for book and verse.
       Making them low means that they will be coerced to Gen1:1 if
       the user gives a null input.
       Making the book illegal means that it won't match any existing book,
       which keeps the pretty-printing logic happy.
       */
    brl_book = -1;
    brl_chapter = 1;
    brl_verse = 0;
} /* brl_init */


brl_close()
/*----------------------------------------------------------------------
|   NAME:
|       brl_close
|
|   ALGORITHM:
|       Close the library.
|
|   HISTORY:
|
\*----------------------------------------------------------------------*/
{
    tsl_close();
} /* brl_close */


