/* -*-C-*-
*******************************************************************************
*
* File:         bible.c
* RCS:          $Header: bible.c,v 1.4 89/09/14 20:33:35 chip Rel $
* Description:  Write Bible text to stdout.
* Author:       Chip Chapin, Hewlett Packard Company
* Created:      Jan 21 1989
* Modified:     Thu Sep 14 13:30:39 1989 (Chip Chapin) chip@hpcllcc
* Language:     C
* Package:      Bible Retrieval System
* Status:       Experimental (Do Not Distribute)
*
* $Log:	bible.c,v $
 * Revision 1.4  89/09/14  20:33:35  20:33:35  chip (Chip Chapin)
 * Release 1-2.  Supports -f and -l options for formatting the output.
 * Updates primarily brl.c, bible.c, and bible.1.
 * 
 * Revision 1.3  89/09/13  21:48:26  21:48:26  chip (Chip Chapin)
 * Implement -f and -l options for pretty-printing and linewidth limitation.
 * 
 * Revision 1.2  89/09/08  09:00:26  09:00:26  chip (Chip Chapin)
 * Bug fix and simplification: send whole input lines or arguments to BRL,
 * and let BRL worry about multiple references.  We don't care.
 * 
 * Revision 1.1  89/09/05  17:47:27  17:47:27  chip (Chip Chapin)
 * Initial revision
 * 
*
*******************************************************************************
*/


/*----------------------------------------------------------------------
|   NAME:
|       bible.c
|
|   PURPOSE:
|       Reads verse specs from the command line or from stdin and
|       writes Bible verses to stdout.  Uses the Bible Retrieval
|       Library for all text access functions.
|
|   FUNCTIONS:
|       main
|       get_verse
|
|   HISTORY:
|       890121 cc Initial Creation
|       890824 cc Updated to use new brl_verse_text.
|       890829 cc Updated to think about buffer size.
|
\*----------------------------------------------------------------------*/

#include <ctype.h>
#include <stdio.h>
#include "brl.h"

#define VSPECSZ 132
#define VBUFFSZ 32768
#define FALSE	(0)
#define TRUE	(1)

static char rcs_ident[]="@(#)$Header: bible.c,v 1.4 89/09/14 20:33:35 chip Rel $";

char *strtok();

char *myname;			/* Program Name */
int  mem_limit;			/* Limit on buffer space used by BRL */
int  pretty_printing=FALSE;	/* Format the output */
int  line_width=FALSE;		/* Line width to use in formatting */


get_verse( vs )
/*----------------------------------------------------------------------
|   NAME:
|       get_verse
|
|   ALGORITHM:
|       Retrieve the text of a Bible verse, and stuff it in a
|       buffer.  Also writes the text to stdout.
|       
|       vs	Verse Specification, e.g. "jn3:16".
|
|   HISTORY:
|       890121 cc Initial Creation
|
\*----------------------------------------------------------------------*/

char *vs;
{
    char vbuff[VBUFFSZ];

    if (brl_getverse( vs, vbuff, VBUFFSZ )) {
	printf( "%s\n", vbuff );
    }
} /* get_verse */



printverse( vs )
/*----------------------------------------------------------------------
|   NAME:
|       printverse
|
|   ALGORITHM:
|       Print text of bible verses referenced by "vs" to stdout.
|       
|       vs	Verse Specification, e.g. "Jn3:16-18".
|
|   HISTORY:
|       890902 cc Created, replacing buffered "get_verse".
|       890912 cc Added pretty_printing and line_width params to call.
|
\*----------------------------------------------------------------------*/

char *vs;
{
    brl_printverse( vs, pretty_printing, line_width );
} /* printverse */



usage()
/*----------------------------------------------------------------------
|   NAME:
|       usage
|
|   ALGORITHM:
|       Print usage message to stderr, then exit program.
|
|   HISTORY:
|       890830 cc Created.
|       890912 cc Added -f, -l.
|
\*----------------------------------------------------------------------*/

{
    fprintf( stderr, "Usage: %s [-f][-l[cols]][-m mem][-p path][-d file][<verse spec>...]\n",
	    myname );
    fprintf( stderr, "\n" );
    fprintf( stderr, "      -d file  Override default datafile name\n" );
    fprintf( stderr, "      -f       Format the output\n" );
    fprintf( stderr, "      -l[cols] Set line width limit (defaults to envar COLUMNS.\n" );
    fprintf( stderr, "      -m mem   Specify maximum buffer memory usage\n" );
    fprintf( stderr, "               in Kbytes.  Defaults to 1024K.\n" );
    fprintf( stderr, "      -p path  Override default datafile search path.\n" );
    exit(1);
} /* usage */



main(argc, argv)
/*----------------------------------------------------------------------
|   NAME:
|       main
|
|   ALGORITHM:
|       Main Program.
|       
|       Handle command line options.
|       Initialize the Bible Retrieval Library.
|       Read Bible references from the command line or from stdin.
|       For each line of input, call printverse to retrieve and display
|       the text.  Multiple references are possible on each input
|       line, but we don't care: the BRL routines take care of that. 
|
|   HISTORY:
|       890830 cc Added options processing.
|       890908 cc Send whole lines to printverse instead of
|       	partially parsing them.
|       890912 cc Added -f, -l options.
|
\*----------------------------------------------------------------------*/

int argc;
char **argv;
{
  char line[VSPECSZ];
  char *dfname, *dfpath, *s;

  mem_limit = 1024;		/* Default 1024K of buffer space */
  dfname = dfpath = NULL;	/* Use library's default values */
    
  myname = *argv++; argc--;	/* Program name */
  
#define ARGVAL()  (*++(*argv) || (--argc && *++argv))
  for (;argc && **argv == '-'; argc--, argv++) {
      /* Got an option flag */
      while (*++(*argv)) {
	  /* Process all flags in this argument */
	  switch (**argv) {
	    case 'd':
	      if (!ARGVAL()) {
		  fprintf( stderr, "%s: -d Missing datafile-name\n", myname );
		  usage();
	      }
	      dfname = *argv;
	      goto nextarg;
	    case 'f':
	      pretty_printing = TRUE;
	      if (line_width == 0) {
		  /*
		    This forces pretty-printing to be done with a
		    restricted line width.  If somebody doesn't like that,
		    they can use "-l999" or something.
		    */
		  if ((s=getenv("COLUMNS")) == NULL)
		      line_width = 79;	/* Take a guess.  Oh well */
		  else
		      line_width = atoi(s) -1;
	      }
	      break;
	    case 'l':
	      if (isdigit(*(*argv+1))) {
		  line_width = atoi(++(*argv));
		  goto nextarg;
	      } else if (--argc && isdigit(*(argv[1]))) {
		  line_width = atoi(*++argv);
		  goto nextarg;
	      } else {
		  argc++;	/* hack-alert!  Fix error from above */
		  /* Set line width to COLUMNS-1.  Avoids auto-newline. */
		  if ((s=getenv("COLUMNS")) == NULL)
		      line_width = 79;	/* Take a guess.  Oh well */
		  else
		      line_width = atoi(s) -1;
	      }
	      break;
	    case 'm':
	      if (!ARGVAL() || !isdigit(**argv)) {
		  fprintf(stderr, "%s: -m Missing memory-limit\n", myname);
		  usage();
	      }
	      mem_limit = atoi(*argv);
	      goto nextarg;
	    case 'p':
	      if (!ARGVAL()) {
		  fprintf( stderr, "%s: -p Missing path-list\n", myname );
		  usage();
	      }
	      dfpath = *argv;
	      goto nextarg;
	    default:
	      fprintf(stderr, "%s: Unknown flag: '%c'\n", myname, **argv);
	      usage();
	  } /* switch */
      } /* while */
    nextarg: continue;
  } /* for */
  
  brl_init( dfname, dfpath, mem_limit );   /* Initialize Bible Retrieval Lib */

  if (argc)
    /* read references from command line */
    while (argc--) printverse( *argv++ );
  else
    /* read references from stdin */
    while(fgets(line, VSPECSZ, stdin) != NULL)
	printverse( line );

  brl_close();
} /* main */
