/*
 * Copyright (c) 1991, 1992 The Regents of the University of California.
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that: (1) source code distributions
 * retain the above copyright notice and this paragraph in its entirety, (2)
 * distributions including binary code include the above copyright notice and
 * this paragraph in its entirety in the documentation or other materials
 * provided with the distribution, and (3) all advertising materials mentioning
 * features or use of this software display the following acknowledgement:
 * ``This product includes software developed by the University of California,
 * Lawrence Berkeley Laboratory and its contributors.'' Neither the name of
 * the University nor the names of its contributors may be used to endorse
 * or promote products derived from this software without specific prior
 * written permission.
 * THIS SOFTWARE IS PROVIDED ``AS IS'' AND WITHOUT ANY EXPRESS OR IMPLIED
 * WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 */
#ifndef lint
char copyright[] =
    "@(#) Copyright (c) 1991, 1992 The Regents of the University of California.\nAll rights reserved.\n";
static  char rcsid[] =
    "@(#)$Header: runexp.c,v 1.8 92/01/26 14:13:23 leres Exp $ (LBL)";
#endif

/*
 * runexp - process explicit expires
 */
#include <stdio.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#ifdef HAVE_TIMEB
#include <sys/timeb.h>
#endif
#ifndef SYSV
#include <strings.h>
#else
#include <string.h>
#define index strchr
#define rindex strrchr
#endif
#include <ctype.h>
#include <errno.h>

#include "dexpire.h"
#include "util.h"

/* Converts 512 byte blocks to Kbytes */
#define BLK2KB(b) ((b) / 2)

/* Private data */
static char *expires = EXPIRES;
static char *nexpires = NEXPIRES;
static char *spool_dir = SPOOL_DIR;

static time_t currenttime;	/* current timestamp */
#ifdef HAVE_TIMEB
static struct timeb currenttimeb;
#else
struct timeb {
	time_t  time;           /* time, seconds since the epoch */
	unsigned short millitm; /* 1000 msec of additional accuracy */
	short   timezone;       /* timezone, minutes west of GMT */
	short   dstflag;        /* daylight savings when appropriate? */
};
static struct timeb currenttimeb;
#endif

/* Public data */
char *prog;

/* Forwards */
static time_t getnumdate();

int debug = 0;			/* debugging modes */
int verbose = 0;		/* verbose information to stdout */
int nflag = 0;			/* non-destructive mode */

/* Statistics */
int bad = 0;
int ndeleted = 0;		/* number of articles deleted */
int emitted = 0;		/* number of lines written to new file */
static long freed_blk = 0;	/* number of 512 bytes blocks freed */

/* External data */

extern char *optarg;
extern int optind, opterr;
extern int getopt();

extern int errno;

void
main(argc, argv)
	int argc;
	char **argv;
{
	register int op;
	register char *cp;
	register FILE *fin, *fout;
	int status;
	char *usage = "usage: %s [-dnv] [expires]\n";

	if (cp = rindex(argv[0], '/'))
		prog = cp + 1;
	else
		prog = argv[0];

	/* Process arguments */
	while ((op = getopt(argc, argv, "dnv")) != EOF)
		switch (op) {

		case 'd':
			++debug;
			break;

		case 'n':
			++nflag;
			break;

		case 'v':
			++verbose;
			break;

		default:
			(void) fprintf(stderr, usage, prog);
			exit(1);
			/* NOTREACHED */
		}

	/* Optional expires file */
	if (optind < argc) {
		expires = argv[optind];
		++optind;
	}

	if (optind != argc) {
		(void) fprintf(stderr, usage, prog);
		exit(1);
		/* NOTREACHED */
	}

	/* Fetch current time (used in various calculations) */
#ifndef SYSV
	ftime(&currenttimeb);
#else
	currenttimeb.time=time(0);
#endif
	currenttime = currenttimeb.time;

	/* Report various times */
	if (verbose) {
		(void) printf("%s: Current time: %s\n",
		    prog, fmtdate(currenttime));
		(void) fflush(stdout);
	}

	/* Open the expires file */
	if ((fin = fopen(expires, "r")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(expires);
		exit(1);
		/* NOTREACHED */
	}

	/* Create the new expires file */
	(void) unlink(nexpires);
	if ((fout = fopen(nexpires, "w")) == 0) {
		(void) fprintf(stderr, "%s: fopen(): ", prog);
		perror(nexpires);
		exit(1);
		/* NOTREACHED */
	}

	/* Process the expires file */
	status = runexp(fin, fout);

	/* Check for errors and close the files */
	if (ferror(fin)) {
		(void) fprintf(stderr, "%s: ferror(): ", prog);
		perror(expires);
		status |= 1;
	}
	if (ferror(fout)) {
		(void) fprintf(stderr, "%s: ferror(): ", prog);
		perror(nexpires);
		status |= 1;
	}

	(void)fclose(fin);
	(void)fclose(fout);

	/* Rename the explicit expires file */
	if (status == 0 && !nflag &&
	    rename(nexpires, expires) < 0 &&
	    unlink(expires) < 0 &&
	    rename(nexpires, expires) < 0) {
		(void)fprintf(stderr, "%s: rename(): %s -> ", prog, nexpires);
		perror(expires);
		status |= 1;
	}

	if (verbose) {
		(void) fflush(stderr);
		(void) printf("%s: %d bad\n", prog, bad);
		(void) printf("%s: %d emitted\n", prog, emitted);

		(void)printf(
		    "%s: %seleted %d article%s (%ld Kbyte%s), took %s\n",
		    prog, nflag ? "Would have d" : "D",
		    ndeleted, PLURAL(ndeleted),
		    BLK2KB(freed_blk),
		    PLURAL(BLK2KB(freed_blk)),
		    fmtdelta(time(0) - currenttime));
	}

	exit(status);
}

int
runexp(fin, fout)
	FILE *fin, *fout;
{
	register char *cp, *cp2;
	register int n;
	register time_t t;
	register char *art;
	char line[1024];
	char stamp[128];
	struct stat sbuf;

	n = 1;

	while (fgets(line, sizeof(line), fin)) {
		/* Step over message id */
		cp = index(line, '\t');
		if (cp == 0) {
			(void) fprintf(stderr,
			    "%s: parse error #1 line %d\n", prog, n);
			++bad;
			goto emit;
		}

		/* Find arrival/expires seperator */
		cp = index(cp, '~');
		if (cp == 0) {
			(void) fprintf(stderr,
			    "%s: parse error #2 line %d\n", prog, n);
			++bad;
			goto emit;
		}

		/* Step over '~' */
		++cp;

		/* Look for explicit expires time and date */
		/* Better have an explicit timestamp */
		if (*cp == '-') {
			(void) fprintf(stderr,
			    "%s: parse error #3 line %d\n", prog, n);
			++bad;
			goto emit;
		}

		/* Find newsgroups while copying timestamp out */
		cp2 = stamp;
		while (*cp != '\t' && *cp != '\0')
			*cp2++ = *cp++;
		*cp2 = '\0';

		/* Better be some newsgroups */
		if (*cp == '\0') {
			(void) fprintf(stderr,
			    "%s: parse error #4 line %d\n", prog, n);
			++bad;
			goto emit;
		}

		/* Step over tab */
		++cp;

		/* Terminate possible INN style date ("arrive~expire~posted") */
		cp2 = index(stamp, '~');
		if (cp2)
			*cp2 = '\0';

		/* Parse expires timestamp; try for numeric date first */
		t = getnumdate(stamp);
#ifdef CNEWS
		/* Try to parse timestamp using C News routine */
		if (t < 0)
			t = getdate(stamp, &currenttimeb);
#endif
#ifdef BNEWS
		/* Try to parse timestamp using B News routine */
		/* XXX how the heck do we do that? */
#endif
		if (t < 0) {
			/* Extract message id */
			cp2 = index(line, '\t');
			if (cp2)
				*cp2 = '\0';
			(void) fprintf(stderr,
			    "%s: Bad date \"%s\" in %s\n", prog, stamp, line);
			++bad;
			/* Throw this one away */
			continue;
		}

#ifdef notdef
		/* Complain if from the first year of Unix history */
		if (t < 365 * 24 * 60 * 60)
			(void)fprintf(stderr, "%s: Warning: art is old: %s\n",
			    prog, fmtdate(t));
#endif

		/* Keep this entry if it hasn't expired yet */
		if (t > currenttime)
			goto emit;

		/* Loop through newsgroup(s) */
		while (*cp != '\0') {
			/* Construct article pathname */
			art = artpath(spool_dir, cp);

			/* Step over this article */
			while (*cp != '\0' && !isspace(*cp))
				++cp;
			while (isspace(*cp))
				++cp;

			if (stat(art, &sbuf) < 0) {
				/* Ok if it's already gone */
				if (errno == ENOENT)
					continue;
				(void)fprintf(stderr, "%s: stat(): ", prog);
				perror(art);
				goto emit;
			}
			if (nflag) {
				if (debug > 1) {
					(void) printf(
					    "%s: wanted to unlink(%s)\n",
					    prog, art);
					(void) fflush(stdout);
				}
			} else if (unlink(art) < 0) {
				    (void)fprintf(stderr, "%s: unlink(): ",
					prog);
				    perror(art);
				    goto emit;
			}
			if (sbuf.st_nlink <= 1) {
				++ndeleted;
				freed_blk += sbuf.st_blocks;
			}
		}

		/* Eat this expires line */
		continue;

emit:
		/* Output current expires line */
		fputs(line, fout);
		n++;
	}
	emitted = n - 1;
	return(0);
}

static time_t
getnumdate(date)
	register char *date;
{
	register int n;
	register long t;
	register char *cp;

	/* Must be all digits and not too long */
	n = 0;
	for (cp = date; isdigit(*cp); ++cp)
		++n;

	if (n > 10 || *cp != '\0')
		return((long)-1);

	t = atol(date);

	/* Not a numeric date if from the first year of Unix history */
	if (t < 365 * 24 * 60 * 60)
		return((long)-1);

	return(t);
}
