// luc.cpp - written and placed in the public domain by Wei Dai

#include "misc.h"
#include "asn.h"
#include "nbtheory.h"

#include "luc.h"

LUCPublicKey::LUCPublicKey(const Integer &n, const Integer &e)
    : n(n), e(e), modulusLen(n.ByteCount())
{
}

LUCPublicKey::LUCPublicKey(BufferedTransformation &bt)
{
    BERSequenceDecoder seq(bt);
    n.BERDecode(seq);
    modulusLen = n.ByteCount();
    e.BERDecode(seq);
}

void LUCPublicKey::DEREncode(BufferedTransformation &bt) const
{
    DERSequenceEncoder seq(bt);
    n.DEREncode(seq);
    e.DEREncode(seq);
}

void LUCPublicKey::RawEncrypt(const Integer &in, Integer &out) const
{
    out = Lucas(e, in, n);
}

// *****************************************************************************
// private key operations:

LUCPrivateKey::LUCPrivateKey(const Integer &nIn, const Integer &eIn,
                             const Integer &pIn, const Integer &qIn, const Integer &uIn)
    : LUCPublicKey(nIn, eIn), p(pIn), q(qIn), u(uIn)
{
    assert(p*q==n);
    assert(u*q%p==1);
}

// generate a random private key
LUCPrivateKey::LUCPrivateKey(RandomNumberGenerator &rng, unsigned int keybits, Integer eStart)
{
	assert(keybits >= 16);
    // generate 2 random primes of suitable size
	if (keybits%2==0)
	{
		const Integer minP = Integer(182) << (keybits/2-8);
		const Integer maxP = Integer::Power2(keybits/2)-1;
		p.Randomize(rng, minP, maxP, Integer::PRIME);
		q.Randomize(rng, minP, maxP, Integer::PRIME);
	}
	else
	{
		const Integer minP = Integer::Power2((keybits-1)/2);
		const Integer maxP = Integer(181) << ((keybits+1)/2-8);
		p.Randomize(rng, minP, maxP, Integer::PRIME);
		q.Randomize(rng, minP, maxP, Integer::PRIME);
	}

    // pre-calculate some other data for faster speed
    const Integer lcm = LCM(LCM(p-1, q-1), LCM(p+1, q+1));
	// make sure e starts odd
    for (e = eStart+(1-eStart%2); GCD(e, lcm)!=1; ++e, ++e);
    u = EuclideanMultiplicativeInverse(q, p);
    n = p * q;
	assert(n.BitCount() == keybits);
    modulusLen = n.ByteCount();
}

LUCPrivateKey::LUCPrivateKey(BufferedTransformation &bt)
{
    BERSequenceDecoder seq(bt);

    Integer version(seq);
    if (!!version)  // make sure version is 0
        BERDecodeError();

    n.BERDecode(seq);
    modulusLen = n.ByteCount();
    e.BERDecode(seq);
    p.BERDecode(seq);
    q.BERDecode(seq);
    u.BERDecode(seq);
}

void LUCPrivateKey::DEREncode(BufferedTransformation &bt) const
{
    DERSequenceEncoder seq(bt);

    const byte version[] = {INTEGER, 1, 0};
    seq.Put(version, sizeof(version));
    n.DEREncode(seq);
    e.DEREncode(seq);
    p.DEREncode(seq);
    q.DEREncode(seq);
    u.DEREncode(seq);
}

void LUCPrivateKey::RawDecrypt(const Integer &in, Integer &out) const
{
    // here we follow the notation of PKCS #1 and let u=q inverse mod p
    // but in InverseLucas, u=p inverse mod q, so we reverse the order of p and q
    out = InverseLucas(e, in, q, p, u);
}
